## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL,
    warning = FALSE
)

## ----"citation"---------------------------------------------------------------
## Citation info
citation("HiCParser")

## ----"start", message=FALSE---------------------------------------------------
library("HiCParser")

## ----coolFormat---------------------------------------------------------------
# Path to each file
paths <- c(
    "path/to/condition-1.replicate-1.cool",
    "path/to/condition-1.replicate-2.cool",
    "path/to/condition-1.replicate-3.cool",
    "path/to/condition-2.replicate-1.cool",
    "path/to/condition-2.replicate-2.cool",
    "path/to/condition-2.replicate-3.cool"
)

# For the sake of the example, we will use the same file, several times
paths <- rep(
    system.file("extdata",
        "hicsample_21.cool",
        package = "HiCParser"
    ),
    6
)

# Condition and replicate of each file. Can be names instead of numbers.
conditions <- c(1, 1, 1, 2, 2, 2)
replicates <- c(1, 2, 3, 1, 2, 3)

# Instantiation of data set
hic.experiment <- parseCool(
    paths,
    conditions = conditions,
    replicates = replicates
)

## ----mcoolFormat--------------------------------------------------------------
# Path to each file
paths <- c(
    "path/to/condition-1.replicate-1.mcool",
    "path/to/condition-1.replicate-2.mcool",
    "path/to/condition-1.replicate-3.mcool",
    "path/to/condition-2.replicate-1.mcool",
    "path/to/condition-2.replicate-2.mcool",
    "path/to/condition-2.replicate-3.mcool"
)

# For the sake of the example, we will use the same file, several times
paths <- rep(
    system.file("extdata",
        "hicsample_21.mcool",
        package = "HiCParser"
    ),
    6
)

# Condition and replicate of each file. Can be names instead of numbers.
conditions <- c(1, 1, 1, 2, 2, 2)
replicates <- c(1, 2, 3, 1, 2, 3)

# mcool files can store several resolutions.
# We will mention the one we need.
binSize <- 5000000

# Instantiation of data set
# The same function "parseCool" is used for cool and mcool files
hic.experiment <- parseCool(
    paths,
    conditions = conditions,
    replicates = replicates,
    binSize = binSize # Specified for .mcool files.
)

## ----hicFormat----------------------------------------------------------------
# Path to each file
paths <- c(
    "path/to/condition-1.replicate-1.hic",
    "path/to/condition-1.replicate-2.hic",
    "path/to/condition-2.replicate-1.hic",
    "path/to/condition-2.replicate-2.hic",
    "path/to/condition-3.replicate-1.hic"
)

# For the sake of the example, we will use the same file, several times
paths <- rep(
    system.file("extdata",
        "hicsample_21.hic",
        package = "HiCParser"
    ),
    6
)

# Condition and replicate of each file. Can be names instead of numbers.
conditions <- c(1, 1, 1, 2, 2, 2)
replicates <- c(1, 2, 3, 1, 2, 3)

# hic files can store several resolutions.
# We will mention the one we need.
binSize <- 5000000

# Instantiation of data set
hic.experiment <- parseHiC(
    paths,
    conditions = conditions,
    replicates = replicates,
    binSize = binSize
)

## ----hicproFormat-------------------------------------------------------------
# Path to each matrix file
matrixPaths <- c(
    "path/to/condition-1.replicate-1.matrix",
    "path/to/condition-1.replicate-2.matrix",
    "path/to/condition-1.replicate-3.matrix",
    "path/to/condition-2.replicate-1.matrix",
    "path/to/condition-2.replicate-2.matrix",
    "path/to/condition-2.replicate-3.matrix"
)

# For the sake of the example, we will use the same file, several times
matrixPaths <- rep(
    system.file("extdata",
        "hicsample_21.matrix",
        package = "HiCParser"
    ),
    6
)

# Path to each bed file
bedPaths <- c(
    "path/to/condition-1.replicate-1.bed",
    "path/to/condition-1.replicate-2.bed",
    "path/to/condition-1.replicate-3.bed",
    "path/to/condition-2.replicate-1.bed",
    "path/to/condition-2.replicate-2.bed",
    "path/to/condition-2.replicate-3.bed"
)

# Alternatively, if the same bed file is used, we can provide it only once
bedPaths <- system.file("extdata",
    "hicsample_21.bed",
    package = "HiCParser"
)

# Condition and replicate of each file. Can be names instead of numbers.
conditions <- c(1, 1, 1, 2, 2, 2)
replicates <- c(1, 2, 3, 1, 2, 3)

# Instantiation of data set
hic.experiment <- parseHiCPro(
    matrixPaths = matrixPaths,
    bedPaths = bedPaths,
    conditions = conditions,
    replicates = replicates
)

## ----tabFormat----------------------------------------------------------------
hic.experiment <- parseTabular(
    system.file("extdata",
        "hicsample_21.tsv",
        package = "HiCParser"
    ),
    sep = "\t"
)

## -----------------------------------------------------------------------------
library("HiCParser")
hicFilePath <- system.file("extdata", "hicsample_21.hic", package = "HiCParser")
hic.experiment <- parseHiC(
    paths = rep(hicFilePath, 6),
    binSize = 5000000,
    conditions = rep(seq(2), each = 3),
    replicates = rep(seq(3), 2)
)
hic.experiment

## -----------------------------------------------------------------------------
SummarizedExperiment::colData(hic.experiment)

## -----------------------------------------------------------------------------
head(SummarizedExperiment::assay(hic.experiment))

## -----------------------------------------------------------------------------
InteractionSet::interactions(hic.experiment)

## -----------------------------------------------------------------------------
path <- system.file("extdata", "hicsample_21.cool", package = "HiCParser")
object1 <- parseCool(path, conditions = 1, replicates = 1)
# Creating an object with a different condition
object2 <- parseCool(path, conditions = 2, replicates = 1)

## -----------------------------------------------------------------------------
objectMerged <- mergeInteractionSet(object1, object2)
SummarizedExperiment::colData(objectMerged)
head(SummarizedExperiment::assay(objectMerged))

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

