/*****************************************************************************
 * Copyright (c) 2023-2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - bug 582853
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue GL-8, GL-9
 *  Pauline Deville (CEA LIST) pauline.deville@cea.fr - Issue GL-10
 *****************************************************************************/
package org.eclipse.papyrus.model2doc.markup.emf.template2structure.mapping;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.commonmark.Extension;
import org.commonmark.ext.gfm.tables.TableBlock;
import org.commonmark.ext.gfm.tables.TableCell;
import org.commonmark.ext.gfm.tables.TableHead;
import org.commonmark.ext.gfm.tables.TablesExtension;
import org.commonmark.node.Node;
import org.commonmark.parser.Parser;
import org.commonmark.renderer.html.AttributeProvider;
import org.commonmark.renderer.html.AttributeProviderContext;
import org.commonmark.renderer.html.AttributeProviderFactory;
import org.commonmark.renderer.html.HtmlRenderer;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.papyrus.model2doc.core.generatorconfiguration.IDocumentStructureGeneratorConfiguration;
import org.eclipse.papyrus.model2doc.core.generatorconfiguration.accessors.IOutputFileAccessor;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.DocumentTemplate;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.IBodySectionPartTemplate;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.utils.DocumentStructureTemplateUtils;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractBodyPartTemplateToStructureMapper;
import org.eclipse.papyrus.model2doc.markup.emf.documentstructuretemplate.IMarkupToFileBodyPartTemplate;

/**
 * Abstract class for mapper converting mardown into html file (it also works for html to html)
 *
 * @param <INPUT>
 *            an implementation of {@link IMarkupToFileBodyPartTemplate}
 */
public abstract class AbstractMarkdownToHtmlFileMapper<INPUT extends IMarkupToFileBodyPartTemplate> extends AbstractBodyPartTemplateToStructureMapper<INPUT> {

	/**
	 * the extension of the output file
	 */
	protected static final String FILE_EXTENSION = "html"; //$NON-NLS-1$

	/**
	 * the string identifying the markdown format
	 */
	private static final String MARKDOWN_FORMAT = "markdown"; //$NON-NLS-1$

	/**
	 * another string to identify the markdown format
	 */
	private static final String MD_FORMAT = "md"; //$NON-NLS-1$

	/**
	 * the string identifying the html format
	 */
	private static final String HTML_FORMAT = "html";//$NON-NLS-1$

	/**
	 * empty stirng
	 */
	protected static final String EMPTY_STRING = ""; //$NON-NLS-1$

	/**
	 * undescrore string
	 */
	protected static final String UNDESCORE = "_"; //$NON-NLS-1$

	/**
	 * the list of the supported output format for the current mapper
	 */
	private static final Collection<String> SUPPORTED_INPUT_FORMAT = new ArrayList<>();

	static {
		SUPPORTED_INPUT_FORMAT.add(HTML_FORMAT);
		SUPPORTED_INPUT_FORMAT.add(MARKDOWN_FORMAT);
		SUPPORTED_INPUT_FORMAT.add(MD_FORMAT);
	}

	/**
	 * Constructor.
	 *
	 * @param inputEClass
	 * @param outputClass
	 */
	public <T> AbstractMarkdownToHtmlFileMapper(EClass inputEClass, Class<T> outputClass) {
		super(inputEClass, outputClass);
	}

	/**
	 * @see org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper#doHandlesInput(org.eclipse.emf.ecore.EObject)
	 *
	 * @param eobject
	 * @return
	 */
	@Override
	protected boolean doHandlesInput(final INPUT eobject) {
		final String inputFormat = eobject.getInputFormat() != null ? eobject.getInputFormat().toLowerCase() : EMPTY_STRING;
		final String outputFormat = eobject.getOutputFomat() != null ? eobject.getOutputFomat().toLowerCase() : EMPTY_STRING;
		return HTML_FORMAT.equals(outputFormat) && SUPPORTED_INPUT_FORMAT.contains(inputFormat);
	}


	/**
	 * Create the html file with the htmlEquation as unique paragraph
	 *
	 * @param bodyPartTemplate
	 *            the bodyPartTemplate
	 * @param markupText
	 *            the text to convert into an html file
	 * @param fileName
	 *            the file name
	 * @param logger
	 *            the logger used to log the messages during the process
	 * @return the uri of the newly created file (platform:/resource)
	 */
	protected String createHTMLFile(final IBodySectionPartTemplate bodyPartTemplate, final String markupText, final String fileName, final IModel2DocLogger logger) {
		// Get file path (saved in the same folder than images)
		final DocumentTemplate documentTemplate = DocumentStructureTemplateUtils.getDocumentTemplate(bodyPartTemplate);
		final IDocumentStructureGeneratorConfiguration configuration = documentTemplate.getDocumentStructureGeneratorConfiguration();
		final IOutputFileAccessor accessor = configuration.createImageOutputAccessor();
		final URI uri = accessor.createOutputFileURI(fileName, FILE_EXTENSION);
		final String filePath = uri.toString();

		// we check all folders tree already exists, and we create them if not
		// then write the file
		if (uri.segmentCount() > 1) {
			URI folderURI = uri.trimSegments(1);
			IPath folderPath = new org.eclipse.core.runtime.Path(folderURI.toPlatformString(true));
			IFolder folder = ResourcesPlugin.getWorkspace().getRoot().getFolder(folderPath);

			createFolders(folder, logger);

			try {
				final String realFileName = uri.lastSegment();
				IFile file = folder.getFile(realFileName);
				if (file.exists()) {
					file.delete(true, new NullProgressMonitor());
				}

				String result = stringToHTML(markupText);

				StringBuilder fileContent = new StringBuilder();
				fileContent.append("<!DOCTYPE html>\n"); //$NON-NLS-1$
				fileContent.append("<html>\n"); //$NON-NLS-1$
				fileContent.append("<body>\n"); //$NON-NLS-1$

				// ensure the good rendering of the accentuated char
				fileContent.append("<meta charset=\"UTF-8\">"); //$NON-NLS-1$

				// Here we should not add <p> otherwise the used style is not the default one
				fileContent.append(result);
				fileContent.append("\n"); //$NON-NLS-1$

				fileContent.append("</body>\n"); //$NON-NLS-1$
				fileContent.append("</html>\n"); //$NON-NLS-1$

				file.create(new ByteArrayInputStream(fileContent.toString().getBytes()), false, new NullProgressMonitor());
			} catch (CoreException e) {
				logger.error(e);
			}
		}
		return filePath;
	}

	/**
	 * Converts all markdown in a String into HTML
	 *
	 * @param content
	 *            a markdown String to convert
	 * @return a String with all markdown converted to HTML
	 */
	private String stringToHTML(String content) {
		List<Extension> extensions = Arrays.asList(TablesExtension.create());
		final Parser parser = Parser.builder().extensions(extensions).build();
		final HtmlRenderer renderer = HtmlRenderer.builder()
				.escapeHtml(true)
				.extensions(extensions)
				.attributeProviderFactory(new TableStyeAttributeProviderFactory())
				.build();

		Node document = parser.parse(content);
		if (document == null) {
			document = parser.parse("Couldn't load file's content."); //$NON-NLS-1$
		}

		return renderer.render(document);
	}


	/**
	 * If the folder does not exist we create it and its parents if necessary
	 *
	 * @param folder
	 *            the folder to create
	 * @param logger
	 *            the logger used to log the messages during the process
	 */
	private static void createFolders(IContainer folder, IModel2DocLogger logger) {
		if (false == folder.exists()) {
			try {
				if (false == folder.getParent().exists()) {
					createFolders(folder.getParent(), logger);
				}
				if (folder instanceof IFolder) {
					((IFolder) folder).create(true, true, new NullProgressMonitor());
				} else if (folder instanceof IProject) {
					((IProject) folder).create(new NullProgressMonitor());
				}
			} catch (CoreException e) {
				logger.error(e);
			}
		}
	}


	/**
	 *
	 * Returns the XMI ID of the given {@link EObject} or <code>null</code> if it cannot be resolved.
	 *
	 * @param object
	 *            Object which we seek the XMI ID of.
	 * @return <code>object</code>'s XMI ID, <code>null</code> if not applicable.
	 */
	protected static final String getXMIID(final EObject object) {
		String objectID = null;
		if (object != null && object.eResource() instanceof XMIResource) {
			objectID = ((XMIResource) object.eResource()).getID(object);
		}
		return objectID;
	}

	private static class TableStyeAttributeProviderFactory implements AttributeProviderFactory {

		@Override
		public AttributeProvider create(AttributeProviderContext context) {
			return new AttributeProvider() {
				@Override
				public void setAttributes(Node node, String tagName, Map<String, String> attributes) {
					if (node instanceof TableBlock) {
						attributes.put("border", "1"); //$NON-NLS-1$ //$NON-NLS-2$
						attributes.put("style", "border-collapse: collapse;"); //$NON-NLS-1$ //$NON-NLS-2$
					} else if (node instanceof TableCell) {
						attributes.put("style", "padding: 4px"); //$NON-NLS-1$ //$NON-NLS-2$
					} else if (node instanceof TableHead) {
						attributes.put("style", "background:#D9D9D9"); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			};
		}

	}
}
