/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.olingo2;

import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import java.util.Properties;

import org.apache.camel.CamelContext;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.apache.camel.test.junit5.TestSupport;
import org.junit.jupiter.api.Assumptions;

/**
 * Abstract base class for Olingo Integration tests generated by Camel API component maven plugin.
 */
public class AbstractOlingo2TestSupport extends CamelTestSupport {

    private static final String TEST_OPTIONS_PROPERTIES = "/test-options.properties";
    private Properties properties = new Properties();

    protected static String getWebResource() {
        final URL resource = AbstractOlingo2TestSupport.class.getResource("/olingo2_ref");

        Assumptions.assumeTrue(resource != null, "The web app cannot be found");
        try {
            return resource.toURI().getPath();
        } catch (URISyntaxException e) {
            Assumptions.abort("Unable to convert the resource to URI");
        }

        return null;
    }

    protected void setDefaultTestProperty(String key, String value) {
        properties.setProperty(key, value);
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {

        final CamelContext context = super.createCamelContext();

        // read Olingo component configuration from TEST_OPTIONS_PROPERTIES
        TestSupport.loadExternalProperties(properties, getClass(), TEST_OPTIONS_PROPERTIES);

        final Olingo2Configuration configuration = new Olingo2Configuration();
        configuration.setServiceUri(properties.getProperty("serviceUri"));
        configuration.setContentType(properties.getProperty("contentType"));

        // add OlingoComponent to Camel context
        final Olingo2Component component = new Olingo2Component(context);
        component.init();
        component.setConfiguration(configuration);
        context.addComponent("olingo2", component);

        return context;
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBodyAndHeaders(String endpointUri, Object body, Map<String, Object> headers)
            throws CamelExecutionException {
        return (T) template().requestBodyAndHeaders(endpointUri, body, headers);
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBody(String endpoint, Object body) throws CamelExecutionException {
        return (T) template().requestBody(endpoint, body);
    }
}
