/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.sql.engine.exec.rel;

import static org.apache.ignite.internal.util.CollectionUtils.nullOrEmpty;

import java.util.ArrayList;
import java.util.List;
import org.apache.ignite.internal.lang.IgniteStringBuilder;
import org.apache.ignite.internal.sql.engine.exec.ExecutionContext;

/**
 * Table spool node.
 */
public class TableSpoolNode<RowT> extends AbstractNode<RowT> implements SingleNode<RowT>, Downstream<RowT> {
    /** How many rows are requested by downstream. */
    private int requested;

    /** How many rows are we waiting for from the upstream. {@code -1} means end of source stream. */
    private int waiting;

    /** Index of the current row to push. */
    private int rowIdx;

    /** Rows buffer. */
    private final List<RowT> rows;

    /**
     * If {@code true} this spool should emit rows as soon as it stored. If {@code false} the spool have to collect all
     * rows from underlying input.
     */
    private final boolean lazyRead;

    /**
     * Flag indicates that spool pushes row to downstream. Need to check a case when a downstream produces requests on
     * push.
     */
    private boolean inLoop;

    /**
     * Constructor.
     * TODO Documentation https://issues.apache.org/jira/browse/IGNITE-15859
     *
     * @param ctx Execution context.
     * @param lazyRead Lazy read flag.
     */
    public TableSpoolNode(ExecutionContext<RowT> ctx, boolean lazyRead) {
        super(ctx);

        this.lazyRead = lazyRead;

        rows = new ArrayList<>();
    }

    /** {@inheritDoc} */
    @Override
    protected void rewindInternal() {
        requested = 0;
        rowIdx = 0;
    }

    /** {@inheritDoc} */
    @Override
    public void rewind() {
        rewindInternal();
    }

    /** {@inheritDoc} */
    @Override
    protected Downstream<RowT> requestDownstream(int idx) {
        if (idx != 0) {
            throw new IndexOutOfBoundsException();
        }

        return this;
    }

    /** {@inheritDoc} */
    @Override
    public void request(int rowsCnt) throws Exception {
        assert !nullOrEmpty(sources()) && sources().size() == 1;
        assert rowsCnt > 0;

        requested += rowsCnt;

        if ((waiting == NOT_WAITING || rowIdx < rows.size()) && !inLoop) {
            this.execute(this::doPush);
        } else if (waiting == 0) {
            source().request(waiting = inBufSize);
        }
    }

    private void doPush() throws Exception {
        if (!lazyRead && waiting != NOT_WAITING) {
            return;
        }

        int processed = 0;
        inLoop = true;
        try {
            while (requested > 0 && rowIdx < rows.size()) {
                if (processed++ >= inBufSize) {
                    // Allow others to do their job
                    this.execute(this::doPush);

                    return;
                }

                downstream().push(rows.get(rowIdx));

                rowIdx++;
                requested--;
            }
        } finally {
            inLoop = false;
        }

        if (rowIdx >= rows.size() && waiting == NOT_WAITING && requested > 0) {
            requested = 0;
            downstream().end();
        }
    }

    /** {@inheritDoc} */
    @Override
    public void push(RowT row) throws Exception {
        assert downstream() != null;
        assert waiting > 0;

        waiting--;

        rows.add(row);

        if (waiting == 0) {
            source().request(waiting = inBufSize);
        }

        if (requested > 0 && rowIdx < rows.size()) {
            doPush();
        }
    }

    /** {@inheritDoc} */
    @Override
    public void end() throws Exception {
        assert downstream() != null;
        assert waiting > 0;

        waiting = NOT_WAITING;

        this.execute(this::doPush);
    }

    @Override
    protected void dumpDebugInfo0(IgniteStringBuilder buf) {
        buf.app("class=").app(getClass().getSimpleName())
                .app(", requested=").app(requested)
                .app(", waiting=").app(waiting);
    }
}
