/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.nifi.processors.slack.util;

import org.apache.nifi.logging.ComponentLog;

import java.time.Duration;
import java.util.Date;
import java.util.concurrent.atomic.AtomicLong;

public class RateLimit {
    private final AtomicLong nextRequestTime = new AtomicLong(0L);
    private final ComponentLog logger;

    public RateLimit(final ComponentLog logger) {
        this.logger = logger;
    }

    public void retryAfter(final Duration duration) {
        final long timeOfNextRequest = System.currentTimeMillis() + duration.toMillis();
        nextRequestTime.getAndUpdate(currentTime -> Math.max(currentTime, timeOfNextRequest));
    }

    public boolean isLimitReached() {
        final long nextTime = nextRequestTime.get();
        if (nextTime > 0 && System.currentTimeMillis() < nextTime) {
            logger.debug("Will not interact with Slack until {} due to Slack's Rate Limit", new Date(nextTime));
            return true;
        } else if (nextTime > 0) {
            // Set nextRequestTime to 0 so that we no longer bother to make system calls to System.currentTimeMillis()
            nextRequestTime.compareAndSet(nextTime, 0);
        }

        return false;
    }

}
