﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/ExportTaskS3Location.h>
#include <aws/ec2/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes an export image task.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportImageTask">AWS
 * API Reference</a></p>
 */
class ExportImageTask {
 public:
  AWS_EC2_API ExportImageTask() = default;
  AWS_EC2_API ExportImageTask(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API ExportImageTask& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>A description of the image being exported.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ExportImageTask& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the export image task.</p>
   */
  inline const Aws::String& GetExportImageTaskId() const { return m_exportImageTaskId; }
  inline bool ExportImageTaskIdHasBeenSet() const { return m_exportImageTaskIdHasBeenSet; }
  template <typename ExportImageTaskIdT = Aws::String>
  void SetExportImageTaskId(ExportImageTaskIdT&& value) {
    m_exportImageTaskIdHasBeenSet = true;
    m_exportImageTaskId = std::forward<ExportImageTaskIdT>(value);
  }
  template <typename ExportImageTaskIdT = Aws::String>
  ExportImageTask& WithExportImageTaskId(ExportImageTaskIdT&& value) {
    SetExportImageTaskId(std::forward<ExportImageTaskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the image.</p>
   */
  inline const Aws::String& GetImageId() const { return m_imageId; }
  inline bool ImageIdHasBeenSet() const { return m_imageIdHasBeenSet; }
  template <typename ImageIdT = Aws::String>
  void SetImageId(ImageIdT&& value) {
    m_imageIdHasBeenSet = true;
    m_imageId = std::forward<ImageIdT>(value);
  }
  template <typename ImageIdT = Aws::String>
  ExportImageTask& WithImageId(ImageIdT&& value) {
    SetImageId(std::forward<ImageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The percent complete of the export image task.</p>
   */
  inline const Aws::String& GetProgress() const { return m_progress; }
  inline bool ProgressHasBeenSet() const { return m_progressHasBeenSet; }
  template <typename ProgressT = Aws::String>
  void SetProgress(ProgressT&& value) {
    m_progressHasBeenSet = true;
    m_progress = std::forward<ProgressT>(value);
  }
  template <typename ProgressT = Aws::String>
  ExportImageTask& WithProgress(ProgressT&& value) {
    SetProgress(std::forward<ProgressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the destination Amazon S3 bucket.</p>
   */
  inline const ExportTaskS3Location& GetS3ExportLocation() const { return m_s3ExportLocation; }
  inline bool S3ExportLocationHasBeenSet() const { return m_s3ExportLocationHasBeenSet; }
  template <typename S3ExportLocationT = ExportTaskS3Location>
  void SetS3ExportLocation(S3ExportLocationT&& value) {
    m_s3ExportLocationHasBeenSet = true;
    m_s3ExportLocation = std::forward<S3ExportLocationT>(value);
  }
  template <typename S3ExportLocationT = ExportTaskS3Location>
  ExportImageTask& WithS3ExportLocation(S3ExportLocationT&& value) {
    SetS3ExportLocation(std::forward<S3ExportLocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the export image task. The possible values are
   * <code>active</code>, <code>completed</code>, <code>deleting</code>, and
   * <code>deleted</code>.</p>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  ExportImageTask& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status message for the export image task.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  inline bool StatusMessageHasBeenSet() const { return m_statusMessageHasBeenSet; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  ExportImageTask& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any tags assigned to the export image task.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  ExportImageTask& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  ExportImageTask& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_description;

  Aws::String m_exportImageTaskId;

  Aws::String m_imageId;

  Aws::String m_progress;

  ExportTaskS3Location m_s3ExportLocation;

  Aws::String m_status;

  Aws::String m_statusMessage;

  Aws::Vector<Tag> m_tags;
  bool m_descriptionHasBeenSet = false;
  bool m_exportImageTaskIdHasBeenSet = false;
  bool m_imageIdHasBeenSet = false;
  bool m_progressHasBeenSet = false;
  bool m_s3ExportLocationHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_statusMessageHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
