#
# Copyright (c) 2021 Contributors to the Eclipse Foundation
#
# This program and the accompanying materials are made
# available under the terms of the Eclipse Public License 2.0
# which is available at https://www.eclipse.org/legal/epl-2.0/
#
# SPDX-License-Identifier: EPL-2.0
#

import threading, random, asyncio, queue, ctypes, datetime, subprocess, os, io, json, traceback, sys, re, time
from typing import Dict, Callable, List, Optional, Union, Type
from snakes.nets import PetriNet


class WalkRecorder():
    def __init__(self) -> None:
        self.logger = None
        self.saveInFile = ""
        self.rerunFromFile = ""
        self.file = None
        self.extend = False
        self.continu = False
        self.minimize = False
        self.minimizer : TraceMinimizer = None

    def log(self, logger):
        self.logger = logger

    def saveAs(self, file: str):
        if file != None:
            self.saveInFile = file
            self.file = open(file, "w")
    
    def stop(self):
        self.saveInFile = ""
        self.rerunFromFile = ""
        if self.file != None and not self.file.closed:
            self.file.flush()
            self.file.close()
            self.file = None

    def record(self, step: WalkerStep):
        if self.saveInFile != "":
            self.file.write(f"{str(step.event)}\n")

    def playFrom(self, file: str, extend: bool, continu: bool, minimize: bool):
        if file != None:
            self.extend = extend
            self.continu = continu
            self.minimize = minimize
            self.rerunFromFile = file
            self.file = open(file, "r")

    def rerun(self, steps: List[WalkerStep]) -> List[WalkerStep]:
        line = ""
        if self.rerunFromFile == "":
            return steps
        else:
            line = self.file.readline().strip()
            newLine = line
        rStep = []
        for step in steps:
            parts = line.split(' ')
            if len(parts) > 2:
                step.breakStep = True
                newLine = parts[0] + ' ' + parts[1].strip()
            if str(step.event) == newLine:
                rStep.append(step)
        if rStep == [] and line != "":
            self.logger("Cannot select step in {}. Is the SUT non-deterministic?".format(line))
            rStep = None
        if rStep != None and rStep == [] and line == "":
            self.logger("All recorded lines have been executed successfully")
            if self.extend:
                self.logger("Continue testing and append to recorded file")
                self.saveInFile = self.rerunFromFile
                self.rerunFromFile = ""
                self.file = open(self.saveInFile, "a")
                rStep = steps
            else:
                if self.continu:
                    self.logger("Continue testing")
                    self.rerunFromFile = ""
                    rStep = steps
                else:
                    if self.minimize and self.minimizer != None:
                        self.minimizer.closeFiles
                        self.minimizer = None
                    rStep = None
        return rStep

class TestMove():
    def __init__(self, source:str, target:str, clause:str) -> None:
        self.source_state = source
        self.target_state = target
        self.clause = clause
        self.joker_move = True # by default a joker move
        self.event = None

class JokerStrategyMultiInterface():
    def __init__(self, walker: Walker) -> None:
        self.walker = walker
        self.goal_clause = None
        self.moves = {}
        self.unseen_clauses = {}
        walker.joker_strategy = self
        for key in nets.keys():
            self.moves[key] = test_moves(nets[key]())
            self.unseen_clauses[key] = [self.place_to_clause_key(p, key) for p in nets[key]().place() if p.meta['type'] == 'clause' and 'sourceline' in p.meta]

    def step_taken(self, step: WalkerStep):
        if self.goal_clause != None and self.expected_clauses != None and step.event.kind != EventType.Command:
            if step.event.port == self.port and step.event.component == self.component:
                if not step.clause in self.expected_clauses:
                    self.goal_clause = None # abandon the goal
                    connection = (self.port, self.component)
                    if len(self.unseen_clauses[connection]) > 0:
                        elem = self.unseen_clauses[connection][0]
                        self.unseen_clauses[connection].remove(elem)
                        self.unseen_clauses[connection].append(elem)
        if step.event != None and step.event.kind != EventType.Command:
            net = nets[step.event.connectionKey()]()
            clause_string = step.event.interface + "." + step.event.port + "." + step.event.component + "." + step.clause + "." + net.place(step.clause).meta['sourceline']
            if clause_string in self.unseen_clauses[step.event.connectionKey()]:
                self.unseen_clauses[step.event.connectionKey()].remove(clause_string)
    
    def next_step(self, steps: list[WalkerStep], walker: Walker) -> WalkerStep:
        # check if there is a running goal
        if self.goal_clause is None:
            (connection, steps_on_connection) = self.select_connection(steps)
            self.goal_connection = connection
            return self.select_new_step(connection, steps_on_connection)
        else:
            # there is a running goal
            # check if the goal is achieved
            if not self.goal_full_clause in self.unseen_clauses[self.goal_connection]:
                # goal achieved
                self.goal_clause = None
                (connection, steps_on_connection) = self.select_connection(steps)
                self.goal_connection = connection
                return self.select_new_step(connection, steps_on_connection)
            connection_states = self.walker.states[self.goal_connection]
            current_state = "S_" + connection_states[list(connection_states.keys())[0]]
            # check if goal state is the same as the current state
            if current_state == self.goal_state:
                candidate_steps = []
                for step in steps:
                    if self.goal_move.event != None and self.goal_move.event.method == step.event.method:
                        candidate_steps.append(step)
                if len(candidate_steps) > 0:
                    result = random.choice(candidate_steps)
                    self.expected_clauses = [self.goal_move.clause]
                    return result
                else:
                    self.goal_clause = None
                    (connection, steps_on_connection) = self.select_connection(steps)
                    self.goal_connection = connection
                    return self.select_new_step(connection, steps_on_connection)
            else:
                # consult strategy
                outgoing_moves = {m.event.method for m in self.current_strategy if m.source_state == current_state and m.event != None}
                candidate_steps = []
                for step in steps:
                    for m in outgoing_moves:
                        if step.event.method == m and self.port == step.event.port and self.component == step.event.component:
                            candidate_steps.append(step)
                if len(candidate_steps) > 0:
                    result = random.choice(candidate_steps)
                    self.expected_clauses = [m.clause for m in self.current_strategy if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                    return result
                else:
                    self.goal_clause = None
                    (connection, steps_on_connection) = self.select_connection(steps)
                    self.goal_connection = connection
                    return self.select_new_step(connection, steps_on_connection)

    def select_new_step(self, connection, steps: list[WalkerStep]) -> WalkerStep:
        if len(self.unseen_clauses[connection]) == 0: # all visited
            # pick up a random step
            return random.choice(steps)
        else: # there are uncovered clauses
            i = 0
            while i < len(self.unseen_clauses[connection]):
                clause = self.unseen_clauses[connection][0]
                self.goal_full_clause = clause
                fragments = self.goal_full_clause.split(".", 3)
                self.port = fragments[1]
                self.component = fragments[2]
                last_dot = fragments[3].rfind(".")
                self.goal_clause = (fragments[3])[:last_dot]
                self.goal_move = [m for m in self.moves[connection] if m.clause == self.goal_clause][0]
                self.goal_state = self.goal_move.source_state
                # get current state, for now work for a single machine
                connection_states = self.walker.states[connection]
                current_state = "S_" + connection_states[list(connection_states.keys())[0]]
                # check if goal state is the same as the current state
                if current_state == self.goal_state:
                    candidate_steps = []
                    for step in steps:
                        if (self.goal_move.event != None and self.goal_move.event.method == step.event.method) or (self.goal_move.event == None and self.goal_move.clause == step.clause):
                            candidate_steps.append(step)
                    if len(candidate_steps) > 0:
                        result = random.choice(candidate_steps)
                        self.expected_clauses = [self.goal_move.clause]
                        return result
                else:
                    # calculate strategy
                    self.current_strategy = calculate_strategy(current_state, self.goal_state, self.moves[connection])
                    outgoing_moves = {m.event.method for m in self.current_strategy if m.source_state == current_state and m.event != None}
                    candidate_steps = []
                    for step in steps:
                        for m in outgoing_moves:
                            if step.event.method == m and self.port == step.event.port and self.component == step.event.component:
                                candidate_steps.append(step)
                    if len(candidate_steps) > 0:
                        result = random.choice(candidate_steps)
                        self.expected_clauses = [m.clause for m in self.current_strategy if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                        return result
                i = i + 1
                elem = self.unseen_clauses[connection][0]
                self.unseen_clauses[connection].remove(elem)
                self.unseen_clauses[connection].append(elem)
            return random.choice(steps)

    def select_connection(self, steps: list[WalkerStep]):
        while True:
            # find a connection on which a step can be taken
            # alternative of random choice of connection is to select a clause among all clauses
            # thus making selection chance proportional to the number of clauses per connection
            connection = random.choice(list(nets.keys()))
            steps_on_connection = [s for s in steps if s.event.port == connection[0] and s.event.component == connection[1]]
            if len(steps_on_connection) > 0:
                return (connection, steps_on_connection)

    def place_to_clause_key(self, place: Place, connection: str) -> str:
        return f"{place.meta['interface']}.{connection[0]}.{connection[1]}.{place.name}.{place.meta['sourceline']}"

class JokerStrategyScenarios(JokerStrategyMultiInterface):
    
    class StrategyState(enum.Enum):
        Idle = 0
        PreparingScenario = 1
        RunningScenario = 2
        JokerMove = 3

    def __init__(self, walker: Walker) -> None:
        JokerStrategyMultiInterface.__init__(self, walker)
        self.scenario_lengths = {}
        self.unexecuted_scenarios = []
        self.executed_scenarios = []
        self.unexecuted_scenarios_connection = {}
        self.start_states_scenario = {}
        self.current_state = self.StrategyState.Idle
        self.index_next_event = 0
        for connection in nets.keys():
            self.unexecuted_scenarios_connection[connection] = []
            net = nets[connection]()
            for t in net.transition():
                if t.meta['type'] == 'event':
                    if t.meta['event'].scenario is None:
                        continue
                    if not t.meta['event'].scenario.startswith("0_"):
                        continue
                    v = [arc[0] for arc in t.input() if arc[0].meta['type'] == 'state']
                    if len(v) != 1:
                        continue
                    fragments = t.meta['event'].scenario.split("_", 1)
                    scenario_name = fragments[1]
                    if not scenario_name in self.unexecuted_scenarios:
                        self.unexecuted_scenarios.append(scenario_name)
                        self.unexecuted_scenarios_connection[connection].append(scenario_name)
                        self.scenario_lengths[scenario_name] = len(set([t.meta['event'].scenario for t in net.transition() if t.meta['type'] == 'event' and t.meta['event'].scenario != None and t.meta['event'].scenario.endswith(scenario_name)]))
                        self.start_states_scenario[scenario_name] = v[0].name

    def initiate_scenario(self, connection, steps, walker):
        for scenario in self.unexecuted_scenarios_connection[connection]:
            self.start_state_scenario = self.start_states_scenario[scenario]
            self.scenario_connection = connection
            connection_states = self.walker.states[connection]
            current_state = "S_" + connection_states[list(connection_states.keys())[0]]
            if current_state == self.start_state_scenario:
                self.current_state = self.StrategyState.PreparingScenario
                self.running_scenario_name = scenario
                return self.next_step(steps, walker)
            self.current_strategy_scenario = calculate_strategy(current_state, self.start_state_scenario, self.moves[connection])
            outgoing_moves = {m.event.method for m in self.current_strategy_scenario if m.source_state == current_state and m.event != None}
            candidate_steps = []
            for step in steps:
                for m in outgoing_moves:
                    if step.event.method == m and connection[0] == step.event.port and connection[1] == step.event.component:
                        candidate_steps.append(step)
            if len(candidate_steps) > 0:
                result = random.choice(candidate_steps)
                self.current_state = self.StrategyState.PreparingScenario
                self.running_scenario_name = scenario
                self.expected_clauses = [m.clause for m in self.current_strategy_scenario if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                return result
        # no scenario can be initiated with the given steps on connection
        return random.choice(steps)
    
    def step_taken(self, step: WalkerStep):
        JokerStrategyMultiInterface.step_taken(self, step)
        if self.current_state != self.StrategyState.PreparingScenario or step.event.kind == EventType.Command:
            return
        if step.event.port == self.scenario_connection[0] and step.event.component == self.scenario_connection[1]:
            if not step.clause in self.expected_clauses:
                # abandon scenario
                self.current_state = self.StrategyState.Idle
                # move the scenario to the end
                self.unexecuted_scenarios_connection[self.scenario_connection].remove(self.running_scenario_name)
                self.unexecuted_scenarios_connection[self.scenario_connection].append(self.running_scenario_name)

    def next_step(self, steps: list[WalkerStep], walker: Walker) -> WalkerStep:
        if self.current_state == self.StrategyState.Idle:
            if len(self.unexecuted_scenarios) > 0:
                (connection, steps_on_connection) = self.select_connection(steps)
                if len(self.unexecuted_scenarios_connection[connection]) > 0:
                    return self.initiate_scenario(connection, steps_on_connection, walker)
                else:
                    # we return a random step
                    # alternative is to execute a joker move on this connection
                    return random.choice(steps_on_connection)
            else:
                self.current_state = self.StrategyState.JokerMove
                return JokerStrategyMultiInterface.next_step(self, steps, walker)
        elif self.current_state == self.StrategyState.PreparingScenario:
            connection_states = self.walker.states[self.scenario_connection]
            current_state = "S_" + connection_states[list(connection_states.keys())[0]]
            if current_state == self.start_state_scenario: # start state of scenario reached
                # check if we can start the scenario
                for step in steps:
                    if step.event.port == self.scenario_connection[0] and step.event.component == self.scenario_connection[1] and step.event.scenario != None:
                        fragments = step.event.scenario.split("_", 1)
                        scenario_name = fragments[1]
                        if fragments[0] == "0":
                            # start event of a scenario
                            self.index_next_event = 1
                            if self.index_next_event == self.scenario_lengths[scenario_name]: # scenario of just a single step
                                self.current_state = self.StrategyState.Idle
                                self.unexecuted_scenarios.remove(scenario_name)
                                self.executed_scenarios.append(scenario_name)
                                self.unexecuted_scenarios_connection[self.scenario_connection].remove(scenario_name)
                            else:
                                self.current_state = self.StrategyState.RunningScenario
                                self.running_scenario_name = scenario_name
                            return step
                # we cannot start the scenario
                self.current_state = self.StrategyState.Idle
                # move the scenario to the end
                self.unexecuted_scenarios_connection[self.scenario_connection].remove(self.running_scenario_name)
                self.unexecuted_scenarios_connection[self.scenario_connection].append(self.running_scenario_name)
                # we return a random step instead
                return random.choice(steps)
                # return self.next_step(steps, walker)
            else: # target start state of scenario not reached yet
                # consult strategy
                outgoing_moves = {m.event.method for m in self.current_strategy_scenario if m.source_state == current_state and m.event != None}
                candidate_steps = []
                for step in steps:
                    for m in outgoing_moves:
                        if step.event.method == m and self.scenario_connection[0] == step.event.port and self.scenario_connection[1] == step.event.component:
                            candidate_steps.append(step)
                if len(candidate_steps) > 0:
                    result = random.choice(candidate_steps)
                    self.expected_clauses = [m.clause for m in self.current_strategy_scenario if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                    return result
                else:
                    # cannot reach the start state of scenario -> abandon scenario
                    self.current_state = self.StrategyState.Idle
                    # move the scenario to the end
                    self.unexecuted_scenarios_connection[self.scenario_connection].remove(self.running_scenario_name)
                    self.unexecuted_scenarios_connection[self.scenario_connection].append(self.running_scenario_name)
                    return self.next_step(steps, walker)
        elif self.current_state == self.StrategyState.RunningScenario:
            for step in steps:
                if step.event.scenario is not None:
                    fragments = step.event.scenario.split("_", 1)
                    scenario_name = fragments[1]
                    if scenario_name == self.running_scenario_name:
                        if int(fragments[0]) == self.index_next_event:
                            self.index_next_event = self.index_next_event + 1
                            if self.index_next_event == self.scenario_lengths[scenario_name]:
                                # end of scenario
                                self.current_state = self.StrategyState.Idle
                                self.unexecuted_scenarios.remove(scenario_name)
                                self.executed_scenarios.append(scenario_name)
                                self.unexecuted_scenarios_connection[self.scenario_connection].remove(scenario_name)
                            return step
            # the next step is not possible -> abandon scenario
            # we assume that once scenario has started, steps are in sequence without interleavings
            self.current_state = self.StrategyState.Idle
            # move the scenario to the end
            self.unexecuted_scenarios_connection[self.scenario_connection].remove(self.running_scenario_name)
            self.unexecuted_scenarios_connection[self.scenario_connection].append(self.running_scenario_name)
            return self.next_step(steps, walker)
        else:
            return JokerStrategyMultiInterface.next_step(self, steps, walker)

class JokerStrategy():
    def __init__(self, walker: Walker) -> None:
        self.walker = walker
        self.moves = {}
        for key in nets.keys():
            self.moves[key] = test_moves(nets[key]())
        self.seen_clauses = []
        self.unseen_clauses = list(self.walker.all_clauses)
        self.goal_clause = None
        self.goal_state = None
        self.goal_full_clause = None
        self.port = None
        self.component = None
        self.current_strategy = None
        self.expected_clauses = None
        walker.joker_strategy = self

    def step_taken(self, step: WalkerStep):
        if self.goal_clause != None and self.expected_clauses != None and step.event.kind != EventType.Command:
            if not (step.clause in self.expected_clauses and step.event.port == self.port and step.event.component == self.component):
                self.goal_clause = None # abandon the goal
                elem = self.unseen_clauses[0]
                self.unseen_clauses.remove(elem)
                self.unseen_clauses.append(elem)
        if step.event != None and step.event.kind != EventType.Command:
            net = nets[step.event.connectionKey()]()
            clause_string = step.event.interface + "." + step.event.port + "." + step.event.component + "." + step.clause + "." + net.place(step.clause).meta['sourceline']
            if clause_string in self.unseen_clauses:
                self.unseen_clauses.remove(clause_string)

    def select_new_step(self, steps: list[WalkerStep]) -> WalkerStep:
        if len(self.unseen_clauses) == 0: # all visited
            # pick up a random step
            return random.choice(steps)
        else: # there are uncovered clauses
            # for clause in self.unseen_clauses:
            i = 0
            while i < len(self.unseen_clauses):
                clause = self.unseen_clauses[0]
                self.goal_full_clause = clause
                fragments = self.goal_full_clause.split(".", 3)
                self.port = fragments[1]
                self.component = fragments[2]
                last_dot = fragments[3].rfind(".")
                self.goal_clause = (fragments[3])[:last_dot]
                self.goal_move = [m for m in self.moves[(self.port, self.component)] if m.clause == self.goal_clause][0]
                self.goal_state = self.goal_move.source_state
                # get current state, first work for a single machine
                connection_states = self.walker.states[(self.port, self.component)]
                current_state = "S_" + connection_states[list(connection_states.keys())[0]]
                # check if goal state is the same as the current state
                if current_state == self.goal_state:
                    candidate_steps = []
                    for step in steps:
                        if (self.goal_move.event != None and self.goal_move.event.method == step.event.method) or (self.goal_move.event == None and self.goal_move.clause == step.clause):
                            candidate_steps.append(step)
                    if len(candidate_steps) > 0:
                        result = random.choice(candidate_steps)
                        self.expected_clauses = [self.goal_move.clause]
                        return result
                else:
                    # calculate strategy
                    self.current_strategy = calculate_strategy(current_state, self.goal_state, self.moves[(self.port, self.component)])
                    outgoing_moves = {m.event.method for m in self.current_strategy if m.source_state == current_state and m.event != None}
                    candidate_steps = []
                    for step in steps:
                        for m in outgoing_moves:
                            if step.event.method == m and self.port == step.event.port and self.component == step.event.component:
                                candidate_steps.append(step)
                    if len(candidate_steps) > 0:
                        result = random.choice(candidate_steps)
                        self.expected_clauses = [m.clause for m in self.current_strategy if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                        return result
                i = i + 1
                elem = self.unseen_clauses[0]
                self.unseen_clauses.remove(elem)
                self.unseen_clauses.append(elem)
            return random.choice(steps)

    def next_step(self, steps: list[WalkerStep], walker: Walker) -> WalkerStep:
        if self.goal_clause == None:
            return self.select_new_step(steps)        
        else:
            # there is a goal
            # check if the goal is achieved
            if not self.goal_full_clause in self.unseen_clauses:
                # goal achieved
                self.goal_clause = None
                return self.select_new_step(steps)
            connection_states = self.walker.states[(self.port, self.component)]
            current_state = "S_" + connection_states[list(connection_states.keys())[0]]
            # check if goal state is the same as the current state
            if current_state == self.goal_state:
                candidate_steps = []
                for step in steps:
                    if self.goal_move.event != None and self.goal_move.event.method == step.event.method:
                        candidate_steps.append(step)
                if len(candidate_steps) > 0:
                    result = random.choice(candidate_steps)
                    self.expected_clauses = [self.goal_move.clause]
                    return result
                else:
                    self.goal_clause = None
                    return self.select_new_step(steps)
            else:
                # consult strategy
                outgoing_moves = {m.event.method for m in self.current_strategy if m.source_state == current_state and m.event != None}
                candidate_steps = []
                for step in steps:
                    for m in outgoing_moves:
                        if step.event.method == m and self.port == step.event.port and self.component == step.event.component:
                            candidate_steps.append(step)
                if len(candidate_steps) > 0:
                    result = random.choice(candidate_steps)
                    self.expected_clauses = [m.clause for m in self.current_strategy if m.source_state == current_state and m.event != None and m.event.method == result.event.method]
                    return result
                else:
                    self.goal_clause = None
                    return self.select_new_step(steps)
                
def state_places(net:PetriNet):
    return [place for place in net.place() if place.meta['type'] == 'state']

def test_moves_for_state(state: str, net: PetriNet):
    result = []
    transition_states = [[s for s in net.post(t) if net.place(s).meta['type'] == 'transition'][0] for t in net.post(state)]
    non_triggered_transitions = [t for t in net.post(state) if net.transition(t).meta['type'] == 'none']
    for s in transition_states:
        clauses = []
        for first_transition in net.post(s):
            event = None
            root_transition = net.transition(list(net.pre(s))[0])
            if root_transition.meta['type'] == 'event':
                event = root_transition.meta['event']
            place_found = False
            clause_set = False
            place: str
            clause: str
            t = first_transition
            while not place_found:
                place = [p for p in net.post(t) if net.place(p).meta['type'] == 'state' or net.place(p).meta['type'] == 'clause'][0]
                if net.place(place).meta['type'] == 'state':
                    place_found = True
                else:
                    t = list(net.post(place))[0]
                if net.place(place).meta['type'] == 'clause' and not clause_set:
                    clause_set = True
                    clause = place
            test_move = TestMove(state, place, clause)
            test_move.event = event
            clauses.append(test_move)
            result.append(test_move)

        if len(non_triggered_transitions) == 0:
            if len(clauses) == 1: # single clause
                clauses[0].joker_move = False
            else: # multiple clauses: check if all go to the same state
                same_state = True
                target_state = clauses[0].target_state
                for move in clauses:
                    if move.target_state != target_state:
                        same_state = False
                        break
                if same_state:
                    for move in clauses:
                        move.joker_move = False
        elif len(non_triggered_transitions) == 1 and len(non_triggered_transitions) == len(transition_states):
            if len(clauses) == 1: # single clause
                test_move.joker_move = False
            else: # multiple clauses: check if all go to the same state
                same_state = True
                target_state = clauses[0].target_state
                for move in clauses:
                    if move.target_state != target_state:
                        same_state = False
                        break
                if same_state:
                    for move in clauses:
                        move.joker_move = False
    return result

def test_moves(net: PetriNet):
    result = []
    for place in state_places(net):
        for m in test_moves_for_state(place.name, net):
            result.append(m)
    return result

def calculate_strategy(source_state: str, target_state: str, test_moves: List[TestMove]):
    # the result is a set of test moves that can be used starting from source_state
    moves = set()
    visited_states = {target_state}
    new_states = {target_state}

    # first find the attractors
    goal_reached = False
    while new_states != set():
        while new_states != set():
            predecessor_states = set()
            for state in new_states:
                for move in test_moves:
                    if move.target_state == state and not move.joker_move and move.source_state not in visited_states:
                        moves.add(move)
                        if move.source_state != source_state:
                            predecessor_states.add(move.source_state)
                        else:
                            goal_reached = True
            new_states = predecessor_states
            visited_states = visited_states.union(predecessor_states)
        # find the one-step joker moves
        new_states = set()
        for state in visited_states:
            for move in test_moves:
                if move.joker_move and move.target_state == state and move.source_state not in visited_states:
                    moves.add(move)
                    if move.source_state != source_state:
                        new_states.add(move.source_state)
                    else:
                        goal_reached = True
        visited_states = visited_states.union(new_states)
    if goal_reached:
        return moves
    else:
        return None # source state never reached from target

class ScenarioRandom():
    def __init__(self) -> None:
        # we assume:
        # possibly multiple scenarios per interface
        # unique scenario names across interfaces
        # single connection per port
        
        self.scenario_lengths = dict()
        self.scenario_executed = dict()
        self.executed_scenarios = []
        self.scenario_running = False
        self.index_next_event = 0

    def next_step(self, steps: list[WalkerStep], walker: Walker) -> WalkerStep:
        if self.scenario_running:
            for step in steps:
                if step.event.scenario is not None:
                    fragments = step.event.scenario.split("_", 1)
                    scenario_name = fragments[1]
                    if scenario_name == self.running_scenario_name:
                        if int(fragments[0]) == self.index_next_event:
                            self.index_next_event = self.index_next_event + 1
                            if self.index_next_event == self.scenario_lengths[scenario_name]:
                                # end of scenario
                                self.scenario_running = False
                                self.scenario_executed[scenario_name] = True
                                self.executed_scenarios.append(scenario_name)
                            return step
            # the next step is not possible -> abandon scenario
            # we assume that once scenario has started, steps are in sequence without interleavings
            self.scenario_running = False
        else:
            for step in steps:
                if step.event.scenario is not None:
                    fragments = step.event.scenario.split("_", 1)
                    scenario_name = fragments[1]
                    if not scenario_name in self.scenario_lengths:
                        self.scenario_executed[scenario_name] = False
                        self.scenario_lengths[scenario_name] = len(set([t.meta['event'].scenario for t in nets[step.event.connectionKey()]().transition() if t.meta['type'] == 'event' and t.meta['event'].scenario != None and t.meta['event'].scenario.endswith(scenario_name)]))
                    if fragments[0] == "0" and not self.scenario_executed[scenario_name]: # scenario can be executed only once, otherwise a cycle can happen
                        # start event of a scenario
                        self.index_next_event = 1
                        if self.index_next_event == self.scenario_lengths[scenario_name]:
                            self.scenario_executed[scenario_name] = True
                            self.executed_scenarios.append(scenario_name)
                        else:
                            self.scenario_running = True
                            self.running_scenario_name = scenario_name
                        return step
        return random.choice(steps)

class TestStrategy():
    def __init__(self, strategy: str, recorder: Callable[['WalkRecorder'], None], log: Callable[[str], None], debugger: Callable[['Debugger'], None], strategy_implementation) -> None:
        self.strategy = strategy
        self.recorder = recorder
        self.log = log
        self.debugger = debugger
        self.taken_transitions = []
        self.strategy_implementation = strategy_implementation

    def next_step(self, walker: Dict[str, Callable[[], Walker]], take_reply_to_cmd: Optional[Event]):
        if self.strategy == "Random":
            return self.next_step_orig(walker, take_reply_to_cmd)
        elif self.strategy == "Prioritize non-selected":
            return self.next_step_improved(walker, take_reply_to_cmd)
        elif self.strategy == "Joker" or self.strategy == "Scenario Random" or self.strategy == "Scenario Joker":
            return self.next_step_others(walker, take_reply_to_cmd)
        else:
            assert False

    def next_step_orig(self, walker, take_reply_to_cmd):
        steps = []
        if take_reply_to_cmd != None:
            steps = walker.next_steps((take_reply_to_cmd.port, take_reply_to_cmd.component))
            steps = [c for c in steps if c.event.port == take_reply_to_cmd.port and c.event.component == take_reply_to_cmd.component
                and c.event.kind == EventType.Reply and c.event.method == take_reply_to_cmd.method]
        else:
            connections = list(walker.nets.keys())
            for connection in connections:
                steps.extend(walker.next_steps(connection))
        step = None
        if len(steps) != 0:
            steps = self.recorder.rerun(steps)
            if steps == None:
                return None
            steps = self.debugger.debug_next_step(steps)
            step = random.choice(steps)
            assert step.event != None
        return step
    
    def next_step_improved(self, walker, take_reply_to_cmd):
        steps = []
        if take_reply_to_cmd != None:
            steps = walker.next_steps((take_reply_to_cmd.port, take_reply_to_cmd.component))
            steps = [c for c in steps if c.event.port == take_reply_to_cmd.port and c.event.component == take_reply_to_cmd.component
                and c.event.kind == EventType.Reply and c.event.method == take_reply_to_cmd.method]
        else:
            connections = list(walker.nets.keys())
            for connection in connections:
                steps.extend(walker.next_steps(connection))
        step = None
        if len(steps) != 0:
            steps = self.recorder.rerun(steps)
            if steps == None:
                return None
            rSteps = []
            for step in steps: # If possible, choose a step that has not been taken before
                net = walker.nets[step.event.connectionKey()]
                try:
                    if "_join" in step.clause:
                        step.clause = step.clause[0: (step.clause.rfind("_join"))]
                    if "_split" in step.clause:
                        step.clause = step.clause[0: (step.clause.rfind("_split"))]
                    if not step.clause.endswith("_0"):
                        step.clause = step.clause[0: (step.clause.rfind("_")+1)] + "0"
                    clause_str = step.event.interface + "." + step.event.port + "." + step.clause + "." + net._place[step.clause].meta['sourceline']
                except Exception as e:
                    # self.log(f"step: '{step}', clause: '{net._place[step.clause]}'")
                    # self.log(f"clause.meta: '{net._place[step.clause].meta['sourceline']}'")
                    raise Exception("New exception")
                if clause_str not in walker.seen_clauses:
                    rSteps.append(step)
            if rSteps == []:
                rSteps = steps
            rSteps = self.debugger.debug_next_step(rSteps)
            step = random.choice(rSteps)
        return step

    def next_step_others(self, walker, take_reply_to_cmd):
        steps = []
        if take_reply_to_cmd != None:
            steps = walker.next_steps((take_reply_to_cmd.port, take_reply_to_cmd.component))
            steps = [c for c in steps if c.event.port == take_reply_to_cmd.port and c.event.component == take_reply_to_cmd.component
                and c.event.kind == EventType.Reply and c.event.method == take_reply_to_cmd.method]
        else:
            connections = list(walker.nets.keys())
            for connection in connections:
                steps.extend(walker.next_steps(connection))
        step = None
        if len(steps) != 0:
            steps = self.recorder.rerun(steps)
            if steps == None:
                return None
            steps = self.debugger.debug_next_step(steps)
            step = self.strategy_implementation.next_step(steps, walker)
            assert step.event != None
        return step

class Debugger():
    def __init__(self):
        self.choice_func = None
        self.lock = threading.Lock()
        self.enable_lock = threading.Lock()
        self.wait = False
        self.enabled = False
        self.take_step = -1

    def debug_register_choice_func(self, func):
        self.choice_func = func

    def debug_set(self, state: bool):
        self.enable_lock.acquire()
        self.enabled = state
        self.enable_lock.release()

    def debug_next_step(self, steps: List[Type['Event']]):
        self.lock.acquire()
        for step in steps:
            self.enable_lock.acquire()
            enabled = self.enabled
            self.enable_lock.release()
            if self.enabled and step.event.has_breakpoint():
                self.wait = True
            if self.enabled and step.breakStep:
                self.wait = True
        has_to_wait = self.wait
        if has_to_wait and self.choice_func != None:
             self.choice_func(steps)
        self.lock.release()
        while has_to_wait:
            time.sleep(0.1)
            self.lock.acquire()
            has_to_wait = self.wait
            self.lock.release()
        self.lock.acquire()
        if self.take_step != -1:
            self.wait = True
            tmpSteps = []
            tmpSteps.extend(steps)
            steps = []
            steps.append(tmpSteps[self.take_step])
        else:
            self.wait = False
        self.lock.release()
        return steps

    def debug_pause(self):
        self.lock.acquire()
        self.wait = True
        self.lock.release()

    def debug_continue(self):
        self.lock.acquire()
        self.wait = False
        self.take_step = -1
        self.lock.release()       

    def debug_step(self, index: str):
        self.lock.acquire()
        self.wait = False
        self.take_step = int(index)
        self.lock.release()

    def debug_timeout(self, timeout: int):
        # self.lock.acquire()
        # if self.wait or self.take_step != -1:
        #     timeout = None # Infinite timeout during debugging
        # self.lock.release()
        return timeout


class TestApplicationWalker():
    def __init__(self, nets: Dict[str, Callable[[], PetriNet]], constraints: List[Type['Constraint']], send_event: Callable[['Event'], None], 
                 stopped: Callable[[Optional[str]], None], strategy: str, log: Callable[[str], None], recorder: Callable[['WalkRecorder'], None], 
                 debugger: Callable[[Optional['Debugger']], None]) -> None:
        self.send_event = send_event
        self.stopped = stopped
        self.walker = Walker(nets, constraints, log)
        self.event_queue: queue.Queue[Union[Event, None]] = queue.Queue()
        self.thread: Optional[threading.Thread] = None
        self.stop_requested = False
        self.recorder = recorder
        if strategy == "Joker":
            self.strategy_implementation = JokerStrategyMultiInterface(self.walker)
        elif strategy == "Scenario Random":
            self.strategy_implementation = ScenarioRandom()
        elif strategy == "Scenario Joker":
            self.strategy_implementation = JokerStrategyScenarios(self.walker)
        else:
            self.strategy_implementation = None
        self.test_strategy = TestStrategy(strategy, self.recorder, log, debugger, self.strategy_implementation)
        self.debugger = debugger

    def start(self):
        assert self.thread == None, "Already running"
        self.thread = threading.Thread(target=self.__run_non_async)
        self.thread.start()
    
    def stop(self):
        if self.thread != None:
            self.stop_requested = True
            self.event_queue.put(None) # Force run to stop
            if threading.current_thread() != self.thread: self.thread.join()
            self.stop_requested = False

    def received_event(self, event: 'Event'):
        self.event_queue.put(event)

    def __run_non_async(self):
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        loop.run_until_complete(self.__run())
        loop.close()

    async def __run(self):
        port_notification_during_command_transition: Dict[str, List[Event]] = {}
        take_reply_to_cmd: Optional[Event] = None
        stop_on_no_events = False
        error: Optional[str] = None

        self.walker.log("Initial states:")
        for (connection, states) in self.walker.states.items():
            for (machine, state) in states.items():
                self.walker.log(f"Connection ({connection[1]}, {connection[0]}) , machine '{machine}' is in state '{state}'")

        try:
            while not self.stop_requested and error == None:
                event: Optional[Event] = None
                try:
                    timeout = NO_EVENTS_TIMEOUT if stop_on_no_events else DEFAULT_TIMEOUT
                    timeout = self.debugger.debug_timeout(timeout)
                    if self.event_queue.qsize() == 0:
                        event = self.event_queue.get(True, timeout)
                    else:
                        event = self.event_queue.get()
                    if event == None: continue # None means we have to stop (added in stop())
                except queue.Empty:
                    self.walker.inc_events_timeout_counter()

                if event != None:
                    stop_on_no_events = False
                    parameter_place_name = f"P_{event.method}{'_reply' if event.kind == EventType.Reply else ''}"
                    # self.walker.log(f"Process event: '{str(event)}'")
                    if not event.connectionKey() in self.walker.nets:
                        error = f"Received event '{str(event)}' from unknown port '{event.port}'"
                        continue
                    if not parameter_place_name in self.walker.nets[event.connectionKey()]._place:
                        error = f"Event '{event.method}' is unknown for port '{event.port}'"
                        continue
                    place = self.walker.nets[event.connectionKey()]._place[parameter_place_name]
                    # self.walker.log(f"Process place: '{str(place)}'")
                    place.add([Parameters([p.value for p in event.parameters])])
                    # self.walker.log(f"Process parameters: '{str(place.meta)}'")
                    steps = [e for e in self.walker.next_steps(event.connectionKey()) if e.event == event]
                    if len(steps) == 0:
                        if event.kind == EventType.Notification and event.port in port_notification_during_command_transition:
                            port_notification_during_command_transition[event.port].append(event)
                        else:
                            error = f"Event '{str(event)}' is not possible"
                    else:
                        step = random.choice(steps)
                        # self.walker.log(f"Process step: '{str(step)}'")
                        self.walker.take_step(step)
                        if event.kind == EventType.Reply:
                            for notification in port_notification_during_command_transition[event.port]:
                                steps = [e for e in self.walker.next_steps(event.connectionKey()) if e.event == notification]
                                if len(steps) == 0:
                                    error = f"Event '{str(notification)}' is not possible"
                                    break
                                else:
                                    self.walker.take_step(random.choice(steps))
                            del port_notification_during_command_transition[event.port]  
                        elif event.kind == EventType.Command:
                            take_reply_to_cmd = event
                else:
                    step = self.test_strategy.next_step(self.walker, take_reply_to_cmd)
                    # self.walker.log(f"Process step: '{str(step)}'")
                    if step == None:
                        if stop_on_no_events:
                            error = "No next steps possible from test application"
                        else:
                            stop_on_no_events = True
                    else:
                        if step.event.kind == EventType.Command:
                            port_notification_during_command_transition[step.event.port] = []
                        self.send_event(step.event)
                        self.recorder.record(step)
                        take_reply_to_cmd = None
                        self.walker.take_step(step)
        except Exception as e:
            error = f"Error while running: {repr(e)}, event: {str(event)}, place: {str(place)}"
            traceback.print_exc()

        if not self.stop_requested:
            self.stopped(error)
            