/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.ast.NamingHelper;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import java.util.Comparator;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationType;
import org.eclipse.xtext.common.types.JvmAnnotationValue;
import org.eclipse.xtext.common.types.JvmCustomAnnotationValue;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeAnnotationValue;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.XAbstractFeatureCall;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A collection of utilities around language {@link Aspect}s,
 * and the associated {@link JvmTypeReference}s.
 */
@SuppressWarnings("all")
public class AspectExtensions {
  @Inject
  @Extension
  private IQualifiedNameConverter _iQualifiedNameConverter;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;
  
  @Inject
  @Extension
  private NamingHelper _namingHelper;
  
  public final static String ASPECT_ANNOTATION_FQN = "fr.inria.diverse.k3.al.annotationprocessor.Aspect";
  
  public final static String ASPECT_ANNOTATION_PARAMETER = "className";
  
  /**
   * Checks whether the given {@link Aspect} {@code asp} can be resolved
   * and processed as a JvmDeclaredType
   */
  public boolean isValid(final Aspect asp) {
    boolean _and = false;
    JvmTypeReference _aspectTypeRef = asp.getAspectTypeRef();
    JvmType _type = null;
    if (_aspectTypeRef!=null) {
      _type=_aspectTypeRef.getType();
    }
    boolean _tripleNotEquals = (_type != null);
    if (!_tripleNotEquals) {
      _and = false;
    } else {
      JvmType _type_1 = asp.getAspectTypeRef().getType();
      _and = (_type_1 instanceof JvmDeclaredType);
    }
    return _and;
  }
  
  /**
   * Checks whether the given aspect {@code asp} has an @Aspect annotation
   */
  public boolean hasAspectAnnotation(final Aspect asp) {
    boolean _xifexpression = false;
    boolean _isValid = this.isValid(asp);
    if (_isValid) {
      _xifexpression = this.hasAspectAnnotation(this.asJvmType(asp));
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Checks whether the given JVM type {@code aspType} has an @Aspect annotation
   */
  public boolean hasAspectAnnotation(final JvmDeclaredType aspType) {
    final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final JvmAnnotationReference it) {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectExtensions.ASPECT_ANNOTATION_FQN));
      }
    };
    return IterableExtensions.<JvmAnnotationReference>exists(aspType.getAnnotations(), _function);
  }
  
  /**
   * Returns the fully qualified name of the class on which the aspect
   * pointed by the given {@link JvmTypeReference} {@code typeRef} is woven
   * (ie. the value of its 'className=' annotation parameter), or null
   * if it cannot be retrieved
   */
  public String getAspectAnnotationValue(final JvmTypeReference typeRef) {
    boolean _or = false;
    JvmType _type = null;
    if (typeRef!=null) {
      _type=typeRef.getType();
    }
    boolean _tripleEquals = (_type == null);
    if (_tripleEquals) {
      _or = true;
    } else {
      JvmType _type_1 = typeRef.getType();
      boolean _not = (!(_type_1 instanceof JvmDeclaredType));
      _or = _not;
    }
    if (_or) {
      return null;
    }
    JvmType _type_2 = typeRef.getType();
    QualifiedName _extractAspectAnnotationValue = this.extractAspectAnnotationValue(((JvmDeclaredType) _type_2));
    String _string = null;
    if (_extractAspectAnnotationValue!=null) {
      _string=_extractAspectAnnotationValue.toString();
    }
    return _string;
  }
  
  /**
   * Returns the simple name of the class on which the aspect
   * pointed by the given {@link JvmTypeReference} {@code typeRef} is woven
   * (ie. the value of its 'className=' annotation parameter), or null
   * if it cannot be retrieved
   */
  public String getSimpleAspectAnnotationValue(final JvmTypeReference typeRef) {
    boolean _or = false;
    JvmType _type = null;
    if (typeRef!=null) {
      _type=typeRef.getType();
    }
    boolean _tripleEquals = (_type == null);
    if (_tripleEquals) {
      _or = true;
    } else {
      JvmType _type_1 = typeRef.getType();
      boolean _not = (!(_type_1 instanceof JvmDeclaredType));
      _or = _not;
    }
    if (_or) {
      return null;
    }
    JvmType _type_2 = typeRef.getType();
    QualifiedName _extractAspectAnnotationValue = this.extractAspectAnnotationValue(((JvmDeclaredType) _type_2));
    String _lastSegment = null;
    if (_extractAspectAnnotationValue!=null) {
      _lastSegment=_extractAspectAnnotationValue.getLastSegment();
    }
    String _string = null;
    if (_lastSegment!=null) {
      _string=_lastSegment.toString();
    }
    return _string;
  }
  
  /**
   * Returns the fully qualified name of the class on which the aspect
   * {@code asp} is woven (ie. the value of its 'className=' annotation
   * parameter), or null if it cannot be retrieved
   */
  public String getTargetedClassFqn(final Aspect asp) {
    String _xifexpression = null;
    boolean _isValid = this.isValid(asp);
    if (_isValid) {
      _xifexpression = this.extractAspectAnnotationValue(this.asJvmType(asp)).toString();
    }
    return _xifexpression;
  }
  
  /**
   * Checks whether the aspect pointed by the {@code aspectTypeRef} reference
   * is defined over the namespace defined by the {@code mm} {@link Metamodel}
   */
  public boolean isDefinedOver(final JvmTypeReference aspectTypeRef, final Metamodel mm) {
    final Function1<GenPackage, String> _function = new Function1<GenPackage, String>() {
      @Override
      public String apply(final GenPackage it) {
        return AspectExtensions.this._namingHelper.getPackageNamespace(it);
      }
    };
    final Function1<String, Boolean> _function_1 = new Function1<String, Boolean>() {
      @Override
      public Boolean apply(final String ns) {
        String _string = AspectExtensions.this.getTargetedNamespace(aspectTypeRef).toString();
        return Boolean.valueOf(Objects.equal(ns, _string));
      }
    };
    return IterableExtensions.<String>exists(IterableExtensions.<GenPackage, String>map(this._modelingElementExtensions.getAllGenPkgs(mm), _function), _function_1);
  }
  
  /**
   * Checks whether the given {@code aspectTypeRef} aspect reference can be
   * copied for the {@code mm} {@link Metamodel}
   */
  public boolean canBeCopiedFor(final JvmTypeReference aspectTypeRef, final Metamodel mm) {
    return true;
  }
  
  /**
   * Tries to replace the {@link JvmTypeReference} {@code asp} points to with
   * a new {@link JvmTypeReference} pointing to its new qualified name in
   * the runtime project of its containing language.
   */
  public void tryUpdateAspect(final Aspect asp) {
    Language language = asp.getOwningLanguage();
    final JvmTypeReference newRef = this._languageExtensions.getCopiedAspectRefFor(language, asp.getAspectTypeRef().getSimpleName());
    if ((newRef != null)) {
      asp.setAspectTypeRef(newRef);
    }
  }
  
  /**
   * Returns the qualified name of the base package of the class on which
   * the aspect pointed by the {@link JvmTypeReference} {@code aspectTypeRef}
   * reference points, or an empty {@link QualifiedName}
   */
  private QualifiedName getTargetedNamespace(final JvmTypeReference aspectTypeRef) {
    JvmType _type = aspectTypeRef.getType();
    final QualifiedName aavt = this.extractAspectAnnotationValue(((JvmDeclaredType) _type));
    QualifiedName _xifexpression = null;
    if ((aavt != null)) {
      _xifexpression = aavt.skipLast(1);
    } else {
      _xifexpression = QualifiedName.create();
    }
    return _xifexpression;
  }
  
  /**
   * Sorts the {@code aspects} list by overriding priority, ie. if an aspect
   * A potentially overrides an aspect B, it will be earlier in the list.
   */
  public Iterable<Aspect> sortByOverridingPriority(final Iterable<Aspect> aspects) {
    final Comparator<Aspect> _function = new Comparator<Aspect>() {
      @Override
      public int compare(final Aspect aspA, final Aspect aspB) {
        final EClass clsA = aspA.getAspectedClass();
        final EClass clsB = aspB.getAspectedClass();
        boolean _contains = clsA.getEAllSuperTypes().contains(clsB);
        if (_contains) {
          return (-1);
        } else {
          boolean _contains_1 = clsB.getEAllSuperTypes().contains(clsA);
          if (_contains_1) {
            return 1;
          } else {
            return 0;
          }
        }
      }
    };
    return IterableExtensions.<Aspect>sortWith(aspects, _function);
  }
  
  /**
   * Returns the underlying {@link JvmDeclaredType} corresponding to
   * the aspect {@code asp} or null if it cannot be determined
   */
  private JvmDeclaredType asJvmType(final Aspect asp) {
    JvmDeclaredType _xifexpression = null;
    boolean _isValid = this.isValid(asp);
    if (_isValid) {
      JvmType _type = asp.getAspectTypeRef().getType();
      _xifexpression = ((JvmDeclaredType) _type);
    }
    return _xifexpression;
  }
  
  /**
   * Parses the given {@link JvmDeclaredType} {@code t} to extract the value
   * of its {@code className} annotation parameter in the form of a
   * {@link QualifiedName}
   */
  public QualifiedName extractAspectAnnotationValue(final JvmDeclaredType t) {
    final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final JvmAnnotationReference it) {
        JvmAnnotationType _annotation = it.getAnnotation();
        String _qualifiedName = null;
        if (_annotation!=null) {
          _qualifiedName=_annotation.getQualifiedName();
        }
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectExtensions.ASPECT_ANNOTATION_FQN));
      }
    };
    final JvmAnnotationReference aspAnn = IterableExtensions.<JvmAnnotationReference>findFirst(t.getAnnotations(), _function);
    EList<JvmAnnotationValue> _values = null;
    if (aspAnn!=null) {
      _values=aspAnn.getValues();
    }
    JvmAnnotationValue _findFirst = null;
    if (_values!=null) {
      final Function1<JvmAnnotationValue, Boolean> _function_1 = new Function1<JvmAnnotationValue, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationValue it) {
          String _valueName = it.getValueName();
          return Boolean.valueOf(Objects.equal(_valueName, AspectExtensions.ASPECT_ANNOTATION_PARAMETER));
        }
      };
      _findFirst=IterableExtensions.<JvmAnnotationValue>findFirst(_values, _function_1);
    }
    final JvmAnnotationValue aspClassName = _findFirst;
    String _switchResult = null;
    boolean _matched = false;
    if (aspClassName instanceof JvmTypeAnnotationValue) {
      _matched=true;
      EList<JvmTypeReference> _values_1 = ((JvmTypeAnnotationValue)aspClassName).getValues();
      JvmTypeReference _head = null;
      if (_values_1!=null) {
        _head=IterableExtensions.<JvmTypeReference>head(_values_1);
      }
      String _qualifiedName = null;
      if (_head!=null) {
        _qualifiedName=_head.getQualifiedName();
      }
      _switchResult = _qualifiedName;
    }
    if (!_matched) {
      if (aspClassName instanceof JvmCustomAnnotationValue) {
        _matched=true;
        String _xblockexpression = null;
        {
          EList<EObject> _values_1 = ((JvmCustomAnnotationValue)aspClassName).getValues();
          EObject _head = null;
          if (_values_1!=null) {
            _head=IterableExtensions.<EObject>head(_values_1);
          }
          final XAbstractFeatureCall feature = ((XAbstractFeatureCall) _head);
          _xblockexpression = feature.getFeature().getQualifiedName();
        }
        _switchResult = _xblockexpression;
      }
    }
    final String aspVal = _switchResult;
    QualifiedName _qualifiedName = null;
    if (aspVal!=null) {
      _qualifiedName=this._iQualifiedNameConverter.toQualifiedName(aspVal);
    }
    return _qualifiedName;
  }
}
