"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.isSameResource = isSameResource;
exports.isParentDir = isParentDir;
exports.computeRelativePath = computeRelativePath;
exports.looksLikePathToResource = looksLikePathToResource;
exports.looksLikeMarkdownUri = looksLikeMarkdownUri;
exports.looksLikeMarkdownFilePath = looksLikeMarkdownFilePath;
exports.parseLocationInfoFromFragment = parseLocationInfoFromFragment;
const path = require("path");
const vscode_uri_1 = require("vscode-uri");
const schemes_1 = require("./schemes");
function isSameResource(a, b) {
    return a.toString() === b.toString();
}
function isParentDir(parent, maybeChild) {
    if (parent.scheme === maybeChild.scheme && parent.authority === maybeChild.authority) {
        const relative = path.relative(parent.path, maybeChild.path);
        return !relative.startsWith('..');
    }
    return false;
}
function computeRelativePath(fromDoc, toDoc, preferDotSlash = false) {
    if (fromDoc.scheme === toDoc.scheme && fromDoc.scheme !== schemes_1.Schemes.untitled) {
        const rootDir = vscode_uri_1.Utils.dirname(fromDoc);
        let newLink = path.posix.relative(rootDir.path, toDoc.path);
        if (preferDotSlash && !(newLink.startsWith('../') || newLink.startsWith('..\\'))) {
            newLink = './' + newLink;
        }
        return newLink;
    }
    return undefined;
}
/**
 * Check if `path` looks like it points to `target`.
 *
 * Handles cases where `path` doesn't have a file extension but `target` does.
 */
function looksLikePathToResource(configuration, path, target) {
    if (path.fsPath === target.fsPath) {
        return true;
    }
    return configuration.markdownFileExtensions.some(ext => path.with({ path: path.path + '.' + ext }).fsPath === target.fsPath);
}
function looksLikeMarkdownUri(config, resolvedHrefPath) {
    return looksLikeMarkdownExt(config, vscode_uri_1.Utils.extname(resolvedHrefPath));
}
function looksLikeMarkdownFilePath(config, fileName) {
    return looksLikeMarkdownExt(config, path.extname(fileName));
}
function looksLikeMarkdownExt(config, rawExt) {
    return config.markdownFileExtensions.includes(rawExt.toLowerCase().replace('.', ''));
}
/**
 * Extract position info from link fragments that look like `#L5,3`
 */
function parseLocationInfoFromFragment(fragment) {
    const match = fragment.match(/^L(\d+)(?:,(\d+))?$/i);
    if (!match) {
        return undefined;
    }
    const line = +match[1] - 1;
    if (isNaN(line)) {
        return undefined;
    }
    const column = +match[2] - 1;
    return { line, character: isNaN(column) ? 0 : column };
}
//# sourceMappingURL=path.js.map