/* verify.c -  signature verification
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is safe_free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "gpgme.h"

struct verify_result_s {
    struct verify_result_s * next;
    gpgme_sigstat_t status;
    gpgme_data_t notation; /* we store an XML fragment here */
    int collecting;       /* private to finish_sig() */
    int notation_in_data; /* private to add_notation() */
    char fpr[41];    /* fingerprint of a good signature or keyid of a bad one*/
    ulong timestamp; /* signature creation time */
    ulong exptime;   /* signature expiration time */
    unsigned expired:1;
    int trust;    
    int key_algo;
    int md_algo;
    int sig_class;
    char * user_id;
    char * policy_url;
    char * key_server;
    char * file_name;
};


void
_gpgme_release_verify_result(_verify_result_t res)
{
    while (res) {
        _verify_result_t r2 = res->next;
        gpgme_data_release (res->notation);
	safe_free (res->user_id);
	safe_free (res->policy_url );
	safe_free (res->file_name);
	safe_free (res->key_server);
        safe_free (res);
        res = r2;
    }
} /* _gpgme_release_verify_result */


/* 
 * finish a pending signature info collection and prepare for a new
 * signature info collection
 */
static void
finish_sig( gpgme_ctx_t ctx, int stop )
{
    if( stop )
        return; /* nothing to do */

    if( ctx->result.verify->collecting ) {
        _verify_result_t res2;

        ctx->result.verify->collecting = 0;
        /* create a new result structure */
        res2 = calloc ( 1, sizeof *res2 );
        if( !res2 ) {
            ctx->out_of_core = 1;
            return;
        }
        res2->next = ctx->result.verify;
        ctx->result.verify = res2;
    }
    
    ctx->result.verify->collecting = 1;
} /* finish_sig */


static char *
read_val( const char * args, int *ret_i )
{
    static char buf[32];    
    int i = *ret_i, pos = 0;

    for( ; args[i] && args[i] != ' '; i++ )
	buf[pos++] = args[i];
    buf[pos] = '\0';
    *ret_i = ++i;
    return buf;
} /* read_val */


static int
read_int_val( const char * args, int *ret_i )
{
    char * p = read_val( args, ret_i );    
    return atol( p );
} /* read_int_val */


static unsigned
read_unsigned_val( const char * args, int *ret_i )
{
    char * p = read_val( args, ret_i );
    return strtoul( p, NULL, 10 );
} /* read_unsigned_val */


static void
verify_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char * args )
{
    gpgme_data_t in;
    char * p;
    int i, j, rc = 0;

    if( ctx->out_of_core )
        return;
    if( ctx->result_type == RESULT_TYPE_NONE ) {
        assert ( !ctx->result.verify );
        ctx->result.verify = calloc( 1, sizeof *ctx->result.verify );
        if( !ctx->result.verify ) {
            ctx->out_of_core = 1;
            return;
        }
        ctx->result_type = RESULT_TYPE_VERIFY;
    }
    assert( ctx->result_type == RESULT_TYPE_VERIFY );

    if( code == STATUS_GOODSIG 
	|| code == STATUS_REVKEYSIG
	|| code == STATUS_EXPKEYSIG
	|| code == STATUS_BADSIG 
	|| code == STATUS_ERRSIG ) {
        finish_sig( ctx, 0 );
        if( ctx->out_of_core )
            return;
    }

    _gpgme_sigtrust_status_handler (code, args, &ctx->result.verify->trust);

    switch( code ) {
    case STATUS_FILE_START:
	if( *args++ == '1' ) {
	    args++;
	    p = ctx->result.verify->file_name = strdup( args );
	    if( !p ) {
		ctx->out_of_core = 1;
		return;
	    }
	}
	break;

    case STATUS_FILE_DONE:
	break;

    case STATUS_GOODSIG: /* use it to get the user-id */
	ctx->result.verify->status = GPGME_SIG_STAT_GOOD;
	i = 0;
	while( args[i] && args[i] == ' ' ) /* skip keyid */
	    i++;
	while( args[i] && args[i] != ' ' )
	    i++;
	ctx->result.verify->user_id = p = calloc( 1, strlen( args+i ) + 2 );
	if( !p ) {
	    ctx->out_of_core = 1;
	    return;
	}
	j = 0;
	while( args[i] )
	    p[j++] = args[i++];
	p[j++] = '\0';
        break;

    case STATUS_BADSIG:
        ctx->result.verify->status = GPGME_SIG_STAT_BAD;
        p = ctx->result.verify->fpr; /* store the keyID in the fpr field */
        for( i = 0; i < DIM(ctx->result.verify->fpr)
                 && args[i] && args[i] != ' ' ; i++ )
            *p++ = args[i];
        *p = '\0';
	p = ctx->result.verify->user_id = calloc( 1, strlen( args+i ) + 2 );
	if( !p ) {
	    ctx->out_of_core = 1;
	    return;
	}
	j = 0;
	while( args[i] )
	    p[j++] = args[i++];
	p[j++] = '\0';
        break;

    case STATUS_EXPKEYSIG:
    case STATUS_REVKEYSIG:
	if( code == STATUS_EXPKEYSIG )
	    ctx->result.verify->status = GPGME_SIG_STAT_E_GOOD;
	else if( code == STATUS_REVKEYSIG )
	    ctx->result.verify->status = GPGME_SIG_STAT_R_GOOD;	
	for( i=0, p=ctx->result.verify->fpr;
	     args[i] && args[i] != ' '; i++ )
	    *p++ = args[i];
	*p = '\0';
	p = ctx->result.verify->user_id = calloc( 1, strlen( args+i ) + 2 );
	if( !p ) {
	    ctx->out_of_core = 1;
	    return;
	}
	j=0;
	while( args[i] )
	    p[j++] = args[i++];
	p[j++] = '\0';
	break;

    case STATUS_VALIDSIG:
        p = ctx->result.verify->fpr;
        for( i = 0; i < DIM(ctx->result.verify->fpr)
                 && args[i] && args[i] != ' ' ; i++ )
            *p++ = args[i];
        *p = 0;
        /* skip the formatted date */
        while( args[i] && args[i] == ' ')
            i++;
        while( args[i] && args[i] != ' ')
            i++;
        /* and get the timestamp */
        ctx->result.verify->timestamp = strtoul( args+i, NULL, 10 );
	/* and the expire timestamp */
	while( args[i] && args[i] == ' ' )
	    i++;
	while( args[i] && args[i] != ' ' )
	    i++;
	ctx->result.verify->exptime = strtoul( args+i, NULL, 10 );
        break;

    case STATUS_ERRSIG:
        ctx->result.verify->status = GPGME_SIG_STAT_ERROR;
        p = ctx->result.verify->fpr;
        for( i = 0; i < DIM(ctx->result.verify->fpr )
	    && args[i] && args[i] != ' ' ; i++ )
            *p++ = args[i];
	*p = 0;
	i++;
	ctx->result.verify->key_algo = read_int_val( args, &i );
	ctx->result.verify->md_algo = read_int_val( args, &i );
	ctx->result.verify->sig_class = read_int_val( args, &i );
	ctx->result.verify->timestamp = read_unsigned_val( args, &i );
	rc = read_int_val( args, &i );
	if( rc == 9 )
	    ctx->result.verify->status = GPGME_SIG_STAT_NOKEY;
        break;

    case STATUS_EXPSIG:
	ctx->result.verify->expired = 1;
	break;
        
    case STATUS_NO_PUBKEY:
        ctx->result.verify->status = GPGME_SIG_STAT_NOKEY;
        break;

    case STATUS_NOTATION_NAME:	
	if( !ctx->result.verify->notation )
	    gpgme_data_new( &ctx->result.verify->notation );
	in = ctx->result.verify->notation;
	gpgme_data_write( in, args, strlen( args ) );
	gpgme_data_write( in, "=", 1 );
	break;

    case STATUS_NOTATION_DATA:
	in = ctx->result.verify->notation;
	for( i = 0; args[i] && in; i++ ) {
	    if( args[i] == '%' ) {
		gpgme_data_putc( in, ' ' );
		i += 2;
	    }
	    else		
		gpgme_data_putc( in, args[i] );
	}
	break;

    case STATUS_SIG_SUBPACKET:
	i=0;
	read_int_val (args, &i);
	read_int_val (args, &i);
	j = read_int_val (args, &i);
	p = ctx->result.verify->key_server = calloc (1, j+1);
	if (!p) {
	    ctx->out_of_core = 1;
	    break;
	}	
	memcpy (p, args+i, j);
	break;

    case STATUS_POLICY_URL:
	p = ctx->result.verify->policy_url = strdup (args);
	if (!p)
	    ctx->out_of_core = 1;
        break;

    case STATUS_EOF:
        finish_sig( ctx, 1 );
        break;
    }
} /* verify_status_handler */


static const char *
verify_command_handler( void *opaque, gpg_status_code_t code, const char *key )
{
    gpgme_ctx_t ctx = opaque;
    
    if( !code || !key )
        return NULL;

    if( code == STATUS_GET_LINE && !strcmp( key, "detached_signature.filename" ) )
	return ctx->tmp_id;
    
    return NULL;
} /* verify_command_handler */


static long
verify_get_sig_date( gpgme_data_t sig )
{
    gpgme_ctx_t ctx;
    gpgme_data_t out;
    gpgme_error_t rc;
    const char * s;
    char buf[128], tbuf[32];
    size_t nread = 0, i = 0;
    long timestamp = 0;

    rc = gpgme_new( &ctx );
    if( rc )
	return 0;
    gpgme_data_rewind( sig );
    gpgme_data_read( sig, buf, 40, &nread );

    _gpgme_gpg_new( &ctx->gpg );
    _gpgme_gpg_add_arg( ctx->gpg, "--list-packets" );
    gpgme_data_new( &out );    
    _gpgme_data_set_mode( out, GPGME_DATA_MODE_IN );

    _gpgme_gpg_add_arg( ctx->gpg, "--output" );        
    _gpgme_gpg_add_arg( ctx->gpg, "-" );    
    _gpgme_gpg_add_data( ctx->gpg, out, 1 );
    _gpgme_gpg_add_arg( ctx->gpg, "--" );    
    _gpgme_gpg_add_data( ctx->gpg, sig, 0 );
    
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( !rc ) {
	gpgme_wait( ctx, 1 );
	nread = gpgme_data_readline( out, buf, sizeof buf-1 );
	if( !nread )
	    goto end;   
	nread = gpgme_data_readline( out, buf, sizeof buf -1 );
	if( !nread )
	    goto end;
	s = buf;	    	
	while( s && (*s == ' ' || * s == '\t') )
	    s++;
	if( strstr( s, "version" ) && strlen( s ) > 20 ) {
	    s += 19;
	    while( isdigit( *s ) ) {
		tbuf[i++] = *s;
		s++;
	    }
	    tbuf[i++] = '\0';
	    timestamp = atol( tbuf );
	}
    }

end:
    gpgme_data_release( out );
    gpgme_release( ctx );
    return timestamp;
}


static gpgme_error_t
verify_start (gpgme_ctx_t ctx,  gpgme_data_t sig, gpgme_data_t text)
{
    gpgme_error_t rc = 0;

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    _gpgme_release_result( ctx );
    ctx->out_of_core = 0;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new ( &ctx->gpg );
    if( rc )
        goto leave;

    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, verify_status_handler, ctx );

    /* build the commandline */
    _gpgme_gpg_add_arg ( ctx->gpg, "--verify" );
    /* Check the supplied data */
    if( gpgme_data_get_type( sig ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    if( text && gpgme_data_get_type( text ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode( sig, GPGME_DATA_MODE_OUT );
    if (text) /* detached signature */
        _gpgme_data_set_mode (text, GPGME_DATA_MODE_OUT);
    /* Tell the gpg object about the data */
    _gpgme_gpg_add_arg (ctx->gpg, "--");
    _gpgme_gpg_add_data (ctx->gpg, sig, -1);
    if (text) {
	_gpgme_gpg_add_arg (ctx->gpg, "-");
	_gpgme_gpg_add_data (ctx->gpg, text, 0);   
    }

    /* and kick off the process */
    rc = _gpgme_gpg_spawn (ctx->gpg, ctx);

leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg ); 
    }

    return rc;
} /* verify_start */


static gpgme_error_t
file_verify_start( gpgme_ctx_t ctx, gpgme_sigmode_t sigmode ,
		   const char ** sigfile, size_t nfiles, const char * datfile )
{
    gpgme_error_t rc = 0;	    
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc ) {
        _gpgme_gpg_release( &ctx->gpg );
        return rc;
    }
    
    _gpgme_gpg_set_status_handler( ctx->gpg, verify_status_handler, ctx );
    _gpgme_gpg_set_command_handler( ctx->gpg, verify_command_handler, ctx );
    
    _gpgme_gpg_add_arg( ctx->gpg, "--yes" );
    if( ctx->pipemode || nfiles > 1 )
	_gpgme_gpg_add_arg( ctx->gpg, "--verify-files" );
    else if( sigmode != GPGME_SIG_MODE_DETACH )
        _gpgme_gpg_add_arg( ctx->gpg, "--verify" );
    else {
	ctx->tmp_id = strdup( datfile );
	if( !ctx->tmp_id ) {
	    ctx->out_of_core = 1;
	    return mk_error( Out_Of_Core );
	}
    }

    while( nfiles-- )
	_gpgme_gpg_add_arg( ctx->gpg, *sigfile++ );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    
    return rc;
} /* gpgme_file_verify_start */


/* 
 * Figure out a common status value for all signatures 
 */
static gpgme_sigstat_t
intersect_stati( _verify_result_t res )
{
    gpgme_sigstat_t status = res->status;

    for( res = res->next; res; res = res->next ) {
        if( status != res->status )
            return GPGME_SIG_STAT_DIFF;
    }

    return status;
} /* intersect_stati */


static char*
get_sig_userid( gpgme_ctx_t ctx, int idx )
{
    _verify_result_t res;

    if( !ctx )
	return NULL;
    for( res = ctx->result.verify; res && idx>0; res = res->next, idx-- )
	;
    if( !res )
	return NULL;
    return res->user_id? strdup( res->user_id ) : NULL;
} /* get_sig_userid */


static gpgme_error_t
read_sig_results( gpgme_ctx_t ctx, gpgme_sig_t * r_sigctx )
{
    gpgme_sig_t root = NULL, node;
    gpgme_error_t err;
    size_t i, n;

    gpgme_get_sig_ctx( ctx, -1, &n, NULL );
    for( i = 0; i < n; i++ ) {
        err = gpgme_get_sig_ctx( ctx, i, NULL, &node );
	if( !err ) {
	    if( node->sigstat != GPGME_SIG_STAT_NOKEY )	    
		err = gpgme_get_sig_key( ctx, i, &node->key );
	}
        if( !root )
    	    root = node;
	else {
	    gpgme_sig_t s;
	    for( s = root; s->next; s = s->next )
		;
	    s->next = node;   
	}
    }
    *r_sigctx = root;
    return err;
} /* read_sig_results */


/**
 * gpgme_op_verify:
 * @c: the context
 * @sig: the signature data
 * @text: the signed text
 * 
 * Perform a signature check on the signature given in @sig. Currently it is
 * assumed that this is a detached signature for the material given in @text.
 * The result of this operation is returned via gpgme_get_sig_ctx ().
 *
 * Return value: 0 on success or an errorcode if something not related to
 *               the signature itself did go wrong.
 **/
gpgme_error_t
gpgme_op_verify (gpgme_ctx_t ctx, gpgme_data_t sig, gpgme_data_t text)
{
    gpgme_error_t rc;

    gpgme_data_release (ctx->notation);
    ctx->notation = NULL;
    
    rc = verify_start( ctx, sig, text );
    if( !rc ) {
        gpgme_wait( ctx, 1 );
        if( ctx->result_type != RESULT_TYPE_VERIFY )			
            rc = mk_error( General_Error );
        else if ( ctx->out_of_core )
            rc = mk_error( Out_Of_Core );
        else {
            assert ( ctx->result.verify );
            if( ctx->result.verify->notation ) {
                gpgme_data_t dh = ctx->result.verify->notation;
                ctx->notation = dh;
                ctx->result.verify->notation = NULL;
            }
	    if( intersect_stati( ctx->result.verify ) == GPGME_SIG_STAT_BAD )
		ctx->result.verify->timestamp = verify_get_sig_date( sig );
        }
        ctx->pending = 0;
    }

    return rc;
} /* gpgme_op_verify */


gpgme_error_t
gpgme_op_file_verify( gpgme_ctx_t ctx, gpgme_sigmode_t sigmode, 
		      gpgme_sig_t * r_sigctx,
		      const char * sigfile, const char * datfile )
		      
{
    const char * s[1];
    gpgme_error_t rc = 0;    
    
    s[0] = sigfile;
    rc = file_verify_start( ctx, sigmode, s, 1, datfile );
    if( !rc ) {
        gpgme_wait( ctx, 1 );
	ctx->pending = 0;
        if( ctx->result_type != RESULT_TYPE_VERIFY )
            rc = mk_error( General_Error );
        else if( ctx->out_of_core )
            rc = mk_error( Out_Of_Core );
        else {
            assert( ctx->result.verify );
            if( ctx->result.verify->notation ) {
                gpgme_data_t dh = ctx->result.verify->notation;
                ctx->notation = dh;
                ctx->result.verify->notation = NULL;
            }
	    if( intersect_stati( ctx->result.verify ) == GPGME_SIG_STAT_BAD ) {
		gpgme_data_t sig;
		rc = gpgme_data_new_from_file (&sig, sigfile);
		if( !rc ) {
		    ctx->result.verify->timestamp = verify_get_sig_date( sig );
		    gpgme_data_release( sig );
		}
	    }
	    rc = read_sig_results( ctx, r_sigctx );
        }
    }	
    
    return rc;
} /* gpgme_op_file_verify */


gpgme_error_t
gpgme_op_files_verify( gpgme_ctx_t ctx, const char ** files, size_t nfiles,
		       gpgme_sig_t * ret_sig )
{
    gpgme_error_t rc;

    rc = file_verify_start( ctx, 0, files, nfiles, NULL );
    if( !rc ) {
	gpgme_wait( ctx, 1 );
	ctx->pending = 0;
	if( ctx->result_type != RESULT_TYPE_VERIFY )
	    rc = mk_error( General_Error );
	else if( ctx->out_of_core )
	    rc = mk_error( Out_Of_Core );
	else
	    read_sig_results( ctx, ret_sig );
    }
    return rc;
}


/**
 * gpgme_get_sig_ctx:
 * @ctx: Context
 * @idx: Index of the signature starting at 0
 * @r_stat: Returns the status
 * @r_created: Returns the creation timestamp
 * 
 * Return information about an already verified signatures. 
 * 
 * Return value: The fingerprint or NULL in case of an problem or
 *               when there are no more signatures.
 **/
gpgme_error_t
gpgme_get_sig_ctx (gpgme_ctx_t ctx, int idx, size_t * r_ncount, 
		   gpgme_sig_t * r_sig )
{
    _verify_result_t res;
    gpgme_sig_t sig;
    gpgme_error_t err;
    int idx_old = idx;

    if( idx == -1 && !r_sig && r_ncount ) {
	size_t n = 0;
	for( res = ctx->result.verify; res; res = res->next )
	    n++;
	*r_ncount = n;
    }

    if (!r_sig)
	return mk_error (Invalid_Value);
    if (!ctx || ctx->pending || ctx->result_type != RESULT_TYPE_VERIFY)
        return mk_error (General_Error); /* No results yet or verification error */

    for (res = ctx->result.verify; res && idx > 0 ; res = res->next, idx--)
        ;
    if (!res)
        return mk_error (Invalid_Value); /* No more signatures */

    *r_sig = NULL;
    err = gpgme_sig_new (&sig);
    if (err)
	return err;	

    sig->flags.expired = res->expired;
    sig->sigstat = res->status;
    sig->created = res->timestamp;
    sig->expired = res->exptime;
    strcpy( sig->id, res->fpr );
    sig->notation = gpgme_get_notation( ctx );
    sig->trust = res->trust;
    sig->key_algo = res->key_algo;
    sig->md_algo = res->md_algo;
    sig->sigclass = res->sig_class;
    sig->user_id = get_sig_userid( ctx, idx_old );
    sig->policy_url = res->policy_url? strdup (res->policy_url) : NULL;
    sig->file_name = res->file_name? strdup (res->file_name) : NULL;
    sig->key_server = res->key_server? strdup (res->key_server) : NULL;
    *r_sig = sig;
    return 0;
} /* gpgme_get_sig_ctx */


/**
 * gpgme_get_sig_key:
 * @c: context
 * @idx: Index of the signature starting at 0
 * @r_key: Returns the key object
 * 
 * Return a key object which was used to check the signature. 
 * 
 * Return value: An Errorcode or 0 for success. GPGME_EOF is returned to
 *               indicate that there are no more signatures. 
 **/
gpgme_error_t
gpgme_get_sig_key( gpgme_ctx_t ctx, int idx, gpgme_key_t * r_key )
{
    _verify_result_t res;
    gpgme_ctx_t listctx = NULL;
    gpgme_error_t err;

    if( !ctx || !r_key )
        return mk_error( Invalid_Value );
    if( ctx->pending || ctx->result_type != RESULT_TYPE_VERIFY )
        return mk_error( Busy );

    for( res = ctx->result.verify; res && idx>0 ; res = res->next, idx-- )
        ;
    if( !res )
        return mk_error( EOF );

    if( strlen( res->fpr ) < 16 ) /* we have at least an key ID */
        return mk_error( Invalid_Key );
    
    if( ctx->key_lookup )
        err = gpgme_keycache_find_key( ctx->key_lookup, res->fpr, 0, r_key );
    else {
	err = gpgme_new( &listctx );
        if( !err ) {            
	    gpgme_control( listctx, GPGME_CTRL_LISTMODE, listctx->keylist_mode );
	    err = gpgme_op_keylist_start( listctx, res->fpr, 0 );
	}
        if( !err )
            err = gpgme_op_keylist_next( listctx, r_key );
        gpgme_release( listctx );
    }

    return err;
} /* gpgme_get_sig_key */


gpgme_error_t
gpgme_op_clip_verify (gpgme_keycache_t cache, gpgme_sig_t * r_sigctx)
{
    gpgme_error_t err;	
    gpgme_data_t sig = NULL;
    gpgme_ctx_t ctx = NULL;
    
    
    err = gpgme_new (&ctx);
    if (!err) {
	if (cache)
	    gpgme_set_cache_ctx (ctx, cache);
    }
    if (!err)
	err = gpgme_data_new_from_clipboard (&sig);
    if (!err)
	err = gpgme_op_verify (ctx, sig, NULL);
    if (!err)
	err = read_sig_results (ctx, r_sigctx);
    gpgme_data_release (sig);
    gpgme_release (ctx);

    return err;
} /* gpgme_op_clip_verify */


gpgme_error_t
gpgme_op_clip_verify_detached( gpgme_keycache_t cache, gpgme_sig_t * r_sigctx,
			       const char * data, size_t len )
{
    gpgme_error_t err;
    gpgme_data_t sig = NULL;
    gpgme_data_t text = NULL;
    gpgme_ctx_t ctx = NULL;
    
    err = gpgme_new( &ctx );
    if( !err )        
	err = gpgme_data_new_from_mem( &text, data, len, 1 );
    if( !err )        
    if( !err && cache )
        gpgme_set_cache_ctx( ctx, cache );
    if( !err )
	err = gpgme_data_new_from_clipboard (&sig);
    if( !err )        
	err = gpgme_op_verify( ctx, sig, text );
    if( !err )
	err = read_sig_results( ctx, r_sigctx );
    
    gpgme_data_release( sig );
    gpgme_data_release( text );
    gpgme_release( ctx );

    return err;
} /* gpgme_op_clip_verify_detached */
