/* wptFileStatDlg.cpp - (File Manager) Show file statistics
 *	Copyright (C) 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <time.h>
#include <sys/types.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptDlgs.h"
#include "wptErrors.h"
#include "wptW32API.h"
#include "openpgp.h"

static const char*
get_pubkey_algo( int algo )
{	
    if ( algo == 17 )
	return "DSA";	
   else if ( is_ELGAMAL(algo) )
	return "ELG";	
    else if ( is_RSA(algo) )
	return "RSA";
    return NULL;
} /* get_algo */

static const char*
get_symkey_algo (int algo)
{
    switch (algo)
    {
    case 0: return "PLAINTEXT";
    case 1: return "IDEA";
    case 2: return "3DES"; 
    case 3: return "CAST5";
    case 4: return "BLOWFISH";
    case 5: return "RESERVED";
    case 6: return "RESERVED";
    case 7: return "AES";
    case 8: return "AES-192";
    case 9: return "AES-256";
    }    
    return "UNKNOWN";
} /* get_symkey_algo */

static const char*
get_digest_algo (int algo)
{
    switch (algo)
    {
    case 0: return "NONE";
    case 1: return "MD5";
    case 2: return "SHA1";
    case 3: return "RMD160";
    case 4: return "RESERVED";
    case 5: return "MD2";
    case 6: return "RESERVED";
    case 7: return "RESERVED";
    }
    return "UNKNOWN";
} /* get_digest_algo */

static const char*
get_compress_algo (int algo)
{
    switch (algo) 
    {
    case 0: return "NONE";
    case 1: return "ZIB";
    case 2: return "ZLIB";
    case 3: return "BZIP2";
    }
    return "UNKNOWN";
} /* get_compress_algo */


static const char*
get_timestring (long timestamp)
{
    static char ts[64];
    struct tm *ltm;

    ltm = localtime( &timestamp );
    _snprintf( ts, sizeof ts -1, "%04d-%02d-%02d",
		ltm->tm_year + 1900, ltm->tm_mon + 1, ltm->tm_mday );
    return ts;
} /* get_timestring */


static int
do_list_packets (HWND list, const char *file)
{
    PACKET *pkt = (PACKET *)calloc( 1, sizeof *pkt );
    gpg_iobuf_t inp = NULL;
    armor_filter_context_t afx;
    int rc = 0;
    char inf[1024];

    if( !list || file == NULL ) {
	safe_free( pkt );
	return -1;
    }

    inp = gpg_iobuf_open( file );
    if( inp == NULL ) {
	log_box( _("File Manager"), MB_ERR, "%s: %s", file, 
		 winpt_strerror( WPTERR_FILE_OPEN ) );
	safe_free( pkt );
	return -1;
    }
    gpg_iobuf_ioctl( inp, 3, 1, NULL ); /* disable cache */
    if( gpg_use_armor_filter( inp ) ) {
	memset( &afx, 0, sizeof afx );
	gpg_iobuf_push_filter( inp, gpg_armor_filter, &afx );
    }	
    gpg_init_packet( pkt );
    while ( (rc = gpg_parse_packet( inp, pkt )) != -1 ) {
	switch ( pkt->pkttype ) {	
	case PKT_PUBKEY_ENC:
	    {PKT_pubkey_enc *enc = pkt->pkt.pubkey_enc;
	    if (!enc)
		break;
	    _snprintf( inf, sizeof(inf)-1, 	
		"public key encryted packet: version %d, algo %s, keyid 0x%08X",		
		enc->version, get_pubkey_algo(enc->pubkey_algo), enc->keyid[1]);
	    listbox_add_string( list, inf );}
	    break;
		
	case PKT_SYMKEY_ENC:
	    {PKT_symkey_enc *enc = pkt->pkt.symkey_enc;	
	    if (!enc)
		break;
	    _snprintf( inf, sizeof(inf)-1,	
		"symmetric key encrypted packet: version %d, cipher %s, s2k %d, hash %s",		
		enc->version, get_symkey_algo(enc->cipher_algo),		
		enc->s2k.mode, get_digest_algo(enc->s2k.hash_algo) );
	    listbox_add_string( list, inf ); }
	    break;

	case PKT_ENCRYPTED:	
	case PKT_ENCRYPTED_MDC:
	    {PKT_encrypted *enc = pkt->pkt.encrypted;
	    if (!enc)
		break;
	    _snprintf( inf, sizeof(inf)-1,	
		"encrypted data packet: mdc method %d, length %d",		
		enc->mdc_method, enc->len );
	    listbox_add_string( list, inf ); }
	    break;

	case PKT_PUBLIC_KEY:
	case PKT_PUBLIC_SUBKEY:
	    {PKT_public_key *pk = pkt->pkt.public_key;
	    if (!pk)
		break;
	    _snprintf( inf, sizeof(inf)-1,
		"public key packet: version %d, algo %s, created %s",
		pk->version, get_pubkey_algo(pk->pubkey_algo),
		get_timestring(pk->timestamp));
	    listbox_add_string( list, inf ); }
	    break;

	case PKT_SECRET_KEY:
	case PKT_SECRET_SUBKEY:	    
	    {PKT_secret_key *sk = pkt->pkt.secret_key;
	    if (!sk)
		break;
	    _snprintf( inf, sizeof(inf)-1,
		"secret key packet: version %d, algo %s, created %s mode %d",
		sk->version,  get_pubkey_algo( sk->pubkey_algo ),
		get_timestring( sk->timestamp ), 
		sk->protect.s2k.mode );
	    listbox_add_string( list, inf ); }
	    break;

		
	case PKT_SIGNATURE:
	    {PKT_signature *sig = pkt->pkt.signature;
	    if (!sig)
		break;
	    _snprintf( inf, sizeof(inf)-1,
		"signature packet: version %d, algo %s, keyid 0x%08X, created %s",
		sig->version, get_pubkey_algo(sig->pubkey_algo), sig->keyid[1],
		get_timestring(sig->timestamp));
	    listbox_add_string( list, inf ); }
	    break;

		
	case PKT_USER_ID:
	    {PKT_user_id *id = pkt->pkt.user_id;
	    if (!id)
		break;
	    _snprintf( inf, sizeof(inf)-1, "user id packet: %s", id->name );
	    listbox_add_string( list, inf ); }
	    break;
	    
	case PKT_COMPRESSED:
	    {PKT_compressed *zip = pkt->pkt.compressed;
	    if (!zip)
		break;
	    _snprintf( inf, sizeof(inf)-1, "compressed packet: algo %s, length %d",
		get_compress_algo(zip->algorithm), zip->len );
	    listbox_add_string( list, inf ); }
	    break;
	}
	gpg_free_packet( pkt );
	gpg_init_packet( pkt );
    }
    gpg_iobuf_close( inp );
    safe_free( pkt );
    return rc;
} /* do_list_packets */


BOOL CALLBACK
file_stat_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    const char *file;
    
    switch ( msg ) {
    case WM_INITDIALOG:
	#ifndef LANG_DE
        SetWindowText( dlg, _("File Status") );
	#endif
        file = (const char*)lparam;
        if( file == NULL )
            dlg_fatal_error( dlg, "Could not get dialog state!" );
	do_list_packets( GetDlgItem( dlg, IDC_FILE_STAT_LIST ), file );
        SetForegroundWindow( dlg );
        break;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDOK:
            EndDialog( dlg, TRUE );
            break;
        }
        break;
    }
    
    return FALSE;
} /* file_stat_dlg_proc */
