/* wptImportList.cpp
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <sys/types.h>

#include "wptTypes.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptUTF8.h"
#include "wptW32API.h"
#include "wptRegistry.h"
#include "openpgp.h"


static char *
key_description( gpgme_key_t key )
{
    char * p;
    const char * type, * state;
    int pkalgo, subalgo;
    
    p = new char[64+1];
    if( !p )
	BUG( NULL );
    pkalgo = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, 0 );
    subalgo = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, 1 );
    if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_REVOKED, NULL, 0 ) )
	state = _("Revoked" );
    else if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_EXPIRED, NULL, 0 ) )
	state = _("Expired" );
    else
	state = "";
    if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_IS_SECRET, NULL, 0 ) )
	type = _("secret key");
    else
	type = _("public key");

    if( !subalgo )
	_snprintf( p, 64, "%s %s %s", state,
		    gpgme_key_expand_attr( GPGME_ATTR_ALGO, pkalgo ), type );
    else
	_snprintf( p, 64, "%s %s/%s %s", state,
		    gpgme_key_expand_attr( GPGME_ATTR_ALGO, pkalgo ),
		    gpgme_key_expand_attr( GPGME_ATTR_ALGO, subalgo ), type );

    return p;
} /* key_description */


static int
implist_add_key( listview_ctrl_t lv, int pos, gpgme_key_t key )
{	
    char *uid = NULL;
    char buf[128], * desc;
    const char *t;
    u32 tt, tt2;
	
    if( listview_add_item( lv, " " ) )
	return WPTERR_GENERAL;

    t = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
    if( !t || strlen( t ) < 5 ) {
	t = _("Invalid user ID");
	listview_add_sub_item( lv, pos, 0, t );	
    }
    else {
	uid = utf8_to_wincp (t, strlen (t));
	if (uid) {
	    listview_add_sub_item( lv, pos, 0, uid );
	    free( uid );
	}
    }

    tt = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, 0 );
    tt2 = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, 1 );
    if( tt && tt2 )
	_snprintf( buf, sizeof buf - 1, "%d/%d", tt, tt2 );
    else
	_snprintf( buf, sizeof buf-1, "%d", tt );
    listview_add_sub_item( lv, pos, 1, buf );

    t = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
    if( !t || strlen( t ) < 8 )
	t = "DEADBEEFDEADBEEF";
    _snprintf( buf, sizeof buf -1, "0x%s", t + 8 );
    listview_add_sub_item( lv, pos, 2, buf );

	
    tt = gpgme_key_get_ulong_attr( key, GPGME_ATTR_CREATED, NULL, 0 );
    t = get_key_created( tt );
    if( !t )
	t = "????-??-??";
    listview_add_sub_item( lv, pos, 3, (char *)t );

    desc = key_description( key );
    if( desc ) {    
	listview_add_sub_item( lv, pos, 4, desc );
	free_if_alloc( desc );
    }

    return 0;
} /* implist_add_key */


int
implist_build( listview_ctrl_t *lv, HWND ctrl )
{
    struct listview_ctrl_s *c;
    struct listview_column_s implist[] = {
	{ 0, 190, (char *)_("User ID") },	
	{ 1,  66, (char *)_("Size") },
	{ 2,  80, (char *)_("Key ID") },
	{ 3,  72, (char *)_("Creation") },
	{ 4, 132, (char *)_("Type") },
	{ 0,   0, NULL }
	};	
    int j, rc = 0;
	
    rc = listview_new( &c );
    if( rc )
	return rc;
    c->ctrl = ctrl;
    for( j = 0; implist[j].fieldname != NULL; j++ )
	listview_add_column( c, &implist[j] );
    listview_set_ext_style( c );
    *lv = c;
    return 0;
} /* implist_build */


int
implist_load( listview_ctrl_t lv, const char *file, 
	      int *r_revcerts, int *r_seckeys )
{
    gpgme_ctx_t ctx;
    gpgme_data_t dh, list;
    gpgme_error_t err;    
    gpgme_key_t pubk, key;
    int rc = 0;
    char *p, keyid[16+1];
    char * pending_line = NULL;
	
    if( file && *file ) {
	err = gpgme_data_new_from_file (&dh, file);
	if( err ) {
	    msg_box( NULL, gpgme_strerror( err ), _("Import"), MB_ERR );
	    return WPTERR_GENERAL;
	}	
    }
    else {
	err = gpgme_data_new_from_clipboard (&dh);
	if( err ) {
	    msg_box( NULL, gpgme_strerror( err ), _("Import"), MB_ERR );
	    return WPTERR_CLIP_OPEN;
	}
    }
		
    p = gpgme_data_get_as_string( dh );
    /* fixme: the code could need a little more work because it's
              no very flexible. */
    if( p && stristr( p, _("A revocation certificate should follow") ) ) {
	gpgme_revcert_t cert;
	gpgme_revcert_new( &cert );
	rc = gpgme_revcert_parse( p, cert, keyid );
	if( rc ) {
	    msg_box( NULL, gpgme_strerror( (gpgme_error_t)rc ), _("Revocation"), MB_ERR );
	    free( p );
	    return WPTERR_GENERAL;
	}
	if( get_pubkey( keyid, &pubk ) )
	    BUG( NULL );
	gpgme_implist_create_revcert( &key, cert, pubk );
	implist_add_key( lv, 0, key );
	gpgme_key_release( key );
	gpgme_revcert_release( cert );
	if( r_revcerts )
	    *r_revcerts = 1;
	free( p );
	return 0;
    }
    safe_free (p);
    err = gpgme_data_new( &list );
    if( err )
	BUG( NULL );
    err = gpgme_new( &ctx );
    if( err )
	BUG( NULL );    

    err = gpgme_op_import_list( ctx, dh, list );
    if( err ) {
	gpgme_release( ctx );
	gpgme_data_release( dh );
	gpgme_data_release( list );
	if (err == GPGME_General_Error) {
	    msg_box (NULL, _("It is possible that the ASCII-Armor is damaged\n"
			     "and thus a CRC error occurs."), _("Import"), MB_ERR);
	}
	else
	    msg_box( NULL, gpgme_strerror( err ), _("Import"), MB_ERR );
	return WPTERR_GENERAL;	
    }

    gpgme_release( ctx );
    gpgme_data_release( dh );

    if( r_seckeys )
	*r_seckeys = 0;
    while( !rc ) {
	/* xxx: if the key has a direct key signature, the user-id field 
	        in the --with-colons mode is empty! */
	rc = gpgme_op_import_list_next( list, &pending_line, &key );
	if( !rc )
	    rc = implist_add_key( lv, 0, key );
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_IS_SECRET, NULL, 0 ) )
	    (*r_seckeys)++;
	gpgme_key_release( key );
    }
    gpgme_data_release( list );

    return rc;
} /* implist_load */


void 
implist_delete( listview_ctrl_t lv )
{
    if( lv ) {
	listview_release( lv );	
    }
} /* implist_delete */
