/* sig.c - General signature API
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "gpgme.h"
#include "context.h"
#include "util.h"


gpgme_error_t
gpgme_sig_new( gpgme_sig_t *r_sig )
{
    gpgme_sig_t sig;
    
    sig = calloc( 1, sizeof *sig );
    if( !sig )
        return mk_error( Out_Of_Core );	
    *r_sig = sig;
    return 0;
} /* gpgme_sig_new */


void
gpgme_sig_release (gpgme_sig_t sig)
{
    if( !sig )
	return;    
    if( sig->ref_count > 0 ) {
	sig->ref_count--;
	return;
    }
    safe_free( sig->notation );
    safe_free( sig->policy_url );
    safe_free( sig->user_id );
    safe_free( sig->file_name );
    safe_free (sig->key_server);
    safe_free( sig );
} /* gpgme_sig_release */


void
_gpgme_sig_ref( gpgme_sig_t sig )
{
    if( sig )
	sig->ref_count++;
} /* _gpgme_sig_ref */


void
gpgme_sig_set_key_ctx( gpgme_sig_t sig, gpgme_keycache_t cache )
{
    if( sig && cache )        
	gpgme_keycache_find_key( cache, sig->id, 0, &sig->key );
} /* gpgme_sig_set_key_ctx */


ulong
gpgme_sig_get_ulong_attr (gpgme_sig_t sig, int idx, int what)
{
    gpgme_sig_t s;
    int pos = 0;
     
    if (!sig)
	return 0;

    switch( what ) {
    case GPGME_ATTR_EXPIRES:  return sig->expired;
    case GPGME_ATTR_EXPIRE:   return sig->flags.expired;;
    case GPGME_ATTR_VALIDITY: return sig->sigstat;
    case GPGME_ATTR_OTRUST:   return sig->trust;
    case GPGME_ATTR_CREATED:  return sig->created;
    case GPGME_ATTR_TYPE:     return (ulong)sig->key;	
    case GPGME_ATTR_ALGO:     return sig->key_algo;
    case GPGME_ATTR_OPAQUE:
	for( s = sig; s && idx > 0; s=s->next, idx-- )
	    ;
	return (ulong)s;

    case GPGME_ATTR_LEVEL:
	for (s = sig; s; s = s->next)
	    pos++;
	return pos;
    case GPGME_ATTR_KEYID: 
        if( strlen( sig->id ) == 40 ) pos = 32;            
        else if( strlen( sig->id ) == 16 ) pos = 8;
        return strtoul( sig->id + pos, NULL, 16 );
    }
    return 0;
} /* gpgme_sig_get_ulong_attr */


const char*
gpgme_sig_get_string_attr( gpgme_sig_t sig, int what )
{
    if( !sig )
	return NULL;

    switch( what ) {
    case GPGME_ATTR_NOTATION:   
	return sig->notation;
    case GPGME_ATTR_NAME:       
	return sig->file_name;
    case GPGME_ATTR_USERID:     
	return sig->user_id;
    case GPGME_ATTR_POLICY_URL: 
	return sig->policy_url;
    case GPGME_ATTR_KEYSERVER:  
	return sig->key_server;
    case GPGME_ATTR_KEYID:
	if( strlen( sig->id ) == 40 )
	    return sig->id + 24;
	return sig->id;
    }
    return NULL;
} /* gpgme_sig_get_string_attr */

