/* ciphersuites.c - cipher suite helpers
 * Copyright (C) 2006-2014, Brainspark B.V.
 * Copyright (C) 2014 g10 code GmbH
 *
 * This file is part of NTBTLS
 *
 * NTBTLS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * NTBTLS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * This file was part of PolarSSL (http://www.polarssl.org).  Former
 * Lead Maintainer: Paul Bakker <polarssl_maintainer at polarssl.org>.
 * Please do not file bug reports to them but to the address given in
 * the file AUTHORS in the top directory of NTBTLS.
 */

#include <config.h>
#include <stdlib.h>
#include <string.h>

#include "ntbtls-int.h"
#include "ciphersuites.h"

/*
 * This structure is used for storing ciphersuite information
 */
struct _ntbtls_ciphersuite_s
{
  int tlsid;
  const char *name;
  cipher_algo_t cipher;
  cipher_mode_t ciphermode;
  mac_algo_t mac;
  key_exchange_type_t key_exchange;
  unsigned char flags;
  unsigned char min_major_ver;
  unsigned char min_minor_ver;
  unsigned char max_major_ver;
  unsigned char max_minor_ver;
};



/*
 * Ordered from most preferred to least preferred in terms of security.
 *
 * Current rule:
 * 1. By key exchange:
 *    Forward-secure non-PSK > forward-secure PSK > other non-PSK > other PSK
 * 2. By key length and cipher:
 *    AES-256 > Camellia-256 > AES-128 > Camellia-128 > 3DES
 * 3. By cipher mode when relevant GCM > CCM > CBC > CCM_8
 * 4. By hash function used when relevant
 * 5. By key exchange/auth again: EC > non-EC
 */
static const int ciphersuite_preference[] = {

  /* All AES-256 ephemeral suites */
  TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
  TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
  TLS_DHE_RSA_WITH_AES_256_GCM_SHA384,
  TLS_ECDHE_ECDSA_WITH_AES_256_CCM,
  TLS_DHE_RSA_WITH_AES_256_CCM,
  TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384,
  TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384,
  TLS_DHE_RSA_WITH_AES_256_CBC_SHA256,
  TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
  TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
  TLS_DHE_RSA_WITH_AES_256_CBC_SHA,
  TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8,
  TLS_DHE_RSA_WITH_AES_256_CCM_8,

  /* All CAMELLIA-256 ephemeral suites */
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256,
  TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA,

  /* All AES-128 ephemeral suites */
  TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
  TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
  TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
  TLS_ECDHE_ECDSA_WITH_AES_128_CCM,
  TLS_DHE_RSA_WITH_AES_128_CCM,
  TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256,
  TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256,
  TLS_DHE_RSA_WITH_AES_128_CBC_SHA256,
  TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
  TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
  TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
  TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8,
  TLS_DHE_RSA_WITH_AES_128_CCM_8,

  /* All CAMELLIA-128 ephemeral suites */
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA,

  /* All remaining >= 128-bit ephemeral suites */
  TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA,
  TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA,
  TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA,

  /* The PSK ephemeral suites */
  TLS_DHE_PSK_WITH_AES_256_GCM_SHA384,
  TLS_DHE_PSK_WITH_AES_256_CCM,
  TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384,
  TLS_DHE_PSK_WITH_AES_256_CBC_SHA384,
  TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA,
  TLS_DHE_PSK_WITH_AES_256_CBC_SHA,
  TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_DHE_PSK_WITH_AES_256_CCM_8,

  TLS_DHE_PSK_WITH_AES_128_GCM_SHA256,
  TLS_DHE_PSK_WITH_AES_128_CCM,
  TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256,
  TLS_DHE_PSK_WITH_AES_128_CBC_SHA256,
  TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA,
  TLS_DHE_PSK_WITH_AES_128_CBC_SHA,
  TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_DHE_PSK_WITH_AES_128_CCM_8,

  TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA,
  TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA,

  /* All AES-256 suites */
  TLS_RSA_WITH_AES_256_GCM_SHA384,
  TLS_RSA_WITH_AES_256_CCM,
  TLS_RSA_WITH_AES_256_CBC_SHA256,
  TLS_RSA_WITH_AES_256_CBC_SHA,
  TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
  TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384,
  TLS_ECDH_RSA_WITH_AES_256_CBC_SHA,
  TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384,
  TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384,
  TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA,
  TLS_RSA_WITH_AES_256_CCM_8,

  /* All CAMELLIA-256 suites */
  TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256,
  TLS_RSA_WITH_CAMELLIA_256_CBC_SHA,
  TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,

  /* All AES-128 suites */
  TLS_RSA_WITH_AES_128_GCM_SHA256,
  TLS_RSA_WITH_AES_128_CCM,
  TLS_RSA_WITH_AES_128_CBC_SHA256,
  TLS_RSA_WITH_AES_128_CBC_SHA,
  TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
  TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
  TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
  TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256,
  TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256,
  TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA,
  TLS_RSA_WITH_AES_128_CCM_8,

  /* All CAMELLIA-128 suites */
  TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_RSA_WITH_CAMELLIA_128_CBC_SHA,
  TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,

  /* All remaining >= 128-bit suites */
  TLS_RSA_WITH_3DES_EDE_CBC_SHA,
  TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA,
  TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA,

  /* The RSA PSK suites */
  TLS_RSA_PSK_WITH_AES_256_GCM_SHA384,
  TLS_RSA_PSK_WITH_AES_256_CBC_SHA384,
  TLS_RSA_PSK_WITH_AES_256_CBC_SHA,
  TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384,

  TLS_RSA_PSK_WITH_AES_128_GCM_SHA256,
  TLS_RSA_PSK_WITH_AES_128_CBC_SHA256,
  TLS_RSA_PSK_WITH_AES_128_CBC_SHA,
  TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256,

  TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA,

  /* The PSK suites */
  TLS_PSK_WITH_AES_256_GCM_SHA384,
  TLS_PSK_WITH_AES_256_CCM,
  TLS_PSK_WITH_AES_256_CBC_SHA384,
  TLS_PSK_WITH_AES_256_CBC_SHA,
  TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384,
  TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384,
  TLS_PSK_WITH_AES_256_CCM_8,

  TLS_PSK_WITH_AES_128_GCM_SHA256,
  TLS_PSK_WITH_AES_128_CCM,
  TLS_PSK_WITH_AES_128_CBC_SHA256,
  TLS_PSK_WITH_AES_128_CBC_SHA,
  TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256,
  TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256,
  TLS_PSK_WITH_AES_128_CCM_8,

  TLS_PSK_WITH_3DES_EDE_CBC_SHA,

  0
};



static const struct _ntbtls_ciphersuite_s ciphersuite_definitions[] = {

  {TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
   "TLS-ECDHE-ECDSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
   "TLS-ECDHE-ECDSA-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256,
   "TLS-ECDHE-ECDSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
   "TLS-ECDHE-ECDSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384,
   "TLS-ECDHE-ECDSA-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
   "TLS-ECDHE-ECDSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_256_CCM,
   "TLS-ECDHE-ECDSA-WITH-AES-256-CCM",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8,
   "TLS-ECDHE-ECDSA-WITH-AES-256-CCM-8",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_128_CCM,
   "TLS-ECDHE-ECDSA-WITH-AES-128-CCM",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8,
   "TLS-ECDHE-ECDSA-WITH-AES-128-CCM-8",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-ECDHE-ECDSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-ECDHE-ECDSA-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-ECDHE-ECDSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-ECDHE-ECDSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-ECDHE-ECDSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
   "TLS-ECDHE-RSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
   "TLS-ECDHE-RSA-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256,
   "TLS-ECDHE-RSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
   "TLS-ECDHE-RSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384,
   "TLS-ECDHE-RSA-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
   "TLS-ECDHE-RSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-ECDHE-RSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-ECDHE-RSA-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-ECDHE-RSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-ECDHE-RSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-ECDHE-RSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_256_GCM_SHA384,
   "TLS-DHE-RSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
   "TLS-DHE-RSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_128_CBC_SHA256,
   "TLS-DHE-RSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_256_CBC_SHA256,
   "TLS-DHE-RSA-WITH-AES-256-CBC-SHA256",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
   "TLS-DHE-RSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_256_CBC_SHA,
   "TLS-DHE-RSA-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_256_CCM,
   "TLS-DHE-RSA-WITH-AES-256-CCM",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_256_CCM_8,
   "TLS-DHE-RSA-WITH-AES-256-CCM-8",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_128_CCM,
   "TLS-DHE-RSA-WITH-AES-128-CCM",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_AES_128_CCM_8,
   "TLS-DHE-RSA-WITH-AES-128-CCM-8",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-DHE-RSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256,
   "TLS-DHE-RSA-WITH-CAMELLIA-256-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA,
   "TLS-DHE-RSA-WITH-CAMELLIA-128-CBC-SHA",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA,
   "TLS-DHE-RSA-WITH-CAMELLIA-256-CBC-SHA",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-DHE-RSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-DHE-RSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-DHE-RSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_256_GCM_SHA384,
   "TLS-RSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_128_GCM_SHA256,
   "TLS-RSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_128_CBC_SHA256,
   "TLS-RSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_256_CBC_SHA256,
   "TLS-RSA-WITH-AES-256-CBC-SHA256",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_128_CBC_SHA,
   "TLS-RSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_256_CBC_SHA,
   "TLS-RSA-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_256_CCM,
   "TLS-RSA-WITH-AES-256-CCM",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_256_CCM_8,
   "TLS-RSA-WITH-AES-256-CCM-8",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_128_CCM,
   "TLS-RSA-WITH-AES-128-CCM",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_AES_128_CCM_8,
   "TLS-RSA-WITH-AES-128-CCM-8",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-RSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  { TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256,
    "TLS-RSA-WITH-CAMELLIA-256-CBC-SHA256",
    GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
    KEY_EXCHANGE_RSA,
    0,
    TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
    TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_CAMELLIA_128_CBC_SHA,
   "TLS-RSA-WITH-CAMELLIA-128-CBC-SHA",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_CAMELLIA_256_CBC_SHA,
   "TLS-RSA-WITH-CAMELLIA-256-CBC-SHA",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-RSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-RSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-RSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
   "TLS-ECDH-RSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_256_CBC_SHA,
   "TLS-ECDH-RSA-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
   "TLS-ECDH-RSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
   "TLS-ECDH-RSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384,
   "TLS-ECDH-RSA-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
   "TLS-ECDH-RSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-ECDH-RSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-ECDH-RSA-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-ECDH-RSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-ECDH-RSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-ECDH-RSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDH_RSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

   {TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA,
   "TLS-ECDH-ECDSA-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDH_ECDSA,
    0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

   {TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA,
    "TLS-ECDH-ECDSA-WITH-AES-256-CBC-SHA",
    GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
    KEY_EXCHANGE_ECDH_ECDSA,
    0,
    TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
    TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

   {TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256,
   "TLS-ECDH-ECDSA-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_ECDSA,
    0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256,
   "TLS-ECDH-ECDSA-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384,
   "TLS-ECDH-ECDSA-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384,
   "TLS-ECDH-ECDSA-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},


  {TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-ECDH-ECDSA-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-ECDH-ECDSA-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-ECDH-ECDSA-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-ECDH-ECDSA-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA,
   "TLS-ECDH-ECDSA-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDH_ECDSA,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_128_GCM_SHA256,
   "TLS-PSK-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_256_GCM_SHA384,
   "TLS-PSK-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_128_CBC_SHA256,
   "TLS-PSK-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_256_CBC_SHA384,
   "TLS-PSK-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_128_CBC_SHA,
   "TLS-PSK-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_256_CBC_SHA,
   "TLS-PSK-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_256_CCM,
   "TLS-PSK-WITH-AES-256-CCM",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_256_CCM_8,
   "TLS-PSK-WITH-AES-256-CCM-8",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_128_CCM,
   "TLS-PSK-WITH-AES-128-CCM",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_AES_128_CCM_8,
   "TLS-PSK-WITH-AES-128-CCM-8",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-PSK-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-PSK-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-PSK-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-PSK-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_PSK_WITH_3DES_EDE_CBC_SHA,
   "TLS-PSK-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_128_GCM_SHA256,
   "TLS-DHE-PSK-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_256_GCM_SHA384,
   "TLS-DHE-PSK-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_128_CBC_SHA256,
   "TLS-DHE-PSK-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_256_CBC_SHA384,
   "TLS-DHE-PSK-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_128_CBC_SHA,
   "TLS-DHE-PSK-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_256_CBC_SHA,
   "TLS-DHE-PSK-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_256_CCM,
   "TLS-DHE-PSK-WITH-AES-256-CCM",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_256_CCM_8,
   "TLS-DHE-PSK-WITH-AES-256-CCM-8",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_128_CCM,
   "TLS-DHE-PSK-WITH-AES-128-CCM",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_AES_128_CCM_8,
   "TLS-DHE-PSK-WITH-AES-128-CCM-8",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   CIPHERSUITE_FLAG_SHORT_TAG,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-DHE-PSK-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-DHE-PSK-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-DHE-PSK-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-DHE-PSK-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA,
   "TLS-DHE-PSK-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_DHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256,
   "TLS-ECDHE-PSK-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384,
   "TLS-ECDHE-PSK-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA,
   "TLS-ECDHE-PSK-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA,
   "TLS-ECDHE-PSK-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-ECDHE-PSK-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-ECDHE-PSK-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA,
   "TLS-ECDHE-PSK-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_ECDHE_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_1,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_128_GCM_SHA256,
   "TLS-RSA-PSK-WITH-AES-128-GCM-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_256_GCM_SHA384,
   "TLS-RSA-PSK-WITH-AES-256-GCM-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_128_CBC_SHA256,
   "TLS-RSA-PSK-WITH-AES-128-CBC-SHA256",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_256_CBC_SHA384,
   "TLS-RSA-PSK-WITH-AES-256-CBC-SHA384",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_128_CBC_SHA,
   "TLS-RSA-PSK-WITH-AES-128-CBC-SHA",
   GCRY_CIPHER_AES128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_AES_256_CBC_SHA,
   "TLS-RSA-PSK-WITH-AES-256-CBC-SHA",
   GCRY_CIPHER_AES256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256,
   "TLS-RSA-PSK-WITH-CAMELLIA-128-CBC-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384,
   "TLS-RSA-PSK-WITH-CAMELLIA-256-CBC-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256,
   "TLS-RSA-PSK-WITH-CAMELLIA-128-GCM-SHA256",
   GCRY_CIPHER_CAMELLIA128, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA256,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384,
   "TLS-RSA-PSK-WITH-CAMELLIA-256-GCM-SHA384",
   GCRY_CIPHER_CAMELLIA256, GCRY_CIPHER_MODE_GCM, GCRY_MAC_HMAC_SHA384,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA,
   "TLS-RSA-PSK-WITH-3DES-EDE-CBC-SHA",
   GCRY_CIPHER_3DES, GCRY_CIPHER_MODE_CBC, GCRY_MAC_HMAC_SHA1,
   KEY_EXCHANGE_RSA_PSK,
   0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_0,
   TLS_MAJOR_VERSION_3, TLS_MINOR_VERSION_3},

  {0, "", 0, 0, 0, 0, 0, 0, 0, 0}
};


static int supported_ciphersuites[DIM (ciphersuite_definitions)];
static int supported_init = 0;


/* Return an array with all supported cipher suites.  */
const int *
_ntbtls_ciphersuite_list (void)
{
  /*
   * On initial call filter out all ciphersuites not supported by current
   * build based on presence in the ciphersuite_definitions.
   */
  //FIXME: We need to check that algorithms are enabled in libgcrypt
  //       and we need to protect this with a mutex.
  if (!supported_init)
    {
      ciphersuite_t suite;
      int i, j;

      for (i=j=0; (ciphersuite_preference[i]
                   && j < DIM(ciphersuite_definitions)-1); i++)
        {
          if ((suite = _ntbtls_ciphersuite_from_id (ciphersuite_preference[i])))
            {
              /*FIXME: CCM are not yet ready for us - disable.  */
              if (suite->ciphermode != GCRY_CIPHER_MODE_CCM
                  && suite->key_exchange != KEY_EXCHANGE_ECDH_RSA
                  && suite->key_exchange != KEY_EXCHANGE_ECDHE_ECDSA
                  && suite->key_exchange != KEY_EXCHANGE_ECDH_ECDSA)
                supported_ciphersuites[j++] = ciphersuite_preference[i];
            }
        }
      supported_ciphersuites[j] = 0;

      supported_init = 1;
    }

  return supported_ciphersuites;
}


ciphersuite_t
_ntbtls_ciphersuite_from_id (int suite_id)
{
  ciphersuite_t suite;

  for (suite = ciphersuite_definitions; suite->tlsid; suite++)
    if (suite->tlsid == suite_id)
      return suite;

  return NULL;
}


const char *
_ntbtls_ciphersuite_get_name (int suite_id)
{
  ciphersuite_t suite;

  suite = _ntbtls_ciphersuite_from_id (suite_id);
  return suite? suite->name : "unknown";
}


cipher_algo_t
_ntbtls_ciphersuite_get_cipher (ciphersuite_t suite, cipher_mode_t *r_mode)
{
  if (!suite)
    return 0;
  *r_mode = suite->ciphermode;
  return suite->cipher;
}


mac_algo_t
_ntbtls_ciphersuite_get_mac (ciphersuite_t suite)
{
  return suite? suite->mac : 0;
}


key_exchange_type_t
_ntbtls_ciphersuite_get_kex (ciphersuite_t suite)
{
  return suite? suite->key_exchange : 0;
}


unsigned int
_ntbtls_ciphersuite_get_flags (ciphersuite_t suite)
{
  return suite? suite->flags : 0;
}


pk_algo_t
_ntbtls_ciphersuite_get_sig_pk_alg (ciphersuite_t suite)
{
  if (!suite)
    return 0;

  switch (suite->key_exchange)
    {
    case KEY_EXCHANGE_RSA:
    case KEY_EXCHANGE_DHE_RSA:
    case KEY_EXCHANGE_ECDHE_RSA:
    case KEY_EXCHANGE_RSA_PSK:
      return GCRY_PK_RSA;

    case KEY_EXCHANGE_ECDHE_ECDSA:
      return GCRY_PK_ECC; /* Actually ECDSA */

    case KEY_EXCHANGE_ECDH_RSA:
    case KEY_EXCHANGE_ECDH_ECDSA:
      return GCRY_PK_ECC;

    default:
      return 0;
    }
}


int
_ntbtls_ciphersuite_version_ok (ciphersuite_t suite,
                                int min_minor_ver, int max_minor_ver)
{
  if (!suite)
    return 0;

  return (suite->min_minor_ver <= max_minor_ver
          && suite->max_minor_ver >= min_minor_ver);
}


int
_ntbtls_ciphersuite_uses_ec (ciphersuite_t suite)
{
  if (!suite)
    return 0;

  switch (suite->key_exchange)
    {
    case KEY_EXCHANGE_ECDHE_RSA:
    case KEY_EXCHANGE_ECDHE_ECDSA:
    case KEY_EXCHANGE_ECDHE_PSK:
    case KEY_EXCHANGE_ECDH_RSA:
    case KEY_EXCHANGE_ECDH_ECDSA:
      return 1;

    default:
      return 0;
    }
}


int
_ntbtls_ciphersuite_uses_psk (ciphersuite_t suite)
{
  if (!suite)
    return 0;

  switch (suite->key_exchange)
    {
    case KEY_EXCHANGE_PSK:
    case KEY_EXCHANGE_RSA_PSK:
    case KEY_EXCHANGE_DHE_PSK:
    case KEY_EXCHANGE_ECDHE_PSK:
      return 1;

    default:
      return 0;
    }
}
