//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.bitvectors;

import java.math.BigInteger;

import org.eclipse.escet.cif.bdd.spec.CifBddDomain;
import org.eclipse.escet.common.java.Assert;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDFactory;

/**
 * Signed BDD bit vector in two's complement representation.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Two%27s_complement">Two&apos;s complement (Wikipedia)</a>
 */
public class SignedBddBitVector extends BddBitVector<SignedBddBitVector, SignedBddBitVectorAndCarry> {
    /** The minimum length (in number of bits) of any {@link SignedBddBitVector}. */
    public static final int MINIMUM_LENGTH = 2;

    /**
     * Constructor for the {@link SignedBddBitVector} class.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    private SignedBddBitVector(BDDFactory factory, int length) {
        super(factory, length);
    }

    @Override
    protected int getMinimumLength() {
        return MINIMUM_LENGTH;
    }

    /**
     * Returns the minimum length (in number of bits) needed to represent the given integer value as a
     * {@link SignedBddBitVector}.
     *
     * @param value The integer value.
     * @return The minimum length.
     */
    public static int getMinimumLength(int value) {
        if (value >= 0) {
            // Needs extra sign bit, besides the number of bits required for the unsigned bit vector. At least two
            // bits are required.
            return Math.max(2, 1 + UnsignedBddBitVector.getMinimumLength(value));
        } else {
            // Requires sign bit, besides number of bits to represent the inverse, while accounting for the fact that we
            // can represent one more negative number than we can positive ones. At least two bits are required.
            return Math.max(2, 1 + UnsignedBddBitVector.getMinimumLength(-(value + 1)));
        }
    }

    @Override
    protected SignedBddBitVector createEmpty(int length) {
        return new SignedBddBitVector(this.factory, length);
    }

    /**
     * Creates a {@link SignedBddBitVector}. Initializes the bits of the bit vector to 'false'.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    public static SignedBddBitVector create(BDDFactory factory, int length) {
        return create(factory, length, false);
    }

    /**
     * Creates a {@link SignedBddBitVector}. Initializes each bit of the bit vector to the given boolean value.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @param value The value to use for each bit.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    public static SignedBddBitVector create(BDDFactory factory, int length, boolean value) {
        // Create.
        SignedBddBitVector vector = new SignedBddBitVector(factory, length);

        // Initialize.
        for (int i = 0; i < vector.bits.length; i++) {
            vector.bits[i] = value ? factory.one() : factory.zero();
        }

        // Return.
        return vector;
    }

    /**
     * Creates a {@link SignedBddBitVector} from an integer value. Initializes the bits of the bit vector to the given
     * integer value. Uses an as small as possible bit vector to represent the integer value.
     *
     * @param factory The BDD factory to use.
     * @param value The integer value to represent using a bit vector.
     * @return The created bit vector.
     */
    public static SignedBddBitVector createFromInt(BDDFactory factory, int value) {
        int length = getMinimumLength(value);
        return createFromInt(factory, length, value);
    }

    /**
     * Creates a {@link SignedBddBitVector} from an integer value. Initializes the bits of the bit vector to the given
     * integer value, creating a bit vector of the given length. If the requested length is larger than the needed
     * number of bits, the remaining/highest bits are 'false' if the value is non-negative and 'true' if the value is
     * negative.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @param value The integer value to represent using a bit vector.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is insufficient to store the given value.
     */
    public static SignedBddBitVector createFromInt(BDDFactory factory, int length, int value) {
        // Precondition check.
        if (length < getMinimumLength(value)) {
            throw new IllegalArgumentException("Length is insufficient.");
        }

        // Create.
        SignedBddBitVector vector = new SignedBddBitVector(factory, length);

        // Initialize.
        for (int i = 0; i < vector.bits.length; i++) {
            vector.bits[i] = ((value & 0x1) != 0) ? factory.one() : factory.zero();
            value >>= 1;
        }
        Assert.check(value == 0 || value == -1, value); // All zeros or all ones.

        // Return.
        return vector;
    }

    /**
     * Creates a {@link SignedBddBitVector} from a {@link CifBddDomain}. Initializes the bits of the bit vector to the
     * variables of the given domain. The length of the bit vector is the number of variables in the given domain.
     *
     * @param domain The domain to use.
     * @return The created bit vector.
     */
    public static SignedBddBitVector createFromDomain(CifBddDomain domain) {
        // Create bit vector.
        int varCnt = domain.getVarCount();
        SignedBddBitVector vector = new SignedBddBitVector(domain.getFactory(), varCnt);

        // Initialize.
        int[] vars = domain.getVarIndices();
        for (int i = 0; i < vars.length; i++) {
            vector.bits[i] = vector.factory.ithVar(vars[i]);
        }

        // Return.
        return vector;
    }

    /**
     * Creates a {@link SignedBddBitVector} from an {@link UnsignedBddBitVector}. The signed bit vector is one bit
     * longer than the unsigned bit vector, to account for the sign bit, which is always 'false' as unsigned bit vectors
     * in all cases represent non-negative values. The bits of the unsigned bit vector are copied to the new signed bit
     * vector; the unsigned bit vector is not modified.
     *
     * @param unsignedVector The unsigned bit vector to use.
     * @return The created signed bit vector.
     */
    public static SignedBddBitVector createFromUnsignedBitVector(UnsignedBddBitVector unsignedVector) {
        // Create.
        SignedBddBitVector signedVector = new SignedBddBitVector(unsignedVector.factory, unsignedVector.length() + 1);

        // Initialize non-sign bits.
        for (int i = 0; i < unsignedVector.length(); i++) {
            signedVector.bits[i] = unsignedVector.getBit(i).id();
        }

        // Set sign bit to 'false' (no negative values).
        signedVector.bits[signedVector.bits.length - 1] = signedVector.factory.zero();

        // Return.
        return signedVector;
    }

    @Override
    public BigInteger getLower() {
        return BigInteger.TWO.pow(bits.length - 1).negate();
    }

    @Override
    public int getLowerInt() {
        return getLower().intValueExact();
    }

    @Override
    public BigInteger getUpper() {
        return BigInteger.TWO.pow(bits.length - 1).subtract(BigInteger.ONE);
    }

    @Override
    public int getUpperInt() {
        return getUpper().intValueExact();
    }

    @Override
    public Integer getInt() {
        // Check for enough room to represent the value.
        if (bits.length > 32) {
            throw new IllegalStateException("More than 32 bits in vector.");
        }

        // Return value.
        Long value = getLong();
        return (value == null) ? null : (int)(long)value;
    }

    @Override
    public Long getLong() {
        // Check for enough room to represent the value.
        if (bits.length > 64) {
            throw new IllegalStateException("More than 64 bits in vector.");
        }

        // Initialize all result bits with the sign bit.
        long value;
        int bitIndex = bits.length - 1;
        if (bits[bitIndex].isOne()) {
            // Negative value: set all bits to '1'.
            value = -1;
        } else if (bits[bitIndex].isZero()) {
            // Non-negative value: set all bits to '0'.
            value = 0;
        } else {
            // Not a constant value.
            return null;
        }
        bitIndex--;

        // Shift remaining bits in from the right.
        for (; bitIndex >= 0; bitIndex--) {
            if (bits[bitIndex].isOne()) {
                // Shift already-considered higher bits. Add current '1' bit.
                value = (value << 1) | 1;
            } else if (bits[bitIndex].isZero()) {
                // Shift already-considered higher bits. Nothing to add for '0' bit.
                value = (value << 1);
            } else {
                // Not a constant value.
                return null;
            }
        }

        // Return the value.
        return value;
    }

    @Override
    public void setInt(int value) {
        // Precondition check.
        if (bits.length < getMinimumLength(value)) {
            throw new IllegalArgumentException("Length is insufficient.");
        }

        // Set value.
        for (int i = 0; i < bits.length; i++) {
            bits[i].free();
            bits[i] = ((value & 0x1) != 0) ? factory.one() : factory.zero();
            value >>= 1;
        }
        Assert.check(value == 0 || value == -1, value); // All zeros or all ones.
    }

    /**
     * {@inheritDoc}
     *
     * <p>
     * For signed bit vectors, the additional (most significant) bits are set to the sign bit.
     * </p>
     *
     * @throws IllegalArgumentException If the new length is less than two.
     */
    @Override
    public void resize(int length) {
        // Optimization.
        if (length == bits.length) {
            return;
        }

        // Precondition check.
        if (length < 2) {
            throw new IllegalArgumentException("Length is less than two.");
        }

        // Allocate new bits.
        BDD[] newBits = new BDD[length];

        // Copy the common bits.
        int numberOfCommonBits = Math.min(bits.length, length);
        System.arraycopy(bits, 0, newBits, 0, numberOfCommonBits);

        // If new length is larger, set additional bits to the sign bit.
        BDD signBit = bits[bits.length - 1];
        for (int i = numberOfCommonBits; i < length; i++) {
            newBits[i] = signBit.id();
        }

        // If new length is smaller, free dropped bits.
        for (int i = numberOfCommonBits; i < bits.length; i++) {
            bits[i].free();
        }

        // Replace the bits.
        bits = newBits;
    }

    @Override
    public SignedBddBitVector shrink() {
        // Get the minimum-required length of this vector. A two's complement bit vector can be sign extended: by adding
        // additional most-significant bits that are equal to the sign bit, the vector's value(s) is/are preserved.
        // Therefore, as long as the two most-significant bits are equal, we can drop the most-significant bit. We do
        // ensure we keep the mimimum-required length of the bit vector representation into account as well.
        int length = bits.length;
        while (length > MINIMUM_LENGTH && bits[length - 1].equalsBDD(bits[length - 2])) {
            length--;
        }

        // Resize to the minimum-required length.
        resize(length);

        // Return this bit vector, for chaining.
        return this;
    }

    @Override
    public SignedBddBitVectorAndCarry negate() {
        // To negate a two's complement bit vector, we negate each bit and perform an addition with 'one'. This
        // implementation inlines the 'one' addition and merges negating the bits with the addition, for better
        // performance.

        // Compute resulting vector.
        SignedBddBitVector rslt = new SignedBddBitVector(factory, bits.length);
        BDD carry = factory.one(); // Add 'one' using the carry-in of bit 0.
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // notThisI = !this[i]
            BDD notThisI = this.bits[i].not();

            // rslt[i] = !this[i] ^ 0 ^ carry = ![this] ^ carry = notThisI ^ carry
            rslt.bits[i] = notThisI.id().xorWith(carry.id());

            // carry = (!this[i] & 0) | (carry & (!this[i] | 0)) = carry & !this[i] = carry & notThisI
            carry = carry.andWith(notThisI);

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new SignedBddBitVectorAndCarry(rslt, carry);
    }

    @Override
    public SignedBddBitVectorAndCarry abs() {
        // Compute 'this < 0'. We can use the sign bit, as it is 'true' when the value is negative and 'false' when it
        // is non-negative.
        BDD cmp = bits[bits.length - 1];

        // Compute '-this', including overflow.
        SignedBddBitVectorAndCarry negated = negate();
        BDD overflow = negated.carry;

        // Compute result vector 'if this < 0: -this else this end'.
        SignedBddBitVector resultVector = ifThenElse(cmp, negated.vector, this);
        negated.vector.free();

        // Return the result.
        return new SignedBddBitVectorAndCarry(resultVector, overflow);
    }

    @Override
    public SignedBddBitVector sign() {
        // Compute 'this < 0'. We can use the sign bit, as it is 'true' when the value is negative and 'false' when it
        // is non-negative.
        BDD isNeg = bits[bits.length - 1];

        // Compute 'this = 0'.
        BDD isZero = factory.one();
        for (BDD bit: bits) {
            isZero = isZero.andWith(bit.not());
        }

        // Compute result vector 'if this < 0: -1 elif this = 0 then 0 else 1 end'.
        SignedBddBitVector negOne = createFromInt(factory, 2, -1);
        SignedBddBitVector zero = createFromInt(factory, 2, 0);
        SignedBddBitVector posOne = createFromInt(factory, 2, 1);
        SignedBddBitVector elifResult = ifThenElse(isZero, zero, posOne);
        SignedBddBitVector result = ifThenElse(isNeg, negOne, elifResult);

        // Cleanup.
        isZero.free();
        negOne.free();
        zero.free();
        posOne.free();
        elifResult.free();

        // Return the result.
        return result;
    }

    @Override
    public SignedBddBitVectorAndCarry add(SignedBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute resulting vector.
        SignedBddBitVector rslt = new SignedBddBitVector(factory, bits.length);
        BDD carry = factory.zero();
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // rslt[i] = this[i] ^ other[i] ^ carry
            rslt.bits[i] = this.bits[i].xor(other.bits[i]).xorWith(carry.id());

            // carry = (this[i] & other[i]) | (carry & (this[i] | other[i]))
            carry = this.bits[i].and(other.bits[i]).orWith(carry.andWith(this.bits[i].or(other.bits[i])));

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new SignedBddBitVectorAndCarry(rslt, carry);
    }

    @Override
    public SignedBddBitVectorAndCarry subtract(SignedBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute resulting vector.
        SignedBddBitVector rslt = new SignedBddBitVector(factory, bits.length);
        BDD carry = factory.zero();
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // rslt[i] = this[i] ^ other[i] ^ carry
            rslt.bits[i] = this.bits[i].xor(other.bits[i]).xorWith(carry.id());

            // carry = (this[n] & other[n] & carry) | (!this[n] & (other[n] | carry))
            BDD tmp1 = other.bits[i].or(carry);
            BDD tmp2 = this.bits[i].apply(tmp1, BDDFactory.less);
            tmp1.free();
            carry = this.bits[i].and(other.bits[i]).andWith(carry).orWith(tmp2);

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new SignedBddBitVectorAndCarry(rslt, carry);
    }

    @Override
    public SignedBddBitVectorAndCarry multiply(SignedBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Sign-extend both vectors.
        int length = this.bits.length;
        int doubleLength = length * 2;
        SignedBddBitVector left = this.copy();
        SignedBddBitVector right = other.copy();
        left.resize(doubleLength);
        right.resize(doubleLength);

        // Perform multiplication.
        SignedBddBitVector result = create(factory, doubleLength, false);

        for (int i = 0; i < doubleLength; i++) {
            SignedBddBitVectorAndCarry added = result.add(left);

            for (int j = 0; j < doubleLength; j++) {
                BDD bit = result.bits[j];
                result.bits[j] = right.bits[i].ite(added.vector.bits[j], bit);
                bit.free();
            }

            added.vector.free();
            added.carry.free();

            SignedBddBitVector oldLeft = left;
            left = left.shiftLeft(1, factory.zero());
            oldLeft.free();
        }

        left.free();
        right.free();

        // Compute overflow. The upper 'length' bits must all be sign-extended from the sign bit of the lower 'length'
        // bits, for there to be no overflow. If any of the upper 'length' bits is different from that sign bit, it
        // indicates overflow.
        BDD signBit = result.bits[length - 1];
        BDD overflow = factory.zero();
        for (int i = length; i < doubleLength; i++) {
            overflow = overflow.orWith(result.bits[i].xor(signBit));
        }

        // Truncate to the right length.
        result.resize(length);

        // Return the result.
        return new SignedBddBitVectorAndCarry(result, overflow);
    }

    /**
     * {@inheritDoc}
     *
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the highest bit of the {@link #abs absolute value} of this vector is not
     *     'false', and the highest two bits of the divisor are not 'false' (based on the length of this vector).
     */
    @Override
    public SignedBddBitVector div(int divisor) {
        return divmod(divisor, true);
    }

    /**
     * {@inheritDoc}
     *
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalStateException If the highest bit of the {@link #abs absolute value} of this bit vector is not
     *     'false'.
     */
    @Override
    public SignedBddBitVector mod(int divisor) {
        return divmod(divisor, false);
    }

    /**
     * Computes the quotient ('div' result) or remainder ('mod' result) of dividing this vector (the dividend) by the
     * given value (the divisor). This operation returns a new bit vector. The bit vector on which the operation is
     * performed is neither modified nor {@link #free freed}.
     *
     * @param divisor The value by which to divide this bit vector.
     * @param isDiv Whether to compute and return the quotient/'div' ({@code true}) or remainder/'mod' ({@code false}).
     * @return The quotient ('div' result) or remainder ('mod' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the quotient/'div' is computed, and the highest bit of the {@link #abs
     *     absolute value} of this vector is not 'false', and the highest two bits of the divisor are not 'false' (based
     *     on the length of this vector).
     * @throws IllegalStateException If the remainder/'mod' is computed, and the highest bit of the {@link #abs absolute
     *     value} of this bit vector is not 'false'.
     */
    private SignedBddBitVector divmod(int divisor, boolean isDiv) {
        // Get absolute value.
        SignedBddBitVectorAndCarry absThis = abs();
        absThis.carry.free();

        // Compute unsigned 'div' or 'mod' result on absolute value.
        SignedBddBitVector resultUnsigned = absThis.vector.divmodUnsigned(divisor, isDiv);
        absThis.vector.free();

        // Compute signed 'div' or 'mod' result: 'if signBit: -resultUnsigned else resultUnsigned end'.
        BDD signBit = bits[bits.length - 1];
        SignedBddBitVectorAndCarry resultUnsignedNeg = resultUnsigned.negate();
        SignedBddBitVector resultSigned = ifThenElse(signBit, resultUnsignedNeg.vector, resultUnsigned);
        resultUnsigned.free();
        resultUnsignedNeg.vector.free();
        resultUnsignedNeg.carry.free();

        // Return signed 'div' or 'mod' result.
        return resultSigned;
    }

    /**
     * Computes the quotient ('div' result) or remainder ('mod' result) of dividing this vector (the dividend) by the
     * given value (the divisor), considering the bit vectors as being unsigned. This operation returns a new bit
     * vector. The bit vector on which the operation is performed is neither modified nor {@link #free freed}.
     *
     * @param divisorValue The value by which to divide this bit vector.
     * @param isDiv Whether to compute and return the quotient/'div' ({@code true}) or remainder/'mod' ({@code false}).
     * @return The quotient ('div' result) or remainder ('mod' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the quotient/'div' is computed, and the highest bit of this bit vector is not
     *     'false', and the highest two bits of the divisor are not 'false' (based on the length of this vector).
     * @throws IllegalStateException If the remainder/'mod' is computed, and the highest bit of this bit vector is not
     *     'false'.
     */
    private SignedBddBitVector divmodUnsigned(int divisorValue, boolean isDiv) {
        // Precondition checks.
        if (divisorValue <= 0) {
            throw new IllegalArgumentException("Divisor is not positive.");
        }
        if (bits.length < getMinimumLength(divisorValue)) {
            throw new IllegalArgumentException("Divisor doesn't fit.");
        }

        // Get divisor vector.
        SignedBddBitVector divisor = createFromInt(factory, bits.length, divisorValue);

        // More precondition checks.
        if (isDiv && !bits[bits.length - 1].isZero()
                && (!divisor.bits[divisor.length() - 1].isZero() || !divisor.bits[divisor.length() - 2].isZero()))
        {
            throw new IllegalArgumentException("Computing the quotient/'div', the highest bit of the dividend's "
                    + "absolute value is not 'false', and the highest two bits of the divisor are not both "
                    + "'false'.");
        }
        if (!isDiv && !bits[bits.length - 1].isZero()) {
            throw new IllegalStateException("Computing the remainder/'mod', and the highest bit of the dividend's "
                    + "absolute value is not 'false'.");
        }

        // Create result vectors.
        SignedBddBitVector quotient = shiftLeft(1, factory.zero());
        SignedBddBitVector remainderZero = create(factory, bits.length);
        SignedBddBitVector remainder = remainderZero.shiftLeft(1, bits[bits.length - 1]);
        remainderZero.free();

        // Compute result.
        divModUnsignedRecursive(divisor, quotient, remainder, bits.length);
        divisor.free();

        // Return requested result.
        if (isDiv) {
            remainder.free();
            return quotient;
        } else {
            quotient.free();
            SignedBddBitVector shiftedRemainder = remainder.shiftRight(1, factory.zero());
            remainder.free();
            return shiftedRemainder;
        }
    }

    /**
     * Computes the quotient ('div' result) and remainder ('mod' result) of dividing a bit vector (the dividend) by the
     * given other bit vector (the divisor), considering the bit vectors as being unsigned.
     *
     * @param divisor The divisor bit vector. Is not modified.
     * @param quotient The quotient/'div' bit vector, as computed so far. Is modified in-place.
     * @param remainder The remainder/'mod' bit vector, as computed so far. Is modified in-place.
     * @param step The number of steps to perform.
     */
    private void divModUnsignedRecursive(SignedBddBitVector divisor, SignedBddBitVector quotient,
            SignedBddBitVector remainder, int step)
    {
        int divLen = divisor.bits.length;
        BDD isSmaller = divisor.lessOrEqual(remainder);
        SignedBddBitVector newQuotient = quotient.shiftLeft(1, isSmaller);

        SignedBddBitVector sub = create(factory, divLen);
        for (int i = 0; i < divLen; i++) {
            sub.bits[i] = isSmaller.ite(divisor.bits[i], factory.zero());
        }

        SignedBddBitVectorAndCarry tmp = remainder.subtract(sub);
        SignedBddBitVector newRemainder = tmp.vector.shiftLeft(1, quotient.bits[divLen - 1]);

        if (step > 1) {
            divModUnsignedRecursive(divisor, newQuotient, newRemainder, step - 1);
        }

        tmp.vector.free();
        tmp.carry.free();
        sub.free();
        isSmaller.free();

        quotient.replaceBy(newQuotient);
        remainder.replaceBy(newRemainder);
    }

    @Override
    public BDD lessThan(SignedBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute carry.
        BDD carry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // carry = (!this[i] & other[i]) | biimp(this[i], other[i]) & carry
            BDD lt = this.bits[i].apply(other.bits[i], BDDFactory.less);
            BDD eq = this.bits[i].biimp(other.bits[i]);
            carry = lt.orWith(eq.andWith(carry));
        }

        // Compute result: (this[n-1] & !other[n-1]) | ((this[n-1] | !other[n-1]) & carry)
        BDD thisSignBit = this.bits[this.bits.length - 1];
        BDD otherSignBit = other.bits[other.bits.length - 1];
        BDD lhs = thisSignBit.id().andWith(otherSignBit.not());
        BDD rhs = thisSignBit.id().orWith(otherSignBit.not()).andWith(carry);
        return lhs.orWith(rhs);
    }

    @Override
    public BDD lessOrEqual(SignedBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute carry.
        BDD carry = factory.one();
        for (int i = 0; i < bits.length; i++) {
            // carry = (!this[i] & other[i]) | biimp(this[i], other[i]) & carry
            BDD lt = this.bits[i].apply(other.bits[i], BDDFactory.less);
            BDD eq = this.bits[i].biimp(other.bits[i]);
            carry = lt.orWith(eq.andWith(carry));
        }

        // Compute result: (this[n-1] & !other[n-1]) | ((this[n-1] | !other[n-1]) & carry)
        BDD thisSignBit = this.bits[this.bits.length - 1];
        BDD otherSignBit = other.bits[other.bits.length - 1];
        BDD lhs = thisSignBit.id().andWith(otherSignBit.not());
        BDD rhs = thisSignBit.id().orWith(otherSignBit.not()).andWith(carry);
        return lhs.orWith(rhs);
    }

    @Override
    public SignedBddBitVector min(SignedBddBitVector other) {
        // Compute 'this <= other'.
        BDD cmp = this.lessOrEqual(other);

        // Compute result: 'if this <= other: this else other end'.
        SignedBddBitVector result = ifThenElse(cmp, this, other);

        // Cleanup.
        cmp.free();

        // Return the result.
        return result;
    }

    @Override
    public SignedBddBitVector max(SignedBddBitVector other) {
        // Compute 'this >= other'.
        BDD cmp = this.greaterOrEqual(other);

        // Compute result: 'if this >= other: this else other end'.
        SignedBddBitVector result = ifThenElse(cmp, this, other);

        // Cleanup.
        cmp.free();

        // Return the result.
        return result;
    }
}
