/*******************************************************************************
 * Copyright (c) 2000, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.dltk.internal.testing.ui;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.dltk.core.ISourceRange;
import org.eclipse.dltk.internal.testing.model.TestElement;
import org.eclipse.dltk.internal.testing.util.LegacyTestElementResolver;
import org.eclipse.dltk.testing.DLTKTestingMessages;
import org.eclipse.dltk.testing.ITestElementResolver;
import org.eclipse.dltk.testing.TestElementResolution;
import org.eclipse.dltk.ui.DLTKUIPlugin;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * Abstract Action for opening a Java editor.
 */
public class OpenTestAction extends Action {
	private final TestElement fTestElement;
	protected TestRunnerViewPart fTestRunner;
	private final boolean fActivate;

	protected OpenTestAction(TestRunnerViewPart testRunner,
			TestElement testElement) {
		this(testRunner, testElement, true);
	}

	public OpenTestAction(TestRunnerViewPart testRunner,
			TestElement testElement, boolean activate) {
		super(DLTKTestingMessages.OpenEditorAction_action_label);
		fTestElement = testElement;
		fTestRunner = testRunner;
		fActivate = activate;
		PlatformUI.getWorkbench().getHelpSystem().setHelp(this,
				IDLTKTestingHelpContextIds.OPENTEST_ACTION);
	}

	/*
	 * @see IAction#run()
	 */
	public void run() {
		final ITestElementResolver resolver = acquireResolver();
		if (resolver == null) {
			return;
		}
		final TestElementResolution resolution = resolver
				.resolveElement(fTestElement);
		if (resolution == null) {
			MessageDialog
					.openError(
							getShell(),
							DLTKTestingMessages.OpenEditorAction_error_cannotopen_title,
							DLTKTestingMessages.OpenEditorAction_error_cannotopen_message);
			return;
		}
		try {
			final ITextEditor textEditor = (ITextEditor) DLTKUIPlugin
					.openInEditor(resolution.getElement(), fActivate, false);
			if (textEditor != null) {
				final ISourceRange range = resolution.getRange();
				if (range != null) {
					textEditor.selectAndReveal(range.getOffset(), range
							.getLength());
				}
			} else {
				fTestRunner
						.registerInfoMessage(DLTKTestingMessages.OpenEditorAction_message_cannotopen);
			}
		} catch (CoreException e) {
			ErrorDialog.openError(getShell(),
					DLTKTestingMessages.OpenEditorAction_error_dialog_title,
					DLTKTestingMessages.OpenEditorAction_error_dialog_message,
					e.getStatus());
		}
	}

	/**
	 * @return
	 */
	private ITestElementResolver acquireResolver() {
		final ITestElementResolver resolver = (ITestElementResolver) fTestRunner
				.getTestRunnerUI().getAdapter(ITestElementResolver.class);
		if (resolver != null) {
			return resolver;
		} else {
			final ILaunchConfiguration launchConfiguration = fTestRunner
					.getLaunch().getLaunchConfiguration();
			if (launchConfiguration == null) {
				return null;
			}
			return new LegacyTestElementResolver(getLaunchedProject(),
					launchConfiguration);
		}
	}

	protected Shell getShell() {
		return fTestRunner.getSite().getShell();
	}

	/**
	 * @return the project, or <code>null</code>
	 */
	protected IScriptProject getLaunchedProject() {
		return fTestRunner.getLaunchedProject();
	}

	protected TestElement getTestElement() {
		return fTestElement;
	}

}
