/* This is -*- C -*- */
/* $Id: core.c,v 1.12 2001/05/06 07:56:19 trow Exp $ */

/*
 * guppi-seq-scalar-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include <guppi-data-impl-plug-in.h>
#include "core.h"
#include "scm-seq-scalar-core.h"

#ifdef HAVE_PYTHON
void python_seq_scalar_core_init (void);
#endif

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_scalar_core_impl_get_arg (GtkObject * obj, GtkArg * arg,
				    guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_core_impl_set_arg (GtkObject * obj, GtkArg * arg,
				    guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_core_impl_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_seq_scalar_core_impl_finalize (GtkObject * obj)
{
  GuppiSeqScalarCoreImpl *impl = GUPPI_SEQ_SCALAR_CORE_IMPL (obj);

  guppi_unref0 (impl->garray);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/**************************************************************************/

static double
v_seq_scalar_get (const GuppiSeqScalarImpl * impl, gint i)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  const double *data;

  data = (const double *) guppi_garray_data (core->garray);
  i -= core->index_basis;
  return data[i];
}

static void
v_seq_scalar_set (GuppiSeqScalarImpl * impl, gint i, double x)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double *data;

  data = (double *) guppi_garray_data (core->garray);
  i -= core->index_basis;

  data[i] = x;

  guppi_seq_impl_set_missing (GUPPI_SEQ_IMPL (impl), i + core->index_basis,
			      FALSE);
}

static void
v_seq_scalar_set_many (GuppiSeqScalarImpl * impl, gint i,
		       gconstpointer ptr, gint stride, gsize N)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double *data;
  const guchar *byte_ptr;
  gint j;

  if (core->size == 0)
    core->index_basis = i;

  i -= core->index_basis;

  if (guppi_garray_size (core->garray) < (i + N - 1))
    guppi_garray_set_size (core->garray, i + N - 1);

  data = (double *) guppi_garray_data (core->garray);

  byte_ptr = (const guchar *) ptr;
  for (j = 0; j < N; ++j) {
    double x = *(const double *) byte_ptr;
    data[i + j] = x;
    byte_ptr += stride;
  }

  guppi_seq_impl_set_many_missing (GUPPI_SEQ_IMPL (impl), i, N, FALSE);
}

static void
v_seq_scalar_insert (GuppiSeqScalarImpl * impl, gint i, double x)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double *data;
  gint j;

  if (core->size == 0)
    core->index_basis = i;

  i -= core->index_basis;

  if (guppi_garray_size (core->garray) <= core->size)
    guppi_garray_set_size (core->garray, MAX (20, 2 * core->size));

  data = (double *) guppi_garray_data (core->garray);
  for (j = core->size - 1; i <= j; --j)
    data[j + 1] = data[j];
  data[i] = x;
  ++core->size;

  guppi_seq_impl_insert_missing (GUPPI_SEQ_IMPL (impl), i + core->index_basis,
				 FALSE, 1);
}

static void
v_seq_scalar_insert_many (GuppiSeqScalarImpl * impl, gint i,
			  gconstpointer ptr, gint stride, gsize N)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double *data;
  const guchar *byte_ptr;
  gint j;

  if (core->size == 0)
    core->index_basis = i;

  i -= core->index_basis;

  if (guppi_garray_size (core->garray) < core->size + N)
    guppi_garray_set_size (core->garray,
			   MAX (MAX (20, 2 * core->size),
				core->size + 2 * N));

  data = (double *) guppi_garray_data (core->garray);

  for (j = core->size - 1; i <= j; --j)
    data[j + N] = data[j];

  byte_ptr = (const guchar *) ptr;
  for (j = 0; j < N; ++j) {
    double x = *(const double *) byte_ptr;
    data[i + j] = x;
    byte_ptr += stride;
  }

  core->size += N;

  guppi_seq_impl_insert_missing (GUPPI_SEQ_IMPL (impl), i + core->index_basis,
				 FALSE, N);
}

static gconstpointer
v_seq_scalar_raw (const GuppiSeqScalarImpl * impl, gint * stride)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  gint8 *p;

  p = (gint8 *) guppi_garray_data (core->garray);
  p -= sizeof (double) * core->index_basis;
  *stride = sizeof (double);

  return (gconstpointer) p;
}


static void
v_seq_size_hint (GuppiSeqImpl * impl, gsize n)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);

  if (guppi_garray_size (core->garray) < n)
    guppi_garray_set_size (core->garray, n);
}

static void
v_seq_get_bounds (const GuppiSeqImpl * impl, gint * min, gint * max)
{
  const GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);

  if (min)
    *min = core->index_basis;

  if (max)
    *max = core->index_basis - 1 + core->size;

}

static void
v_seq_shift_indices (GuppiSeqImpl * impl, gint delta)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  core->index_basis += delta;
}

static void
v_seq_insert_missing (GuppiSeqImpl * impl, gint i)
{
  v_seq_scalar_insert (GUPPI_SEQ_SCALAR_IMPL (impl), i, 0);
  guppi_seq_impl_set_missing (impl, i, TRUE);
}

static void
v_seq_delete_many (GuppiSeqImpl * impl, gint i, gsize N)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double *data;
  gint j;

  data = (double *) guppi_garray_data (core->garray);
  i -= core->index_basis;

  for (j = i; j + N < core->size; ++j)
    data[j] = data[j + N];

  core->size -= N;

  guppi_seq_impl_delete_missing (impl, i + core->index_basis, N);
}

static void
v_seq_grow_to_include (GuppiSeqImpl * impl, gint j0, gint j1)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  double default_value = 0;

  if (core->size == 0) {
    v_seq_scalar_insert_many (GUPPI_SEQ_SCALAR_IMPL (impl),
			      0, &default_value, 0, j1 - j0 + 1);
    core->index_basis = j0;
    return;
  }

  /* Grow to the front */
  if (j0 < core->index_basis) {
    v_seq_scalar_insert_many (GUPPI_SEQ_SCALAR_IMPL (impl), core->index_basis,
			      &default_value, 0, core->index_basis - j0);
    core->index_basis = j0;
  }

  /* Grow the back */
  if (j1 >= core->index_basis + (gint) core->size) {
    v_seq_scalar_insert_many (GUPPI_SEQ_SCALAR_IMPL (impl),
			      core->index_basis + core->size,
			      &default_value, 0,
			      1 + j1 - (gint) core->size + core->index_basis);
  }
}

static GuppiDataImpl *
v_data_copy (GuppiDataImpl * impl)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  GuppiSeqScalarCoreImpl *copy;
  gint i;
  double *data;
  double *copy_data;

  copy =
    GUPPI_SEQ_SCALAR_CORE_IMPL (guppi_type_new
				(GUPPI_TYPE_SEQ_SCALAR_CORE_IMPL));

  copy->index_basis = core->index_basis;
  copy->size = core->size;
  guppi_garray_set_size (copy->garray, copy->size);

  data = (double *) guppi_garray_data (core->garray);
  copy_data = (double *) guppi_garray_data (copy->garray);
  for (i = 0; i < core->size; ++i)
    copy_data[i] = data[i];

  return GUPPI_DATA_IMPL (copy);
}

static gint
v_data_size_in_bytes (GuppiDataImpl * impl)
{
  GuppiSeqScalarCoreImpl *core = GUPPI_SEQ_SCALAR_CORE_IMPL (impl);
  return guppi_garray_size (core->garray) * sizeof (double) +
    sizeof (GuppiSeqScalarCoreImpl);
}


/**************************************************************************/

static void
guppi_seq_scalar_core_impl_class_init (GuppiSeqScalarCoreImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiSeqImplClass *seq_class = GUPPI_SEQ_IMPL_CLASS (klass);
  GuppiSeqScalarImplClass *seq_scalar_class =
    GUPPI_SEQ_SCALAR_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ_SCALAR_IMPL);

  object_class->get_arg = guppi_seq_scalar_core_impl_get_arg;
  object_class->set_arg = guppi_seq_scalar_core_impl_set_arg;
  object_class->destroy = guppi_seq_scalar_core_impl_destroy;
  object_class->finalize = guppi_seq_scalar_core_impl_finalize;

  data_class->impl_name = _("Core Scalar Sequence");

  seq_scalar_class->get = v_seq_scalar_get;
  seq_scalar_class->set = v_seq_scalar_set;
  seq_scalar_class->set_many = v_seq_scalar_set_many;
  seq_scalar_class->insert = v_seq_scalar_insert;
  seq_scalar_class->insert_many = v_seq_scalar_insert_many;

  seq_scalar_class->raw_access = v_seq_scalar_raw;

  seq_class->size_hint = v_seq_size_hint;
  seq_class->get_bounds = v_seq_get_bounds;
  seq_class->shift_indices = v_seq_shift_indices;
  seq_class->insert_missing = v_seq_insert_missing;

  seq_class->delete_many = v_seq_delete_many;
  seq_class->grow_to_include = v_seq_grow_to_include;

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_size_in_bytes;
}

static void
guppi_seq_scalar_core_impl_init (GuppiSeqScalarCoreImpl * obj)
{
  obj->index_basis = 0;
  obj->size = 0;
  obj->garray = guppi_garray_new (sizeof (double));
}

GtkType guppi_seq_scalar_core_impl_get_type (void)
{
  static GtkType guppi_seq_scalar_core_impl_type = 0;
  if (!guppi_seq_scalar_core_impl_type) {
    static const GtkTypeInfo guppi_seq_scalar_core_impl_info = {
      "GuppiSeqScalarCoreImpl",
      sizeof (GuppiSeqScalarCoreImpl),
      sizeof (GuppiSeqScalarCoreImplClass),
      (GtkClassInitFunc) guppi_seq_scalar_core_impl_class_init,
      (GtkObjectInitFunc) guppi_seq_scalar_core_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_scalar_core_impl_type =
      gtk_type_unique (GUPPI_TYPE_SEQ_SCALAR_IMPL,
		       &guppi_seq_scalar_core_impl_info);
  }
  return guppi_seq_scalar_core_impl_type;
}


/**************************************************************************/

static GuppiDataImpl *
make_impl (void)
{
  return
    GUPPI_DATA_IMPL (guppi_type_new (guppi_seq_scalar_core_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  scm_seq_scalar_core_init ();

#ifdef HAVE_PYTHON
  python_seq_scalar_core_init ();
#endif

  return pi;
}



/* $Id: core.c,v 1.12 2001/05/06 07:56:19 trow Exp $ */
