/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-barchart-register.c
 *
 * Copyright (C) 2000 Helix Code, Inc.
 *
 * Developed by Jody Goldberg <jgoldberg@home.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include "guppi-chart-type.h"
#include "guppi-barchart-register.h"
#include "guppi-XLplots-common.h"
#include "bar.xpm"
#include "column.xpm"

#include <guppi-seq-style.h>
#include <guppi-useful.h>
#include <guppi-root-group-state.h>
#include <guppi-root-group-view.h>

typedef enum
{
  SHAPE_SQUARE,
  SHAPE_CUBE,
  SHAPE_TUBE,
  SHAPE_CONE,
  SHAPE_PYRAMID
} BarShape;

typedef enum
{
  TOTAL_NONE,
  TOTAL_STACK,
  TOTAL_PERCENTAGE
} BarTotals;

typedef struct
{
  gboolean	horizontal;
  BarShape	shape;
  BarTotals	totals;
} BarChartDescriptor;

static GuppiRootGroupView *
guppi_barchart_factory (GuppiMinorChartType const *t,
			GuppiSeqData   *series,
			GuppiSeqData   *labels,
			GuppiSeqString *series_names)
{
  BarChartDescriptor const *desc = t->user_data;
  gboolean const is_horizontal = desc->horizontal;
  gboolean const is_stacked = (desc->totals >= TOTAL_STACK);
  gboolean const as_percentage = (desc->totals >= TOTAL_PERCENTAGE);

  GuppiSeqStyle  *styles;

  /* State Objects */
  GuppiElementState* bar_state, *legend_state;

  /* View Objects */
  GuppiElementView* bar_view, *legend_view;

  GuppiElementState* root_state;
  GuppiGroupView* root_view;

  root_state = guppi_root_group_state_new ();
  root_view = GUPPI_GROUP_VIEW (guppi_element_state_make_view (root_state));

  /* Use default styles */
  styles = xlplots_stock_styles (guppi_seq_size (GUPPI_SEQ (series)));

  /* Build our state objects */
  bar_state = guppi_element_state_new("barchart",
				      "data", 			series,
				      "horizontal_bars",	is_horizontal,
				      "stacked",	  	is_stacked,
				      "normalize_stacks",	as_percentage,
				      "use_stock_colors",	0,

				      /* Ideally this would be 0 for an
				       * un-anti-aliased un scale hair width
				       * line */
				      "edge_thickness",		.5,

				      "column_styles",		styles,
				      "column_labels",		series_names,

				      /* What is this measured in ? */
				      "cluster_margin",		0.3,
				      NULL);

  legend_state = xlplots_stock_legend ();
  guppi_element_state_changed_delayed(GUPPI_ELEMENT_STATE(bar_state));
  guppi_element_state_connect(bar_state, SHARED_LABEL_DATA,
			      legend_state, SHARED_LABEL_DATA);
  guppi_element_state_connect(bar_state, SHARED_STYLE_DATA,
			      legend_state, SHARED_STYLE_DATA);

  /* Build our views */
  bar_view = guppi_element_state_make_view(bar_state);
  legend_view = guppi_element_state_make_view(legend_state);

  /* Do our layout */
  guppi_group_view_layout_flush_top(root_view, bar_view, 7.2);
  guppi_group_view_layout_flush_left(root_view, bar_view, 7.2);
  guppi_group_view_layout_flush_bottom(root_view, bar_view, 7.2);

  guppi_group_view_layout_horizontally_adjacent(root_view, bar_view,
						legend_view, 7.2);
  guppi_group_view_layout_natural_width(root_view, legend_view);
  guppi_group_view_layout_natural_height(root_view, legend_view);
  guppi_group_view_layout_flush_right(root_view, legend_view, 7.2);
  guppi_group_view_layout_same_y_center(root_view, bar_view, legend_view);

  guppi_group_view_commit_all_changes(root_view);

  guppi_root_group_view_set_size(GUPPI_ROOT_GROUP_VIEW(root_view),
				 6*72, 4.5*72);

  guppi_unref(root_state);

  return GUPPI_ROOT_GROUP_VIEW(root_view);
}

void
guppi_barchart_register ()
{
  GuppiMajorChartType *bar_types =
      guppi_major_chart_type_register (_("Bar"), bar_icon,
				       GINT_TO_POINTER (FALSE));
  GuppiMajorChartType *column_types =
      guppi_major_chart_type_register ( _("Column"), column_icon,
				       GINT_TO_POINTER (TRUE));

  /* Bar Charts */
  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_SQUARE, TOTAL_NONE };
    guppi_minor_chart_type_register (bar_types,
      _("Adjacent Bars"), "chart_bar_1_1.png",
      _("Adjacent horizontal bars grouped by major and minor categories"),
      1, 1, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_SQUARE, TOTAL_STACK };
    guppi_minor_chart_type_register (bar_types,
      _("Stacked Bars"), "chart_bar_1_2.png",
      _("Minor categories stacked in horizontal bars that are grouped "
	"by major category"),
      1, 2, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_SQUARE, TOTAL_PERCENTAGE };
    guppi_minor_chart_type_register (bar_types,
      _("Percentage Bars"), "chart_bar_1_3.png",
      _("Minor categories stacked as percentages of the minor total, "
	"in horizontal bars, grouped by major category"),
      1, 3, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_CUBE, TOTAL_NONE };
    guppi_minor_chart_type_register (bar_types,
      _("3D Adjacent Bars"), "chart_bar_2_1.png",
      _("Adjacent horizontal 3D bars grouped by major and minor categories"),
      2, 1, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_CUBE, TOTAL_STACK };
    guppi_minor_chart_type_register (bar_types,
      _("3D Stacked Bars"), "chart_bar_2_2.png",
      _("Minor categories stacked in horizontal 3D bars, grouped "
	"by major category"),
      2, 2, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { TRUE, SHAPE_CUBE, TOTAL_PERCENTAGE };
    guppi_minor_chart_type_register (bar_types,
      _("3D Percentage Bars"), "chart_bar_2_3.png",
      _("Minor categories stacked as percentages of the minor total, "
	"in 3D horizontal bars, grouped by major category"),
      2, 3, &guppi_barchart_factory, &desc);
  }

  /* Column Charts */
  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_SQUARE, TOTAL_NONE };
    guppi_minor_chart_type_register (column_types,
      _("Adjacent Columns"), "chart_column_1_1.png",
      _("Adjacent vertical columns grouped by major and minor category"),
      1, 1, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_SQUARE, TOTAL_STACK };
    guppi_minor_chart_type_register (column_types,
      _("Stacked Columns"), "chart_column_1_2.png",
      _("Minor categories stacked in vertical columns that are grouped "
	"by major category"),
      1, 2, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_SQUARE, TOTAL_PERCENTAGE };
    guppi_minor_chart_type_register (column_types,
      _("Percentage Columns"), "chart_column_1_3.png",
      _("Minor categories stacked as percentages of the minor total, "
	"in vertical columns, grouped by major category"),
      1, 3, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_CUBE, TOTAL_NONE };
    guppi_minor_chart_type_register (column_types,
      _("3D Adjacent Columns"), "chart_column_2_1.png",
      _("Adjacent vertical 3D columns grouped by major and minor category"),
      2, 1, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_CUBE, TOTAL_STACK };
    guppi_minor_chart_type_register (column_types,
      _("3D Stacked Columns"), "chart_column_2_2.png",
      _("Minor categories stacked in vertical 3D columns, grouped "
	"by major category"),
      2, 2, &guppi_barchart_factory, &desc);
  }

  {
    static BarChartDescriptor const desc = { FALSE, SHAPE_CUBE, TOTAL_PERCENTAGE };
    guppi_minor_chart_type_register (column_types,
      _("3D Percentage Columns"), "chart_column_2_3.png",
      _("Minor categories stacked as percentages of the minor total, "
	"in 3D vertical bars, grouped by major category"),
      2, 3, &guppi_barchart_factory, &desc);
  }
}
