/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-seq-point.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-memory.h>
#include "guppi-seq-point.h"
#include "guppi-seq-point-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_point_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_point_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_point_finalize (GtkObject *obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_seq_point_class_init (GuppiSeqPointClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GuppiSeqClass *seq_class = GUPPI_SEQ_CLASS (klass);

  parent_class = gtk_type_class(GUPPI_TYPE_SEQ);

  object_class->get_arg = guppi_seq_point_get_arg;
  object_class->set_arg = guppi_seq_point_set_arg;
  object_class->finalize = guppi_seq_point_finalize;
}

static void
guppi_seq_point_init (GuppiSeqPoint *obj)
{

}

GtkType
guppi_seq_point_get_type (void)
{
  static GtkType guppi_seq_point_type = 0;
  if (!guppi_seq_point_type) {
    static const GtkTypeInfo guppi_seq_point_info = {
      "GuppiSeqPoint",
      sizeof(GuppiSeqPoint),
      sizeof(GuppiSeqPointClass),
      (GtkClassInitFunc)guppi_seq_point_class_init,
      (GtkObjectInitFunc)guppi_seq_point_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_point_type = gtk_type_unique (GUPPI_TYPE_SEQ, &guppi_seq_point_info);
  }
  return guppi_seq_point_type;
}

GuppiData *
guppi_seq_point_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_POINT, NULL, 0, NULL);
}

/***************************************************************************/

typedef struct _GuppiDataOp_Point GuppiDataOp_Point;
struct _GuppiDataOp_Point {
  GuppiDataOp op;
  
  gint i;
  gsize N;

  double x, y;
};

static void
op_set (GuppiData *d, GuppiDataOp *op)
{
  GuppiSeqPoint *sp = GUPPI_SEQ_POINT (d);
  GuppiDataOp_Point *point_op = (GuppiDataOp_Point *) op;

  gboolean was_missing;
  double old_x = 0, old_y = 0;
  double x = point_op->x, y = point_op->y;
  gint i = point_op->i;
  
  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  was_missing = guppi_seq_missing (GUPPI_SEQ (d), i);
  if (!was_missing)
    guppi_seq_point_get (sp, i, &old_x, &old_y);

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  impl_class->set (impl, i, x, y);
  
  /* Don't use old values right now. */
}

static void
op_insert (GuppiData *d, GuppiDataOp *op)
{
  GuppiDataOp_Point *point_op = (GuppiDataOp_Point *)op;

  double x = point_op->x, y = point_op->y;
  gint i = point_op->i;

  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert);

  impl_class->insert (impl, i, x, y);
}

/***************************************************************************/

void
guppi_seq_point_get (GuppiSeqPoint *seq, gint i, double *x, double *y)
{
  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT(impl)->klass);

  g_assert (impl_class->get);
  impl_class->get (impl, i, x, y);
}

double
guppi_seq_point_get_x (GuppiSeqPoint *seq, gint i)
{
  double x;
  guppi_seq_point_get (seq, i, &x, NULL);
  return x;
}

double
guppi_seq_point_get_y (GuppiSeqPoint *seq, gint i)
{
  double y;
  guppi_seq_point_get (seq, i, NULL, &y);
  return y;
}

void
guppi_seq_point_set (GuppiSeqPoint *seq, gint i, double x, double y)
{
  GuppiDataOp_Point op;
  double old_x = 0, old_y = 0;
  gboolean was_missing;
  
  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  was_missing = guppi_seq_missing (GUPPI_SEQ (seq), i);
  if (!was_missing) 
    guppi_seq_point_get (seq, i, &old_x, &old_y);

  if (was_missing || x != old_x || y != old_y) {

    op.op.op = op_set;
    op.i = i;
    op.x = x;
    op.y = y;

    guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) &op);
  }
}

void
guppi_seq_point_set_x (GuppiSeqPoint *seq, gint i, double x)
{
  GuppiDataOp_Point op;
  double old_x = 0, old_y = 0;
  gboolean was_missing;
  
  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  was_missing = guppi_seq_missing (GUPPI_SEQ (seq), i);
  if (!was_missing) 
    guppi_seq_point_get (seq, i, &old_x, &old_y);

  if (was_missing || x != old_x) {

    op.op.op = op_set;
    op.i = i;
    op.x = x;
    op.y = old_y;

    guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) &op);
  }
}

void
guppi_seq_point_set_y (GuppiSeqPoint *seq, gint i, double y)
{
  GuppiDataOp_Point op;
  double old_x = 0, old_y = 0;
  gboolean was_missing;
  
  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  was_missing = guppi_seq_missing (GUPPI_SEQ (seq), i);
  if (!was_missing) 
    guppi_seq_point_get (seq, i, &old_x, &old_y);

  if (was_missing || y != old_y) {

    op.op.op = op_set;
    op.i = i;
    op.x = old_x;
    op.y = y;

    guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) &op);
  }
}

void
guppi_seq_point_prepend (GuppiSeqPoint *seq, double x, double y)
{
  gint first;

  first = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_point_insert (seq, first, x, y);
}

void
guppi_seq_point_append (GuppiSeqPoint *seq, double x, double y)
{
  gint last;
  last = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_point_insert (seq, last + 1, x, y);
}

void
guppi_seq_point_insert (GuppiSeqPoint *seq, gint i, double x, double y)
{
  GuppiDataOp_Point op;

  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_insert;
  op.i = i;
  op.x = x;
  op.y = y;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, 1, (GuppiDataOp *) &op);
}

void
guppi_seq_point_get_many (GuppiSeqPoint *seq, gint i0, gint i1,
			  double *x_data, double *y_data)
{
  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  g_return_if_fail (seq && GUPPI_IS_SEQ_POINT (seq));

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->get_many) {

    impl_class->get_many (impl, i0, i1, x_data, y_data);

  } else {
    gint i, j = 0;

    g_assert (impl_class->get);
    for (i = i0; i <= i1; ++i) {
      impl_class->get (impl, i,
		       x_data ? x_data + j : NULL, y_data ? y_data + j : NULL);
      ++j;
    }
  }

}

gconstpointer
guppi_seq_point_raw_x (GuppiSeqPoint *seq, gint *stride)
{
  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  g_return_val_if_fail (seq && GUPPI_IS_SEQ_POINT (seq), NULL);
  g_return_val_if_fail (stride, NULL);

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  return impl_class->raw_x ? impl_class->raw_x (impl, stride) : NULL;
}

gconstpointer
guppi_seq_point_raw_y (GuppiSeqPoint *seq, gint *stride)
{
  GuppiSeqPointImpl *impl;
  GuppiSeqPointImplClass *impl_class;

  g_return_val_if_fail (seq && GUPPI_IS_SEQ_POINT (seq), NULL);
  g_return_val_if_fail (stride, NULL);

  impl = GUPPI_SEQ_POINT_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_POINT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  return impl_class->raw_y ? impl_class->raw_y (impl, stride) : NULL;
}
