/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Document.h"
#include <fstream>

namespace Bakery
{

Document::Document()
{
  m_bIsNew = true;
  m_bModified = false;
  m_bReadOnly = false;
  m_pView = 0;
}

Document::~Document()
{
}

std::string Document::get_filepath() const
{
  return m_strFilePath;
}

void Document::set_filepath(const std::string& strVal, bool bEnforceFileExtension /* = false */)
{
  if(strVal != m_strFilePath)
    set_modified(); //Ready to save() for a Save As.

  m_strFilePath = strVal;

  //Enfore file extension:
  if(bEnforceFileExtension)
  {
    if(m_strFileExtension.size() != 0)  //If there is an extension to enforce.
    {
      bool bAddExt = false;
      std::string strExt = "." + get_file_extension();    
      
      if(m_strFilePath.size() < strExt.size()) //It can't have the ext already if it's not long enough.
      {
        bAddExt = true; //It isn't there already.
      }
      else
      {
        std::string strEnd = m_strFilePath.substr(m_strFilePath.size()-strExt.size());
        if(strEnd != strExt) //If it doesn't already have the extension
          bAddExt = true;
      }
 
      //Add extension if necessay.
      if(bAddExt)
        m_strFilePath += strExt;
        
      //Note that this does not replace existing extensions, so it could be e.g. 'something.blah.theext'
    }
  }
}

void Document::set_contents(const std::string& strVal)
{
  m_strContents = strVal;
}

std::string Document::get_contents() const
{
  return m_strContents;
}

void Document::set_modified(bool bVal /* = true */)
{
  m_bModified = bVal;

  if(m_bModified)
  {
    m_bIsNew = false; //Can't be new if it's been modified.
    signal_modified.emit();
  } 
}

bool Document::get_modified() const
{
  return m_bModified;
}

bool Document::load()
{
  m_strContents.erase();

  //Read the text from the disk:
  //TODO: Check for read only.
  std::ifstream fStream(m_strFilePath.c_str());
  if(fStream.is_open())
  {
    while(!(fStream.eof()))
    {
      char chTemp = fStream.get();
      if(!(fStream.eof()))
      m_strContents += chTemp;
    }
		
    set_modified(false);
		
    //Tell the View to show the new data:
    if(m_pView)
    m_pView->load_from_document();
		
    return true; //Success.
  }
  else
  {
    return false; //Failed.
  }
}


bool Document::save()
{
  //Tell the view to update the data in this document.
  if(m_pView)
		  m_pView->save_to_document();
		
  //Write the changed data to disk:
  if(get_modified())
  {
    std::ofstream file(m_strFilePath.c_str());
    if(file)
    {
      file << m_strContents;
  		
      set_modified(false);
      set_read_only(false); //It's obviously not read-only if we can write to it.
  
      return true; //Success.
    }
    else
    {
      return false; //Failed.
    }
  }
  else
  {
    return true; //Success. (At doing nothing, because nothing needed to be done.)
  }
}

std::string Document::get_name() const
{
  return util_filepath_get_name(m_strFilePath);
}

std::string Document::util_filepath_get_name(const std::string& filePath)
{
  std::string strResult;

  if(filePath.size())
  {
    //Find position of last '/' character.
    std::string::size_type posLastSlash = filePath.find_last_of("/");
    if(posLastSlash == std::string::npos)
    {
      //No '/' found. The file must be in the current or top-level directory.
      strResult = filePath;
    }
    else
    {
      //'/' found. The filename is the text after this: e.g. sub/sub/filename:
      strResult = filePath.substr(posLastSlash+1);
    }
  }

  //Show untitled explicitly:
  //Also happens for filepaths with path but no name. e.g. /sub/sub/, which shouldn't happen.
  if(strResult.size() == 0)
    strResult = _("Untitled");

  return strResult;
}

void Document::set_view(ViewBase* pView)
{
  m_pView = pView;
}

ViewBase* Document::get_view()
{
  return m_pView;
}

bool Document::get_read_only() const
{
  return m_bReadOnly;
}

void Document::set_read_only(bool bVal)
{
  m_bReadOnly = bVal;
}

bool Document::get_is_new() const
{
  return m_bIsNew;
}

void Document::set_is_new(bool bVal)
{
  if(bVal)
    set_modified(false); //can't be modified if it is new.
    
  m_bIsNew = bVal; 
}

void Document::set_file_extension(const std::string& strVal)
{
  m_strFileExtension = strVal;
}

std::string Document::get_file_extension() const
{
  return m_strFileExtension;
}



} //namespace
