//
// CrawlTask.cs
//
// Copyright (C) 2005 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

using System;
using System.IO;

using Beagle.Util;
using Beagle.Daemon;

namespace Beagle.Daemon.FileSystemQueryable {

	public class CrawlTask : Scheduler.Task {

		FileSystemQueryable queryable;
		IIndexableGenerator current_generator;

		public CrawlTask (FileSystemQueryable queryable)
		{
			this.queryable = queryable;
			this.Tag = "File System Crawler";
			this.Priority = Scheduler.Priority.Generator;
		}

		private class PostCrawlClosure {
			public FileSystemModel Model;
			public FileSystemModel.Directory Directory;
			public DateTime CrawlTime;

			public void Hook ()
			{
				Model.MarkAsCrawled (Directory, CrawlTime);
			}
		}
			

		protected override void DoTaskReal ()
		{
			// If our last generator is still doing stuff, just reschedule
			// and return.  This keeps us from generating more tasks until
			// the last one we started runs to completion.
			if (current_generator != null && current_generator.HasNextIndexable ()) {
				Reschedule = true;
				return;
			}

			FileSystemModel model = queryable.Model;

			current_generator = null;

			FileSystemModel.Directory next_dir = model.GetNextDirectoryToCrawl ();
			if (next_dir == null)
				return;

			int uncrawled, dirty;
			model.GetUncrawledCounts (out uncrawled, out dirty);
			this.Description = String.Format ("last={0}, uncrawled={1}, dirty={2})",
							  next_dir.FullName, uncrawled, dirty);

			Logger.Log.Debug ("Crawl Task Scheduling {0} (state={1})", next_dir.FullName, next_dir.State);

			// Check the next directory for new subdirectories.  If we find any,
			// add them to the model.
			try {
				foreach (DirectoryInfo subdir in DirectoryWalker.GetDirectoryInfos (next_dir.FullName)) {
					Logger.Log.Debug ("Looking at {0} in {1}", subdir.Name, next_dir.FullName);
					if (! next_dir.HasChildWithName (subdir.Name)
					    && ! model.Ignore (subdir.FullName)) {
						Logger.Log.Debug ("Found new subdir {0} under {1}",
								  subdir.Name, next_dir.FullName);
						model.AddChild (next_dir, subdir.Name);
					}
				}
			} catch (DirectoryNotFoundException ex) {
				Logger.Log.Debug ("Caught exception, deleting {0} from model", next_dir.FullName);
				model.Delete (next_dir);
				next_dir = null;
			}

			// We want this task to get re-scheduled after it is run.
			Reschedule = true;

			// ...but if we are crawling a possibly-clean directory,
			// maybe wait a little bit extra.
			if (Environment.GetEnvironmentVariable ("BEAGLE_EXERCISE_THE_DOG") == null)
				TriggerTime = DateTime.Now.AddSeconds (5);

			if (next_dir == null)
				return;

			// Set up a task group to mark the time on the directory
			// after we finish crawling it.
			PostCrawlClosure closure = new PostCrawlClosure ();
			closure.Model = model;
			closure.Directory = next_dir;
			closure.CrawlTime = DateTime.Now;
			Scheduler.TaskGroup group = Scheduler.NewTaskGroup ("Crawl " + next_dir.FullName, null,
									    new Scheduler.Hook (closure.Hook));

			// Construct an indexable generator and add it to the scheduler
			current_generator = new DirectoryIndexableGenerator (queryable, next_dir);
			Scheduler.Task task = queryable.NewAddTask (current_generator);
			task.AddTaskGroup (group);
			ThisScheduler.Add (task, Scheduler.AddType.OptionallyReplaceExisting);
		}
	}
}
