//
// BlamQueryable.cs
//
// Copyright (C) 2004 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

using System;
using System.IO;
using System.Collections;
using System.Threading;

using System.Xml;
using System.Xml.Serialization;
	
using Beagle.Daemon;
using Beagle.Util;

namespace Beagle.Daemon.BlamQueryable {

	[QueryableFlavor (Name="Blam", Domain=QueryDomain.Local, RequireInotify=false)]
	public class BlamQueryable : LuceneQueryable, IIndexableGenerator {

		private static Logger log = Logger.Get ("BlamQueryable");

		string blam_dir;
		const string blam_file = "collection.xml";

		public BlamQueryable () : base ("BlamIndex")
		{
			blam_dir = Path.Combine (Path.Combine (PathFinder.HomeDir, ".gnome2"), "blam");
		}

		/////////////////////////////////////////////////

		public override void Start () 
		{			
			base.Start ();

			ExceptionHandlingThread.Start (new ThreadStart (StartWorker));
		}

		private void StartWorker ()
		{
			if (!Directory.Exists (blam_dir)) {
				GLib.Timeout.Add (60000, new GLib.TimeoutHandler (CheckForExistence));
				return;
			}

			if (Inotify.Enabled) {
				Inotify.EventType mask = Inotify.EventType.CloseWrite;
				Inotify.Subscribe (blam_dir, OnInotifyEvent, mask);
			} else {
				FileSystemWatcher fsw = new FileSystemWatcher ();
			       	fsw.Path = blam_dir;
				fsw.Filter = blam_file;
				
				fsw.Changed += new FileSystemEventHandler (OnChangedEvent);
				fsw.Created += new FileSystemEventHandler (OnChangedEvent);
				
				fsw.EnableRaisingEvents = true;
			}

			if (File.Exists (Path.Combine (blam_dir, blam_file)))
				Index ();
		}

		private bool CheckForExistence ()
                {
                        if (!Directory.Exists (blam_dir))
                                return true;

                        this.Start ();

                        return false;
                }

		/////////////////////////////////////////////////

		// Modified event using Inotify
		private void OnInotifyEvent (Inotify.Watch watch,
					     string path,
					     string subitem,
					     string srcpath,
					     Inotify.EventType type)
		{
			if (subitem != blam_file)
				return;

			Index ();
		}

		// Modified/Created event using FSW
		private void OnChangedEvent (object o, FileSystemEventArgs args)
		{
			Index ();		
		}

		/////////////////////////////////////////////////

		private void Index ()
		{
			log.Debug ("Creating blam indexable generator");
			Scheduler.Task task = NewAddTask (this);
			task.Tag = "Blam";
			ThisScheduler.Add (task);
		}

		/////////////////////////////////////////////////
		
		// IIndexableGenerator implementation

		private ChannelCollection collection = null;
		private IEnumerator channel_enumerator = null;
		private IEnumerator item_enumerator = null;

		public Indexable GetNextIndexable ()
		{
			Channel channel = (Channel) this.channel_enumerator.Current;
			Item item = (Item) this.item_enumerator.Current;

			Uri uri = new Uri (String.Format ("feed:{0};item={1}", channel.Url, item.Id));

			Indexable indexable = new Indexable (uri);
			indexable.MimeType = "text/html";
			indexable.Type = "FeedItem";
			indexable.Timestamp = item.PubDate;
					
			indexable.AddProperty (Property.New ("dc:title", item.Title));
			indexable.AddProperty (Property.New ("fixme:author", item.Author));
			indexable.AddProperty (Property.NewDate ("fixme:published", item.PubDate));
			indexable.AddProperty (Property.NewKeyword ("fixme:itemuri", item.Link));
			indexable.AddProperty (Property.NewKeyword ("fixme:webloguri", channel.Url));

			string img = null;
			int i = item.Text.IndexOf ("<img src=\"");
			if (i != -1) {
				i += "<img src=\"".Length;
				int j = item.Text.IndexOf ("\"", i);
				if (j != -1)
					img = item.Text.Substring (i, j-i);
			}

			if (img != null) {
				string path = Path.Combine (Path.Combine (blam_dir, "Cache"),
							    img.GetHashCode ().ToString ());
				indexable.AddProperty (Property.NewUnsearched ("fixme:cachedimg", path));
			}

			StringReader reader = new StringReader (item.Text);
			indexable.SetTextReader (reader);

			return indexable;
		}

		public bool HasNextIndexable ()
		{
			if (this.collection == null) {
				FileInfo file = new FileInfo (Path.Combine (blam_dir, blam_file));

				if (this.FileAttributesStore.IsUpToDate (file.FullName))
					return false;

				try {
					this.collection = ChannelCollection.LoadFromFile (file.FullName);
				} catch (Exception e) {
					log.Warn ("Could not open Blam! channel list: " + e);
					return false;
				}

				this.FileAttributesStore.AttachTimestamp (file.FullName, file.LastWriteTime);

				if (this.collection.Channels == null || this.collection.Channels.Count == 0) {
					this.collection = null;
					return false;
				}

				this.channel_enumerator = this.collection.Channels.GetEnumerator ();
			}

			while (this.item_enumerator == null || !this.item_enumerator.MoveNext ()) {
				Channel channel;
				
				do {
					if (!this.channel_enumerator.MoveNext ()) {
						this.collection = null;
						this.channel_enumerator = null;
						return false;
					}
				
					channel = (Channel) this.channel_enumerator.Current;
				} while (channel.Items == null || channel.Items.Count == 0);

				this.item_enumerator = channel.Items.GetEnumerator ();
			}

			return true;
		}

		public string StatusName {
			get { return null; }
		}
	}

	/////////////////////////////////////////////////

	// Classes from Blam! sources for deserialization

	public class ChannelCollection {

		private ArrayList mChannels;
		
		[XmlElement ("Channel", typeof (Channel))]
		public ArrayList Channels {
			get { return mChannels; }
			set { mChannels = value; }
		}
		
		public static ChannelCollection LoadFromFile (string filename)
		{
			XmlSerializer serializer = new XmlSerializer (typeof (ChannelCollection));
			ChannelCollection collection;

			Stream stream = new FileStream (filename,
							FileMode.Open,
							FileAccess.Read,
							FileShare.ReadWrite);
			XmlTextReader reader = new XmlTextReader (stream);

			collection = (ChannelCollection) serializer.Deserialize (reader);
			reader.Close ();
			stream.Close ();

			return collection;
		}
	}
	
	public class Channel
	{
		[XmlAttribute] public string Name = "";
		[XmlAttribute] public string Url = "";

		[XmlAttribute] public string LastModified = "";
		[XmlAttribute] public string ETag = "";
	     
		ArrayList mItems;
	    
		[XmlElement ("Item", typeof (Item))]
		public ArrayList Items {
			get { return mItems; }
			set { mItems = value; }
		}
	}
	
	public class Item
	{
		[XmlAttribute] public string   Id = "";
		[XmlAttribute] public bool     Unread = true;		
		[XmlAttribute] public string   Title = "";
		[XmlAttribute] public string   Text = "";
		[XmlAttribute] public string   Link = "";
		[XmlAttribute] public DateTime PubDate;
		[XmlAttribute] public string   Author = "";
  	}
}
