//
// FileNameFilter.cs
//
// Copyright (C) 2004 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Text.RegularExpressions;
using System.IO;

using Beagle.Util;

namespace Beagle.Daemon.FileSystemQueryable {

	public class FileNameFilter {
		
		private FileSystemModel model = null;

		//////////////////////////////////////////////////////////////////////
		
		private ArrayList excludes = new ArrayList ();
		private ArrayList default_excludes = new ArrayList ();

		private void SetupDefaultPatternsToIgnore ()
		{
			// Add our default skip patterns.
			// FIXME: This probably shouldn't be hard-wired.  Or should it?
			AddDefaultPatternToIgnore (new string [] {
				                   ".*",
						   "*~",
						   "#*#",
						   "*.o",
						   "*.a",
						   "*.S",
						   "*.la",
						   "*.lo",
						   "*.loT",
						   "*.so",
						   "*.exe",
						   "*.dll",
						   "*.mdb",
						   "*.com",
						   "*.csproj",
						   "*.dsp",
						   "*.dsw",
						   "*.m4",
						   "*.pc",
						   "*.pc.in",
						   "*.in.in",
						   "*.omf",
						   "*.aux",
						   "*.tmp",
						   "autom4te.cache",
						   "po",
						   "aclocal",
						   "Makefile",
						   "Makefile.am",
						   "Makefile.in",
						   "CVS",
						   "SCCS",
						   // Garbage generated by the autotools
						   "conftest",
						   "confdefs.h",
						   "conftest.*",
						   "confstat*",
						   "/conf[0-9]+.sh/",
						   "/conf[0-9]+.file/"
			});
		}
		
		public void AddDefaultPatternToIgnore (IEnumerable patterns)
		{
			foreach (string pattern in patterns)
				default_excludes.Add (new ExcludeItem (ExcludeType.Pattern, pattern));
		}
		
		/////////////////////////////////////////////////////////////

		public void AddPatternToIgnore (string pattern)
		{
			AddExclude (new ExcludeItem (ExcludeType.Pattern, pattern));
		}
		
		public void AddPatternToIgnore (IEnumerable patterns)
		{
			foreach (string pattern in patterns)
				AddPatternToIgnore (pattern);
		}

		public void AddPathToIgnore (string path) 
		{
			AddExclude (new ExcludeItem (ExcludeType.Path, path));
		}

		public void AddPathToIgnore (IEnumerable paths)
		{
			foreach (string path in paths)
				AddPathToIgnore (path);
		}

		/////////////////////////////////////////////////////////////

		public void AddExclude (ExcludeItem exclude)
		{
			if (exclude.Type != ExcludeType.Path && exclude.Type != ExcludeType.Pattern)
				return;

			Logger.Log.Debug ("FileNameFilter: Adding ExcludeItem (value={0}, type={1})", exclude.Value, exclude.Type);

			if (!excludes.Contains (exclude))
				excludes.Add (exclude);
		}

		public bool RemoveExclude (ExcludeItem exclude)
		{
			if (!excludes.Contains (exclude))
				return false;

			Logger.Log.Debug ("FileNameFilter: Removing ExcludeItem (value={0}, type={1})", exclude.Value, exclude.Type);

			excludes.Remove (exclude);
			return true;
		}
		
		/////////////////////////////////////////////////////////////

		public FileNameFilter (FileSystemModel model)
		{
			this.model = model;

			SetupDefaultPatternsToIgnore ();

			LoadConfiguration ();
		}

		private void LoadConfiguration () 
		{
			foreach (ExcludeItem exclude in Conf.Indexing.Excludes)
				AddExclude (exclude);

			Conf.Subscribe (typeof (Conf.IndexingConfig), OnConfigurationChanged);
		}

		private void OnConfigurationChanged (Conf.Section section)
		{
			IList excludes_to_add, excludes_to_remove;

			ArrayFu.IntersectListChanges (Conf.Indexing.Excludes, excludes, out excludes_to_add, out excludes_to_remove);

			foreach (ExcludeItem exclude in excludes_to_remove)
				RemoveExclude (exclude);

			foreach (ExcludeItem exclude in excludes_to_add)
				AddExclude (exclude);

			// FIXME: Send a re-scan event here when ignore patterns are updated
			// and do a model.Remove/Add on the directories in question
		}

		/////////////////////////////////////////////////////////////

		// FIXME: We could make this more efficient by storing more information.
		// In particular, if ~/foo is an IgnoreAll directory, we could store
		// that info in the PerDirectoryInfo of subdir ~/foo/bar so that
		// we could avoid walking up the chain of directories.
		public bool Ignore (string path)
		{
			if (! Path.IsPathRooted (path))
				path = Path.GetFullPath (path);

			foreach (ExcludeItem exclude in excludes)
				if (exclude.Type == ExcludeType.Path && exclude.IsMatch (path))
					return true;

			foreach (FileSystemModel.Directory root in model.Roots)
				if (root.FullName == path)
					return false;

			string name = Path.GetFileName (path);

			// Ugh..

			foreach (ExcludeItem exclude in excludes)
				if (exclude.Type == ExcludeType.Pattern && exclude.IsMatch (name))
					return true;

			foreach (ExcludeItem exclude in default_excludes)
				if (exclude.IsMatch (name))
					return true;

			if (path == Path.GetPathRoot (path))
				return true;

			string dir = Path.GetDirectoryName (path);

			// A file should be ignored if any of its parent directories
			// is ignored.
			return Ignore (dir);
		}

		public bool Ignore (FileSystemInfo info)
		{
			return Ignore (info.FullName);
		}
	}		
}
