/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <libsoup/soup-logger.h>
#include <libsoup/soup-session-async.h>
#include <libsoup/soup-gnome.h>
#include <json-glib/json-parser.h>
#include <json-glib/json-generator.h>
#include "couchdb-glib.h"
#include "utils.h"

struct _CouchDBDocument {
	GObject parent;

	CouchDB *couchdb;
	char *dbname;
	JsonNode *root_node;
};

G_DEFINE_TYPE(CouchDBDocument, couchdb_document, G_TYPE_OBJECT);

static void
couchdb_document_finalize (GObject *object)
{
	CouchDBDocument *document = COUCHDB_DOCUMENT (object);

	g_free (document->dbname);
	json_node_free (document->root_node);

	G_OBJECT_CLASS (couchdb_document_parent_class)->finalize (object);
}

static void
couchdb_document_class_init (CouchDBDocumentClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = couchdb_document_finalize;
}

static void
couchdb_document_init (CouchDBDocument *document)
{
}

CouchDBDocument *
couchdb_document_new (CouchDB *couchdb)
{
	CouchDBDocument *document;

	document = g_object_new (COUCHDB_TYPE_DOCUMENT, NULL);
	document->couchdb = couchdb;
	document->dbname = NULL;
	document->root_node = json_node_new (JSON_NODE_OBJECT);

	return document;
}

CouchDBDocument *
couchdb_document_get (CouchDB *couchdb,
		      const char *dbname,
		      const char *docid,
		      GError **error)
{
	char *url;
	JsonParser *parser;
	CouchDBDocument *document = NULL;

	g_return_val_if_fail (COUCHDB_IS (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);
	g_return_val_if_fail (docid != NULL, NULL);

	url = g_strdup_printf ("http://%s/%s/%s", couchdb->hostname, dbname, docid);
	parser = send_message_and_parse (couchdb, SOUP_METHOD_GET, url, error);
	if (parser) {
		document = g_object_new (COUCHDB_TYPE_DOCUMENT, NULL);
		document->couchdb = couchdb;
		document->dbname = g_strdup (dbname);

		document->root_node = json_node_copy (json_parser_get_root (parser));
		g_object_unref (G_OBJECT (parser));
	}

	g_free (url);

	return document;
}

const char *
couchdb_document_get_id (CouchDBDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node &&
	    json_node_get_node_type (document->root_node) == JSON_NODE_OBJECT) {
		return json_object_get_string_member (
			json_node_get_object (document->root_node),
			"_id");
	}

	return NULL;
}

void
couchdb_document_set_id (CouchDBDocument *document, const char *id)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (id != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node),
				       "_id",
				       id);
}

const char *
couchdb_document_get_revision (CouchDBDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node &&
	    json_node_get_node_type (document->root_node) == JSON_NODE_OBJECT) {
		return json_object_get_string_member (
			json_node_get_object (document->root_node),
			"_rev");
	}

	return NULL;
}

gboolean
couchdb_document_get_boolean_field (CouchDBDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), FALSE);
	g_return_val_if_fail (field != NULL, FALSE);

	return json_object_get_boolean_member (json_node_get_object (document->root_node),
					       field);
}

void
couchdb_document_set_boolean_field (CouchDBDocument *document, const char *field, gboolean value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_boolean_member (json_node_get_object (document->root_node),
					field,
					value);
}

gint
couchdb_document_get_int_field (CouchDBDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), -1);
	g_return_val_if_fail (field != NULL, -1);

	return json_object_get_int_member (json_node_get_object (document->root_node),
					   field);
}

void
couchdb_document_set_int_field (CouchDBDocument *document, const char *field, gint value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_int_member (json_node_get_object (document->root_node),
				    field,
				    value);
}

gdouble
couchdb_document_get_double_field (CouchDBDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), -1);
	g_return_val_if_fail (field != NULL, -1);

	return json_object_get_double_member (json_node_get_object (document->root_node),
					      field);
}

void
couchdb_document_set_double_field (CouchDBDocument *document, const char *field, gdouble value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_double_member (json_node_get_object (document->root_node),
				       field,
				       value);
}

const char *
couchdb_document_get_string_field (CouchDBDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (field != NULL, NULL);

	return json_object_get_string_member (json_node_get_object (document->root_node),
					      field);
}

void
couchdb_document_set_string_field (CouchDBDocument *document, const char *field, const char *value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node),
				       field,
				       value);
}

char *
couchdb_document_to_string (CouchDBDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node) {
		JsonGenerator *generator;
		char *str;
		gsize size;

		generator = json_generator_new ();
		json_generator_set_root (generator, document->root_node);

		str = json_generator_to_data (generator, &size);
		g_object_unref (G_OBJECT (generator));

		return str;
	}

	return NULL;
}
