/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <uuid/uuid.h>
#include <string.h>
#include <libsoup/soup-session-async.h>
#include "couchdb-glib.h"
#include "utils.h"
#ifdef HAVE_OAUTH
#include <time.h>
#include "oauth.h"
#endif

static JsonParser *
parse_json_response (SoupMessage *http_message, GError **error)
{
	SoupBuffer *buffer;
        GString *str = NULL;
        goffset offset = 0;
	JsonParser *json_parser = NULL;

	while ((buffer = soup_message_body_get_chunk (http_message->response_body, offset))) {
		if (!str)
                        str = g_string_new ("");
                g_string_append_len (str, buffer->data, buffer->length);

                offset += buffer->length;
                soup_buffer_free (buffer);
	}

	if (str && str->len > 0) {
		g_debug ("Response body: %s", str->str);
		json_parser = json_parser_new ();
		if (!json_parser_load_from_data (json_parser,
						 (const gchar *) str->str,
						 str->len,
						 error)) {
			g_object_unref (G_OBJECT (json_parser));
			json_parser = NULL;

			g_set_error (error, COUCHDB_ERROR, -1, "Invalid JSON response");
		}

		g_string_free (str, TRUE);
	}

	return json_parser;
}

GQuark
couchdb_error_quark (void)
{
	static GQuark error;

	if (!error)
		error = g_quark_from_static_string ("couchdb_glib");

	return error;
}

#ifdef HAVE_OAUTH
static void
add_oauth_signature (CouchDB *couchdb, SoupMessage *http_message, const char *method, const char *url)
{
	char *signed_url;

	signed_url = oauth_sign_url2 (url, NULL, OA_HMAC, method,
				      couchdb->oauth_consumer_key,
				      couchdb->oauth_consumer_secret,
				      couchdb->oauth_token_key,
				      couchdb->oauth_token_secret);
	if (signed_url != NULL) {
		char **parsed_url;
		GString *header = NULL;

		/* Get the OAuth signature from the signed URL */
		parsed_url = g_strsplit (signed_url, "?", 2);
		if (parsed_url != NULL) {
			gchar **params;
			int i;

			params = g_strsplit (parsed_url[1], "&", 0);
#ifdef DEBUG_OAUTH
			g_debug ("Parsing %s", parsed_url[1]);
#endif
			for (i = 0; params[i] != NULL; i++) {
				gchar **url_param;

				if (!g_str_has_prefix (params[i], "oauth_"))
					continue;

				url_param = g_strsplit (params[i], "=", 2);
				if (url_param == NULL)
					continue;

				if (header != NULL)
					header = g_string_append (header, ", ");
				else
					header = g_string_new ("OAuth ");

				header = g_string_append (header, url_param[0]);
				header = g_string_append (header, "=\"");
				header = g_string_append (header, url_param[1]);
				header = g_string_append (header, "\"");

				g_strfreev (url_param);
			}

			if (params)
				g_strfreev (params);

			g_strfreev (parsed_url);
		}

		if (header != NULL) {
			soup_message_headers_append (http_message->request_headers, "Authorization", header->str);

			g_string_free (header, TRUE);
		}

		free (signed_url);
	}
}
#endif

static void
debug_print_headers (const char *name, const char *value, gpointer user_data)
{
	g_print ("\t%s: %s\n", name, value);
}

JsonParser *
send_message_and_parse (CouchDB *couchdb, const char *method, const char *url, const char *body, GError **error)
{
	SoupMessage *http_message;
	guint status;
	JsonParser *parser = NULL;

	http_message = soup_message_new (method, url);
	if (body != NULL) {
		soup_message_set_request (http_message, "application/json", SOUP_MEMORY_COPY,
					  body, strlen (body));
	}

#ifdef HAVE_OAUTH
	if (couchdb->oauth_enabled)
		add_oauth_signature (couchdb, http_message, method, url);
#endif

	g_debug ("Sending %s to %s... with headers\n: ", method, url);
	soup_message_headers_foreach (http_message->request_headers,
				      (SoupMessageHeadersForeachFunc) debug_print_headers,
				      NULL);

	status = soup_session_send_message (couchdb->http_session, http_message);
	if (SOUP_STATUS_IS_SUCCESSFUL (status)) {
	       	parser = parse_json_response (http_message, error);
	} else {
		g_set_error (error, COUCHDB_ERROR, status, "%s", http_message->reason_phrase);
	}

	return parser;
}

char *
generate_uuid (void)
{
	uuid_t uuid;
	char uuid_string[37];

	uuid_generate_random (uuid);
	uuid_unparse (uuid, uuid_string);

	g_print ("Generated %s uuid", uuid_string);
	return g_strdup (uuid_string);
}
