/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include "gailtexthelper.h"

static void gail_text_helper_class_init      (GailTextHelperClass *klass);

static void gail_text_helper_init            (GailTextHelper      *texthelper);
static void gail_text_helper_finalize        (GObject             *object);


static void get_pango_text_offsets           (PangoLayout         *layout,
                                              GtkTextBuffer       *buffer,
                                              GailOffsetType      function,
                                              AtkTextBoundary     boundary_type,
                                              gint                offset,
                                              gint                *start_offset,
                                              gint                *end_offset,
                                              GtkTextIter         *start_iter,
                                              GtkTextIter         *end_iter);
static gpointer parent_class = NULL;

GType
gail_text_helper_get_type(void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailTextHelperClass), /* class size */
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_text_helper_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof(GailTextHelper), /*instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) gail_text_helper_init, /* instance init */
      NULL, /* value table */
    };

    type = g_type_register_static (G_TYPE_OBJECT, "GailTextHelper", &tinfo, 0);
  }
  return type;
}

GailTextHelper*
gail_text_helper_new (void)
{
  return GAIL_TEXT_HELPER (g_object_new (GAIL_TYPE_TEXT_HELPER, NULL));
}

static void
gail_text_helper_init (GailTextHelper *texthelper)
{
  texthelper->buffer = NULL;
}

static void
gail_text_helper_class_init (GailTextHelperClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_ref (G_TYPE_OBJECT);

  gobject_class->finalize = gail_text_helper_finalize;
}

static void
gail_text_helper_finalize (GObject *object)
{
  GailTextHelper *texthelper = GAIL_TEXT_HELPER (object);

  if (texthelper->buffer)
    g_object_unref (texthelper->buffer);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

void
gail_text_helper_text_setup     (GailTextHelper *texthelper,
                                 const gchar    *text)
{
  g_return_if_fail (GAIL_IS_TEXT_HELPER(texthelper));

  if (texthelper->buffer != NULL)
    {
      g_object_unref (texthelper->buffer);
      texthelper->buffer = NULL;
    }
  if (text == NULL)
    return;

  texthelper->buffer = gtk_text_buffer_new (NULL);
  gtk_text_buffer_set_text (texthelper->buffer, text, -1);
}

void
gail_text_helper_buffer_setup  (GailTextHelper  *texthelper,
                                GtkTextBuffer   *buffer)
{
  g_return_if_fail (GAIL_IS_TEXT_HELPER (texthelper));

  texthelper->buffer = g_object_ref(buffer);
}

/**
 * gail_text_helper_get_text:
 * @texthelper: a #GailTextHelper
 * @layout: A gpointer which is a PangoLayout, a GtkTreeView of NULL
 * @function: enumeration specifying whether to return the text before, at, or
 *   after the offset.
 * @boundary_type: the boundary type.
 * @offset: the offset 
 * @start_offset: passes back the start offset
 * @end_offset: passes back the end offset
 *
 * The layout is used for getting the text on a line. The value is NULL for a
 * GtkTextView which is not wrapped, is a GtkTextView for a GtkTextView which
 * is wrapped and is a PangoLayout otherwise.
 *
 * Returns: the substring specified
 **/
gchar*
gail_text_helper_get_text (GailTextHelper  *texthelper,
                           gpointer        layout,
                           GailOffsetType  function,
                           AtkTextBoundary boundary_type,
                           gint            offset,
                           gint            *start_offset,
                           gint            *end_offset)
{

  GtkTextIter start, end, empty_check;
  gint line_number;
  gboolean not_empty;
  GtkTextBuffer *buffer;

  g_return_val_if_fail (GAIL_IS_TEXT_HELPER(texthelper), NULL);

  buffer = texthelper->buffer;
  if (buffer == NULL)
    {
      *start_offset = 0;
      *end_offset = 0;
      return NULL;
    }

  gtk_text_buffer_get_iter_at_offset (buffer, &start, offset);

  gtk_text_buffer_get_iter_at_offset (buffer, &empty_check, 0);
  not_empty = gtk_text_iter_forward_char (&empty_check);

  if (!not_empty)
    {
      *start_offset = 0;
      *end_offset = 0;
      return g_strdup ("");
    }
    
  end = start;

  switch (function)
    {
    case GAIL_BEFORE_OFFSET:
      switch (boundary_type)
        {
        case ATK_TEXT_BOUNDARY_CHAR:
          gtk_text_iter_backward_char(&start);
          break;
        case ATK_TEXT_BOUNDARY_WORD_START:
          if (!gtk_text_iter_starts_word (&start))
            gtk_text_iter_backward_word_start (&start);
          end = start;
          gtk_text_iter_backward_word_start(&start);
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          if (gtk_text_iter_inside_word (&start) &&
              !gtk_text_iter_starts_word (&start))
            gtk_text_iter_backward_word_start (&start);
          while (!gtk_text_iter_ends_word (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          end = start;
          gtk_text_iter_backward_word_start(&start);
          while (!gtk_text_iter_ends_word (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          if (!gtk_text_iter_starts_sentence (&start))
            gtk_text_iter_backward_sentence_start (&start);
          end = start;
          gtk_text_iter_backward_sentence_start (&start);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          if (gtk_text_iter_inside_sentence (&start) &&
              !gtk_text_iter_starts_sentence (&start))
            gtk_text_iter_backward_sentence_start (&start);
          while (!gtk_text_iter_ends_sentence (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          end = start;
          gtk_text_iter_backward_sentence_start (&start);
          while (!gtk_text_iter_ends_sentence (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
            {
              line_number = gtk_text_iter_get_line (&start);
              if (line_number == 0)
                {
                  gtk_text_buffer_get_iter_at_offset (buffer,
                    &start, 0);
                }
              else
                {
                  gtk_text_iter_backward_line (&start);
                  gtk_text_iter_forward_line (&start);
                }
              end = start;
              gtk_text_iter_backward_line (&start);
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              g_warning ("atk_text_get_text_before_offset() not implemented"
                         " for ATK_TEXT_BOUNDARY_LINE_START\n");
            }
          else if (PANGO_IS_LAYOUT (layout))
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
            {
              line_number = gtk_text_iter_get_line (&start);
              if (line_number == 0)
                {
                  gtk_text_buffer_get_iter_at_offset (buffer,
                    &start, 0);
                  end = start;
                }
              else
                {
                  gtk_text_iter_backward_line (&start);
                  end = start;
                  while (!gtk_text_iter_ends_line (&start))
                    {
                      if (!gtk_text_iter_backward_char (&start))
                        break;
                    }
                  gtk_text_iter_forward_to_line_end (&end);
                }
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              g_warning ("atk_text_get_text_before_offset() not implemented"
                         " for ATK_TEXT_BOUNDARY_LINE_END\n");
            }
          else if (PANGO_IS_LAYOUT (layout))
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);
          break;
        }
      break;
 
    case GAIL_AT_OFFSET:
      switch (boundary_type)
        {
        case ATK_TEXT_BOUNDARY_CHAR:
          gtk_text_iter_forward_char (&end);
          break;
        case ATK_TEXT_BOUNDARY_WORD_START:
          if (!gtk_text_iter_starts_word (&start))
            gtk_text_iter_backward_word_start (&start);
          if (gtk_text_iter_inside_word (&end))
            gtk_text_iter_forward_word_end (&end);
          while (!gtk_text_iter_starts_word (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
                break;
            }
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          if (gtk_text_iter_inside_word (&start) &&
              !gtk_text_iter_starts_word (&start))
            gtk_text_iter_backward_word_start (&start);
          while (!gtk_text_iter_ends_word (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          gtk_text_iter_forward_word_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          if (!gtk_text_iter_starts_sentence (&start))
            gtk_text_iter_backward_sentence_start (&start);
          if (gtk_text_iter_inside_sentence (&end))
            gtk_text_iter_forward_sentence_end (&end);
          while (!gtk_text_iter_starts_sentence (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
                break;
            }
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          if (gtk_text_iter_inside_sentence (&start) &&
              !gtk_text_iter_starts_sentence (&start))
            gtk_text_iter_backward_sentence_start (&start);
          while (!gtk_text_iter_ends_sentence (&start))
            {
              if (!gtk_text_iter_backward_char (&start))
                break;
            }
          gtk_text_iter_forward_sentence_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
            {
              line_number = gtk_text_iter_get_line (&start);
              if (line_number == 0)
                {
                  gtk_text_buffer_get_iter_at_offset (buffer,
                    &start, 0);
                }
              else
                {
                  gtk_text_iter_backward_line (&start);
                  gtk_text_iter_forward_line (&start);
                }
              gtk_text_iter_forward_line (&end);
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              /*
               * There is a problem with 
               * gtk_text_layout_move_iter_to_previous_line()
               * and gtk_text_layout_move_iter_to_next_line()
               * which I have raised in bug 62594.
               */
              GtkTextView *view = GTK_TEXT_VIEW (layout);
              GtkTextIter temp;

              temp = start;
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
              gtk_text_view_backward_display_line (view, &start);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
              /*
               * This call seems to go to the start of the paragraph or 
               * document
               */
              gtk_text_view_forward_display_line (view, &start);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
              /*
               * Iterate through the lines until we get beyond where we started
               */
              while (gtk_text_iter_compare (&temp, &start) > 0)
{
                gtk_text_view_forward_display_line (view, &start);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
}
              /*
               * Now go back to the previous line. We have a while rather than
               * an if here because if we have gone to the next pagagraph
               * we may need to step back twice.
               */
              while (gtk_text_iter_compare (&temp, &start) <= 0)
{
                gtk_text_view_backward_display_line (view, &start);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
}
              if (gtk_text_iter_compare (&start, &end) > 0)
                {
                  /*
                   * start > end => no previous line
                   */
                  start = end;
                  gtk_text_view_backward_display_line_start (view, &start);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&start));
                }
              temp = end;
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&end));
              gtk_text_view_forward_display_line (view, &end);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&end));
              while (gtk_text_iter_compare (&temp, &end) > 0)
{
                /*
                 * First time this function goes to the start of the second
                 * line
                 */
                gtk_text_view_forward_display_line (view, &end);
              g_print ("%d: offset: %d\n", __LINE__, gtk_text_iter_get_offset(&end));
}
            }
          else if PANGO_IS_LAYOUT (layout)
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);

          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
            {
              line_number = gtk_text_iter_get_line (&start);
              if (line_number == 0)
                {
                  gtk_text_buffer_get_iter_at_offset (buffer,
                    &start, 0);
                }
              else
                {
                  gtk_text_iter_backward_line (&start);
                  gtk_text_iter_forward_line (&start);
                }
              while (!gtk_text_iter_ends_line (&start))
                {
                  if (!gtk_text_iter_backward_char (&start))
                    break;
                }
              gtk_text_iter_forward_to_line_end (&end);
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              g_warning ("atk_text_get_text_at_offset() not implemented"
                         " for ATK_TEXT_BOUNDARY_LINE_END\n");
            }
          else if PANGO_IS_LAYOUT (layout)
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);
          break;
        }
      break;
  
    case GAIL_AFTER_OFFSET:
      switch (boundary_type)
        {
        case ATK_TEXT_BOUNDARY_CHAR:
          gtk_text_iter_forward_char(&start);
          gtk_text_iter_forward_chars(&end, 2);
          break;
        case ATK_TEXT_BOUNDARY_WORD_START:
          if (gtk_text_iter_inside_word (&end))
            gtk_text_iter_forward_word_end (&end);
          while (!gtk_text_iter_starts_word (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
                break;
            }
          start = end;
          if (!gtk_text_iter_is_end (&end))
            {
              gtk_text_iter_forward_word_end (&end);
              while (!gtk_text_iter_starts_word (&end))
                {
                  if (!gtk_text_iter_forward_char (&end))
                    break;
                }
            }
          break;
        case ATK_TEXT_BOUNDARY_WORD_END:
          gtk_text_iter_forward_word_end (&end);
          start = end;
          if (!gtk_text_iter_is_end (&end))
            gtk_text_iter_forward_word_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_START:
          if (gtk_text_iter_inside_sentence (&end))
            gtk_text_iter_forward_sentence_end (&end);
          while (!gtk_text_iter_starts_sentence (&end))
            {
              if (!gtk_text_iter_forward_char (&end))
                break;
            }
          start = end;
          if (!gtk_text_iter_is_end (&end))
            {
              gtk_text_iter_forward_sentence_end (&end);
              while (!gtk_text_iter_starts_sentence (&end))
                {
                  if (!gtk_text_iter_forward_char (&end))
                    break;
                }
            }
          break;
        case ATK_TEXT_BOUNDARY_SENTENCE_END:
          gtk_text_iter_forward_sentence_end (&end);
          start = end;
          if (!gtk_text_iter_is_end (&end))
            gtk_text_iter_forward_sentence_end (&end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_START:
          if (layout == NULL)
            {
              gtk_text_iter_forward_line (&end);
              start = end;
              gtk_text_iter_forward_line (&end);
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              g_warning ("atk_text_get_text_after_offset() not implemented"
                         " for ATK_TEXT_BOUNDARY_LINE_START\n");
            }
          else if (PANGO_IS_LAYOUT (layout))
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);
          break;
        case ATK_TEXT_BOUNDARY_LINE_END:
          if (layout == NULL)
            {
              gtk_text_iter_forward_line (&start);
              end = start;
              if (!gtk_text_iter_is_end (&start))
                { 
                  while (!gtk_text_iter_ends_line (&start))
                  {
                    if (!gtk_text_iter_backward_char (&start))
                      break;
                  }
                  gtk_text_iter_forward_to_line_end (&end);
                }
            }
          else if GTK_IS_TEXT_VIEW (layout)
            {
              g_warning ("atk_text_get_text_after_offset() not implemented"
                         " for ATK_TEXT_BOUNDARY_LINE_END\n");
            }
          else if (PANGO_IS_LAYOUT (layout))
            get_pango_text_offsets (PANGO_LAYOUT (layout),
                                    buffer,
                                    function,
                                    boundary_type,
                                    offset,
                                    start_offset,
                                    end_offset,
                                    &start,
                                    &end);
          break;
        }
      break;
    }
  *start_offset = gtk_text_iter_get_offset (&start);
  *end_offset = gtk_text_iter_get_offset (&end);

  return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
}

/**
 * get_substring:
 * @texthelper: a #GailTextHelper
 * @start_pos: the start position of the substring
 * @end_pos: the end position of the substring.
 *
 * Gets the substring indicated by @start_pos and @end_pos
 *
 * Returns: the substring indicated by @start_pos and @end_pos
 **/
gchar*
gail_text_helper_get_substring (GailTextHelper *texthelper,
                                gint           *start_pos, 
                                gint           *end_pos)
{
  GtkTextIter start, end;
  GtkTextBuffer *buffer;

  g_return_val_if_fail(GAIL_IS_TEXT_HELPER (texthelper), NULL);

  buffer = texthelper->buffer;
  if (buffer == NULL)
     return NULL;

  gtk_text_buffer_get_iter_at_offset (buffer, &start, *start_pos);
  gtk_text_buffer_get_iter_at_offset (buffer, &end, *end_pos);

  return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
}

/**
 * get_extents_from_pango_rectangle:
 * @widget: the widget that contains the PangoLayout, that contains
 *   the PangoRectangle
 * @char_rect: the #PangoRectangle from which to calculate extents
 * @x_layout: the x-offset at which the widget displays the
 *   PangoLayout that contains the PangoRectangle, relative to @widget
 * @y_layout: the y-offset at which the widget displays the
 *   PangoLayout that contains the PangoRectangle, relative to @widget
 * @x: the x-position of the #PangoRectangle relative to @coords
 * @y: the y-position of the #PangoRectangle relative to @coords
 * @width: the width of the #PangoRectangle
 * @height: the  gailheight of the #PangoRectangle
 * @coords: an #AtkCoordType enumeration
 *
 * Gets the extents of @char_rect in device coordinates,
 * relative to either top-level window or screen coordinates as
 * specified by @coords.
 **/
void
gail_text_helper_get_extents_from_pango_rectangle (GtkWidget      *widget,
                                                   PangoRectangle *char_rect,
                                                   gint           x_layout,
                                                   gint           y_layout,
                                                   gint           *x,
                                                   gint           *y,
                                                   gint           *width,
                                                   gint           *height,
                                                   AtkCoordType   coords)
{
  gint x_window, y_window, x_widget, y_widget;

  gail_widget_get_origins (widget, &x_widget, &y_widget, &x_window,
    &y_window);

  if (coords == ATK_XY_WINDOW)
    {
      *x = (char_rect->x / PANGO_SCALE) + x_layout + (x_widget - x_window);
      *y = (char_rect->y / PANGO_SCALE) + y_layout + (y_widget - y_window);
    }
  else if (coords == ATK_XY_SCREEN)
    {
      *x = (char_rect->x / PANGO_SCALE) + x_layout + x_widget;
      *y = (char_rect->y / PANGO_SCALE) + y_layout + y_widget;
    }
  else
    {
      *x = 0;
      *y = 0;
      *height = 0;
      *width = 0;
      return;
    }
  *height = char_rect->height / PANGO_SCALE;
  *width = char_rect->width / PANGO_SCALE;

  return;
}

/**
 * get_index_at_point_in_layout:
 * @widget: a #GtkWidget
 * @layout: the #PangoLayout from which to get the index at the
 *   specified point.
 * @x_layout: the x-offset at which the widget displays the
 *   #PangoLayout, relative to @widget
 * @y_layout: the y-offset at which the widget displays the
 *   #PangoLayout, relative to @widget
 * @x: the x-coordinate relative to @coords at which to
 *   calculate the index
 * @y: the y-coordinate relative to @coords at which to
 *   calculate the index
 * @coords: an #AtkCoordType enumeration
 *
 * Gets the byte offset at the specified @x and @y in a #PangoLayout.
 *
 * Returns: the byte offset at the specified @x and @y in a
 *   #PangoLayout
 **/
gint
gail_text_helper_get_index_at_point_in_layout (GtkWidget *widget,
                                              PangoLayout *layout,
                                              gint        x_layout,
                                              gint        y_layout,
                                              gint        x,
                                              gint        y,
                                              AtkCoordType coords)
{
  gint index, x_window, y_window, x_widget, y_widget;

  gail_widget_get_origins (widget, &x_widget, &y_widget, &x_window,
    &y_window);
  if (coords == ATK_XY_WINDOW)
    pango_layout_xy_to_index (layout,
      (x - x_layout - (x_widget - x_window)) * PANGO_SCALE,
      (y - y_layout - (y_widget - y_window)) * PANGO_SCALE,
      &index, NULL);
  else if (coords == ATK_XY_SCREEN)
    pango_layout_xy_to_index (layout,
      (x - x_layout - x_widget) * PANGO_SCALE,
      (y - y_layout - y_widget) * PANGO_SCALE,
      &index, NULL);
  else
    return -1;

  return index;
}

/**
 * gail_text_helper_add_attribute:
 * @attrib_set: The #AtkAttributeSet to add the attribute to
 * @attr: The AtkTextAttrribute which identifies the attribut
 * @value: The attribute value
 *
 * Creates an #AtkAttribute from @attr and @value, and adds it
 * to @attrib_set. 
 *
 * Returns: A pointer to the new #AtkAttributeSet.
 **/
AtkAttributeSet*
gail_text_helper_add_attribute (AtkAttributeSet *attrib_set,
                                AtkTextAttribute attr,
                                gchar           *value)
{
  AtkAttributeSet *return_set;
  AtkAttribute *at = g_malloc (sizeof (AtkAttribute));
  at->name = g_strdup (atk_attribute_get_name (attr));
  at->value = value;
  return_set = g_slist_prepend(attrib_set, at);
  return return_set;
}

/**
 * gail_text_helper_get_run_attributes:
 * @attrib_set: The #AtkAttributeSet to add the attribute to
 * @layout: The PangoLayout from which the attributes will be obtained
 * @text: The text 
 * @offset: The offset at which the attributes are required
 * @start_offset: The start offset of the current run
 * @end_offset: The end offset of the current run
 *
 * Returns: A pointer to the #AtkAttributeSet.
 **/
AtkAttributeSet* 
gail_text_helper_get_run_attributes (AtkAttributeSet *attrib_set,
                                     PangoLayout     *layout,
                                     gchar           *text,
                                     gint            offset,
                                     gint            *start_offset,
                                     gint            *end_offset)
{
  PangoAttrIterator *iter;
  PangoAttrList *attr;  
  PangoAttrString *pango_string;
  PangoAttrInt *pango_int;
  PangoAttrColor *pango_color;
  PangoAttrLanguage *pango_lang;
  PangoAttrFloat *pango_float;
  gint index, start_index, end_index;
  gboolean is_next = TRUE;
  gchar *value = NULL;
  glong len;

  len = g_utf8_strlen (text, -1);
  /* Grab the attributes of the PangoLayout, if any */
  if ((attr = pango_layout_get_attributes (layout)) == NULL)
    {
      *start_offset = 0;
      *end_offset = len;
      return attrib_set;
    }
  iter = pango_attr_list_get_iterator (attr);
  /* Get invariant range offsets */
  /* If offset out of range, set offset in range */
  if (offset > len)
    offset = len;
  else if (offset < 0)
    offset = 0;

  index = g_utf8_offset_to_pointer (text, offset) - text;
  pango_attr_iterator_range (iter, &start_index, &end_index);
  while (is_next)
    {
      if (index >= start_index && index < end_index)
        {
          *start_offset = g_utf8_pointer_to_offset (text, 
                                                    text + start_index);  
          if (end_index == G_MAXINT)
          /* Last iterator */
            end_index = len;
      
          *end_offset = g_utf8_pointer_to_offset (text, 
                                                  text + end_index);  
          break;
        }  
      is_next = pango_attr_iterator_next (iter);
      pango_attr_iterator_range (iter, &start_index, &end_index);
    }
  /* Get attributes */
  if ((pango_string = (PangoAttrString*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_FAMILY)) != NULL)
    {
      value = g_strdup_printf("%s", pango_string->value);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_FAMILY_NAME, 
                                                 value);
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_STYLE)) != NULL)
    {
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_STYLE, 
      g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STYLE, pango_int->value)));
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_WEIGHT)) != NULL)
    {
      value = g_strdup_printf("%i", pango_int->value);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_WEIGHT, 
                                                   value);
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_VARIANT)) != NULL)
    {
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_VARIANT, 
       g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_VARIANT, pango_int->value)));
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_STRETCH)) != NULL)
    {
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_STRETCH, 
       g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STRETCH, pango_int->value)));
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_SIZE)) != NULL)
    {
      value = g_strdup_printf("%i", pango_int->value);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_SIZE,
                                                   value);
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_UNDERLINE)) != NULL)
    {
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                               ATK_TEXT_ATTR_UNDERLINE, 
       g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_UNDERLINE, pango_int->value)));
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_STRIKETHROUGH)) != NULL)
    {
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                               ATK_TEXT_ATTR_STRIKETHROUGH, 
       g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STRIKETHROUGH, pango_int->value)));
    } 
  if ((pango_int = (PangoAttrInt*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_RISE)) != NULL)
    {
      value = g_strdup_printf("%i", pango_int->value);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_RISE,
                                                   value);
    } 
  if ((pango_lang = (PangoAttrLanguage*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_LANGUAGE)) != NULL)
    {
      value = g_strdup( pango_language_to_string( pango_lang->value));
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_LANGUAGE, 
                                                   value);
    } 
  if ((pango_float = (PangoAttrFloat*) pango_attr_iterator_get (iter, 
                                   PANGO_ATTR_SCALE)) != NULL)
    {
      value = g_strdup_printf("%g", pango_float->value);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_SCALE, 
                                                   value);
    } 
  if ((pango_color = (PangoAttrColor*) pango_attr_iterator_get (iter, 
                                    PANGO_ATTR_FOREGROUND)) != NULL)
    {
      value = g_strdup_printf ("%u,%u,%u", 
                               pango_color->color.red, 
                               pango_color->color.green, 
                               pango_color->color.blue);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_FG_COLOR, 
                                                   value);
    } 
  if ((pango_color = (PangoAttrColor*) pango_attr_iterator_get (iter, 
                                     PANGO_ATTR_BACKGROUND)) != NULL)
    {
      value = g_strdup_printf ("%u,%u,%u", 
                               pango_color->color.red, 
                               pango_color->color.green, 
                               pango_color->color.blue);
      attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                                   ATK_TEXT_ATTR_FG_COLOR, 
                                                   value);
    } 
  return attrib_set;
}

/**
 * gail_text_helper_get_default_attributes:
 * @attrib_set: The #AtkAttributeSet to add the attribute to
 * @layout: The PangoLayout from which the attributes will be obtained
 * @widget: The GtkWidget for which the default attributes are required.
 *
 * Returns: A pointer to the #AtkAttributeSet.
 **/
AtkAttributeSet* 
gail_text_helper_get_default_attributes (AtkAttributeSet *attrib_set,
                                         PangoLayout     *layout,
                                         GtkWidget       *widget)
{
  PangoContext *context;
  GtkStyle *style_value;

  attrib_set = gail_text_helper_add_attribute (attrib_set, 
                                               ATK_TEXT_ATTR_DIRECTION,
     g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_DIRECTION, 
                                        gtk_widget_get_direction (widget))));

  context = pango_layout_get_context (layout);
  if (context)
    {
      PangoLanguage* language;
      PangoFontDescription* font;

      language = pango_context_get_language (context);
      if (language)
        {
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_LANGUAGE,
                      g_strdup (pango_language_to_string (language)));
        }
      font = pango_context_get_font_description (context);
      if (font)
        {
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_STYLE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STYLE,
                                   pango_font_description_get_style (font))));
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_VARIANT,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_VARIANT,
                                   pango_font_description_get_variant (font))));
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_STRETCH,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STRETCH,
                                   pango_font_description_get_stretch (font))));
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                     ATK_TEXT_ATTR_FAMILY_NAME,
              g_strdup (pango_font_description_get_family (font)));
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_WEIGHT,
                    g_strdup_printf ("%d",
                                   pango_font_description_get_weight (font)));
          attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                       ATK_TEXT_ATTR_SIZE,
                  g_strdup_printf ("%i",
                                   pango_font_description_get_size (font)));
        }
    }
  style_value = gtk_widget_get_style (widget);
  if (style_value)
    {
      GdkColor color;
      gchar *value;

      color = style_value->base[GTK_STATE_NORMAL];
      value = g_strdup_printf ("%u,%u,%u",
                               color.red, color.green, color.blue);
      attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                   ATK_TEXT_ATTR_BG_COLOR,
                                                   value); 
      color = style_value->text[GTK_STATE_NORMAL];
      value = g_strdup_printf ("%u,%u,%u",
                               color.red, color.green, color.blue);
      attrib_set = gail_text_helper_add_attribute (attrib_set,
                                                   ATK_TEXT_ATTR_FG_COLOR,
                                                   value); 
    }
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_FG_STIPPLE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_FG_STIPPLE, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_BG_STIPPLE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_BG_STIPPLE, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_STRIKETHROUGH,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_STRIKETHROUGH, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_UNDERLINE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_UNDERLINE, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_RISE,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_SCALE,
                                               g_strdup_printf ("%g", 1.0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_BG_FULL_HEIGHT,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_PIXELS_INSIDE_WRAP,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_PIXELS_BELOW_LINES,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_PIXELS_ABOVE_LINES,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_EDITABLE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_EDITABLE, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_INVISIBLE,
              g_strdup (atk_attribute_get_value (ATK_TEXT_ATTR_INVISIBLE, 
                                                 0))); 
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_INDENT,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_RIGHT_MARGIN,
                                               g_strdup_printf ("%i", 0));
  attrib_set = gail_text_helper_add_attribute (attrib_set,
                                               ATK_TEXT_ATTR_LEFT_MARGIN,
                                               g_strdup_printf ("%i", 0));
  return attrib_set;
}

static void
get_pango_text_offsets (PangoLayout         *layout,
                        GtkTextBuffer       *buffer,
                        GailOffsetType      function,
                        AtkTextBoundary     boundary_type,
                        gint                offset,
                        gint                *start_offset,
                        gint                *end_offset,
                        GtkTextIter         *start_iter,
                        GtkTextIter         *end_iter)
{
  PangoLayoutIter *iter;
  PangoLayoutLine *line, *prev_line = NULL, *prev_prev_line = NULL;
  gint index, start_index, end_index;
  const gchar *text;
  gboolean found = FALSE;

  text = pango_layout_get_text (layout);
  index = g_utf8_offset_to_pointer (text, offset) - text;
  iter = pango_layout_get_iter (layout);
  do
    {
      line = pango_layout_iter_get_line (iter);
      start_index = line->start_index;
      end_index = start_index + line->length;

      if (index >= start_index && index <= end_index)
        {
          /*
           * Found line for offset
           */
          switch (function)
            {
            case GAIL_BEFORE_OFFSET:
                  /*
                   * We want the previous line
                   */
              if (prev_line)
                {
                  switch (boundary_type)
                    {
                    case ATK_TEXT_BOUNDARY_LINE_START:
                      end_index = start_index;
                      start_index = prev_line->start_index;
                      break;
                    case ATK_TEXT_BOUNDARY_LINE_END:
                      if (prev_prev_line)
                        start_index = prev_prev_line->start_index + 
                                  prev_prev_line->length;
                      end_index = prev_line->start_index + prev_line->length;
                      break;
                    default:
                      g_assert_not_reached();
                    }
                }
              else
                start_index = end_index = 0;
              break;
            case GAIL_AT_OFFSET:
              switch (boundary_type)
                {
                case ATK_TEXT_BOUNDARY_LINE_START:
                  if (pango_layout_iter_next_line (iter))
                    end_index = pango_layout_iter_get_line (iter)->start_index;
                  break;
                case ATK_TEXT_BOUNDARY_LINE_END:
                  if (prev_line)
                    start_index = prev_line->start_index + 
                                  prev_line->length;
                  break;
                default:
                  g_assert_not_reached();
                }
              break;
            case GAIL_AFTER_OFFSET:
               /*
                * We want the next line
                */
              if (pango_layout_iter_next_line (iter))
                {
                  line = pango_layout_iter_get_line (iter);
                  switch (boundary_type)
                    {
                    case ATK_TEXT_BOUNDARY_LINE_START:
                      start_index = line->start_index;
                      if (pango_layout_iter_next_line (iter))
                        end_index = pango_layout_iter_get_line (iter)->start_index;
                      else
                        end_index = start_index + line->length;
                      break;
                    case ATK_TEXT_BOUNDARY_LINE_END:
                      start_index = end_index;
                      end_index = line->start_index + line->length;
                      break;
                    default:
                      g_assert_not_reached();
                    }
                }
              else
                start_index = end_index;
              break;
            }
          found = TRUE;
          break;
        }
      prev_prev_line = prev_line; 
      prev_line = line; 
    }
  while (pango_layout_iter_next_line (iter));

  if (!found)
    {
      start_index = prev_line->start_index + prev_line->length;
      end_index = start_index;
    }
  pango_layout_iter_free (iter);
  *start_offset = g_utf8_pointer_to_offset (text, text + start_index);
  *end_offset = g_utf8_pointer_to_offset (text, text + end_index);
 
  gtk_text_buffer_get_iter_at_offset (buffer, start_iter, *start_offset);
  gtk_text_buffer_get_iter_at_offset (buffer, end_iter, *end_offset);
}
