/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-glade.h"
#include "galeon-marshal.h"
#include "bookmarks-editor.h"
#include "bookmarks-io.h"
#include "bookmarks-single-editor.h"
#include "bookmarks-clipboard.h"
#include "bookmarks-context-menu.h"
#include "bookmarks-import-druid-mozilla.h"
#include "bookmarks-export-druid-mozilla.h"
#include "bookmarks-import-druid-netscape.h"
#include "bookmarks-export-druid-netscape.h"
#include "bookmarks-import-druid-konqueror.h"
#include "bookmarks-export-druid-konqueror.h"
#include "bookmarks-tree-view.h"
#include "gul-gobject-misc.h"
#include "gul-string.h"
#include "gul-state.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtktreeselection.h>
#include <bonobo/bonobo-window.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-file-selector-util.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GbEditorPrivate 
{
	GbBookmarkSet *set;
	GbTreeModel *btree_model;
	gint block_update;
	
	GtkWidget *main_widget;
	GbTreeView *tree_view;
	
	GbClipboard *clipboard;
	
	GbLocationSource *location_source;
};

/**
 * Private functions, only availble from this file
 */
static void		gb_editor_class_init		(GbEditorClass *klass);
static void		gb_editor_init			(GbEditor *e);
static void		gb_editor_init_widgets		(GbEditor *e);
static void		gb_editor_init_widgets_impl	(GbEditor *e);
static void 		gb_editor_ensure_init_widgets	(GbEditor *e);
static void		gb_editor_finalize_impl		(GObject *o);
static void		gb_editor_selection_changed_cb	(GtkTreeSelection *treeselection, GbEditor *e);
static void		gb_editor_show_data		(GbEditor *e);
static void		gb_editor_add_new		(GbEditor *e, GbBookmark *new_bookmark);
static void 		gb_editor_add_several		(GbEditor *e, GSList *l);
static void		gb_editor_bookmark_activated_cb	(GObject *sender,
							 GbBookmarkEventActivated *ev,
							 GbEditor *e);
static gboolean		gb_editor_window_configure_event_cb (GtkWidget *widget,
							     GdkEventConfigure *event,
							     GbEditor *e);


static void		gb_editor_bonobo_cmd_add_site		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_add_smart_site	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_add_folder		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_add_separator	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_add_alias		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_add_autobookmarks	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_edit_remove	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar* verbname);
static void		gb_editor_bonobo_cmd_file_open		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_file_close		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_properties	(BonoboUIComponent *uic, GbEditor *e, 
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_cut		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_copy		(BonoboUIComponent *uic, GbEditor *e, 
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_paste		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_set_as_default_folder (BonoboUIComponent *uic,
									 GbEditor *e,
									 const gchar *verbname);
static void		gb_editor_bonobo_cmd_import_mozilla	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_import_netscape	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_import_konqueror	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_export_mozilla	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_export_netscape	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_export_konqueror	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_sort_one_level	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_sort_recursively	(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);
static void		gb_editor_bonobo_cmd_sort_one_level_folders_first (BonoboUIComponent *uic,
									   GbEditor *e,
									   const gchar *verbname);
static void		gb_editor_bonobo_cmd_sort_recursively_folders_first (BonoboUIComponent *uic,
									     GbEditor *e,
									     const gchar *verbname);
static void		gb_editor_bonobo_cmd_edit_goto		(BonoboUIComponent *uic, GbEditor *e,
								 const gchar *verbname);


BonoboUIVerb bookmarks_editor_verbs [] = {
        BONOBO_UI_VERB ("AddSite", (BonoboUIVerbFn)gb_editor_bonobo_cmd_add_site),
        BONOBO_UI_VERB ("AddSmartSite", (BonoboUIVerbFn)gb_editor_bonobo_cmd_add_smart_site),
	BONOBO_UI_VERB ("AddFolder", (BonoboUIVerbFn)gb_editor_bonobo_cmd_add_folder),
	BONOBO_UI_VERB ("AddSeparator", (BonoboUIVerbFn) gb_editor_bonobo_cmd_add_separator),
        BONOBO_UI_VERB ("AddAlias", (BonoboUIVerbFn) gb_editor_bonobo_cmd_add_alias),
        BONOBO_UI_VERB ("AddAutobookmarks", (BonoboUIVerbFn) gb_editor_bonobo_cmd_add_autobookmarks),
	/*
	  BONOBO_UI_VERB ("FileNew", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_new),
	*/
        BONOBO_UI_VERB ("FileOpen", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_open),
	/*
	  BONOBO_UI_VERB ("FileSave", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_save),
	  BONOBO_UI_VERB ("FileSaveAs", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_save_as),
	  BONOBO_UI_VERB ("FileRevert", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_revert),
	*/
	BONOBO_UI_VERB ("FileClose", (BonoboUIVerbFn) gb_editor_bonobo_cmd_file_close),
	BONOBO_UI_VERB ("EditProperties", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_properties),
	BONOBO_UI_VERB ("EditCut", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_cut),
        BONOBO_UI_VERB ("EditCopy", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_copy),
        BONOBO_UI_VERB ("EditPaste", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_paste),
        BONOBO_UI_VERB ("EditRemove", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_remove),
	/*
	  BONOBO_UI_VERB ("EditUndo", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_undo),
	  BONOBO_UI_VERB ("EditRedo", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_redo),
	  BONOBO_UI_VERB ("EditFind", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_find),
	*/
	BONOBO_UI_VERB ("EditGoto", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_goto),
	BONOBO_UI_VERB ("EditSetAsDefaultFolder", (BonoboUIVerbFn) 
			gb_editor_bonobo_cmd_edit_set_as_default_folder),
	/*
	  BONOBO_UI_VERB ("EditMoveUp", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_move_up),
	  BONOBO_UI_VERB ("EditMoveDown", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_move_down),
	  BONOBO_UI_VERB ("EditFetchFavicon", (BonoboUIVerbFn) gb_editor_bonobo_cmd_edit_fetch_favicon),
	*/
	BONOBO_UI_VERB ("SortOneLevel", (BonoboUIVerbFn) gb_editor_bonobo_cmd_sort_one_level),
	BONOBO_UI_VERB ("SortOneLevelFoldersFirst", (BonoboUIVerbFn) 
			gb_editor_bonobo_cmd_sort_one_level_folders_first),
	BONOBO_UI_VERB ("SortRecursively", (BonoboUIVerbFn) gb_editor_bonobo_cmd_sort_recursively),
	BONOBO_UI_VERB ("SortRecursivelyFoldersFirst", (BonoboUIVerbFn) 
			gb_editor_bonobo_cmd_sort_recursively_folders_first),
	BONOBO_UI_VERB ("ImportMozilla", (BonoboUIVerbFn) gb_editor_bonobo_cmd_import_mozilla),
	BONOBO_UI_VERB ("ImportNetscape", (BonoboUIVerbFn) gb_editor_bonobo_cmd_import_netscape),
	BONOBO_UI_VERB ("ImportKonqueror", (BonoboUIVerbFn) gb_editor_bonobo_cmd_import_konqueror),
	BONOBO_UI_VERB ("ExportMozilla", (BonoboUIVerbFn) gb_editor_bonobo_cmd_export_mozilla),
	BONOBO_UI_VERB ("ExportNetscape", (BonoboUIVerbFn) gb_editor_bonobo_cmd_export_netscape),
	BONOBO_UI_VERB ("ExportKonqueror", (BonoboUIVerbFn) gb_editor_bonobo_cmd_export_konqueror),
	
        BONOBO_UI_VERB_END
};

static gpointer g_object_class;

/* signals enums and ids */
enum GbEditorSignalsEnum {
	GB_EDITOR_BOOKMARK_ACTIVATED,
	GB_EDITOR_LAST_SIGNAL
};
static gint GbEditorSignals[GB_EDITOR_LAST_SIGNAL];

/**
 * Editor object
 */

MAKE_GET_TYPE (gb_editor, "GbEditor", GbEditor, gb_editor_class_init, 
	       gb_editor_init, G_TYPE_OBJECT);

static void
gb_editor_class_init (GbEditorClass *klass)
{
	klass->gb_editor_init_widgets = gb_editor_init_widgets_impl;
	G_OBJECT_CLASS (klass)->finalize = gb_editor_finalize_impl;
	
	GbEditorSignals[GB_EDITOR_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbEditorClass, gb_editor_bookmark_activated), 
		NULL, NULL, 
		g_cclosure_marshal_VOID__POINTER,
		G_TYPE_NONE, 1, G_TYPE_POINTER);
	
	g_object_class = g_type_class_peek_parent (klass);
}

static void 
gb_editor_init (GbEditor *e)
{
	GbEditorPrivate *p = g_new0 (GbEditorPrivate, 1);
	e->priv = p;
	
}

static void
gb_editor_finalize_impl (GObject *o)
{
	GbEditor *e = GB_EDITOR (o);
	GbEditorPrivate *p = e->priv;
	
	gb_editor_set_location_source (e, NULL);
	
	if (p->btree_model) g_object_unref (p->btree_model);
	if (p->clipboard) g_object_unref (p->clipboard);
	if (p->set) g_object_unref (p->set);
	if (p->tree_view) g_object_unref (p->tree_view);

	g_free (p);
	
	DEBUG_MSG (("GbEditor finalized\n"));
	
	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

static void
gb_editor_init_widgets (GbEditor *e)
{
	GbEditorClass *klass = GB_EDITOR_GET_CLASS (e);
	return klass->gb_editor_init_widgets (e);
}

static void 
gb_editor_ensure_init_widgets (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	if (!p->main_widget)
	{
		GtkTreeSelection *s;
		p->clipboard = gb_clipboard_get_shared ();
		
		gb_editor_init_widgets (e);
		
		s = gtk_tree_view_get_selection (GTK_TREE_VIEW (p->tree_view));
		gtk_tree_selection_set_mode (s, GTK_SELECTION_MULTIPLE);
		g_signal_connect (s, "changed", G_CALLBACK (gb_editor_selection_changed_cb), e);
	
		gb_editor_show_data (e);
	}
}


GbTreeView *
gb_editor_get_tree_view	(GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	if (!p->tree_view)
	{
		p->tree_view = gb_tree_view_new ();
		if (p->location_source)
		{
			gb_tree_view_set_location_source (p->tree_view, p->location_source);
		}
	
		g_signal_connect (p->tree_view, "bookmark-activated", 
				  G_CALLBACK (gb_editor_bookmark_activated_cb), e);

		gtk_widget_show (GTK_WIDGET (p->tree_view));
		g_object_ref (p->tree_view);
		gtk_object_sink (GTK_OBJECT (p->tree_view));
	}
	return p->tree_view;
}

void
gb_editor_set_main_widget (GbEditor *e, GtkWidget *w)
{
	GbEditorPrivate *p = e->priv;

	g_return_if_fail (p->main_widget == NULL);
	g_return_if_fail (GTK_IS_WIDGET (w));

	p->main_widget = w;
	g_object_ref (p->main_widget);
}

static void
gb_editor_init_widgets_impl (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GtkWidget *contents;
	GtkWidget *mw;
	
	BonoboWindow *win;
	BonoboUIContainer *container;
	BonoboUIComponent *ui_component;
	Bonobo_UIContainer corba_container;
	
	mw = bonobo_window_new ("Galeon Bookmarks Editor", "Bookmarks Editor");
	gb_editor_set_main_widget (e, mw);

	g_object_weak_ref (G_OBJECT (mw), (GWeakNotify) g_object_unref, e);

	/* makes sure the tree view has been created */
	gb_editor_get_tree_view (e);
	
	contents = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (contents), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (contents), GTK_WIDGET (p->tree_view));
	
	gtk_widget_show_all (contents);
	
	win = BONOBO_WINDOW (p->main_widget);
	bonobo_window_set_contents (win, contents);
	container = bonobo_window_get_ui_container (win);
        bonobo_ui_engine_config_set_path (bonobo_window_get_ui_engine (win), 
					  "/galeon/bookmarksEditor/UIConfig/kvps");
        corba_container = BONOBO_OBJREF (container);
	ui_component = bonobo_ui_component_new_default ();
	bonobo_ui_component_set_container (ui_component, corba_container, NULL);
	bonobo_ui_util_set_ui (ui_component, DATADIR, 
			       "galeon-bookmarks-editor-ui.xml", 
			       "galeon-bookmarks", NULL);
	bonobo_ui_component_add_verb_list_with_data (ui_component, bookmarks_editor_verbs, e);
	
	/* window size saving crack */
	gul_state_load_window (p->main_widget, "bookmarks_editor", 400, 300, FALSE);
	
	g_signal_connect (p->main_widget, "configure_event", 
			  G_CALLBACK (gb_editor_window_configure_event_cb), e);
	
}

GbEditor *
gb_editor_new (GbTreeModel *model)
{
	GbEditor *ret = g_object_new (GB_TYPE_EDITOR, NULL);
	gb_editor_set_model (ret, model);
	return ret;
}

GbEditor *
gb_editor_new_for_set (GbBookmarkSet *set)
{
	GbTreeModel *tm;
	GbEditor *e;
	
	g_return_val_if_fail (GB_IS_BOOKMARK_SET (set), NULL);
	g_return_val_if_fail (GB_IS_FOLDER (set->root), NULL);
	
	tm = gb_tree_model_new_for_set (set);
	e = gb_editor_new (tm);
	g_object_unref (tm);
	
	return e;
}

void
gb_editor_set_model (GbEditor *e, GbTreeModel *m)
{
	GbEditorPrivate *p = e->priv;
	
	gb_editor_ensure_init_widgets (e);

	g_return_if_fail (GB_IS_TREE_MODEL (m));
	g_return_if_fail (GB_IS_TREE_VIEW (p->tree_view));
	
	if (p->btree_model)
	{
		g_object_unref (G_OBJECT (p->btree_model));
	}
	if (p->set)
	{
		g_object_unref (G_OBJECT (p->set));
	}
	p->btree_model = m;
	g_object_ref (G_OBJECT (m));
	p->set = gb_tree_model_get_set (m);
	if (p->set)
	{
		g_object_ref (G_OBJECT (p->set));
	}
	
	gb_tree_view_set_model (p->tree_view, 
				p->btree_model);
}

void
gb_editor_set_location_source (GbEditor *e, GbLocationSource *s)
{
	GbEditorPrivate *p = e->priv;
	
	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}
	
	p->location_source = s;
	
	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}
	
	if (p->tree_view && GB_IS_TREE_VIEW (p->tree_view))
	{
		gb_tree_view_set_location_source (p->tree_view, s);
	}
}

static void
gb_editor_selection_changed_cb (GtkTreeSelection *treeselection, GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	p->block_update++;
	
	gb_editor_show_data (e);
	
	p->block_update--;
	
}

static void
gb_editor_show_data (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	const GSList *selected = gb_tree_view_get_selected_list (p->tree_view);
	p->block_update++;
	if (selected && !selected->next)
		/* there's one and only one item selected */
	{
		/* this does not do anything now. Previously, it displayed all the info
		   about the selected bookmark */
		
		/* probably I should update the UI depending of the
		 * type of the selected bookmark */
	}
	p->block_update--;
}


GtkWidget *
gb_editor_get_main_widget (GbEditor *editor)
{
	GbEditorPrivate *p = editor->priv;
	return p->main_widget;
}

void
gb_editor_select (GbEditor *e, GbBookmark *b)
{
	GbEditorPrivate *p = e->priv;
	g_return_if_fail (GB_IS_TREE_VIEW (p->tree_view));
	
	gb_tree_view_select (p->tree_view, b);
}

static gboolean
gb_editor_window_configure_event_cb (GtkWidget *widget,
				     GdkEventConfigure *event,
				     GbEditor *e)
{
	gul_state_save_window (widget, "bookmarks_editor");
	
	return FALSE;
}

static void
gb_editor_bookmark_activated_cb (GObject *sender,
				 GbBookmarkEventActivated *ev,
				 GbEditor *e)
{
	g_signal_emit_by_name (e, "bookmark-activated", ev);
}

static void 
gb_editor_add_new (GbEditor *e, GbBookmark *new_bookmark)
{
	GSList *l = g_slist_prepend (NULL, new_bookmark);
	gb_editor_add_several (e, l);
	g_slist_free (l);
}

static void 
gb_editor_add_several (GbEditor *e, GSList *l)
{
	GbEditorPrivate *p = e->priv;
	GbFolder *parent = NULL;
	GbBookmark *next = NULL;
	gint index;
	GSList *li;
	const GSList *selected = gb_tree_view_get_selected_list (p->tree_view);
	if (selected && !selected->next)
		/* there's one and only one item selected */
	{
		GbBookmark *s = selected->data;
		if (GB_IS_FOLDER (s))
		{
			parent = GB_FOLDER (s);
		} 
		else
		{
			next = s;
			parent = s->parent;
		}
	}
	
	if (parent == NULL)
	{
		GbBookmark *fr = gb_tree_model_get_first_root (p->btree_model);
		if (GB_IS_FOLDER (fr))
		{
			parent = GB_FOLDER (fr);
		}
		else
		{
			/* should not happen, yet */
			g_warning ("couldn't find a parent");
			return;
		}
		next = NULL;
	}
	
	if (next != NULL)
	{
		index = gb_folder_get_child_index (parent, next);
	}
	else
	{
		index = 0;
	}
	
	for (li = l; li; li = li->next)
	{
		gb_folder_add_child (parent, li->data, index++);
	}
	
	if (l && !l->next)
	{
		gb_editor_select (e, l->data);
	}
}

static void
gb_editor_bonobo_cmd_add_site (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_site (e);
}

void 
gb_editor_cmd_add_site (GbEditor *e)
{
	/* see also bookmarks-context-menu.c */
	GbSite *new = gb_site_new (gb_tree_model_get_set (e->priv->btree_model),
				   _("New Site"), "http://");
	gb_editor_add_new (e, GB_BOOKMARK (new));
	g_object_unref (G_OBJECT (new));
}

static void
gb_editor_bonobo_cmd_add_smart_site (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_smart_site (e);
}

void 
gb_editor_cmd_add_smart_site (GbEditor *e)
{
	/* see also bookmarks-context-menu.c */
	GbSmartSite *new = gb_smart_site_new (gb_tree_model_get_set (e->priv->btree_model),
					      _("New Smart Site"), "http://", "http://%s");
	gb_editor_add_new (e, GB_BOOKMARK (new));
	g_object_unref (G_OBJECT (new));
}

static void
gb_editor_bonobo_cmd_add_folder (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_folder (e);
}

void
gb_editor_cmd_add_folder (GbEditor *e)
{
	GbFolder *new = gb_folder_new (gb_tree_model_get_set (e->priv->btree_model),
				       _("New folder"));
	gb_editor_add_new (e, GB_BOOKMARK (new));
	g_object_unref (G_OBJECT (new));
}

static void
gb_editor_bonobo_cmd_add_separator (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_separator (e);
}

void
gb_editor_cmd_add_separator (GbEditor *e)
{
	GbSeparator *new = gb_separator_new (gb_tree_model_get_set (e->priv->btree_model));
	gb_editor_add_new (e, GB_BOOKMARK (new));
	g_object_unref (G_OBJECT (new));
}

static void
gb_editor_bonobo_cmd_add_alias (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_alias (e);
}

void
gb_editor_cmd_add_alias (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GSList *sel = gb_tree_view_get_selected_list_prepared (p->tree_view);
	GSList *seli;
	for (seli = sel; seli; seli = seli->next)
	{
		GbBookmark *b = seli->data;
		GbAliasPlaceholder *ap = gb_alias_placeholder_new (b->set, b->id);
		GbBookmark *new = gb_bookmark_alias_create (seli->data, ap);
		gb_editor_add_new (e, GB_BOOKMARK (new));
		g_object_unref (new);
		g_object_unref (ap);
	}
	g_slist_free (sel);
}

static void
gb_editor_bonobo_cmd_add_autobookmarks (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_add_autobookmarks (e);
}

void
gb_editor_cmd_add_autobookmarks (GbEditor *e)
{
	GbAutoFolder *new = gb_auto_folder_new (gb_tree_model_get_set (e->priv->btree_model), 
						_("New Autobookmarks Folder"));
	gb_editor_add_new (e, GB_BOOKMARK (new));
	g_object_unref (G_OBJECT (new));
}

static void
gb_editor_bonobo_cmd_edit_remove (BonoboUIComponent *uic, GbEditor *e, const gchar* verbname)
{
	gb_editor_cmd_edit_remove (e);
}

void
gb_editor_cmd_edit_remove (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GSList *sel = gb_tree_view_get_selected_list_prepared (p->tree_view);
	GSList *seli;
	for (seli = sel; seli; seli = seli->next)
	{
		gb_bookmark_unparent (seli->data);
	}
	g_slist_free (sel);
}

static void
gb_editor_bonobo_cmd_file_open (BonoboUIComponent *uic, GbEditor *e, const gchar *verbname)
{
	gb_editor_cmd_file_open (e);
}

void
gb_editor_cmd_file_open (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	gchar *dir = eel_gconf_get_string (CONF_STATE_OPEN_DIR);
	//GbBookmarkSet *currentset = p->set;
	char *file;
	
	file = bonobo_file_selector_open 
		(GTK_IS_WINDOW (p->main_widget) ? GTK_WINDOW (p->main_widget) : NULL,
		 FALSE, _("Select the bookmarks file to open"),
		 "XBEL files: application/xbel"
		 "|Mozilla bookmarks files:application/mozilla-bookmarks,text/html"
		 "|Netscape bookmarks files:application/netscape-bookmarks,text/html", 
		 dir);
	g_free (dir);
	
	if (file)
	{
		GbIO *io = gb_io_object_for_file (file);
		GbBookmarkSet *newset = gb_io_load_from_file (io, file);
		GbEditor *e;
		if (newset)
		{
			e = gb_editor_new_for_set (newset);
			g_object_unref (G_OBJECT (newset));
		}
		else
		{
			/* FIXME: show an error message */
			NOT_IMPLEMENTED;
		}
		
		g_free (file);
	}
}


static void
gb_editor_bonobo_cmd_file_close (BonoboUIComponent *uic, GbEditor *e,
				 const gchar *verbname)
{
	gb_editor_cmd_file_close (e);
}

void
gb_editor_cmd_file_close (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	gtk_widget_destroy (p->main_widget);
}

static void
gb_editor_bonobo_cmd_edit_properties (BonoboUIComponent *uic, GbEditor *e, const gchar *verbname)
{
	gb_editor_cmd_edit_properties (e);
}

void
gb_editor_cmd_edit_properties (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	const GSList *sel = gb_tree_view_get_selected_list (p->tree_view);
	while (sel)
	{
		if (!GB_IS_SEPARATOR (sel->data))
		{
			GbSingleEditor *se = gb_single_editor_new ();
			gb_single_editor_set_bookmark (se, sel->data);
			gb_single_editor_show (se);
		}
		sel = sel->next;
	}
}

static void
gb_editor_bonobo_cmd_edit_cut (BonoboUIComponent *uic, GbEditor *e, const gchar *verbname)
{
	gb_editor_cmd_edit_cut (e);
}

void
gb_editor_cmd_edit_cut (GbEditor *e)
{
	gb_editor_cmd_edit_copy (e);
	gb_editor_cmd_edit_remove (e);
}

static void
gb_editor_bonobo_cmd_edit_copy (BonoboUIComponent *uic, GbEditor *e, const gchar *verbname)
{
	gb_editor_cmd_edit_copy (e);
}

void
gb_editor_cmd_edit_copy (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GSList *sel = gb_tree_view_get_selected_list_prepared (p->tree_view);
	GSList *seli;
	gb_clipboard_clear (p->clipboard);
	for (seli = sel; seli; seli = seli->next)
	{
		gb_clipboard_add (p->clipboard, seli->data);
	}
	g_slist_free (sel);
}

static void
gb_editor_bonobo_cmd_edit_paste (BonoboUIComponent *uic, GbEditor *e, const gchar *verbname)
{
	gb_editor_cmd_edit_paste (e);
}

void
gb_editor_cmd_edit_paste (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	if (gb_clipboard_can_paste (p->clipboard))
	{
		GbBookmarkSet *s = gb_clipboard_paste (p->clipboard);
		GSList *l;
		if (!s) return;
		g_return_if_fail (GB_IS_BOOKMARK_SET (s) && GB_IS_FOLDER (s->root));
		l = gb_folder_list_children (s->root);
		gb_editor_add_several (e, l);
		g_slist_free (l);
		g_object_unref (s);
	}
}

static void
gb_editor_bonobo_cmd_edit_set_as_default_folder (BonoboUIComponent *uic, GbEditor *e,
						 const gchar *verbname)
{
	gb_editor_cmd_edit_set_as_default_folder (e);
}

void
gb_editor_cmd_edit_set_as_default_folder (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	const GSList *selected = gb_tree_view_get_selected_list (p->tree_view);
	if (selected && !selected->next && GB_IS_FOLDER (selected->data))
	{
		GbFolder *f = selected->data;
		gb_bookmark_set_set_default_folder (GB_BOOKMARK (f)->set, f);
	}
}

static void
gb_editor_bonobo_cmd_import_mozilla (BonoboUIComponent *uic, GbEditor *e,
				     const gchar *verbname)
{
	gb_editor_cmd_import_mozilla (e);
}

void
gb_editor_cmd_import_mozilla (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbImportDruid *d = GB_IMPORT_DRUID (gb_import_druid_mozilla_new ());
	gb_import_druid_set_dest_bookmark_set (d, p->set);
	gb_import_druid_show (d);
}

static void
gb_editor_bonobo_cmd_import_netscape (BonoboUIComponent *uic, GbEditor *e,
				      const gchar *verbname)
{
	gb_editor_cmd_import_netscape (e);
}

void
gb_editor_cmd_import_netscape (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbImportDruid *d = GB_IMPORT_DRUID (gb_import_druid_netscape_new ());
	gb_import_druid_set_dest_bookmark_set (d, p->set);
	gb_import_druid_show (d);
}

static void
gb_editor_bonobo_cmd_import_konqueror (BonoboUIComponent *uic, GbEditor *e,
				       const gchar *verbname)
{
	gb_editor_cmd_import_konqueror (e);
}

void
gb_editor_cmd_import_konqueror (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbImportDruid *d = GB_IMPORT_DRUID (gb_import_druid_konqueror_new ());
	gb_import_druid_set_dest_bookmark_set (d, p->set);
	gb_import_druid_show (d);
}

static void
gb_editor_bonobo_cmd_sort_one_level (BonoboUIComponent *uic, GbEditor *e,
				     const gchar *verbname)
{
	gb_editor_cmd_sort (e, FALSE, FALSE);
}

static void
gb_editor_bonobo_cmd_sort_recursively (BonoboUIComponent *uic, GbEditor *e,
				       const gchar *verbname)
{
	gb_editor_cmd_sort (e, FALSE, TRUE);
}

static void
gb_editor_bonobo_cmd_sort_one_level_folders_first (BonoboUIComponent *uic, GbEditor *e,
						   const gchar *verbname)
{
	gb_editor_cmd_sort (e, TRUE, FALSE);
}

static void
gb_editor_bonobo_cmd_sort_recursively_folders_first (BonoboUIComponent *uic, GbEditor *e,
						     const gchar *verbname)
{
	gb_editor_cmd_sort (e, TRUE, TRUE);
}

void
gb_editor_cmd_sort (GbEditor *e, gboolean folders_first, 
		    gboolean recursive)
{
	GbEditorPrivate *p = e->priv;
	GSList *sel = gb_tree_view_get_selected_list_prepared (p->tree_view);
	GSList *seli;
	for (seli = sel; seli; seli = seli->next)
	{
		GbFolder *f;
		g_assert (GB_IS_BOOKMARK (seli->data));
		f = GB_IS_FOLDER (seli->data) ? seli->data : GB_BOOKMARK (seli->data)->parent;
		if (f) 
		{
			gb_folder_sort (f, folders_first, recursive);
		}
	}
	g_slist_free (sel);
}

static void
gb_editor_bonobo_cmd_edit_goto (BonoboUIComponent *uic, GbEditor *e,
				const gchar *verbname)
{
	gb_editor_cmd_edit_goto (e);
}

void
gb_editor_cmd_edit_goto (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	const GSList *selected = gb_tree_view_get_selected_list (p->tree_view);

	if (!selected) return;

	if (!selected->next)
	{
		GbBookmark *b = selected->data;
		const gchar *url = GB_IS_SITE (b) ? GB_SITE (b)->url : NULL;
		gb_activated_activate (e, b, url, GB_BAF_DEFAULT);
	}
	else while (selected)
	{
		GbBookmark *b = selected->data;
		const gchar *url = GB_IS_SITE (b) ? GB_SITE (b)->url : NULL;
		gb_activated_activate (e, b, url, GB_BAF_NEW_TAB_OR_WINDOW);
		selected = selected->next;
	}
}

static void
gb_editor_bonobo_cmd_export_mozilla (BonoboUIComponent *uic, GbEditor *e,
				     const gchar *verbname)
{
	gb_editor_cmd_export_mozilla (e);
}

void
gb_editor_cmd_export_mozilla (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbExportDruid *d = GB_EXPORT_DRUID (gb_export_druid_mozilla_new ());
	gb_export_druid_set_bookmark_set (d, p->set);
	gb_export_druid_show (d);
}

static void
gb_editor_bonobo_cmd_export_netscape (BonoboUIComponent *uic, GbEditor *e,
				      const gchar *verbname)
{
	gb_editor_cmd_export_netscape (e);
}

void
gb_editor_cmd_export_netscape (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbExportDruid *d = GB_EXPORT_DRUID (gb_export_druid_netscape_new ());
	gb_export_druid_set_bookmark_set (d, p->set);
	gb_export_druid_show (d);
}

static void
gb_editor_bonobo_cmd_export_konqueror (BonoboUIComponent *uic, GbEditor *e,
				       const gchar *verbname)
{
	gb_editor_cmd_export_konqueror (e);
}

void
gb_editor_cmd_export_konqueror (GbEditor *e)
{
	GbEditorPrivate *p = e->priv;
	GbExportDruid *d = GB_EXPORT_DRUID (gb_export_druid_konqueror_new ());
	gb_export_druid_set_bookmark_set (d, p->set);
	gb_export_druid_show (d);
}

