/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gul-toolbar-item-factory.h"
#include <string.h>

#include "gul-tbi-test.h"
#include "gul-tbi-zoom.h"
#include "gul-tbi-separator.h"
#include "gul-tbi-favicon.h"
#include "gul-tbi-spinner.h"
#include "gul-tbi-location.h"
#include "gul-tbi-navigation-history.h"

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

typedef struct 
{
	const char *type_name;
	GulTbItemConstructor constructor;
} GulTbItemTypeInfo;

static GulTbItemTypeInfo gul_tb_item_default_types[] = 
{
	{ "navigation_history",		(GulTbItemConstructor) gul_tbi_navigation_history_new },
	{ "zoom", 			(GulTbItemConstructor) gul_tbi_zoom_new },
	{ "location", 			(GulTbItemConstructor) gul_tbi_location_new },
	{ "spinner",			(GulTbItemConstructor) gul_tbi_spinner_new },
	{ "favicon", 			(GulTbItemConstructor) gul_tbi_favicon_new },
	{ "separator", 			(GulTbItemConstructor) gul_tbi_separator_new },
	{ "test", 			(GulTbItemConstructor) gul_tbi_test_new },
	{ NULL, 			NULL }
}; 

static GHashTable *gul_tb_item_known_types = NULL;

static void
gul_tb_item_factory_init (void)
{
	if (gul_tb_item_known_types == NULL)
	{
		int i;
		gul_tb_item_known_types = g_hash_table_new (g_str_hash, g_str_equal);

		for (i = 0; gul_tb_item_default_types[i].type_name; ++i)
		{
			gul_toolbar_item_register_type (gul_tb_item_default_types[i].type_name,
							gul_tb_item_default_types[i].constructor);
		}
	}
}

GulTbItem *
gul_toolbar_item_create_from_string (const gchar *str)
{
	GulTbItem *ret = NULL;
	gchar *type;
	gchar *props;
	gchar *id;
	const gchar *rest;
	const gchar *lpar;
	const gchar *rpar;
	const gchar *eq;
	GulTbItemConstructor constructor;
	
	gul_tb_item_factory_init ();

	rest = str;

	eq = strchr (rest, '=');
	if (eq)
	{
		id = g_strndup (rest, eq - rest);
		rest = eq + 1;
	}
	else
	{
		id = NULL;
	}

	lpar = strchr (rest, '(');
	if (lpar)
	{
		type = g_strndup (rest, lpar - rest);
		rest = lpar + 1;
		
		rpar = strchr (rest, ')');
		if (rpar)
		{
			props = g_strndup (rest, rpar - rest);
			rest = rpar + 1;
		}
		else
		{
			props = g_strdup (rest);
		}
	}
	else
	{
		type = g_strdup (rest);
		props = NULL;
	}

	DEBUG_MSG (("gul_toolbar_item_create_from_string id=%s type=%s props=%s\n", id, type, props));

	constructor = g_hash_table_lookup (gul_tb_item_known_types, type);

	if (constructor)
	{
		ret = constructor ();
		if (id)
		{
			gul_tb_item_set_id (ret, id);
		}
		if (props)
		{
			gul_tb_item_parse_properties (ret, props);
		}
	}

	if (!ret)
	{
		g_warning ("Error creating toolbar item of type %s", type);
	}

	if (id)
	{
		g_free (id);
	}
	if (type)
	{
		g_free (type);
	}
	if (props)
	{
		g_free (props);
	}

	return ret;
}

void
gul_toolbar_item_register_type (const gchar *type, GulTbItemConstructor constructor)
{
	gul_tb_item_factory_init ();
	g_hash_table_insert (gul_tb_item_known_types, g_strdup (type), constructor);
}
