// $Id: list_conversion.h,v 1.3 2002/02/17 18:53:39 murrayc Exp $ -*- c++ -*-

/* list_conversion.h
 *
 * Copyright (C) 2000-2002 GConfmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef GCONFMM_LIST_CONVERSION_H
#define GCONFMM_LIST_CONVERSION_H

extern "C"
{
#include <gconf/gconf-value.h>
}

#include <gconfmm/schema.h>
#include <glibmm/sarray.h>


namespace Gnome
{

namespace Conf
{

namespace ListHelpers
{

//TODO:
//This is a mess, because I could not create specialised versions of the functions
//for floats and strings.
//It is also complicated by the two uses:
//Sometimes a gconf method returns a GSList* that needs to be copied and then destroyed.
//and sometimes it returns a GSList* that just needs to be copied.
//Destruction of item members using custom free functions complicates it.

//Conversion between GSLists and STL containers,
//with appropriate memory management.
template<class T_container, class T_citem>
class container_list_converter
{
public:

  typedef void (*citem_free_func)(T_citem);

  //Copy the GSList items to the container.
  //For use with pointer types.
  //Destroys the GSList and its items.
  static T_container container_from_gslist(GSList* pList, citem_free_func funcFree)
  {
    T_container listResult;

    GSList* pListItem = pList;
    while(pListItem)
    {
      T_citem pVal = (T_citem)(pListItem->data);
      if( pVal ) //Check for zero if a free func is specified. That means that it's a pointer.
      {
        typedef typename T_container::value_type type_cppitem;
        listResult.push_back(type_cppitem(pVal));

        //Free the item's data: g_slist_next won't do this.
        (*funcFree)(pVal);
        pListItem->data = 0;
      }

      pListItem = g_slist_next(pListItem);
    }

    g_slist_free(pList);

    return listResult;
  }

  //Copy the GSList items to the container.
  //Destroys the GSList.
  static T_container container_from_gslist(GSList* pList, bool bDestroyList = true, bool bDontCopyNull = false)
  {
    T_container listResult;

    GSList* pListItem = pList;
    while(pListItem)
    {
      T_citem val = (T_citem)(pListItem->data);

      bool bCopy = true;
      if(bDontCopyNull && !val)
        bCopy = false;

      if(bCopy)
      {
        typedef typename T_container::value_type type_cppitem;
        listResult.push_back(type_cppitem(val));
      }

      pListItem = g_slist_next(pListItem);
    }

    if(bDestroyList)
      g_slist_free(pList);

    return listResult;
  }


  //Get a new GSList from a container.
  //Only works for non-pointer basic types.
  static GSList* gslist_from_container(T_container container)
  {
    GSList* pList = 0;
    typedef typename T_container::iterator Titerator;
    for(Titerator iter = container.begin(); iter != container.end(); iter++)
    {
      typedef typename T_container::value_type type_cppitem;
      const type_cppitem& item = *iter;

      pList = g_slist_append(pList, (void*)(T_citem(item)));
    }

    return pList;
  }

  static void gslist_of_pointers_destroy(GSList* pList, citem_free_func funcFree)
  {
    GSList* pListItem = pList;
    while(pListItem)
    {
      T_citem pVal = (T_citem)(pListItem->data);

      if(pVal)
      {
        (*funcFree)(pVal);
        pListItem->data = 0;
      }

      pListItem = g_slist_next(pListItem);
    }

    g_slist_free(pList);
  }

};

typedef std::list<gdouble> type_list_float;
typedef std::list<Glib::ustring> /*Glib::SArray*/ type_list_string;
typedef std::list<Schema> type_list_schema;
typedef std::list<bool> type_list_bool;
typedef std::list<gint> type_list_int;

//Value List versions, used by Value:

static type_list_bool container_from_gslist_of_values_of_bools(GSList* pList)
{
  typedef type_list_bool T_container;
  typedef gboolean T_citem;
  T_container listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    GConfValue* pValue = (GConfValue*)(pListItem->data);
    if(pValue)
    {
      T_citem val = gconf_value_get_bool(pValue); //The only bit that's different.

      typedef T_container::value_type type_cppitem;
      listResult.push_back(type_cppitem(val));

      //Free the item's data: g_slist_next won't do this.
      gconf_value_free(pValue);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}


static type_list_int container_from_gslist_of_values_of_ints(GSList* pList)
{
  typedef type_list_int T_container;
  typedef gint T_citem;

  T_container listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    GConfValue* pValue = (GConfValue*)(pListItem->data);
    if(pValue)
    {
      T_citem val = gconf_value_get_int(pValue); //The only bit that's different.

      typedef T_container::value_type type_cppitem;
      listResult.push_back(type_cppitem(val));

      //Free the item's data: g_slist_next won't do this.
      gconf_value_free(pValue);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}

static type_list_float container_from_gslist_of_values_of_floats(GSList* pList)
{
  typedef type_list_float T_container;
  typedef gdouble T_citem;
  T_container listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    GConfValue* pValue = (GConfValue*)(pListItem->data);
    if(pValue)
    {
      T_citem pVal = gconf_value_get_float(pValue); //The only bit that's different.

      typedef T_container::value_type type_cppitem;
      listResult.push_back(type_cppitem(pVal));

      //Free the item's data: g_slist_next won't do this.
      gconf_value_free(pValue);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}



static type_list_string container_from_gslist_of_values_of_strings(GSList* pList)
{
  typedef type_list_string T_container;
  typedef gchar* T_citem;
  T_container listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    GConfValue* pValue = (GConfValue*)(pListItem->data);
    if(pValue)
    {
      const gchar* pVal = gconf_value_get_string(pValue); //The only bit that's different.
      if( pVal )
      {
        typedef T_container::value_type type_cppitem;
        listResult.push_back(type_cppitem(pVal));
      }

      //Free the item's data: g_slist_next won't do this.
      gconf_value_free(pValue);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}

static type_list_schema container_from_gslist_of_values_of_schemas(GSList* pList)
{
  typedef type_list_schema T_container;
  typedef GConfSchema* T_citem;
  T_container listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    GConfValue* pValue = (GConfValue*)(pListItem->data);
    if(pValue)
    {
      const GConfSchema* pVal = gconf_value_get_schema(pValue); //The only bit that's different.
      if( pVal )
      {
        typedef T_container::value_type type_cppitem;
        listResult.push_back(type_cppitem(pVal));
      }

      //Free the item's data: g_slist_next won't do this.
      gconf_value_free(pValue);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}



//We can't use the template for this, because it
//doesn't allow the same C++ and C types where pointers are used.
static type_list_float container_from_gslist_of_floats(GSList* pList)
{
  type_list_float listResult;

  GSList* pListItem = pList;
  while(pListItem)
  {
    gdouble* val = (gdouble*)(pListItem->data);
    if( val )
    {
      listResult.push_back(*val); //This is the only bit that's different to the converter template

      //Free the item's data: g_slist_next won't do this.
      g_free(val);
      pListItem->data = 0;
    }

    pListItem = g_slist_next(pListItem);
  }

  g_slist_free(pList);

  return listResult;
}


static void gslist_of_values_destroy(GSList* pList)
{
  typedef container_list_converter< std::list<Value>, GConfValue* > type_converter;
  type_converter::gslist_of_pointers_destroy(pList, &gconf_value_free);
}

//I would like to use a template specialization for these,
//but I can't get it to work, so I'm just using a
//different method:

static GSList* gslist_from_container(Glib::SArray container)
{
  GSList* pList = 0;
  for(Glib::SArray::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef Glib::SArray::value_type type_cppitem;
    const type_cppitem& item = *iter;

    pList = g_slist_append(pList, g_strdup((const gchar*)item.c_str ()));
  }

  return pList;
}


static GSList* gslist_of_values_from_container(Glib::SArray container)
{
  GSList* pList = 0;
  for(Glib::SArray::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef Glib::SArray::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfValue* pValue = gconf_value_new(GCONF_VALUE_STRING);
    gconf_value_set_string(pValue, g_strdup(item.c_str ()) );

    pList = g_slist_append(pList, pValue);
  }

  return pList;
}


//typedef std::list<gdouble> type_list_float;
static GSList* gslist_from_container(type_list_float container)
{
  GSList* pList = 0;
  for(type_list_float::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_float::value_type type_cppitem;
    const type_cppitem& item = *iter;

    gdouble* pFloat = (gdouble*)g_malloc(sizeof(gdouble));
    *pFloat = item;
    pList = g_slist_append(pList, pFloat);
  }

  return pList;
}

static GSList* gslist_of_values_from_container(type_list_float container)
{
  GSList* pList = 0;
  for(type_list_float::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_float::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfValue* pValue = gconf_value_new(GCONF_VALUE_FLOAT);
    gconf_value_set_float(pValue, item);

    pList = g_slist_append(pList, pValue);
  }

  return pList;
}

static GSList* gslist_from_container(type_list_schema container)
{
  GSList* pList = 0;
  for(type_list_schema::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_schema::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfSchema* pSchema = gconf_schema_copy(const_cast<GConfSchema*>(item.gobj()));
    pList = g_slist_append(pList, pSchema);
  }

  return pList;
}

static GSList* gslist_of_values_from_container(type_list_schema container)
{
  GSList* pList = 0;
  for(type_list_schema::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_schema::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfValue* pValue = gconf_value_new(GCONF_VALUE_SCHEMA);

    GConfSchema* pSchema = gconf_schema_copy(const_cast<GConfSchema*>(item.gobj()));
    gconf_value_set_schema(pValue, pSchema);

    pList = g_slist_append(pList, pValue);
  }

  return pList;
}

static GSList* gslist_of_values_from_container(type_list_int container)
{
  GSList* pList = 0;
  for(type_list_int::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_int::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfValue* pValue = gconf_value_new(GCONF_VALUE_INT);
    gconf_value_set_int(pValue, item);

    pList = g_slist_append(pList, pValue);
  }

  return pList;
}

static GSList* gslist_of_values_from_container(type_list_bool container)
{
  GSList* pList = 0;
  for(type_list_bool::iterator iter = container.begin(); iter != container.end(); iter++)
  {
    typedef type_list_bool::value_type type_cppitem;
    const type_cppitem& item = *iter;

    GConfValue* pValue = gconf_value_new(GCONF_VALUE_BOOL);
    gconf_value_set_bool(pValue, item);

    pList = g_slist_append(pList, pValue);
  }

  return pList;
}

} /* namespace ListHelpers */
} /* namespace Conf */
} /* namespace Gnome */

#endif //GCONFMM_LIST_CONVERSION_H
