/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * gedit-dialog-replace.c
 * This file is part of gedit
 *
 * Copyright (C) 2001-2002 Paolo Maggi 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

/*
 * Modified by the gedit Team, 1998-2002. See the AUTHORS file for a 
 * list of people on the gedit Team.  
 * See the ChangeLog files for a list of changes. 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <glade/glade-xml.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-entry.h>

#include "gedit2.h"
#include "gedit-mdi.h"
#include "gedit-utils.h"
#include "gedit-dialogs.h"
#include "gedit-document.h"
#include "gedit-mdi-child.h"
#include "gedit-view.h"
#include "gedit-debug.h"
#include "gedit-utils.h"

#define GEDIT_RESPONSE_FIND		101
#define GEDIT_RESPONSE_REPLACE		102
#define GEDIT_RESPONSE_REPLACE_ALL	103

typedef struct _GeditDialogReplace GeditDialogReplace;
typedef struct _GeditDialogFind GeditDialogFind;

struct _GeditDialogReplace {
	GtkWidget *dialog;

	GtkWidget *search_entry;
	GtkWidget *replace_entry;
	GtkWidget *search_entry_list;
	GtkWidget *replace_entry_list;
	GtkWidget *match_case_checkbutton;
	GtkWidget *entire_word_checkbutton;
	GtkWidget *wrap_around_checkbutton;
};

struct _GeditDialogFind {
	GtkWidget *dialog;

	GtkWidget *search_entry;
	GtkWidget *search_entry_list;
	GtkWidget *case_sensitive;
	GtkWidget *match_case_checkbutton;
	GtkWidget *entire_word_checkbutton;
	GtkWidget *wrap_around_checkbutton;
};

static void dialog_destroyed (GtkObject *obj,  void **dialog_pointer);

static void dialog_find_response_handler (GtkDialog *dlg, gint res_id,  GeditDialogFind *find_dialog);

static void find_dlg_find_button_pressed (GeditDialogFind * dialog);

static void replace_dlg_find_button_pressed (GeditDialogReplace * dialog);
static void replace_dlg_replace_button_pressed (GeditDialogReplace * dialog);
static void replace_dlg_replace_all_button_pressed (GeditDialogReplace * dialog);

static GeditDialogReplace *dialog_replace_get_dialog 	(void);
static GeditDialogFind    *dialog_find_get_dialog 	(void);

static GQuark was_wrap_around_id = 0;
static GQuark was_entire_word_id = 0;
static GQuark was_case_sensitive_id = 0;

GQuark 
gedit_was_wrap_around_quark (void)
{
	if (!was_wrap_around_id)
		was_wrap_around_id = g_quark_from_static_string ("GeditWasWrapAround");

	return was_wrap_around_id;
}

static void
dialog_destroyed (GtkObject *obj,  void **dialog_pointer)
{
	gedit_debug (DEBUG_SEARCH, "");

	if (dialog_pointer != NULL)
	{
		g_free (*dialog_pointer);
		*dialog_pointer = NULL;
	}	
}

static void 
text_not_found_dialog (const gchar *text, GtkWindow *parent)
{
	GtkWidget *message_dlg;

	g_return_if_fail (text != NULL);
	
	message_dlg = gtk_message_dialog_new (
			parent,
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_INFO,
			GTK_BUTTONS_OK,
			_("The text \"%s\" was not found."), text);
		
	gtk_dialog_set_default_response (GTK_DIALOG (message_dlg), GTK_RESPONSE_OK);

	gtk_window_set_resizable (GTK_WINDOW (message_dlg), FALSE);
	
	gtk_dialog_run (GTK_DIALOG (message_dlg));
  	gtk_widget_destroy (message_dlg);
}

static void
dialog_replace_response_handler (GtkDialog *dlg, gint res_id,  GeditDialogReplace *replace_dialog)
{
	gedit_debug (DEBUG_SEARCH, "");

	switch (res_id) {
		case GEDIT_RESPONSE_FIND:
			replace_dlg_find_button_pressed (replace_dialog);
			break;
			
		case GEDIT_RESPONSE_REPLACE:
			replace_dlg_replace_button_pressed (replace_dialog);
			break;
			
		case GEDIT_RESPONSE_REPLACE_ALL:
			replace_dlg_replace_all_button_pressed (replace_dialog);
			break;
		default:
			gtk_widget_destroy (replace_dialog->dialog);
	}
}

static void
dialog_find_response_handler (GtkDialog *dlg, gint res_id,  GeditDialogFind *find_dialog)
{
	gedit_debug (DEBUG_SEARCH, "");

	switch (res_id) {
		case GEDIT_RESPONSE_FIND:
			find_dlg_find_button_pressed (find_dialog);
			break;

		default:
			gtk_widget_destroy (find_dialog->dialog);
	}
}

static void 
replace_search_entry_changed (GtkEditable *editable, GeditDialogReplace *dialog)
{
	const gchar *search_string;
	
	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));		
	g_return_if_fail (search_string != NULL);

	if (strlen (search_string) <= 0)
	{
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, FALSE);
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE, FALSE);
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE_ALL, FALSE);
	}
	else
	{
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, TRUE);
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE_ALL, TRUE);
	}
}

static GeditDialogReplace *
dialog_replace_get_dialog (void)
{
	static GeditDialogReplace *dialog = NULL;
	GladeXML *gui;
	GtkWindow *window;
	GtkWidget *content;
	GtkWidget *button;
	GtkWidget *replace_with_label;
	
	gedit_debug (DEBUG_SEARCH, "");
	
	if (dialog != NULL)
	{
		gdk_window_show (dialog->dialog->window);
		gdk_window_raise (dialog->dialog->window);
		gtk_widget_grab_focus (dialog->dialog);

		return dialog;
	}

	gui = glade_xml_new ( GEDIT_GLADEDIR "replace.glade2",
			     "replace_dialog_content", NULL);

	if (!gui) {
		g_warning
		    ("Could not find replace.glade2, reinstall gedit.\n");
		return NULL;
	}

	window = GTK_WINDOW (bonobo_mdi_get_active_window
			     (BONOBO_MDI (gedit_mdi)));

	dialog = g_new0 (GeditDialogReplace, 1);

	dialog->dialog = gtk_dialog_new_with_buttons (_("Replace"),
						      window,						      
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_STOCK_CLOSE,
						      GTK_RESPONSE_CLOSE,
						      NULL);

	g_return_val_if_fail (dialog->dialog != NULL, NULL);

	/* Add Replace All button */
	gtk_dialog_add_button (GTK_DIALOG (dialog->dialog), 
			       _("Replace _All"), GEDIT_RESPONSE_REPLACE_ALL);
	
	/* Add Replace button */
	button = gedit_button_new_with_stock_image (_("_Replace"), GTK_STOCK_FIND_AND_REPLACE);
	g_return_val_if_fail (button != NULL, NULL);

	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

	gtk_widget_show (button);

	gtk_dialog_add_action_widget (GTK_DIALOG (dialog->dialog), 
				      button, GEDIT_RESPONSE_REPLACE);

	/* Add Find button */
	gtk_dialog_add_button (GTK_DIALOG (dialog->dialog), 
			       GTK_STOCK_FIND, GEDIT_RESPONSE_FIND);

	content = glade_xml_get_widget (gui, "replace_dialog_content");
	dialog->search_entry       = glade_xml_get_widget (gui, "search_for_text_entry");
	dialog->replace_entry      = glade_xml_get_widget (gui, "replace_with_text_entry");
	dialog->search_entry_list  = glade_xml_get_widget (gui, "search_for_text_entry_list");
	dialog->replace_entry_list = glade_xml_get_widget (gui, "replace_with_text_entry_list");
	replace_with_label         = glade_xml_get_widget (gui, "replace_with_label");
	dialog->match_case_checkbutton = glade_xml_get_widget (gui, "match_case_checkbutton");
	dialog->wrap_around_checkbutton =  glade_xml_get_widget (gui, "wrap_around_checkbutton");
	dialog->entire_word_checkbutton = glade_xml_get_widget (gui, "entire_word_checkbutton");

	if (!content				||
	    !dialog->search_entry 		||
	    !dialog->replace_entry  		||
	    !dialog->search_entry_list		||
	    !dialog->replace_entry_list		||  
	    !replace_with_label 		||
	    !dialog->match_case_checkbutton 	||
	    !dialog->entire_word_checkbutton 	||
	    !dialog->wrap_around_checkbutton)
	{
		g_print
		    ("Could not find the required widgets inside replace.glade2.\n");
		return NULL;
	}

	gtk_widget_show (replace_with_label);
	gtk_widget_show (dialog->replace_entry);
	gtk_widget_show (dialog->replace_entry_list);
	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox),
			    content, FALSE, FALSE, 0);

	gtk_dialog_set_default_response (GTK_DIALOG (dialog->dialog),
					 GEDIT_RESPONSE_FIND);

	g_signal_connect (G_OBJECT (dialog->search_entry_list), "changed",
			  G_CALLBACK (replace_search_entry_changed), dialog);
	
	g_signal_connect (G_OBJECT (dialog->dialog), "destroy",
			  G_CALLBACK (dialog_destroyed), &dialog);

	g_signal_connect (G_OBJECT (dialog->dialog), "response",
			  G_CALLBACK (dialog_replace_response_handler), dialog);

	g_object_unref (G_OBJECT (gui));

	gtk_window_set_resizable (GTK_WINDOW (dialog->dialog), FALSE);
	
	return dialog;
}

static void 
find_search_entry_changed (GtkEditable *editable, GeditDialogReplace *dialog)
{
	const gchar *search_string;
	
	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));		
	g_return_if_fail (search_string != NULL);

	if (strlen (search_string) <= 0)
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, FALSE);
	else
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, TRUE);
}

static GeditDialogFind *
dialog_find_get_dialog (void)
{
	static GeditDialogFind *dialog = NULL;
	GladeXML *gui;
	GtkWindow *window;
	GtkWidget *content;
	GtkWidget *replace_with_label;
	GtkWidget *replace_entry;
	GtkWidget *table;
	
	gedit_debug (DEBUG_SEARCH, "");

	if (dialog != NULL)
	{
		gdk_window_show (dialog->dialog->window);
		gdk_window_raise (dialog->dialog->window);
		gtk_widget_grab_focus (dialog->dialog);
		return dialog;
	}

	gui = glade_xml_new (GEDIT_GLADEDIR "replace.glade2",
			     "replace_dialog_content", NULL);

	if (!gui) {
		g_warning
		    ("Could not find replace.glade2, reinstall gedit.\n");
		return NULL;
	}

	window = GTK_WINDOW (bonobo_mdi_get_active_window
			     (BONOBO_MDI (gedit_mdi)));

	dialog = g_new0 (GeditDialogFind, 1);

	dialog->dialog = gtk_dialog_new_with_buttons (_("Find"),
						      window,
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_STOCK_CLOSE,
						      GTK_RESPONSE_CLOSE,
						      GTK_STOCK_FIND,
						      GEDIT_RESPONSE_FIND,
						      NULL);

	g_return_val_if_fail (dialog->dialog != NULL, NULL);

	content = glade_xml_get_widget (gui, "replace_dialog_content");

	dialog->search_entry       = glade_xml_get_widget (gui, "search_for_text_entry");
	dialog->search_entry_list  = glade_xml_get_widget (gui, "search_for_text_entry_list");	
	replace_entry      	   = glade_xml_get_widget (gui, "replace_with_text_entry_list");

	replace_with_label         = glade_xml_get_widget (gui, "replace_with_label");
	
	dialog->match_case_checkbutton = glade_xml_get_widget (gui, "match_case_checkbutton");
	dialog->wrap_around_checkbutton =  glade_xml_get_widget (gui, "wrap_around_checkbutton");
	dialog->entire_word_checkbutton = glade_xml_get_widget (gui, "entire_word_checkbutton");

	table                      = glade_xml_get_widget (gui, "table");

	if (!content				||
	    !dialog->search_entry 		||
	    !dialog->search_entry_list 		||
	    !replace_entry			||
	    !replace_with_label 		||
	    !dialog->match_case_checkbutton 	||
	    !dialog->entire_word_checkbutton 	||
	    !dialog->wrap_around_checkbutton	||
	    !table) 
	{
		g_print
		    ("Could not find the required widgets inside replace.glade2.\n");
		return NULL;
	}
	
	gtk_widget_hide (replace_with_label);
	gtk_widget_hide (replace_entry);

	gtk_table_set_row_spacings (GTK_TABLE (table), 0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox),
			    content, FALSE, FALSE, 0);

	gtk_dialog_set_default_response (GTK_DIALOG (dialog->dialog),
					 GEDIT_RESPONSE_FIND);

	g_signal_connect (G_OBJECT (dialog->search_entry_list), "changed",
			  G_CALLBACK (find_search_entry_changed), dialog);

	g_signal_connect(G_OBJECT (dialog->dialog), "destroy",
			 G_CALLBACK (dialog_destroyed), &dialog);

	g_signal_connect(G_OBJECT (dialog->dialog), "response",
			 G_CALLBACK (dialog_find_response_handler), dialog);

	g_object_unref (G_OBJECT (gui));

	gtk_window_set_resizable (GTK_WINDOW (dialog->dialog), FALSE);
	
	return dialog;
}

void
gedit_dialog_find (void)
{
	GeditDialogFind *dialog;
	GeditMDIChild *active_child;
	GeditDocument *doc;
	gchar* last_searched_text;
	gboolean was_wrap_around;
	gboolean was_entire_word;
	gboolean was_case_sensitive;
	gpointer data;
	
	gedit_debug (DEBUG_SEARCH, "");

	dialog = dialog_find_get_dialog ();
	if (dialog == NULL) {
		g_warning ("Cannot create the Find dialog");
		return;
	}

	gtk_window_set_transient_for (GTK_WINDOW (dialog->dialog),
				      GTK_WINDOW
				      (bonobo_mdi_get_active_window
				       (BONOBO_MDI (gedit_mdi))));

	if (GTK_WIDGET_VISIBLE (dialog->dialog))
		return;

	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, FALSE);

	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	doc = active_child->document;
	g_return_if_fail (doc != NULL);

	last_searched_text = gedit_document_get_last_searched_text (doc);
	if (last_searched_text != NULL)
	{
		gtk_entry_set_text (GTK_ENTRY (dialog->search_entry), last_searched_text);
		g_free (last_searched_text);	
	}

	if (!was_wrap_around_id)
		was_wrap_around_id = gedit_was_wrap_around_quark ();

	if (!was_entire_word_id)
		was_entire_word_id = g_quark_from_static_string ("GeditWasEntireWord");

	if (!was_case_sensitive_id)
		was_case_sensitive_id = g_quark_from_static_string ("GeditWasCaseSensitive");

	data = g_object_get_qdata (G_OBJECT (doc), was_wrap_around_id);
	if (data == NULL)
		was_wrap_around = TRUE;
	else
		was_wrap_around = GPOINTER_TO_BOOLEAN (data);

	data = g_object_get_qdata (G_OBJECT (doc), was_entire_word_id);
	if (data == NULL)
		was_entire_word = FALSE;
	else
		was_entire_word = GPOINTER_TO_BOOLEAN (data);

	data = g_object_get_qdata (G_OBJECT (doc), was_case_sensitive_id);
	if (data == NULL)
		was_case_sensitive = FALSE;
	else
		was_case_sensitive = GPOINTER_TO_BOOLEAN (data);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton), 
				      was_case_sensitive);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton),
				      was_entire_word);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->wrap_around_checkbutton),
				      was_wrap_around);

	gtk_widget_grab_focus (dialog->search_entry);
	
	if (!GTK_WIDGET_VISIBLE (dialog->dialog))
		gtk_widget_show (dialog->dialog);
}

void
gedit_dialog_replace (void)
{
	GeditDialogReplace *dialog;
	GeditMDIChild *active_child;
	GeditDocument *doc;
	gchar* last_searched_text;
	gchar* last_replace_text;
	gboolean was_wrap_around;
	gboolean was_entire_word;
	gboolean was_case_sensitive;
	gpointer data;

	gedit_debug (DEBUG_SEARCH, "");

	dialog = dialog_replace_get_dialog ();
	if (dialog == NULL) {
		g_warning ("Cannot create the Replace dialog");
		return;
	}

	gtk_window_set_transient_for (GTK_WINDOW (dialog->dialog),
				      GTK_WINDOW
				      (bonobo_mdi_get_active_window
				       (BONOBO_MDI (gedit_mdi))));

	if (GTK_WIDGET_VISIBLE (dialog->dialog))
		return;

	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_FIND, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE_ALL, FALSE);

	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	doc = active_child->document;
	g_return_if_fail (doc != NULL);

	last_searched_text = gedit_document_get_last_searched_text (doc);
	if (last_searched_text != NULL)
	{
		gtk_entry_set_text (GTK_ENTRY (dialog->search_entry), last_searched_text);
		g_free (last_searched_text);	
	}

	last_replace_text = gedit_document_get_last_replace_text (doc);
	if (last_replace_text != NULL)
	{
		gtk_entry_set_text (GTK_ENTRY (dialog->replace_entry), last_replace_text);
		g_free (last_replace_text);	
	}

	if (!was_wrap_around_id)
		was_wrap_around_id = gedit_was_wrap_around_quark ();

	if (!was_entire_word_id)
		was_entire_word_id = g_quark_from_static_string ("GeditWasEntireWord");

	if (!was_case_sensitive_id)
		was_case_sensitive_id = g_quark_from_static_string ("GeditWasCaseSensitive");

	data = g_object_get_qdata (G_OBJECT (doc), was_wrap_around_id);
	if (data == NULL)
		was_wrap_around = TRUE;
	else
		was_wrap_around = GPOINTER_TO_BOOLEAN (data);

	data = g_object_get_qdata (G_OBJECT (doc), was_entire_word_id);
	if (data == NULL)
		was_entire_word = FALSE;
	else
		was_entire_word = GPOINTER_TO_BOOLEAN (data);

	data = g_object_get_qdata (G_OBJECT (doc), was_case_sensitive_id);
	if (data == NULL)
		was_case_sensitive = FALSE;
	else
		was_case_sensitive = GPOINTER_TO_BOOLEAN (data);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton), 
				      was_case_sensitive);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton),
				      was_entire_word);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->wrap_around_checkbutton),
				      was_wrap_around);

	gtk_widget_grab_focus (dialog->search_entry);
	
	if (!GTK_WIDGET_VISIBLE (dialog->dialog))
		gtk_widget_show (dialog->dialog);
}

static void
find_dlg_find_button_pressed (GeditDialogFind *dialog)
{
	GeditMDIChild *active_child;
	GeditView* active_view;
	GeditDocument *doc;
	const gchar* search_string = NULL;
	gboolean case_sensitive;
	gboolean entire_word;
	gboolean wrap_around;
	gboolean found;

	gedit_debug (DEBUG_SEARCH, "");

	if (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)) == NULL)
		return;
	
	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	active_view = GEDIT_VIEW (bonobo_mdi_get_active_view (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_view != NULL);
	
	doc = active_child->document;
	g_return_if_fail (doc != NULL);
			
	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));		
	g_return_if_fail (search_string != NULL);

	if (strlen (search_string) <= 0)
		return;

	gnome_entry_prepend_history (GNOME_ENTRY (dialog->search_entry_list), TRUE, search_string);
		
	case_sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton));

	entire_word = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton));
	
	wrap_around = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->wrap_around_checkbutton));

	g_object_set_qdata (G_OBJECT (doc), was_wrap_around_id, GBOOLEAN_TO_POINTER (wrap_around ));
	g_object_set_qdata (G_OBJECT (doc), was_entire_word_id, GBOOLEAN_TO_POINTER (entire_word));
	g_object_set_qdata (G_OBJECT (doc), was_case_sensitive_id, GBOOLEAN_TO_POINTER (case_sensitive));

	found = gedit_document_find (doc, search_string, TRUE, case_sensitive, entire_word);

	if (!found && wrap_around)
		found = gedit_document_find (doc, search_string, FALSE, case_sensitive, entire_word);

	if (found)
		gedit_view_scroll_to_cursor (active_view);
	else
		text_not_found_dialog (search_string, (GTK_WINDOW (dialog->dialog)));
}


static void
replace_dlg_find_button_pressed (GeditDialogReplace *dialog)
{
	/* This is basically the same as find_dlg_find_button_pressed */

	GeditMDIChild *active_child;
	GeditView* active_view;
	GeditDocument *doc;
	const gchar* search_string = NULL;
	gboolean case_sensitive;
	gboolean entire_word;
	gboolean wrap_around;
	gboolean found;

	gedit_debug (DEBUG_SEARCH, "");

	if (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)) == NULL)
		return;
	
	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	active_view = GEDIT_VIEW (bonobo_mdi_get_active_view (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_view != NULL);
	
	doc = active_child->document;
	g_return_if_fail (doc != NULL);
			
	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));		
	g_return_if_fail (search_string != NULL);

	if (strlen (search_string) <= 0)
		return;
	
	gnome_entry_prepend_history (GNOME_ENTRY (dialog->search_entry_list), TRUE, search_string);
		
	case_sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton));

	entire_word = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton));

	wrap_around = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->wrap_around_checkbutton));

	g_object_set_qdata (G_OBJECT (doc), was_wrap_around_id, GBOOLEAN_TO_POINTER (wrap_around));
	g_object_set_qdata (G_OBJECT (doc), was_entire_word_id, GBOOLEAN_TO_POINTER (entire_word));
	g_object_set_qdata (G_OBJECT (doc), was_case_sensitive_id, GBOOLEAN_TO_POINTER (case_sensitive));

	found = gedit_document_find (doc, search_string, TRUE, case_sensitive, entire_word);

	if (!found && wrap_around)
		found = gedit_document_find (doc, search_string, FALSE, case_sensitive, entire_word);

	if (found)
		gedit_view_scroll_to_cursor (active_view);
	else
		text_not_found_dialog (search_string, (GTK_WINDOW (dialog->dialog)));

	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
							GEDIT_RESPONSE_REPLACE, found);
}

static void
replace_dlg_replace_button_pressed (GeditDialogReplace *dialog)
{
	GeditMDIChild *active_child;
	GeditView* active_view;
	GeditDocument *doc;
	const gchar* search_string = NULL;
	const gchar* replace_string = NULL;
	gchar* selected_text = NULL;
	gchar *converted_search_string = NULL;
	gboolean case_sensitive;
	gboolean entire_word;
	gint start, end;
	gboolean wrap_around;
	gboolean found;

	gedit_debug (DEBUG_SEARCH, "");

	g_return_if_fail (dialog != NULL);
	
	if (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)) == NULL)
	{
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
							GEDIT_RESPONSE_REPLACE, FALSE);
		return;
	}

	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	active_view = GEDIT_VIEW (bonobo_mdi_get_active_view (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_view != NULL);
	
	doc = active_child->document;
	g_return_if_fail (doc != NULL);

	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));
	replace_string = gtk_entry_get_text (GTK_ENTRY (dialog->replace_entry));

	g_return_if_fail (search_string);
	g_return_if_fail (replace_string);

	if (strlen (search_string) <= 0)
		return;
	
	gnome_entry_prepend_history (GNOME_ENTRY (dialog->search_entry_list), TRUE, search_string);

	if (strlen (replace_string) > 0)
		gnome_entry_prepend_history (GNOME_ENTRY (dialog->replace_entry_list), 
					     TRUE, 
					     replace_string);

	if (gedit_document_get_selection (doc, &start, &end))
		selected_text = gedit_document_get_chars (doc, start, end);
	
	gedit_debug (DEBUG_SEARCH, "Sel text: %s", selected_text ? selected_text : "NULL");
	gedit_debug (DEBUG_SEARCH, "Search string: %s", search_string ? search_string : "NULL");

	case_sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton));

	entire_word = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton));

	wrap_around = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->wrap_around_checkbutton));

	converted_search_string = gedit_utils_convert_search_text (search_string);

	if ((selected_text == NULL) ||
	    (case_sensitive && (strcmp (selected_text, converted_search_string)
	     !=0)) || (!case_sensitive && !g_utf8_caselessnmatch (selected_text, search_string, 
						       strlen (selected_text), 
						       strlen (search_string)) != 0))
	{
		gedit_debug (DEBUG_SEARCH, "selected_text (%s) != search_string (%s)", 
			     selected_text ? selected_text : "NULL",
			     search_string ? search_string : "NULL");

		if (selected_text != NULL)
			g_free (selected_text);

		gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
			GEDIT_RESPONSE_REPLACE, FALSE);
		
		replace_dlg_find_button_pressed (dialog);

		return;
	}
				
	g_free (selected_text);
	g_free (converted_search_string);
		
	gedit_debug (DEBUG_SEARCH, "Replace string: %s", replace_string ? replace_string : "NULL");

	gedit_document_replace_selected_text (doc, replace_string);

	gedit_debug (DEBUG_SEARCH, "Replaced");

	/* go ahead and find the next one */
	found = gedit_document_find (doc, search_string, TRUE, case_sensitive, entire_word);

	if (!found && wrap_around)
		found = gedit_document_find (doc, search_string, FALSE, case_sensitive, entire_word);

	if (found)
		gedit_view_scroll_to_cursor (active_view);
	
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog->dialog), 
							GEDIT_RESPONSE_REPLACE, found);	

	gedit_debug (DEBUG_SEARCH, "END");
}


static void
replace_dlg_replace_all_button_pressed (GeditDialogReplace *dialog)
{
	GeditMDIChild *active_child;
	GeditView* active_view;
	GeditDocument *doc;
	const gchar* search_string = NULL;
	const gchar* replace_string = NULL;
	gboolean case_sensitive;
	gboolean entire_word;
	gint replaced_items;
	GtkWidget *message_dlg;
	
 	gedit_debug (DEBUG_SEARCH, "");

	if (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)) == NULL)
		return;

	active_child = GEDIT_MDI_CHILD (bonobo_mdi_get_active_child (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_child != NULL);

	active_view = GEDIT_VIEW (bonobo_mdi_get_active_view (BONOBO_MDI (gedit_mdi)));
	g_return_if_fail (active_view != NULL);
	
	doc = active_child->document;
	g_return_if_fail (doc != NULL);

	search_string = gtk_entry_get_text (GTK_ENTRY (dialog->search_entry));		
	replace_string = gtk_entry_get_text (GTK_ENTRY (dialog->replace_entry));		

	g_return_if_fail (search_string);
	g_return_if_fail (replace_string);

	if (strlen (search_string) <= 0)
		return;
	
	gnome_entry_prepend_history (GNOME_ENTRY (dialog->search_entry_list), TRUE, search_string);

	if (strlen (replace_string) > 0)
		gnome_entry_prepend_history (GNOME_ENTRY (dialog->replace_entry_list), 
					     TRUE, 
					     replace_string);

	case_sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->match_case_checkbutton));

	entire_word = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->entire_word_checkbutton));

	replaced_items = gedit_document_replace_all (doc, search_string, replace_string, 
						     case_sensitive, entire_word);
		
	if (replaced_items <= 0)
	{
		message_dlg = gtk_message_dialog_new (
			GTK_WINDOW (dialog->dialog),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_INFO,
			GTK_BUTTONS_OK,
			_("The text \"%s\" was not found."), search_string);
	}
	else
	{	
		message_dlg = gtk_message_dialog_new (
			GTK_WINDOW (dialog->dialog),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_INFO,
			GTK_BUTTONS_OK,
			_("Found and replaced %d occurences."), replaced_items);
	}
	
	gtk_dialog_set_default_response (GTK_DIALOG (message_dlg), GTK_RESPONSE_OK);

	gtk_window_set_resizable (GTK_WINDOW (message_dlg), FALSE);

	gtk_dialog_run (GTK_DIALOG (message_dlg));
  	gtk_widget_destroy (message_dlg);
}

