/*
 * Copyright (C) 2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Chema Celorio <chema@celorio.com>
 *   Paolo Borelli <pborelli@katamail.com>
 */

#include "config.h"

#include <string.h>

#include <glib/gi18n-lib.h>

#include "glade.h"
#include "glade-signal.h"
#include "glade-xml-utils.h"

struct _GladeSignalPrivate
{
  gchar    *name;         /* Signal name eg "clicked"            */
  gchar    *handler;      /* Handler function eg "gtk_main_quit" */
  gchar    *userdata;     /* User data signal handler argument   */

  gchar    *support_warning;/* Message to inform the user about signals introduced in future versions */

  guint8    after : 1;    /* Connect after TRUE or FALSE         */
  guint8    swapped : 1;  /* Connect swapped TRUE or FALSE (GtkBuilder only) */
};

enum {
  PROP_0,
  PROP_NAME,
  PROP_HANDLER,
  PROP_USERDATA,
  PROP_SUPPORT,
  PROP_AFTER,
  PROP_SWAPPED
};

static GObjectClass *parent_class;

static void
glade_signal_finalize (GObject *object)
{
  GladeSignal *signal = GLADE_SIGNAL (object);

  g_free (signal->priv->name);
  g_free (signal->priv->handler);
  g_free (signal->priv->userdata);
  g_free (signal->priv->support_warning);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
glade_signal_get_property (GObject * object,
                                  guint prop_id,
                                  GValue * value, GParamSpec * pspec)
{
  GladeSignal *signal = GLADE_SIGNAL (object);

  switch (prop_id)
    {
      case PROP_NAME:
        g_value_set_string (value, signal->priv->name);
        break;
      case PROP_HANDLER:
        g_value_set_string (value, signal->priv->handler);
        break;
      case PROP_USERDATA:
        g_value_set_string (value, signal->priv->userdata);
        break; 
      case PROP_SUPPORT:
        g_value_set_string (value, signal->priv->support_warning);
        break; 
      case PROP_AFTER:
        g_value_set_boolean (value, signal->priv->after);
        break; 
      case PROP_SWAPPED:
        g_value_set_boolean (value, signal->priv->swapped);
        break; 
     default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
glade_signal_set_property (GObject * object,
                                  guint prop_id,
                                  const GValue * value, GParamSpec * pspec)
{
  GladeSignal *signal = GLADE_SIGNAL (object);

  switch (prop_id)
    {
      case PROP_NAME:
	signal->priv->name = g_value_dup_string (value);
        break;
      case PROP_HANDLER:
	glade_signal_set_handler (signal, g_value_get_string (value));
        break;
      case PROP_USERDATA:
	glade_signal_set_userdata (signal, g_value_get_string (value));
        break; 
      case PROP_SUPPORT:
	glade_signal_set_support_warning (signal, g_value_get_string (value));
        break; 
      case PROP_AFTER:
	glade_signal_set_after (signal, g_value_get_boolean (value));
        break; 
      case PROP_SWAPPED:
	glade_signal_set_swapped (signal, g_value_get_boolean (value));
        break; 
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
glade_signal_init (GladeSignal *signal)
{
  signal->priv = G_TYPE_INSTANCE_GET_PRIVATE (signal,
					      GLADE_TYPE_SIGNAL,
					      GladeSignalPrivate);
}

static void
glade_signal_klass_init (GladeSignalKlass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  parent_class = g_type_class_peek_parent (klass);

  object_class->set_property = glade_signal_set_property;
  object_class->get_property = glade_signal_get_property;
  object_class->finalize     = glade_signal_finalize;

  /* Properties */
  g_object_class_install_property (object_class, PROP_NAME,
				   g_param_spec_string
				   ("name", _("Name"),
				    _("The name of this signal"),
				    NULL,
				    G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_HANDLER,
				   g_param_spec_string
				   ("handler", _("Handler"),
				    _("The handler for this signal"),
				    NULL, G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_USERDATA,
				   g_param_spec_string
				   ("userdata", _("User Data"),
				    _("The user data for this signal"),
				    NULL, G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_SUPPORT,
				   g_param_spec_string
				   ("support-warning", _("Support Warning"),
				    _("The versioning support warning for this signal"),
				    NULL, G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_AFTER,
				   g_param_spec_boolean
				   ("after", _("After"),
				    _("Whether this signal is run after default handlers"),
				    FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_SWAPPED,
				   g_param_spec_boolean
				   ("swapped", _("Swapped"),
				    _("Whether the user data is swapped with the instance for the handler"),
				    FALSE, G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (GladeSignalPrivate));
}


GType
glade_signal_get_type (void)
{
  static GType signal_type = 0;

  if (!signal_type)
    {
      static const GTypeInfo signal_info = {
        sizeof (GladeSignalKlass),    /* Klass is our class */
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) glade_signal_klass_init,
        (GClassFinalizeFunc) NULL,
        NULL,                   /* class_data */
        sizeof (GladeSignal),
        0,                      /* n_preallocs */
        (GInstanceInitFunc) glade_signal_init,
      };
      signal_type =
          g_type_register_static (G_TYPE_OBJECT,
                                  "GladeSignal", &signal_info, 0);
    }
  return signal_type;
}


/**
 * glade_signal_new:
 * @name: a name for the signal
 * @handler: a handler function for the signal
 * @userdata: the userdata for this signal
 * @after: whether this handler should be called after the default emission phase
 * @swapped: whether the handler's user data should be swapped with the emitter instance.
 *
 * Creates a new #GladeSignal with the given parameters.
 *
 * Returns: the new #GladeSignal
 */
GladeSignal *
glade_signal_new (const gchar * name,
                  const gchar * handler,
                  const gchar * userdata, 
		  gboolean after, 
		  gboolean swapped)
{
  return (GladeSignal *)g_object_new (GLADE_TYPE_SIGNAL,
				      "name", name,
				      "handler", handler,
				      "userdata", userdata,
				      "after", after,
				      "swapped", swapped,
				      NULL);
}

/**
 * glade_signal_equal:
 * @sig1: a #GladeSignal
 * @sig2: a #GladeSignal
 *
 * Returns: %TRUE if @sig1 and @sig2 have identical attributes, %FALSE otherwise
 */
gboolean
glade_signal_equal (GladeSignal * sig1, GladeSignal * sig2)
{
  gboolean ret = FALSE;

  g_return_val_if_fail (GLADE_IS_SIGNAL (sig1), FALSE);
  g_return_val_if_fail (GLADE_IS_SIGNAL (sig2), FALSE);

  /* Intentionally ignore support_warning */
  if (!strcmp (sig1->priv->name, sig2->priv->name) &&
      !strcmp (sig1->priv->handler, sig2->priv->handler) &&
      sig1->priv->after == sig2->priv->after && sig1->priv->swapped == sig2->priv->swapped)
    {
      if ((sig1->priv->userdata == NULL && sig2->priv->userdata == NULL) ||
          (sig1->priv->userdata != NULL && sig2->priv->userdata != NULL &&
           !strcmp (sig1->priv->userdata, sig2->priv->userdata)))
        ret = TRUE;
    }

  return ret;
}

/**
 * glade_signal_clone:
 * @signal: a #GladeSignal
 *
 * Returns: a new #GladeSignal with the same attributes as @signal
 */
GladeSignal *
glade_signal_clone (const GladeSignal * signal)
{
  GladeSignal *dup;

  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), NULL);

  dup = glade_signal_new (signal->priv->name,
                          signal->priv->handler,
                          signal->priv->userdata, 
			  signal->priv->after, 
			  signal->priv->swapped);

  glade_signal_set_support_warning (dup, signal->priv->support_warning);

  return dup;
}

/**
 * glade_signal_write:
 * @signal: The #GladeSignal
 * @context: A #GladeXmlContext
 * @node: A #GladeXmlNode
 *
 * Writes @signal to @node
 */
void
glade_signal_write (GladeSignal * signal,
                    GladeXmlContext * context, GladeXmlNode * node)
{
  GladeXmlNode *signal_node;
  gchar *name;

  /*  Should assert GLADE_XML_TAG_WIDGET tag here, but no 
   * access to project, so not really seriosly needed 
   */

  name = g_strdup (signal->priv->name);

  /* Now dump the node values... */
  signal_node = glade_xml_node_new (context, GLADE_XML_TAG_SIGNAL);
  glade_xml_node_append_child (node, signal_node);

  glade_xml_node_set_property_string (signal_node, GLADE_XML_TAG_NAME, name);
  glade_xml_node_set_property_string (signal_node, GLADE_XML_TAG_HANDLER,
                                      signal->priv->handler);

  if (signal->priv->userdata)
    glade_xml_node_set_property_string (signal_node,
                                        GLADE_XML_TAG_OBJECT, signal->priv->userdata);

  if (signal->priv->after)
    glade_xml_node_set_property_string (signal_node,
                                        GLADE_XML_TAG_AFTER,
                                        GLADE_XML_TAG_SIGNAL_TRUE);

  /* Always serialize swapped regardless of format (libglade should not complain about this
   * and we prefer to not lose data in conversions).
   */
  glade_xml_node_set_property_string (signal_node,
                                      GLADE_XML_TAG_SWAPPED,
                                      signal->priv->swapped ?
                                      GLADE_XML_TAG_SIGNAL_TRUE :
                                      GLADE_XML_TAG_SIGNAL_FALSE);

  g_free (name);
}


/**
 * glade_signal_read:
 * @node: The #GladeXmlNode to read
 *
 * Reads and creates a ner #GladeSignal based on @node
 *
 * Returns: A newly created #GladeSignal
 */
GladeSignal *
glade_signal_read (GladeXmlNode * node)
{
  GladeSignal *signal;
  gchar *name, *handler;

  g_return_val_if_fail (glade_xml_node_verify_silent
                        (node, GLADE_XML_TAG_SIGNAL), NULL);

  if (!(name =
        glade_xml_get_property_string_required (node, GLADE_XML_TAG_NAME,
                                                NULL)))
    return NULL;
  glade_util_replace (name, '_', '-');

  if (!(handler =
        glade_xml_get_property_string_required (node, GLADE_XML_TAG_HANDLER,
                                                NULL)))
    {
      g_free (name);
      return NULL;
    }

  signal = g_new0 (GladeSignal, 1);
  signal->priv->name = name;
  signal->priv->handler = handler;
  signal->priv->after =
      glade_xml_get_property_boolean (node, GLADE_XML_TAG_AFTER, FALSE);
  signal->priv->userdata = glade_xml_get_property_string (node, GLADE_XML_TAG_OBJECT);
  signal->priv->swapped = glade_xml_get_property_boolean (node, GLADE_XML_TAG_SWAPPED,
                                                    /* If a signal specifies an object, its swapped
                                                     * by default behaviour in GtkBuilder */
                                                    signal->priv->userdata != NULL);

  return signal;
}

G_CONST_RETURN gchar *
glade_signal_get_name (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), NULL);

  return signal->priv->name;
}

void
glade_signal_set_handler (GladeSignal *signal,
			  const gchar *handler)
{
  g_return_if_fail (GLADE_IS_SIGNAL (signal));

  if (g_strcmp0 (signal->priv->handler, handler))
    {
      g_free (signal->priv->handler);
      signal->priv->handler =
          handler ? g_strdup (handler) : NULL;

      g_object_notify (G_OBJECT (signal), "handler");
    }
}

G_CONST_RETURN gchar *
glade_signal_get_handler (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), NULL);

  return signal->priv->handler;
}

void
glade_signal_set_userdata (GladeSignal *signal,
			   const gchar *userdata)
{
  g_return_if_fail (GLADE_IS_SIGNAL (signal));

  if (g_strcmp0 (signal->priv->userdata, userdata))
    {
      g_free (signal->priv->userdata);
      signal->priv->userdata =
          userdata ? g_strdup (userdata) : NULL;

      g_object_notify (G_OBJECT (signal), "userdata");
    }
}

G_CONST_RETURN gchar *
glade_signal_get_userdata (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), NULL);

  return signal->priv->userdata;
}

void
glade_signal_set_after (GladeSignal *signal,
			gboolean     after)
{
  g_return_if_fail (GLADE_IS_SIGNAL (signal));

  if (signal->priv->after != after)
    {
      signal->priv->after = after;

      g_object_notify (G_OBJECT (signal), "after");
    }
}

gboolean
glade_signal_get_after (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), FALSE);

  return signal->priv->after;
}

void
glade_signal_set_swapped (GladeSignal *signal,
			  gboolean     swapped)
{
  g_return_if_fail (GLADE_IS_SIGNAL (signal));

  if (signal->priv->swapped != swapped)
    {
      signal->priv->swapped = swapped;

      g_object_notify (G_OBJECT (signal), "swapped");
    }
}

gboolean
glade_signal_get_swapped (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), FALSE);

  return signal->priv->swapped;
}

void
glade_signal_set_support_warning (GladeSignal *signal,
                                  const gchar *support_warning)
{
  g_return_if_fail (GLADE_IS_SIGNAL (signal));

  if (g_strcmp0 (signal->priv->support_warning, support_warning))
    {
      g_free (signal->priv->support_warning);
      signal->priv->support_warning =
          support_warning ? g_strdup (support_warning) : NULL;

      g_object_notify (G_OBJECT (signal), "support-warning");
    }
}

G_CONST_RETURN gchar *
glade_signal_get_support_warning (GladeSignal *signal)
{
  g_return_val_if_fail (GLADE_IS_SIGNAL (signal), NULL);

  return signal->priv->support_warning;
}
