/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *                         Martin Baulig <martin@home-of-linux.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <config.h>
#include "gnome-build.h"
#include "gbf-project-client.h"

#include <unistd.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <bonobo/bonobo-listener.h>

#include <liboaf/liboaf.h>
#include <libgnome/gnome-util.h>

static void gbf_project_client_class_init (GbfProjectClientClass *class);
static void gbf_project_client_init (GbfProjectClient *prog);
static void gbf_project_client_destroy (GbfProjectClient *prog);

static void project_event_cb (BonoboListener *listener, char *event_name,
                              CORBA_any *any, CORBA_Environment *ev,
                              gpointer user_data);
static GbfProjectClientResult exception_to_result (CORBA_Environment *ev);

static GtkObjectClass *parent_class;

struct _GbfProjectClientPrivate {
};

enum {
    PROJECT_LOADED,
    BUILD_STARTED,
    BUILD_ERROR,
    BUILD_ENDED,
    LAST_SIGNAL
};

static gint project_client_signals[LAST_SIGNAL];

/* private routines */
static void
gbf_project_client_class_init (GbfProjectClientClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;
    
    g_return_if_fail (klass != NULL);
    g_return_if_fail (IS_GBF_PROJECT_CLIENT_CLASS (klass));
    
    parent_class = gtk_type_class (gtk_object_get_type ());
    
    project_client_signals [PROJECT_LOADED] = 
        gtk_signal_new ("project_loaded",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GbfProjectClientClass,
                                           project_loaded),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);

    project_client_signals [BUILD_STARTED] = 
        gtk_signal_new ("build_started",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GbfProjectClientClass,
                                           build_started),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);

    project_client_signals [BUILD_ERROR] = 
        gtk_signal_new ("build_error",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GbfProjectClientClass,
                                           build_error),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);

    project_client_signals [BUILD_ENDED] = 
        gtk_signal_new ("build_ended",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GbfProjectClientClass,
                                           build_ended),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);

    gtk_object_class_add_signals (object_class,
                                  project_client_signals,
                                  LAST_SIGNAL);

    klass->project_loaded = NULL;
    object_class->destroy = (GtkSignalFunc) gbf_project_client_destroy;
}

static void
gbf_project_client_init (GbfProjectClient *client)
{
    client->priv = NULL;
}

GtkType
gbf_project_client_get_type (void)
{
    static GtkType type = 0;
    
    if (!type) {
        GtkTypeInfo info = {
            "GbfProjectClient",
            sizeof (GbfProjectClient),
            sizeof (GbfProjectClientClass),
            (GtkClassInitFunc) gbf_project_client_class_init,
            (GtkObjectInitFunc) gbf_project_client_init,
            NULL,
            NULL,
            (GtkClassInitFunc) NULL
        };
	
        type = gtk_type_unique (gtk_object_get_type (), &info);
    }
    
    return type;
}

static void 
gbf_project_client_construct (GbfProjectClient *prj, 
                              CORBA_Object corba_object)
{
	BonoboListener *listener;
	BonoboObjectClient *cli;
	CORBA_Object source;
	CORBA_Environment ev;

    prj->objref = corba_object;
    prj->priv = g_new0 (GbfProjectClientPrivate, 1);

	CORBA_exception_init (&ev);

	listener = bonobo_listener_new (NULL, NULL);
	gtk_signal_connect (GTK_OBJECT (listener), "event_notify",
			    GTK_SIGNAL_FUNC (project_event_cb), prj);

    cli = bonobo_object_client_from_corba (corba_object);
	source = bonobo_object_client_query_interface (cli, "IDL:Bonobo/EventSource:1.0", &ev);
	if (!CORBA_Object_is_nil (source, &ev) && ev._major == CORBA_NO_EXCEPTION) {
		Bonobo_EventSource_addListener (source, BONOBO_OBJREF (listener), &ev);
	} else {
		g_error ("couldn't get event source for object.");
	}

	CORBA_exception_free (&ev);
}

/**
 * gbf_project_client_new:
 * @iid: OAF IID of the backend to activate.
 *
 * Creates a #GbfProjectClient object.  This object activates a new 
 * GBF::Project server with the given OAF IID.
 *
 * Returns: A newly-created #GbfProjectClient object.
 */
GbfProjectClient *
gbf_project_client_new (const char *iid)
{
    CORBA_Object corba_object;
    GbfProjectClient *ret = NULL;
    CORBA_Environment ev;
    
    CORBA_exception_init (&ev);

    corba_object = oaf_activate_from_id ((const OAF_ActivationID)iid, 
                                         OAF_FLAG_PRIVATE, 
                                         NULL, &ev);    
    
    if (CORBA_Object_is_nil (corba_object, &ev)) {
        CORBA_exception_free (&ev);
        return NULL;
    }
    CORBA_exception_free (&ev);
    
    ret = gtk_type_new (gbf_project_client_get_type ());
 
    gbf_project_client_construct (ret, corba_object);

    gtk_object_ref (GTK_OBJECT (ret));
    gtk_object_sink (GTK_OBJECT (ret));

    return ret;
}

GbfProjectClient *
gbf_project_client_new_from_corba (GBF_Project project) 
{
    GbfProjectClient *ret;
    CORBA_Environment ev;

    g_return_val_if_fail (project != CORBA_OBJECT_NIL, NULL);

    ret = gtk_type_new (gbf_project_client_get_type ());
    gbf_project_client_construct (ret, project);
   
    CORBA_exception_init (&ev);

    Bonobo_Unknown_ref (project, &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);

    gtk_object_ref (GTK_OBJECT (ret));
    gtk_object_sink (GTK_OBJECT (ret));

    return ret;
}

GbfProjectClient *
gbf_project_client_new_for_path (const char *path)
{
    GbfProjectClient *ret = NULL;
    CORBA_Environment ev;
    OAF_ServerInfoList *oaf_result;
    
    CORBA_exception_init (&ev);
    
    oaf_result = oaf_query ("repo_ids.has ('IDL:GNOME/Development/Build:1.0')", NULL, &ev);

    if (ev._major == CORBA_NO_EXCEPTION && oaf_result != NULL && oaf_result->_length >= 1) {
        ret = gbf_project_client_new (oaf_result->_buffer[0].iid);
    }

    if (oaf_result != NULL) {
        CORBA_free (oaf_result);
    }

    return ret;   
}

GbfProjectClientResult 
gbf_project_client_load (GbfProjectClient *prj, const char *path)
{
    CORBA_Environment ev;
    GbfProjectClientResult res;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (path != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GNOME_Development_Project_load_project (prj->objref, path, &ev);

    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_build (GbfProjectClient *prj, GBF_BuildType type)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GNOME_Development_Project_build_project (prj->objref, type, &ev);

    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_get_project_root (GbfProjectClient *prj, CORBA_char **ret)
{
    CORBA_Environment ev;
    GbfProjectClientResult res;
    CORBA_char *corba_ret;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (ret != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    corba_ret = GNOME_Development_Project_get_project_root (prj->objref, &ev);

    if (corba_ret) {
        *ret = corba_ret;
    } else {
        *ret = NULL;
    }
    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_get_targets (GbfProjectClient *prj,
                                GBF_TargetList **targets)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    *targets = GNOME_Development_Project_get_targets (prj->objref, &ev);

    res = exception_to_result (&ev);

    if (res != GBF_PROJECT_CLIENT_OK)
        *targets = NULL;

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_get_targets_of_source (GbfProjectClient *prj,
                                          GBF_Source *source,
                                          GBF_TargetList **targets)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    *targets = GNOME_Development_Project_get_targets_of_source (prj->objref, source, &ev);

    res = exception_to_result (&ev);

    if (res != GBF_PROJECT_CLIENT_OK)
        *targets = NULL;

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult
gbf_project_client_add_target (GbfProjectClient *prj,
                               GBF_Target *target)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GNOME_Development_Project_add_target (prj->objref, target, &ev);

    res = exception_to_result (&ev);

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_remove_target (GbfProjectClient *prj,
                                  GBF_Target *target)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GNOME_Development_Project_remove_target (prj->objref, target, &ev);

    res = exception_to_result (&ev);

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_get_sources (GbfProjectClient *prj,
                                GBF_SourceList **sources)
{
    CORBA_Environment ev;
    GbfProjectClientResult res;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    *sources = GNOME_Development_Project_get_sources (prj->objref, &ev);

    res = exception_to_result (&ev);

    if (res != GBF_PROJECT_CLIENT_OK)
        *sources = NULL;

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_get_target_sources (GbfProjectClient *prj,
                                       GBF_Target *target,
                                       GBF_SourceList **sources)
{
    CORBA_Environment ev;
    GbfProjectClientResult res;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    *sources = GNOME_Development_Project_get_target_sources (prj->objref, target, &ev);

    res = exception_to_result (&ev);

    if (res != GBF_PROJECT_CLIENT_OK)
        *sources = NULL;

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_add_target_source (GbfProjectClient *prj,
                                      GBF_Target *target,
                                      GBF_Source *source)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GNOME_Development_Project_add_target_source (prj->objref, target, source, &ev);

    res = exception_to_result (&ev);

    CORBA_exception_free (&ev);

    return res;
}

GbfProjectClientResult 
gbf_project_client_remove_target_source (GbfProjectClient *prj,
                                         GBF_Target *target,
                                         GBF_Source *source)
{
    CORBA_Environment ev;
    GbfProjectClientResult res = GBF_PROJECT_CLIENT_OK;

	g_return_val_if_fail (prj != NULL, GBF_PROJECT_CLIENT_BAD_PARAMS);
	g_return_val_if_fail (IS_GBF_PROJECT_CLIENT (prj), GBF_PROJECT_CLIENT_BAD_PARAMS);
 
    CORBA_exception_init (&ev);

    GNOME_Development_Project_remove_target_source (prj->objref, target, source, &ev);

    res = exception_to_result (&ev);

    CORBA_exception_free (&ev);

    return res;
}

static GbfProjectClientResult 
exception_to_result (CORBA_Environment *ev)
{
    GbfProjectClientResult res;
    static GHashTable *table = NULL;

    if (!table) {
        table = g_hash_table_new (g_str_hash, g_str_equal);      
        g_hash_table_insert (table, ex_GNOME_Development_Project_DoesntExist,
                             GINT_TO_POINTER (GBF_PROJECT_CLIENT_DOESNT_EXIST));
        g_hash_table_insert (table, ex_GNOME_Development_Project_Malformed,
                             GINT_TO_POINTER (GBF_PROJECT_CLIENT_MALFORMED));
    }

    res = GBF_PROJECT_CLIENT_OK;
    
    if (ev->_major == CORBA_USER_EXCEPTION) {
        res = GPOINTER_TO_INT (g_hash_table_lookup (table, ev->_repo_id));
    } else if (ev->_major == CORBA_SYSTEM_EXCEPTION) {
        g_print ("gbf-project-client.c: CORBA_SYSTEM_EXCEPTION: %s\n", CORBA_exception_id (ev));
        res = -1;
    }

    return res;
}

static void 
project_event_cb (BonoboListener *listener, char *event_name,
                  CORBA_any *any, CORBA_Environment *ev,
                  gpointer user_data)
{
    GbfProjectClient *client = user_data;

	if (!strcmp (event_name, "project-loaded")) {
        gtk_signal_emit (GTK_OBJECT (client),
                         project_client_signals[PROJECT_LOADED]);
	} else if (!strcmp (event_name, "build-started")) {
        gtk_signal_emit (GTK_OBJECT (client),
                         project_client_signals[BUILD_STARTED]);
	} else if (!strcmp (event_name, "build-error")) {
        gtk_signal_emit (GTK_OBJECT (client),
                         project_client_signals[BUILD_ERROR]);
	} else if (!strcmp (event_name, "build-ended")) {
        gtk_signal_emit (GTK_OBJECT (client),
                         project_client_signals[BUILD_ENDED]);
	}
}

static void
gbf_project_client_destroy (GbfProjectClient *client)
{
    CORBA_Environment ev;

    if (client->priv) {
        g_free (client->priv);
        client->priv = NULL;
    }

    CORBA_exception_init (&ev);
    
    Bonobo_Unknown_unref (client->objref, &ev);   
    
    CORBA_exception_free (&ev);

    if (parent_class->destroy)
        parent_class->destroy (GTK_OBJECT (client));
    
}

