/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* board-window.c
 *
 * Copyright (C) 2001  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include "board-window.h"
#include "board_window_menu.h"

static void class_init (BoardWindowClass *klass);
static void init (BoardWindow *window);
static void destroy (GtkObject *object);

static void bw_game_info_cb (GtkWidget *widget, GameInfo *info, gpointer data);
static void bw_switch_page_cb (GtkNotebook *nb, GtkNotebook *page, gint page_num, gpointer user_data);

static GnomeAppClass *parent_class = NULL;

struct _BoardWindowPrivate {
	GtkNotebook *view_nb;
	GtkNotebook *info_nb;

	gint num;
};

GtkType
board_window_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "BoardWindow",
        sizeof (BoardWindow),
        sizeof (BoardWindowClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gnome_app_get_type (), &info);
    }

  return type;
}

static void
class_init (BoardWindowClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gnome_app_get_type ());

	object_class->destroy = destroy;
}


static void
init (BoardWindow *window)
{
	BoardWindowPrivate *priv;
	GtkWidget *view, *vbox;

	gnome_app_construct (GNOME_APP (window), "gnome-chess", NULL); 
	
	priv = g_new0 (BoardWindowPrivate, 1);

	window->priv = priv;

	vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_show (vbox);
	
	priv->view_nb = GTK_NOTEBOOK (gtk_notebook_new ());
	gtk_widget_show (GTK_WIDGET (priv->view_nb));
	gtk_container_add (GTK_CONTAINER (vbox), GTK_WIDGET (priv->view_nb));
	
	priv->info_nb = GTK_NOTEBOOK (gtk_notebook_new ());
	gtk_widget_show (GTK_WIDGET (priv->info_nb));
	gtk_container_add (GTK_CONTAINER (vbox), GTK_WIDGET (priv->info_nb));

	gnome_app_create_menus_with_data (GNOME_APP (window),
					  board_window_menu,
					 (gpointer) window);

	view = game_view_new ();
	board_window_add_view (window, GAME_VIEW (view));

	board_window_set_program_menu (window);
	board_window_set_server_menu (window);

	gtk_signal_connect_after (GTK_OBJECT (priv->view_nb), "switch_page",
				  (GtkSignalFunc) bw_switch_page_cb, window);
	
	gnome_app_set_contents (GNOME_APP (window), 
				GTK_WIDGET (vbox));
}



GtkWidget *
board_window_new (void)
{
	GtkWidget *widget;
	
	widget = gtk_type_new (board_window_get_type ());

	return widget;
}

void 
board_window_add_view (BoardWindow *window, GameView *view)
{
	BoardWindowPrivate *priv;
	Engine *engine;
	GtkWidget *lbl;
	char *txt;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));
	g_return_if_fail (view != NULL);
	g_return_if_fail (IS_GAME_VIEW (view));

	priv = window->priv;

	engine = game_view_get_engine (view);
	
	gtk_signal_connect (GTK_OBJECT (view), "info",
			    (GtkSignalFunc) bw_game_info_cb, window);

	priv->num++;
	txt = g_strdup_printf ("Board %d", priv->num);
	lbl = gtk_label_new (txt);
	g_free (txt);
	gtk_widget_show (lbl);

	gtk_widget_show (GTK_WIDGET (view));

	gtk_notebook_append_page (priv->view_nb, GTK_WIDGET (view), lbl);
	gtk_notebook_set_page (GTK_NOTEBOOK (priv->view_nb), -1);
	board_window_set_game_menu (window);
}

static void
board_window_real_remove (BoardWindow *window, gint page_num)
{
	BoardWindowPrivate *priv;

	priv = window->priv;
	
	gtk_notebook_remove_page (priv->view_nb, page_num);
}

void 
board_window_remove_view (BoardWindow *window, GameView *view)
{
	BoardWindowPrivate *priv;
	gint page_num;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));
	g_return_if_fail (view != NULL);
	g_return_if_fail (IS_GAME_VIEW (view));

	priv = window->priv;
	
	page_num = gtk_notebook_page_num (priv->view_nb, GTK_WIDGET (view));
	board_window_real_remove (window, page_num);
}

void
board_window_remove_current_view (BoardWindow *window)
{
	BoardWindowPrivate *priv;
	gint page_num;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));
	
	priv = window->priv;
	
	page_num = gtk_notebook_get_current_page (priv->view_nb);
	if (page_num <= 0)
		return;
	
	board_window_real_remove (window, page_num);
}

void
board_window_remove_all_views (BoardWindow *window)
{
	BoardWindowPrivate *priv;
	gint page_num;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));

	priv = window->priv;
	
	while ((page_num = gtk_notebook_get_current_page (priv->view_nb)) >= 0)
	       board_window_real_remove (window, page_num);
}

GameView *
board_window_get_current_view (BoardWindow *window)
{
	BoardWindowPrivate *priv;
	GtkWidget *view = NULL;
	gint page_num;

	g_return_val_if_fail (window != NULL, NULL);
	g_return_val_if_fail (IS_BOARD_WINDOW (window), NULL);
	
	priv = window->priv;

	page_num = gtk_notebook_get_current_page (priv->view_nb);
	if (page_num >= 0) {
		view =  gtk_notebook_get_nth_page (GTK_NOTEBOOK (priv->view_nb), page_num);
		return GAME_VIEW (view);
	}
	
	return NULL;
}

void 
board_window_add_info (BoardWindow *window, const char *text, GtkWidget *child)
{
	BoardWindowPrivate *priv;
	GtkWidget *lbl;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));
	g_return_if_fail (text != NULL);
	g_return_if_fail (child != NULL);
	g_return_if_fail (GTK_IS_WIDGET (child));

	priv = window->priv;

	lbl = gtk_label_new (text);
	gtk_widget_show (lbl);

	gtk_widget_show (child);

	gtk_notebook_append_page (priv->info_nb, child, lbl);
}

void 
board_window_remove_info (BoardWindow *window, GtkWidget *child)
{
	BoardWindowPrivate *priv;
	gint page_num;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_BOARD_WINDOW (window));
	g_return_if_fail (child != NULL);
	g_return_if_fail (GTK_IS_WIDGET (child));

	priv = window->priv;
	
	page_num = gtk_notebook_page_num (priv->info_nb, child);
	if (page_num < 0)
		return;
	
	gtk_notebook_remove_page (priv->info_nb, page_num);
}

static void 
destroy (GtkObject *object)
{
	BoardWindow *window;
	BoardWindowPrivate *priv;

	window = BOARD_WINDOW (object);
	priv = window->priv;

	g_free (priv);

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
bw_game_info_cb (GtkWidget *view, GameInfo *info, gpointer data)
{
	BoardWindow *window = BOARD_WINDOW (data);
	BoardWindowPrivate *priv;
	GtkWidget *lbl;
	gchar *text;
		
	priv = window->priv;
	
	lbl = gtk_notebook_get_tab_label (priv->view_nb, view);
	
	text = g_strdup_printf ("%s vs. %s", info->white, info->black);
	gtk_label_set_text (GTK_LABEL (lbl), text);
	g_free (text);
}

static void
bw_switch_page_cb (GtkNotebook *nb, GtkNotebook *page,
		   gint page_num, gpointer user_data)
{
	BoardWindow *window = user_data;
	
	board_window_set_game_menu (window);
}
