/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* engine-ics.c
 *
 * Copyright (C) 2001  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gnome.h>
#include <glade/glade.h>
#include "debug.h"
#include "makros.h"
#include "prefs.h"
#include "child.h"
#include "notation.h"
#include "board-window.h"
#include "dialogs.h"
#include "server-term.h"
#include "engine_null.h"
#include "engine_ics.h"

/* Structs for game info */
typedef struct {
	GameInfo info;
	Engine *engine;
	int game;
	int ply;
} ICSGameInfo;

struct _EngineIcsPrivate 
{
	/* Server terminal */
	ServerTerm *term;
	
	/* User info & status */
	char *user;
	char *pass;
	gboolean loggedin;

	/* Stuff for reading from server */
	GIOChannel *read_chan;
	GIOChannel *write_chan;
	
	pid_t childpid;

	gint read_cb;
	gint err_cb;

	/* List of board windows used by engine */
	GList *list;
};

/* Defines for the board string tokens */
#define BOARD_START 1
#define BOARD_END   8
#define TO_MOVE     9
#define DOUBLE_PUSH 10
#define WHITE_OO    11
#define WHITE_OOO   12
#define BLACK_OO    13
#define BLACK_OOO   14
#define IRREV       15
#define GAME        16
#define WHITE_NAME  17
#define BLACK_NAME  18
#define RELATION    19
#define START_TIME  20
#define START_INC   21
#define WHITE_STR   22
#define BLACK_STR   23
#define WHITE_TIME  24
#define BLACK_TIME  25
#define MOVE_NUM    26
#define VERBOSE     27
#define MOVE_TIME   28
#define NOTATION    29
#define FLIP        30

static GIOChannel *chan;
extern BoardWindow *main_board_window;

static EngineClass *parent_class = NULL;

static void class_init (EngineIcsClass *klass);
static void init (EngineIcs *ics);

static GameView *ei_add_view (EngineIcs *engine);
static void engine_ics_fill_menu (EngineNull *engine, GtkMenuShell *shell, gint pos);
static void engine_ics_move (EngineNull *enull, gint from, gint to);
static void engine_ics_destroy (GtkObject *obj);

static void engine_ics_view_cb (GtkWidget *widget, gpointer data);
static gboolean engine_ics_cb (GIOChannel *source, GIOCondition condition, gpointer data);
static gboolean engine_ics_err_cb (GIOChannel *source, GIOCondition condition, gpointer data);
static void engine_ics_command_cb (GtkWidget *widget, gchar *command, gpointer data);

/* Lex declarations */
extern int yylex (EngineIcs *engine);
void yyerror (char *s);

GtkType
engine_ics_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "EngineIcs",
        sizeof (EngineIcs),
        sizeof (EngineIcsClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (engine_get_type (), &info);
    }

  return type;
}

static void
class_init (EngineIcsClass *klass)
{
	GtkObjectClass *object_class;
	EngineClass *engine_class;
	
	object_class = GTK_OBJECT_CLASS (klass);
	engine_class = ENGINE_CLASS (klass);
	parent_class = gtk_type_class (engine_get_type ());

	object_class->destroy = engine_ics_destroy;
	
	engine_class->engine_fill_menu = (EngineFillMenu)engine_ics_fill_menu;
	engine_class->engine_move = (EngineMove)engine_ics_move;
}


static void
init (EngineIcs *engine)
{
	EngineIcsPrivate *priv;
	
	priv = g_new0 (EngineIcsPrivate, 1);

	engine->priv = priv;

	priv->term = SERVER_TERM (server_term_new ());
	gtk_signal_connect (GTK_OBJECT (priv->term), "command",
			    (GtkSignalFunc) engine_ics_command_cb,
			    engine);
	
	/* Set defaults */
	priv->user = NULL;
	priv->pass = NULL;
	priv->loggedin = FALSE;
	priv->list = NULL;
}



GtkObject *
engine_ics_new (const char *host, 
		const char *port, 
		const char *user,
		const char *pass, 
		const char *telnetprogram)
{		  
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	GameView *view;
	char *arg[4];
	
	engine = gtk_type_new (engine_ics_get_type ());
	priv = engine->priv;
	
	/* Set up the user/pass stuff */
	if (user)  
		priv->user =  g_strdup (user);

	if (pass)
		priv->pass = g_strdup (pass);

	/* Figure out the args to use */
	if (telnetprogram == NULL)
		arg[0] = g_strdup (prefs_get_telnetprogram ());
	else
		arg[0] = g_strdup (telnetprogram);

	arg[1] = g_strdup (host);
	arg[2] = g_strdup (port);
	arg[3] = NULL;
		
	/* Start the child process */
	start_child (arg[0], arg,
		     &priv->read_chan,
		     &priv->write_chan,
		     &priv->childpid);

	priv->read_cb = g_io_add_watch (priv->read_chan, G_IO_IN, engine_ics_cb, engine);
	priv->err_cb = g_io_add_watch (priv->read_chan, G_IO_HUP, engine_ics_err_cb, engine);

	g_free (arg[0]);
	g_free (arg[1]);
	g_free (arg[2]);
	
	view = ei_add_view (engine);
	board_window_add_info (main_board_window, host, 
			       GTK_WIDGET (priv->term));
	
	return GTK_OBJECT(engine);
}

static void
ei_menu_draw (GtkWidget *widget, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	
	engine = ENGINE_ICS (data);
	priv = engine->priv;

	write_child (priv->write_chan, "draw\n");
}

static void
ei_menu_flag (GtkWidget *widget, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	
	engine = ENGINE_ICS (data);
	priv = engine->priv;
	
	write_child (priv->write_chan, "flag\n");
}

static void
ei_menu_resign (GtkWidget *widget, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	
	engine = ENGINE_ICS (data);
	priv = engine->priv;

	write_child (priv->write_chan, "resign\n");
}

static void
ei_menu_rematch (GtkWidget *widget, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	
	engine = ENGINE_ICS (data);
	priv = engine->priv;

	write_child (priv->write_chan, "rematch\n");
}

static void 
engine_ics_fill_menu (EngineNull *enull, GtkMenuShell *shell, gint pos) 
{
	Engine *engine = gtk_object_get_data (GTK_OBJECT (enull), "ics");

	GnomeUIInfo ei_menu[] = {

		{GNOME_APP_UI_ITEM, N_("_Draw"), NULL,
		 ei_menu_draw, engine, NULL, 
		 GNOME_APP_PIXMAP_NONE, NULL, GDK_d, GDK_CONTROL_MASK, NULL},
 
		{GNOME_APP_UI_ITEM, N_("_Flag"), NULL,
		 ei_menu_flag, engine, NULL, 
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TIMER_STOP, 
		 GDK_f, GDK_CONTROL_MASK, NULL},

		{GNOME_APP_UI_ITEM, N_("_Resign"), NULL,
		 ei_menu_resign, engine, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_STOP, 0, 0, NULL},

		GNOMEUIINFO_SEPARATOR, 

		{GNOME_APP_UI_ITEM, N_("Re_match"), NULL,
		 ei_menu_rematch, engine, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},

		GNOMEUIINFO_END
	};

	gnome_app_fill_menu (shell, ei_menu, gtk_accel_group_get_default (), TRUE, pos);
}

static void 
engine_ics_move (EngineNull *enull, gint from, gint to)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	char str[100];

	engine = gtk_object_get_data (GTK_OBJECT (enull), "ics");
	priv = engine->priv;

	move_to_ascii (str, from, to);
	write_child (priv->write_chan, "%s\n", str);
}

static GameView *
ei_add_view (EngineIcs *engine)
{
	EngineIcsPrivate *priv;
	GtkWidget *view;
	ICSGameInfo *info;

	priv = engine->priv;
	
	info = g_new0 (ICSGameInfo, 1);
	info->info.description = g_strdup ("New board");
	info->engine = ENGINE (engine_null_new ());
	info->game = 0;
	info->ply  = 0;
	
	/* Null engine needs info to handle callbacks */
	engine_null_set_fill_menu (ENGINE_NULL (info->engine), 
				   (EngineFillMenu)engine_ics_fill_menu);
	engine_null_set_move (ENGINE_NULL (info->engine),
			      (EngineMove)engine_ics_move);

	gtk_object_set_data (GTK_OBJECT (info->engine), "ics", engine);

	/* Build a new board window */
	view = game_view_new_with_engine (info->engine);
	board_window_add_view (main_board_window, GAME_VIEW (view));

	gtk_object_set_data (GTK_OBJECT (view), "gameinfo", info);

	/* Add it to our list */
	priv->list = g_list_prepend (priv->list, view);
	gtk_signal_connect (GTK_OBJECT (view), "destroy",
			    (GtkSignalFunc) engine_ics_view_cb,
			    engine);
	
	return GAME_VIEW (view);
}

static
ICSGameInfo *ei_get_info (EngineIcs *engine, GameView *view)
{
	return gtk_object_get_data (GTK_OBJECT (view), "gameinfo");
}

static GameView *
ei_get_view (EngineIcs *engine, int game)
{
	EngineIcsPrivate *priv;
	GtkObject *view;
	GtkObject *alt_view;
	ICSGameInfo *info;
	GList *l;
	
	priv = engine->priv;
	
	alt_view = NULL;
	
	for (l = priv->list; l != NULL; l = l->next) {		
		view = GTK_OBJECT (l->data);
		info = ei_get_info (engine, GAME_VIEW (view));

		if (info->game == game)
			return GAME_VIEW (view);
		else if (info->game == 0)
			alt_view = view;
	}	

	if (alt_view)
		return GAME_VIEW (alt_view);
	
	return ei_add_view (engine);
}

void
engine_ics_user (EngineIcs *engine)
{
	EngineIcsPrivate *priv;
	
	priv = engine->priv;

	if (!priv->user || !priv->pass)
		dialog_login (&(priv->user), &(priv->pass));

	write_child (priv->write_chan, "%s\n", priv->user);
}

void 
engine_ics_password (EngineIcs *engine)
{
	EngineIcsPrivate *priv;

	priv = engine->priv;

	write_child (priv->write_chan, "%s\n", priv->pass);
}

void 
engine_ics_invaliduser (EngineIcs *engine) 
{
	EngineIcsPrivate *priv;

	priv = engine->priv;
	
	if (!priv->loggedin) {
		priv->user = NULL;
		priv->pass = NULL;
	}
}

void 
engine_ics_invalidpassword (EngineIcs *engine) 
{
	EngineIcsPrivate *priv;

	priv = engine->priv;

	if (!priv->loggedin)
		priv->pass = NULL;
}

void 
engine_ics_prompt(EngineIcs *engine, gchar *str) 
{
	EngineIcsPrivate *priv;

	priv = engine->priv;

	if (!priv->loggedin) {
		server_term_set_prompt (priv->term, str);
		write_child (priv->write_chan, "set style 12\n");
		priv->loggedin = TRUE;
	}
}

void
engine_ics_result (EngineIcs *engine, char *result)
{
	GameView *view;
	ICSGameInfo *icsinfo;
	char *rstr;
	gint resultval, game;

	/* Figure out what game */
	rstr = strchr(result, ' ');
	rstr++;
	game = atoi(rstr); 

	/* Figure out what happened */
	rstr = strrchr(result, ' ');
	rstr++;
	if (!strcmp(rstr, "1-0"))
		resultval = GAME_WHITE;
	else if(!strcmp(rstr, "0-1"))
		resultval = GAME_BLACK;
	else
		resultval = GAME_DRAW;

	/* Update the info struct */
	view = ei_get_view (engine, game);

	icsinfo = ei_get_info (engine, GAME_VIEW (view));	
	icsinfo->info.result = resultval;

	/* Signal the new info */
	gtk_signal_emit_by_name (GTK_OBJECT (icsinfo->engine),
				 "info", &(icsinfo->info));
}

int 
engine_ics_input (char *buf, int max) 
{
	char b[1024];
	guint len;

	g_io_channel_read (chan, b, sizeof (b), &len);

	if (len > 0)
		memcpy (buf, b, len);
	else if (len == 0)
		debug_print(DEBUG_NORMAL, "Zero length string read.");

	return len;
}

void 
engine_ics_output(EngineIcs *engine, char *str) 
{
	EngineIcsPrivate *priv;
	
	priv = engine->priv;

	server_term_print (priv->term, str);
}

static Position *
engine_ics_board_to_pos (char *b)
{
	int i,j;
	Position *pos;

	pos = POSITION (position_new ());
	for (i = 0; i < 8; i++) {
		for (j = 0; j < 8; j++) {
			int n= A8 - i * 10 + j;                                                                             
			switch (b[(8*i)+j]) {
			case 'R':
				pos->square[n] = WR;
				break;
			case 'N':
				pos->square[n] = WN;
				break;
			case 'B':
				pos->square[n] = WB;
				break;
			case 'Q':
				pos->square[n] = WQ;
				break;
			case 'K':
				pos->square[n] = WK;
				position_set_white_king (pos, n);
				break;
			case 'P':
				pos->square[n] = WP;
				break;
			case 'r':
				pos->square[n] = BR;
				break;
			case 'n':
				pos->square[n] = BN;
				break;
			case 'b':
				pos->square[n] = BB;
				break;
			case 'q':
				pos->square[n] = BQ;
				break;
			case 'k':
				pos->square[n] = BK;
				position_set_black_king (pos, n);
				break;
			case 'p':
				pos->square[n] = BP;
				break;
			case '-':
				pos->square[n] = EMPTY;
				break;
			}
		}
	}	
	return pos;
}

void
engine_ics_update_board (EngineIcs *engine, char *boardstring)
{
	EngineIcsPrivate *priv;
	GameView *view;
	Board *board;
	Position *pos;
	ICSGameInfo *icsinfo;
	gchar **tokens;
	char b[65], m[10];
	int game, colour, ply, rel;
	int from,to;
	int i;

	priv = engine->priv;

	/* Debug info */
	debug_print (DEBUG_NORMAL, "Board String: %s\n", boardstring);
	
	/* Go through the board string */
	tokens = g_strsplit (boardstring, " ", 0);

	/* Game number */
	game = atoi (tokens[GAME]);

	/* Get the board assigned to this engine */
	view = ei_get_view (engine, game);
	board = game_view_get_board (view);

	icsinfo = ei_get_info (engine, GAME_VIEW (view));
	
	/* Find the user's relation to the game */
	rel = atoi (tokens[RELATION]);
		
	/* Flip the board if necessary */ 
	board_set_flip (board, atoi (tokens[FLIP]));

	/* Extract the board string */
	b[0] = '\0';
	for (i = BOARD_START; i <= BOARD_END; i++) {
		strcat (b, tokens[i]);
	}

	/* Get the last move */
	if (!strcmp(tokens[VERBOSE], "o-o") ||
	    !strcmp(tokens[VERBOSE], "o-o-o")) {
		strncpy (m, tokens[VERBOSE], strlen (tokens[VERBOSE])+1);
	} else if (strcmp(tokens[VERBOSE], "none")) {
		strncpy (m, tokens[VERBOSE], 2);
		m[0] = tokens[VERBOSE][2];
		m[1] = tokens[VERBOSE][3];
		m[2] = tokens[VERBOSE][5];
		m[3] = tokens[VERBOSE][6];
		m[4] = '\0';
	} else {
		strcpy (m, "none");
	}
	
	/* Find out who is to move */ 
	if (!strcmp (tokens[TO_MOVE], "W"))
		colour = WHITE;
	else
		colour = BLACK;
		
	/* Figure out what ply this is */
	if (!strcmp(m, "none")) {
		ply =  0;
	} else if (colour == WHITE) {
		ply = atoi (tokens[MOVE_NUM])*2-2;
	} else {
		ply = atoi (tokens[MOVE_NUM])*2-1;
	}

	/* Update the info structure */
	if (!icsinfo->info.white ||
	    !strcmp(icsinfo->info.white, tokens[WHITE_NAME]))
		icsinfo->info.white = g_strdup (tokens[WHITE_NAME]);
	if (!icsinfo->info.black ||
	    !strcmp(icsinfo->info.black, tokens[BLACK_NAME]))
		icsinfo->info.black = g_strdup (tokens[BLACK_NAME]);

	icsinfo->info.white_time = atoi (tokens[WHITE_TIME]);
	icsinfo->info.black_time = atoi (tokens[BLACK_TIME]);

	icsinfo->info.tomove = colour;

	if (rel >= -1 && rel <= 1)
		icsinfo->info.result = GAME_PROGRESS;
	else
		icsinfo->info.result = NONE;
	
	if (icsinfo->info.description)
		g_free (icsinfo->info.description);
	icsinfo->info.description = 
		g_strdup_printf ("%s %s game, %s to move ", 
				 tokens[START_TIME], tokens[START_INC],
				 colour == WHITE ? "white" : "black");
	
	g_strfreev (tokens);

	/* Setup the board and figure out the move if there */
	/* was no dnd update or this is a different game */
	pos = board_get_position (board);
	if (strcmp (m, "none") && ply - icsinfo->ply == 1) {
		icsinfo->ply = ply;
		if (rel != -1) {
			ascii_to_move (pos, m, &from, &to);
			gtk_signal_emit_by_name (GTK_OBJECT (icsinfo->engine),
						 "move", from, to);
		}
	} else {
		pos = engine_ics_board_to_pos (b);
		position_set_color_to_move (pos, colour);
		icsinfo->game = game;				
		icsinfo->ply = ply;
		gtk_signal_emit_by_name (GTK_OBJECT (icsinfo->engine), 
					 "game", ply, pos);
	}
	gtk_signal_emit_by_name (GTK_OBJECT (icsinfo->engine),
				 "info", &(icsinfo->info));
}

static void
engine_ics_destroy (GtkObject *obj)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	GList *l;

	engine = ENGINE_ICS (obj);
	priv = engine->priv;

	g_io_channel_close (priv->read_chan);
	g_io_channel_unref (priv->read_chan);

	g_io_channel_close (priv->write_chan);
	g_io_channel_unref (priv->write_chan);

	/* Destroy the associated boards */
	for (l = priv->list; l != NULL; l = l->next) {
		gtk_object_destroy (GTK_OBJECT (l->data));
	}

	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}

static gboolean
engine_ics_cb (GIOChannel *source,
	       GIOCondition condition,
 	       gpointer data)
{
	int token;

	chan = source;
	token = yylex (ENGINE_ICS (data));

	return TRUE;
}

static gboolean
engine_ics_err_cb (GIOChannel *source,
		   GIOCondition condition,
		   gpointer data)
{
	g_error ("ICS Engine connection died");	
	
	return FALSE;
}

static void
engine_ics_command_cb (GtkWidget *widget, gchar *command, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;

	engine = ENGINE_ICS (data);
	priv = engine->priv;

	write_child (priv->write_chan, "%s\n", command);
}

static void
engine_ics_view_cb (GtkWidget *widget, gpointer data)
{
	EngineIcs *engine;
	EngineIcsPrivate *priv;
	GList *elem;
	ICSGameInfo *info;

	engine = ENGINE_ICS (data);
	priv = engine->priv;

	elem = g_list_find (priv->list, (gpointer)widget);
	if (elem) {
		/* Free the info */
		info = gtk_object_get_data (GTK_OBJECT (elem->data), "gameinfo");
		gtk_object_remove_data (GTK_OBJECT (elem->data), "gameinfo");
		g_free (info);
		
		priv->list = g_list_remove (priv->list, elem->data);
	}
}

