/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-data.h"
#include "gnome-cmd-file-list.h"
#include "gnome-cmd-file-selector-types.h"
#include "gnome-cmd-cmdline.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-device.h"
#include "gnome-cmd-advrename-dialog.h"
#include "gnome-cmd-bookmark-dialog.h"
#include "gnome-cmd-ftp-server-funcs.h"
#include "filter.h"
#include "utils.h"

#define DEFAULT_ANONYMOUS_PW "you@yourhost"

GnomeCmdData *data = NULL;


struct _GnomeCmdDataPrivate
{
	GList               *ftp_servers;            // GnomeCmdFtpServer pointers
	gchar               *ftp_anonymous_password;
	GList               *devices;
	GList               *fav_apps;
	GnomeCmdSizeDispMode size_disp_mode;
	GnomeCmdPermDispMode perm_disp_mode;
	GnomeCmdDateFormat   date_format;
	GnomeCmdLayout       layout;
	GnomeCmdLayoutColors *layout_colors;
	gboolean            type_filter[8];
	gboolean            hidden_filter;
	gint                main_win_width, main_win_height;
	gboolean            case_sens_sort;
	gint                list_row_height;
	gchar               *list_font;
	gchar               *misc_font;
	GnomeCmdRightMouseButtonMode right_mouse_button_mode;
	gboolean            show_toolbar;
	guint               icon_size;
	GdkInterpType       icon_scale_quality;
	gchar               *theme_icon_dir;
	gchar               *document_icon_dir;
	guint               fs_col_width[FILE_LIST_NUM_COLUMNS];
	guint               bookmark_dialog_col_width[BOOKMARK_DIALOG_NUM_COLUMNS];
	gint                cmdline_history_length;
	GList               *cmdline_history;
	GtkReliefStyle      btn_relief;
	FilterType          filter_type;
	gboolean            device_only_icon;
	gint                dir_cache_size;
	gboolean            use_ls_colors;
	gboolean            use_fam;
	SearchDefaults      *search_defaults;
	GnomeCmdFtpServer   *quick_connect_server;
	GnomeCmdBookmarkGroup local_bookmark_group;
	gboolean            honor_expect_uris;
	gboolean            skip_mounting;
	gboolean            toolbar_visibility;
	gboolean            buttonbar_visibility;
	AdvrenameDefaults   *advrename_defaults;
	
	gchar *viewer;
	gchar *editor;
	gchar *differ;
	gchar *term;
};



static GdkColor *gdk_color_new (gushort r, gushort g, gushort b)
{
	GdkColor *c = g_new (GdkColor, 1);
	c->pixel = 0;
	c->red = r;
	c->green = g;
	c->blue = b;
	return c;
}


static void set_string (const gchar *path, const gchar *value)
{
	gnome_config_set_string (path, value);
}

static void set_int    (const gchar *path, int value)
{
	gnome_config_set_int (path, value);
}

static void set_bool (const gchar *path, gboolean value)
{
	gnome_config_set_bool (path, value);
}

static gchar* get_string (const gchar *path, gchar *def)
{
	gboolean b = FALSE;
	gchar *value = gnome_config_get_string_with_default (path, &b);
	if (b)
		return g_strdup (def);	
	return value;
}

static gint get_int (const gchar *path, int def)
{
	gboolean b = FALSE;
	gint value = gnome_config_get_int_with_default (path, &b);
	if (b)
		return def;
	return value;
}

static gboolean get_bool (const gchar *path, gboolean def)
{
	gboolean b = FALSE;
	gboolean value = gnome_config_get_bool_with_default (path, &b);
	if (b)
		return def;
	return value;
}


static void set_color (const gchar *path, GdkColor *color)
{
	gchar *color_str;
	color_str = g_strdup_printf ("%d %d %d", color->red, color->green, color->blue);
	set_string (path, color_str);
	g_free (color_str);
}


static void get_color (const gchar *path, GdkColor *color)
{
	gint red, green, blue;
	gchar *def = g_strdup_printf ("%d %d %d",
								  color->red, color->green, color->blue);
	gchar *color_str = get_string (path, def);
	if (sscanf (color_str, "%u %u %u", &red, &green, &blue) != 3)
		warn_print ("Illegal color in config file\n");

	if (color_str != def)
		g_free (color_str);
	
	color->red   = (gushort)red;
	color->green = (gushort)green;
	color->blue  = (gushort)blue;

	g_free (def);
}


static void create_conf_dir ()
{
	gchar *dir = g_strdup_printf ("%s/%s", g_get_home_dir (), ".gnome-commander");
	if (mkdir (dir, S_IRUSR|S_IWUSR|S_IXUSR) == 0)
		warn_print ("Creating new config directory %s\n", dir);
	g_free (dir);
}


static void write_ftp_servers ()
{
	gchar *path;
	FILE *fd;
	
	path = g_strdup_printf ("%s/.gnome-commander/ftp-servers", g_get_home_dir());
	fd = fopen (path, "w");
	if (fd != NULL) {
		GList *tmp = data->priv->ftp_servers;
		chmod (path, S_IRUSR|S_IWUSR);
		while (tmp) {
			GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)tmp->data;
			if (server) {
				gchar *alias = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_alias (server));
				gchar *hname = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_host_name (server));
				gchar *uname = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_user_name (server));
				gchar *pw = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_pw (server));
				gushort port = gnome_cmd_ftp_server_get_host_port (server);
				GnomeCmdBookmarkGroup *bookmark_group = gnome_cmd_ftp_server_get_bookmarks (server);
				GList *bookmarks = bookmark_group->bookmarks;

				if (pw)
					fprintf (fd, "S: %s %s %d %s %s\n", alias, hname, port, uname, pw);
				else
					fprintf (fd, "S: %s %s %d %s\n", alias, hname, port, uname);

				g_free (alias);
				g_free (hname);
				g_free (uname);
				if (pw) g_free (pw);

				while (bookmarks) {
					GnomeCmdBookmark *bookmark = (GnomeCmdBookmark*)bookmarks->data;
					gchar *name = gnome_vfs_escape_string (bookmark->name);
					gchar *path = gnome_vfs_escape_string (bookmark->path);
					
					fprintf (fd, "B: %s %s\n", name, path);
					
					g_free (name);
					g_free (path);
					bookmarks = bookmarks->next;
				}
			}
			tmp = tmp->next;
		}

		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for writing\n", path);

	g_free (path);
}


static void
write_devices ()
{
	gchar *path;
	FILE *fd;
	
	path = g_strdup_printf ("%s/.gnome-commander/devices", g_get_home_dir());
	fd = fopen (path, "w");
	if (fd != NULL) {
		GList *tmp = data->priv->devices;
		while (tmp) {
			GnomeCmdDevice *device = (GnomeCmdDevice*)tmp->data;
			if (device) {
				gchar *alias = gnome_vfs_escape_string (
					gnome_cmd_device_get_alias (device));
				gchar *device_fn = gnome_vfs_escape_string (
					gnome_cmd_device_get_device_fn (device));
				gchar *mountp = gnome_vfs_escape_string (
					gnome_cmd_device_get_mountp (device));
				gchar *icon_path = gnome_vfs_escape_string (
					gnome_cmd_device_get_icon_path (device));
				
				fprintf (fd, "%s %s %s %s\n", alias, device_fn, mountp, icon_path);

				g_free (alias);
				g_free (device_fn);
				g_free (mountp);
				g_free (icon_path);
			}
			tmp = tmp->next;
		}

		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for writing\n", path);

	g_free (path);
}


static void
write_fav_apps ()
{
	gchar *path;
	FILE *fd;
	
	path = g_strdup_printf ("%s/.gnome-commander/fav-apps", g_get_home_dir());
	fd = fopen (path, "w");
	if (fd != NULL) {
		GList *tmp = data->priv->fav_apps;
		while (tmp) {
			GnomeCmdApp *app = (GnomeCmdApp*)tmp->data;
			if (app) {
				gchar *name = gnome_vfs_escape_string (
					gnome_cmd_app_get_name (app));
				gchar *cmd = gnome_vfs_escape_string (
					gnome_cmd_app_get_command (app));
				gchar *icon_path = gnome_vfs_escape_string (
					gnome_cmd_app_get_icon_path (app));
				gint target = gnome_cmd_app_get_target (app);
				gchar *pattern_string = gnome_vfs_escape_string (
					gnome_cmd_app_get_pattern_string (app));
				gint handles_uris = gnome_cmd_app_get_handles_uris (app);
				gint handles_multiple = gnome_cmd_app_get_handles_multiple (app);
				gint requires_terminal = gnome_cmd_app_get_requires_terminal (app);
				
				fprintf (fd, "%s %s %s %d %s %d %d %d\n",
						 name, cmd, icon_path,
						 target, pattern_string, 
						 handles_uris, handles_multiple, requires_terminal);

				g_free (name);
				g_free (cmd);
				g_free (icon_path);
				g_free (pattern_string);
			}
			tmp = tmp->next;
		}

		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for writing\n", path);

	g_free (path);
}


static void load_ftp_servers ()
{
	gchar *path;
	FILE *fd;

	data->priv->ftp_servers = NULL;
	path = g_strdup_printf ("%s/.gnome-commander/ftp-servers", g_get_home_dir());
	fd = fopen (path, "r");
	if (fd != NULL) {
		gint ret=0;
		GnomeCmdFtpServer *server = NULL;
		
		do {
			gchar line[1024];

			if (fgets (line, 1024, fd) != NULL) {
				if (line[0] == 'S') {
					gchar alias[256], host[256], user[256], pw[256];
					gchar *alias2, *host2, *user2, *pw2=NULL;
					gint port;
					
					ret = sscanf (line, "S: %s %s %d %s %s\n", alias, host, &port, user, pw);
					if (ret == 4 || ret == 5) {
						alias2 = gnome_vfs_unescape_string (alias, NULL);
						host2  = gnome_vfs_unescape_string (host, NULL);
						user2  = gnome_vfs_unescape_string (user, NULL);
						if (ret == 5)
							pw2 = gnome_vfs_unescape_string (pw, NULL);

						server = gnome_cmd_ftp_server_new_with_values (
							alias2,
							host2,
							(gshort)port,
							user2,
							pw2);
						
						data->priv->ftp_servers = g_list_append (
							data->priv->ftp_servers, server);

						g_free (alias2);
						g_free (host2);
						g_free (user2);
						if (pw2) g_free (pw2);
					}
				}
				else if (line[0] == 'B' && server) {
					gchar name[256], path[256];
					ret = sscanf (line, "B: %s %s\n", name, path);
					if (ret == 2) {
						GnomeCmdBookmarkGroup *group = gnome_cmd_ftp_server_get_bookmarks (server);
						GnomeCmdBookmark *bookmark = g_new (GnomeCmdBookmark, 1);
						bookmark->name = gnome_vfs_unescape_string (name, NULL);
						bookmark->path = gnome_vfs_unescape_string (path, NULL);
						bookmark->group = group;

						group->bookmarks = g_list_append (group->bookmarks, bookmark);
					}
				}
				else
					g_warning ("Invalid line in the ftp-servers file - skipping it...\n");
			}
			else
				break;
		} while (1);
		
		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for reading\n", path);

	g_free (path);
}


static void
load_devices ()
{
	gchar *path;
	FILE *fd;

	data->priv->devices = NULL;
	path = g_strdup_printf ("%s/.gnome-commander/devices", g_get_home_dir());
	fd = fopen (path, "r");
	if (fd != NULL) {
		int ret;
		gchar alias[256], device_fn[256], mountp[256], icon_path[256];
		gchar *alias2, *device_fn2, *mountp2, *icon_path2;
		
		do {
			ret = fscanf (fd, "%s %s %s %s\n", alias, device_fn, mountp, icon_path);
			
			if (ret == 4) {
				alias2      = gnome_vfs_unescape_string (alias, NULL);
				device_fn2  = gnome_vfs_unescape_string (device_fn, NULL);
				mountp2     = gnome_vfs_unescape_string (mountp, NULL);
				icon_path2  = gnome_vfs_unescape_string (icon_path, NULL);
				
				data->priv->devices = g_list_append (
					data->priv->devices,
					gnome_cmd_device_new_with_values (
						alias2, device_fn2, mountp2, icon_path2));

				g_free (alias2);
				g_free (device_fn2);
				g_free (mountp2);
				g_free (icon_path2);
			}
		} while (ret == 4);
		
		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for reading\n", path);

	g_free (path);
}


static void
load_fav_apps ()
{
	gchar *path;
	FILE *fd;

	data->priv->fav_apps = NULL;
	path = g_strdup_printf ("%s/.gnome-commander/fav-apps", g_get_home_dir());
	fd = fopen (path, "r");
	if (fd != NULL) {
		int ret;
		gchar name[256], cmd[256], icon_path[256], pattern_string[256];
		gchar *name2, *cmd2, *icon_path2, *pattern_string2;
		gint target, handles_uris, handles_multiple, requires_terminal;
		
		do {
			ret = fscanf (fd, "%s %s %s %d %s %d %d %d\n",
						  name, cmd, icon_path,
						  &target, pattern_string,
						  &handles_uris, &handles_multiple, &requires_terminal);
			
			if (ret == 8) {
				name2      = gnome_vfs_unescape_string (name, NULL);
				cmd2       = gnome_vfs_unescape_string (cmd, NULL);
				icon_path2 = gnome_vfs_unescape_string (icon_path, NULL);
				pattern_string2 = gnome_vfs_unescape_string (pattern_string, NULL);
				
				data->priv->fav_apps = g_list_append (
					data->priv->fav_apps,
					gnome_cmd_app_new_with_values (
						name2, cmd2, icon_path2,
						target, pattern_string2,
						handles_uris, handles_multiple, requires_terminal));

				g_free (name2);
				g_free (cmd2);
				g_free (icon_path2);
				g_free (pattern_string2);
			}
		} while (ret == 8);
		
		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for reading\n", path);

	g_free (path);
}


static void
write_string_history (gchar *format, GList *strings)
{
	gint i = 0;
	gchar key[128];

	while (strings) {
		snprintf (key, 128, format, i);
		set_string (key, (gchar*)strings->data);
		strings = strings->next;
		i++;
	}
}


static void
write_int_array (const gchar *format, gint *array, gint length)
{
	gint i;

	for ( i=0 ; i<length ; i++ ) {
		gchar *name = g_strdup_printf (format, i);
		set_int (name, array[i]);
	}
}


static void
write_cmdline_history ()
{
	data->priv->cmdline_history = gnome_cmd_cmdline_get_history (
		gnome_cmd_main_win_get_cmdline (main_win));

	write_string_history (
		"/gnome-commander/cmdline-history/line%d", data->priv->cmdline_history);
}


static void
write_rename_history ()
{
	static gchar t[2] = "T";
	static gchar f[2] = "F";
	GList *from=NULL, *to=NULL, *csens=NULL;
	GList *tmp = data->priv->advrename_defaults->patterns;

	while (tmp) {
		PatternEntry *entry = (PatternEntry*)tmp->data;
		from = g_list_append (from, entry->from);
		to = g_list_append (to, entry->to);
		csens = g_list_append (csens, entry->case_sens?t:f);
		tmp = tmp->next;
	}

	set_int ("/gnome-commander/options/template-auto-update",
			 data->priv->advrename_defaults->auto_update);
	set_int ("/gnome-commander/advrename/width",
			 data->priv->advrename_defaults->width);
	set_int ("/gnome-commander/advrename/height",
			 data->priv->advrename_defaults->height);

	write_int_array ("/gnome-commander/advrename/pat_col_widths%d",
					 advrename_dialog_default_pat_column_width,
					 ADVRENAME_DIALOG_PAT_NUM_COLUMNS);
	write_int_array ("/gnome-commander/advrename/res_col_widths%d",
					 advrename_dialog_default_res_column_width,
					 ADVRENAME_DIALOG_RES_NUM_COLUMNS);
	
	set_int ("/gnome-commander/advrename/sep_value",
			 data->priv->advrename_defaults->sep_value);

	set_int ("/gnome-commander/options/template-history-size",
			 g_list_length (data->priv->advrename_defaults->templates->ents));
	write_string_history ("/gnome-commander/template-history/template%d",
						  data->priv->advrename_defaults->templates->ents);

	set_int ("/gnome-commander/options/counter_start",
			 data->priv->advrename_defaults->counter_start);
	set_int ("/gnome-commander/options/counter_precision",
			 data->priv->advrename_defaults->counter_precision);
	set_int ("/gnome-commander/options/counter_increment",
			 data->priv->advrename_defaults->counter_increment);

	
	set_int ("/gnome-commander/options/rename-history-size",
			 g_list_length (data->priv->advrename_defaults->patterns));
	write_string_history ("/gnome-commander/rename-history-from/from%d", from);
	write_string_history ("/gnome-commander/rename-history-to/to%d", to);
	write_string_history ("/gnome-commander/rename-history-csens/csens%d", csens);
}


static void
write_bookmarks ()
{
	GList *tmp = data->priv->local_bookmark_group.bookmarks;
	GList *names = NULL;
	GList *paths = NULL;

	while (tmp) {
		GnomeCmdBookmark *bookmark = (GnomeCmdBookmark*)tmp->data;
		names = g_list_append (names, bookmark->name);
		paths = g_list_append (paths, bookmark->path);
		tmp = tmp->next;
	}
	set_int ("/gnome-commander/options/bookmark_count",
			 g_list_length (data->priv->local_bookmark_group.bookmarks));
	write_string_history ("/gnome-commander/local_bookmarks/name%d", names);
	write_string_history ("/gnome-commander/local_bookmarks/path%d", paths);
}


static void
write_search_defaults ()
{
	write_string_history (
		"/gnome-commander/search-history/name_pattern%d",
		data->priv->search_defaults->name_patterns);
	write_string_history (
		"/gnome-commander/search-history/content_pattern%d",
		data->priv->search_defaults->content_patterns);
	write_string_history (
		"/gnome-commander/search-history/directory%d",
		data->priv->search_defaults->directories);
	set_bool (
		"/gnome-commander/search-history/recursive",
		data->priv->search_defaults->recursive);
	set_bool (
		"/gnome-commander/search-history/case_sens",
		data->priv->search_defaults->case_sens);
}


static void
load_int_array (const gchar *format, gint *array, gint length)
{
	gint i;

	for ( i=0 ; i<length ; i++ ) {
		gchar *name = g_strdup_printf (format, i);
		array[i] = get_int (name, array[i]);
	}
}


static GList *
load_string_history (gchar *format, gint size)
{
	gint i = 0;
	gchar *key, *value;
	GList *list = NULL;
	

	while (i < size || size == -1) {
		key = g_strdup_printf (format, i);
		value = get_string (key, NULL);
		g_free (key);
		if (!value)
			break;		
		list = g_list_append (list, value);
		i++;
	}

	return list;
}


static void
load_cmdline_history ()
{
	data->priv->cmdline_history =
		load_string_history ("/gnome-commander/cmdline-history/line%d", -1);
}


static void
load_rename_history ()
{
	gint size;	
	GList *from=NULL, *to=NULL, *csens=NULL;
	GList *tmp_from, *tmp_to, *tmp_csens;
	GList *templates;

	data->priv->advrename_defaults = g_new (AdvrenameDefaults, 1);

	data->priv->advrename_defaults->auto_update = get_int (
		"/gnome-commander/advrename/template-auto-update", TRUE);
	data->priv->advrename_defaults->width = get_int (
		"/gnome-commander/advrename/width", 450);
	data->priv->advrename_defaults->height = get_int (
		"/gnome-commander/advrename/height", 400);

	load_int_array ("/gnome-commander/advrename/pat_col_widths%d",
					advrename_dialog_default_pat_column_width,
					ADVRENAME_DIALOG_PAT_NUM_COLUMNS);
	load_int_array ("/gnome-commander/advrename/res_col_widths%d",
					advrename_dialog_default_res_column_width,
					ADVRENAME_DIALOG_RES_NUM_COLUMNS);
	
	data->priv->advrename_defaults->sep_value = get_int (
		"/gnome-commander/advrename/sep_value", 150);
	
	size = get_int ("/gnome-commander/advrename/template-history-size", 0);
	templates = load_string_history ("/gnome-commander/template-history/template%d", size);
	data->priv->advrename_defaults->templates = history_new (10);
	data->priv->advrename_defaults->templates->ents = templates;
	data->priv->advrename_defaults->templates->pos = templates;

	data->priv->advrename_defaults->counter_start = get_int (
		"/gnome-commander/advrename/counter_start", 1);
	data->priv->advrename_defaults->counter_precision = get_int (
		"/gnome-commander/advrename/counter_precision", 1);
	data->priv->advrename_defaults->counter_increment = get_int (
		"/gnome-commander/advrename/counter_increment", 1);

	
	data->priv->advrename_defaults->patterns = NULL;
	size = get_int ("/gnome-commander/advrename/rename-history-size", 0);
	
	tmp_from = from = load_string_history ("/gnome-commander/rename-history-from/from%d", size);
	tmp_to = to = load_string_history ("/gnome-commander/rename-history-to/to%d", size);
	tmp_csens = csens = load_string_history ("/gnome-commander/rename-history-csens/csens%d", size);

	while (tmp_from && size > 0) {
		PatternEntry *entry = g_new (PatternEntry, 1);
		entry->from = (gchar*)tmp_from->data;
		entry->to = (gchar*)tmp_to->data;
		entry->case_sens = ((gchar*)tmp_csens->data)[0] == 'T';

		tmp_from = tmp_from->next;
		tmp_to = tmp_to->next;
		tmp_csens = tmp_csens->next;
		
		data->priv->advrename_defaults->patterns = g_list_append (
			data->priv->advrename_defaults->patterns, entry);
		size--;
	}

	g_list_free (from);
	g_list_free (to);
	g_list_free (csens);
}


static void
load_bookmarks ()
{
	gint i;
	GList *names, *paths;	
	
	gint size = get_int ("/gnome-commander/options/bookmark_count", 0);
	names = load_string_history ("/gnome-commander/local_bookmarks/name%d", size);
	paths = load_string_history ("/gnome-commander/local_bookmarks/path%d", size);

	data->priv->local_bookmark_group.alias = g_strdup (_("Local"));
	data->priv->local_bookmark_group.bookmarks = NULL;
	data->priv->local_bookmark_group.server = NULL;
	data->priv->local_bookmark_group.data = NULL;
	
	for ( i=0 ; i<size ; i++ ) {
		GnomeCmdBookmark *bookmark = g_new (GnomeCmdBookmark, 1);
		bookmark->name = g_list_nth_data (names, i);
		bookmark->path = g_list_nth_data (paths, i);
		bookmark->group = &data->priv->local_bookmark_group;
		data->priv->local_bookmark_group.bookmarks =
			g_list_append (data->priv->local_bookmark_group.bookmarks, bookmark);
	}
}


static void
load_search_defaults ()
{
	data->priv->search_defaults = g_new (SearchDefaults, 1);

	data->priv->search_defaults->name_patterns = load_string_history (
		"/gnome-commander/search-history/name_pattern%d", -1);
	data->priv->search_defaults->content_patterns = load_string_history (
		"/gnome-commander/search-history/content_pattern%d", -1);
	data->priv->search_defaults->directories = load_string_history (
		"/gnome-commander/search-history/directory%d", -1);
	data->priv->search_defaults->recursive = get_bool (
		"/gnome-commander/search-history/recursive", TRUE);
	data->priv->search_defaults->case_sens = get_bool (
		"/gnome-commander/search-history/case_sens", FALSE);
}


void
gnome_cmd_data_free                      (void)
{
	if (data)
	{
		if (data->priv)
		{
			GList *tmp;
			
			/* free the ftpservers */
			tmp = data->priv->ftp_servers;
			while (tmp)
			{
				GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)tmp->data;
				if (server)
					gnome_cmd_ftp_server_free (server);
				else
					g_warning ("GnomeCmdData: NULL ftpserver in list\n");

				tmp = tmp->next;
			}
			if (data->priv->ftp_servers)
				g_list_free (data->priv->ftp_servers);

			/* free the devices */
			tmp = data->priv->devices;
			while (tmp)
			{
				GnomeCmdDevice *device = (GnomeCmdDevice*)tmp->data;
				if (device)
					gnome_cmd_device_free (device);
				else
					g_warning ("GnomeCmdData: NULL device in list\n");

				tmp = tmp->next;
			}
			if (data->priv->devices)
				g_list_free (data->priv->devices);

			/* free the anonymous password string */
			if (data->priv->ftp_anonymous_password)
				g_free (data->priv->ftp_anonymous_password);

			/* free the dateformat string */
			if (data->priv->date_format)
				g_free (data->priv->date_format);

			/* free the font name strings */
			if (data->priv->list_font)
				g_free (data->priv->list_font);
			if (data->priv->misc_font)
				g_free (data->priv->misc_font);

			/* free the external programs strings */
			if (data->priv->viewer)
				g_free (data->priv->viewer);
			if (data->priv->editor)
				g_free (data->priv->editor);
			if (data->priv->differ)
				g_free (data->priv->differ);
			if (data->priv->term)
				g_free (data->priv->term);

			/* free layout colors */
			g_free (data->priv->layout_colors->norm_fg);
			g_free (data->priv->layout_colors->norm_bg);
			g_free (data->priv->layout_colors->sel_fg);
			g_free (data->priv->layout_colors->sel_bg);
			g_free (data->priv->layout_colors->curs_fg);
			g_free (data->priv->layout_colors->curs_bg);
			g_free (data->priv->layout_colors);
			
			g_free (data->priv);
		}

		g_free (data);
	}
}


void
gnome_cmd_data_save                      (void)
{
	gint i;
	
	for ( i=0 ; i<BOOKMARK_DIALOG_NUM_COLUMNS ; i++ ) {
		gchar *tmp = g_strdup_printf (
			"/gnome-commander-size/column-widths/bookmark_dialog_col_width%d", i);
		set_int (tmp, data->priv->bookmark_dialog_col_width[i]);
		g_free (tmp);
	}
	
	set_string ("/gnome-commander/ftp/anonymous_password",
				data->priv->ftp_anonymous_password);
	set_int    ("/gnome-commander/options/size_disp_mode",
				data->priv->size_disp_mode);
	set_int    ("/gnome-commander/options/perm_disp_mode",
				data->priv->perm_disp_mode);
	set_string ("/gnome-commander/options/date_disp_mode",
				data->priv->date_format);
	set_int    ("/gnome-commander/options/layout",
				data->priv->layout);
	set_int    ("/gnome-commander/options/list_row_height",
				data->priv->list_row_height);

	set_bool   ("/gnome-commander/options/show_unknown",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_UNKNOWN]);
	set_bool   ("/gnome-commander/options/show_regular",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_REGULAR]);
	set_bool   ("/gnome-commander/options/show_directory",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_DIRECTORY]);
	set_bool   ("/gnome-commander/options/show_fifo",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_FIFO]);
	set_bool   ("/gnome-commander/options/show_socket",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_SOCKET]);
	set_bool   ("/gnome-commander/options/show_char_device",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE]);
	set_bool   ("/gnome-commander/options/show_block_device",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_BLOCK_DEVICE]);
	set_bool   ("/gnome-commander/options/show_symbolic_link",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK]);

	set_bool   ("/gnome-commander/options/hidden_filter",
				data->priv->hidden_filter);

	set_bool   ("/gnome-commander/sort/case_sensitive",
				data->priv->case_sens_sort);

	set_color  ("/gnome-commander/colors/norm_fg",
				data->priv->layout_colors->norm_fg);
	set_color  ("/gnome-commander/colors/norm_bg",
				data->priv->layout_colors->norm_bg);
	set_color  ("/gnome-commander/colors/sel_fg",
				data->priv->layout_colors->sel_fg);
	set_color  ("/gnome-commander/colors/sel_bg",
				data->priv->layout_colors->sel_bg);
	set_color  ("/gnome-commander/colors/curs_fg",
				data->priv->layout_colors->curs_fg);
	set_color  ("/gnome-commander/colors/curs_bg",
				data->priv->layout_colors->curs_bg);

	set_string ("/gnome-commander/options/list_font",
				data->priv->list_font);
	set_string ("/gnome-commander/options/misc_font",
				data->priv->misc_font);

	set_int    ("/gnome-commander/options/right_mouse_button_mode",
				data->priv->right_mouse_button_mode);
	set_bool   ("/gnome-commander/options/show_toolbar",
				data->priv->show_toolbar);
	set_int    ("/gnome-commander/options/icon_size",
				data->priv->icon_size);
	set_int    ("/gnome-commander/options/icon_scale_quality",
				data->priv->icon_scale_quality);
	set_string ("/gnome-commander/options/theme_icon_dir",
				data->priv->theme_icon_dir);
	set_string ("/gnome-commander/options/document_icon_dir",
				data->priv->document_icon_dir);
	set_int    ("/gnome-commander/options/cmdline_history_length",
				data->priv->cmdline_history_length);
	set_int    ("/gnome-commander/options/btn_relief",
				data->priv->btn_relief);
	set_int    ("/gnome-commander/options/filter_type",
				data->priv->filter_type);

	set_bool   ("/gnome-commander/programs/honor_expect_uris",
				data->priv->honor_expect_uris);
	set_bool   ("/gnome-commander/programs/skip_mounting",
				data->priv->skip_mounting);
	set_bool   ("/gnome-commander/programs/toolbar_visibility",
				data->priv->toolbar_visibility);
	set_bool   ("/gnome-commander/programs/buttonbar_visibility",
				data->priv->buttonbar_visibility);
	
	set_string ("/gnome-commander/programs/viewer",
				data->priv->viewer);
	set_string ("/gnome-commander/programs/editor",
				data->priv->editor);
	set_string ("/gnome-commander/programs/differ",
				data->priv->differ);
	set_string ("/gnome-commander/programs/term",
				data->priv->term);

	set_bool   ("/gnome-commander/devices/only_icon",
				data->priv->device_only_icon);
	set_int    ("/gnome-commander/options/dir_cache_size",
				data->priv->dir_cache_size);
	set_bool   ("/gnome-commander/colors/use_ls_colors",
				data->priv->use_ls_colors);
	set_bool   ("/gnome-commander/options/use_fam",
				data->priv->use_fam);

	set_string ("/gnome-commander/quick-connect/host",
				gnome_cmd_ftp_server_get_host_name (data->priv->quick_connect_server));
	set_int    ("/gnome-commander/quick-connect/port",		
				gnome_cmd_ftp_server_get_host_port (data->priv->quick_connect_server));
	set_string ("/gnome-commander/quick-connect/user",
				gnome_cmd_ftp_server_get_user_name (data->priv->quick_connect_server));
	
	write_cmdline_history ();
	//write_dir_history ();
	
	create_conf_dir ();
	write_ftp_servers ();
	write_devices ();
	write_fav_apps ();
	write_search_defaults ();
	write_rename_history ();
	write_bookmarks ();
	
	gnome_config_sync ();
}


void
gnome_cmd_data_load                      (void)
{
	gint i;
	gchar *document_icon_dir = g_strdup_printf (
		"%s/share/pixmaps/document-icons/", GNOME_PREFIX);
	gchar *theme_icon_dir = g_strdup_printf (
		"%s/mime-icons", PIXMAPS_DIR);
	
	data = g_new (GnomeCmdData, 1);
	data->priv = g_new (GnomeCmdDataPrivate, 1);

	data->priv->layout_colors = g_new (GnomeCmdLayoutColors, 1);
	data->priv->layout_colors->norm_fg = gdk_color_new (0xffff,0xffff,0xffff);
	data->priv->layout_colors->norm_bg = gdk_color_new (0,0,0x4444);
	data->priv->layout_colors->sel_fg = gdk_color_new (0xffff,0,0);
	data->priv->layout_colors->sel_bg = gdk_color_new (0,0,0x4444);
	data->priv->layout_colors->curs_fg = gdk_color_new (0,0,0);
	data->priv->layout_colors->curs_bg = gdk_color_new (0xaaaa,0xaaaa,0xaaaa);

	data->priv->ftp_anonymous_password = 
		get_string ("/gnome-commander/ftp/anonymous_password", "you@provider.com");

	data->priv->size_disp_mode =
		get_int ("/gnome-commander/options/size_disp_mode",
				 GNOME_CMD_SIZE_DISP_MODE_POWERED);
	data->priv->perm_disp_mode =
		get_int ("/gnome-commander/options/perm_disp_mode",
				 GNOME_CMD_PERM_DISP_MODE_TEXT);

#ifdef HAVE_LOCALE_H
	data->priv->date_format = 
		get_string ("/gnome-commander/options/date_disp_mode", "%H:%M %x");
#else
	data->priv->date_format = 
		get_string ("/gnome-commander/options/date_disp_mode", "%H:%M %d/%m %Y");
#endif

	data->priv->layout = 
		get_int ("/gnome-commander/options/layout", GNOME_CMD_LAYOUT_MIME_ICONS);
	data->priv->list_row_height = 
		get_int ("/gnome-commander/options/list_row_height", 16);


	data->priv->type_filter[GNOME_VFS_FILE_TYPE_UNKNOWN] =
		get_bool ("/gnome-commander/options/show_unknown", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_REGULAR] = 
		get_bool ("/gnome-commander/options/show_regular", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_DIRECTORY] = 
		get_bool ("/gnome-commander/options/show_directory", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_FIFO] = 
		get_bool ("/gnome-commander/options/show_fifo", FALSE);
	
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_SOCKET] =
		get_bool ("/gnome-commander/options/show_socket", FALSE);
							 
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE] =
		get_bool ("/gnome-commander/options/show_char_device", FALSE);
							 
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_BLOCK_DEVICE] =
		get_bool ("/gnome-commander/options/show_block_device", FALSE);
	
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK] =
		get_bool ("/gnome-commander/options/show_symbolic_link", FALSE);

	data->priv->hidden_filter =
		get_bool ("/gnome-commander/options/hidden_filter", TRUE);

	data->priv->case_sens_sort =
		get_bool ("/gnome-commander/sort/case_sensitive", TRUE);

	data->priv->main_win_width =
		get_int ("/gnome-commander-size/main_win/width", 600);
	
	data->priv->main_win_height =
		get_int ("/gnome-commander-size/main_win/height", 400);

	for ( i=0 ; i<FILE_LIST_NUM_COLUMNS ; i++ ) {
		gchar *tmp = g_strdup_printf ("/gnome-commander-size/column-widths/fs_col_width%d", i);
		data->priv->fs_col_width[i] = get_int (tmp, file_list_default_column_width[i]);
		g_free (tmp);
	}
	
	for ( i=0 ; i<BOOKMARK_DIALOG_NUM_COLUMNS ; i++ ) {
		gchar *tmp = g_strdup_printf (
			"/gnome-commander-size/column-widths/bookmark_dialog_col_width%d", i);
		data->priv->bookmark_dialog_col_width[i] =
			get_int (tmp, bookmark_dialog_default_column_width[i]);
		g_free (tmp);
	}
	
	get_color  ("/gnome-commander/colors/norm_fg",
				data->priv->layout_colors->norm_fg);
	get_color  ("/gnome-commander/colors/norm_bg",
				data->priv->layout_colors->norm_bg);
	get_color  ("/gnome-commander/colors/sel_fg",
				data->priv->layout_colors->sel_fg);
	get_color  ("/gnome-commander/colors/sel_bg",
				data->priv->layout_colors->sel_bg);
	get_color  ("/gnome-commander/colors/curs_fg",
				data->priv->layout_colors->curs_fg);
	get_color  ("/gnome-commander/colors/curs_bg",
				data->priv->layout_colors->curs_bg);

	data->priv->list_font = get_string (
		"/gnome-commander/options/list_font",
		"-misc-fixed-medium-r-normal-*-10-*-*-*-c-*-iso8859-1");
	data->priv->misc_font = get_string (
		"/gnome-commander/options/misc_font",
		"-adobe-helvetica-medium-r-normal-*-10-*-*-*-p-*-iso8859-1");

	data->priv->right_mouse_button_mode = get_int (
		"/gnome-commander/options/right_mouse_button_mode",
		RIGHT_BUTTON_POPUPS_MENU);
	data->priv->show_toolbar = get_bool (
		"/gnome-commander/options/show_toolbar", TRUE);
	data->priv->icon_size = get_int (
		"/gnome-commander/options/icon_size", 16);
	data->priv->icon_scale_quality = get_int (
		"/gnome-commander/options/icon_scale_quality", GDK_INTERP_HYPER);
	data->priv->theme_icon_dir = get_string (
		"/gnome-commander/options/theme_icon_dir", theme_icon_dir);
	data->priv->document_icon_dir = get_string (
		"/gnome-commander/options/document_icon_dir", document_icon_dir);
	g_free (document_icon_dir);
	data->priv->cmdline_history_length = get_int (
		"/gnome-commander/options/cmdline_history_length", 16);
	data->priv->btn_relief = get_int (
		"/gnome-commander/options/btn_relief", GTK_RELIEF_NONE);
	data->priv->filter_type = get_int (
		"/gnome-commander/options/filter_type", FILTER_TYPE_FNMATCH);

	data->priv->honor_expect_uris = get_bool (
		"/gnome-commander/programs/honor_expect_uris", FALSE);
	data->priv->skip_mounting = get_bool (
		"/gnome-commander/programs/skip_mounting", FALSE);
	data->priv->toolbar_visibility = get_bool (
		"/gnome-commander/programs/toolbar_visibility", TRUE);
	data->priv->buttonbar_visibility = get_bool (
		"/gnome-commander/programs/buttonbar_visibility", TRUE);

	data->priv->viewer = get_string ("/gnome-commander/programs/viewer",
									 "emacs '%s'");
	data->priv->editor = get_string ("/gnome-commander/programs/editor",
									 "emacs '%s'");
	data->priv->differ = get_string ("/gnome-commander/programs/differ",
									 "gtkdiff '%s' '%s'");
	data->priv->term = get_string (
		"/gnome-commander/programs/term",
		"xterm -e '%s'");

	data->priv->device_only_icon = get_bool (
		"/gnome-commander/devices/only_icon", FALSE);
	data->priv->dir_cache_size = get_int (
		"/gnome-commander/options/dir_cache_size", 10);
	data->priv->use_ls_colors = get_bool (
		"/gnome-commander/colors/use_ls_colors", FALSE);
	data->priv->use_fam = get_bool (
		"/gnome-commander/options/use_fam", TRUE);

	data->priv->quick_connect_server = gnome_cmd_ftp_server_new ();
	gnome_cmd_ftp_server_set_alias (
		data->priv->quick_connect_server, "tmp");

	gnome_cmd_ftp_server_set_host_name (
		data->priv->quick_connect_server,
		get_string ("/gnome-commander/quick-connect/host", "ftp.gnome.org"));
	gnome_cmd_ftp_server_set_host_port (
		data->priv->quick_connect_server,
		get_int ("/gnome-commander/quick-connect/port", 21));
	gnome_cmd_ftp_server_set_user_name (
		data->priv->quick_connect_server,
		get_string ("/gnome-commander/quick-connect/user", "anonymous"));
	
	
	load_cmdline_history ();
	//load_dir_history ();
	load_ftp_servers ();
	load_search_defaults ();
	load_rename_history ();
	load_bookmarks ();
}


void
gnome_cmd_data_load_more (void)
{
	load_devices ();
	load_fav_apps ();
}


void
gnome_cmd_data_add_ftp_server            (GnomeCmdFtpServer *server)
{
	if (server == NULL) {
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->ftp_servers = g_list_append (data->priv->ftp_servers, server);
}


void
gnome_cmd_data_remove_ftp_server         (GnomeCmdFtpServer *server)
{
	if (server == NULL) {
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->ftp_servers = g_list_remove (data->priv->ftp_servers, server);
}


GList*
gnome_cmd_data_get_ftp_servers           (void)
{
	return data->priv->ftp_servers;
}


void
gnome_cmd_data_set_ftp_servers           (GList *servers)
{
	data->priv->ftp_servers = servers;
}


const gchar *
gnome_cmd_data_get_ftp_anonymous_password (void)
{
	return data->priv->ftp_anonymous_password;
}


void
gnome_cmd_data_set_ftp_anonymous_password (const gchar *pw)
{
	if (data->priv->ftp_anonymous_password)
		g_free (data->priv->ftp_anonymous_password);

	data->priv->ftp_anonymous_password = g_strdup (pw);
}


void
gnome_cmd_data_add_device                (GnomeCmdDevice *device)
{
	g_return_if_fail (device != NULL);
	
	data->priv->devices = g_list_append (data->priv->devices, device);
}

void
gnome_cmd_data_remove_device             (GnomeCmdDevice *device)
{
	g_return_if_fail (device != NULL);

	data->priv->devices = g_list_remove (data->priv->devices, device);
}

GList*
gnome_cmd_data_get_devices               (void)
{
	return data->priv->devices;
}

void
gnome_cmd_data_set_devices               (GList *devices)
{
	data->priv->devices = devices;
}


void
gnome_cmd_data_add_fav_app                (GnomeCmdApp *app)
{
	g_return_if_fail (app != NULL);

	data->priv->fav_apps = g_list_append (data->priv->fav_apps, app);
}

void
gnome_cmd_data_remove_fav_app             (GnomeCmdApp *app)
{
	g_return_if_fail (app != NULL);

	data->priv->fav_apps = g_list_remove (data->priv->fav_apps, app);
}

GList*
gnome_cmd_data_get_fav_apps               (void)
{
	return data->priv->fav_apps;
}

void
gnome_cmd_data_set_fav_apps               (GList *apps)
{
	data->priv->fav_apps = apps;
}


GnomeCmdSizeDispMode
gnome_cmd_data_get_size_disp_mode        (void)
{
	return data->priv->size_disp_mode;
}


void
gnome_cmd_data_set_size_disp_mode        (GnomeCmdSizeDispMode mode)
{
	data->priv->size_disp_mode = mode;
}

GnomeCmdPermDispMode
gnome_cmd_data_get_perm_disp_mode        (void)
{
	return data->priv->perm_disp_mode;
}

void
gnome_cmd_data_set_perm_disp_mode        (GnomeCmdPermDispMode mode)
{
	data->priv->perm_disp_mode = mode;
}

GnomeCmdDateFormat
gnome_cmd_data_get_date_format           (void)
{
	return data->priv->date_format;
}

void
gnome_cmd_data_set_date_format           (GnomeCmdDateFormat format)
{
	if (data->priv->date_format)
		g_free (data->priv->date_format);

	data->priv->date_format = g_strdup (format);
}



GnomeCmdLayout
gnome_cmd_data_get_layout                (void)
{
	return data->priv->layout;
}


void
gnome_cmd_data_set_layout                (GnomeCmdLayout layout)
{
	data->priv->layout = layout;
}


gint
gnome_cmd_data_get_list_row_height                (void)
{
	return data->priv->list_row_height;
}


void
gnome_cmd_data_set_list_row_height                (gint height)
{
	data->priv->list_row_height = height;
}


GnomeCmdLayoutColors*
gnome_cmd_data_get_layout_colors    (void)
{
	return data->priv->layout_colors;
}


void
gnome_cmd_data_set_type_filter          (GnomeVFSFileType type, gboolean value)
{
	data->priv->type_filter[type] = value;
}


gboolean
gnome_cmd_data_get_type_filter          (GnomeVFSFileType type)
{
	return data->priv->type_filter[type];
}


void
gnome_cmd_data_set_hidden_filter        (gboolean value)
{
	data->priv->hidden_filter = value;
}


gboolean
gnome_cmd_data_get_hidden_filter        (void)
{
	return data->priv->hidden_filter;
}


void
gnome_cmd_data_set_position             (gint width, gint height)
{
	data->priv->main_win_width = width;
	data->priv->main_win_height = height;
}


void
gnome_cmd_data_get_position             (gint *width, gint *height)
{
	*width = data->priv->main_win_width;
	*height = data->priv->main_win_height;
}


void
gnome_cmd_data_save_position            (void)
{
	gint i;
	
	set_int ("/gnome-commander-size/main_win/width", data->priv->main_win_width);
	set_int ("/gnome-commander-size/main_win/height", data->priv->main_win_height);

	for ( i=0 ; i<FILE_LIST_NUM_COLUMNS ; i++ ) {
		gchar *tmp = g_strdup_printf ("/gnome-commander-size/column-widths/fs_col_width%d", i);
		set_int (tmp, data->priv->fs_col_width[i]);
		g_free (tmp);
	}
}


void
gnome_cmd_data_set_viewer               (const gchar *command)
{
	if (data->priv->viewer)
		g_free (data->priv->viewer);
	data->priv->viewer = g_strdup (command);
}


void
gnome_cmd_data_set_editor               (const gchar *command)
{
	if (data->priv->editor)
		g_free (data->priv->editor);
	data->priv->editor = g_strdup (command);
}


void
gnome_cmd_data_set_differ               (const gchar *command)
{
	if (data->priv->differ)
		g_free (data->priv->differ);
	data->priv->differ = g_strdup (command);
}


void
gnome_cmd_data_set_term                (const gchar *term)
{
	if (data->priv->term)
		g_free (data->priv->term);
	
	data->priv->term = g_strdup (term);
}


const gchar *
gnome_cmd_data_get_viewer               (void)
{
	return data->priv->viewer;
}


const gchar *
gnome_cmd_data_get_editor               (void)
{
	return data->priv->editor;
}


const gchar *
gnome_cmd_data_get_differ               (void)
{
	return data->priv->differ;
}


const gchar *
gnome_cmd_data_get_term                (void)
{
	return data->priv->term;
}


gboolean
gnome_cmd_data_get_case_sens_sort       (void)
{
	return data->priv->case_sens_sort;
}


void
gnome_cmd_data_set_case_sens_sort       (gboolean value)
{
	data->priv->case_sens_sort = value;
}


const gchar *gnome_cmd_data_get_list_font          (void)
{
	return data->priv->list_font;
}


const gchar *gnome_cmd_data_get_misc_font          (void)
{
	return data->priv->misc_font;
}


void gnome_cmd_data_set_list_font            (const gchar *list_font)
{
	if (data->priv->list_font != NULL)
		g_free (data->priv->list_font);
	data->priv->list_font = g_strdup (list_font);
}


void gnome_cmd_data_set_misc_font            (const gchar *misc_font)
{
	if (data->priv->misc_font != NULL)
		g_free (data->priv->misc_font);
	data->priv->misc_font = g_strdup (misc_font);
}


void
gnome_cmd_data_set_right_mouse_button_mode (GnomeCmdRightMouseButtonMode mode)
{
	data->priv->right_mouse_button_mode = mode;
}


GnomeCmdRightMouseButtonMode
gnome_cmd_data_get_right_mouse_button_mode (void)
{
	return data->priv->right_mouse_button_mode;
}


void
gnome_cmd_data_set_show_toolbar        (gboolean value)
{
	data->priv->show_toolbar = value;
}


gboolean
gnome_cmd_data_get_show_toolbar        (void)
{
	return data->priv->show_toolbar;
}


guint
gnome_cmd_data_get_icon_size           (void)
{
	return data->priv->icon_size;
}


void
gnome_cmd_data_set_icon_size           (guint size)
{
	data->priv->icon_size = size;
}


GdkInterpType
gnome_cmd_data_get_icon_scale_quality  (void)
{
	return data->priv->icon_scale_quality;
}


void
gnome_cmd_data_set_icon_scale_quality  (GdkInterpType quality)
{
	data->priv->icon_scale_quality = quality;
}


const gchar *
gnome_cmd_data_get_theme_icon_dir       (void)
{
	return data->priv->theme_icon_dir;
}


void
gnome_cmd_data_set_theme_icon_dir       (const gchar *dir)
{
	if (data->priv->theme_icon_dir)
		g_free (data->priv->theme_icon_dir);
	data->priv->theme_icon_dir = g_strdup (dir);
}


const gchar *
gnome_cmd_data_get_document_icon_dir       (void)
{
	return data->priv->document_icon_dir;
}


void
gnome_cmd_data_set_document_icon_dir       (const gchar *dir)
{
	if (data->priv->document_icon_dir)
		g_free (data->priv->document_icon_dir);
	data->priv->document_icon_dir = g_strdup (dir);
}


void
gnome_cmd_data_set_fs_col_width        (guint column, gint width)
{
	if (column > FILE_LIST_NUM_COLUMNS)
		return;
	
	data->priv->fs_col_width[column] = width;
}


gint
gnome_cmd_data_get_fs_col_width        (guint column)
{
	if (column > FILE_LIST_NUM_COLUMNS)
		return 0;

	return data->priv->fs_col_width[column];
}


void
gnome_cmd_data_set_bookmark_dialog_col_width        (guint column, gint width)
{
	data->priv->bookmark_dialog_col_width[column] = width;
}


gint
gnome_cmd_data_get_bookmark_dialog_col_width        (guint column)
{
	return data->priv->bookmark_dialog_col_width[column];
}


gint
gnome_cmd_data_get_cmdline_history_length (void)
{
	return data->priv->cmdline_history_length;
}


void
gnome_cmd_data_set_cmdline_history_length (gint length)
{
	data->priv->cmdline_history_length = length;
}


GList *
gnome_cmd_data_get_cmdline_history (void)
{
	return data->priv->cmdline_history;
}


void
gnome_cmd_data_set_button_relief       (GtkReliefStyle relief)
{
	data->priv->btn_relief = relief;
}


GtkReliefStyle
gnome_cmd_data_get_button_relief       (void)
{
	return data->priv->btn_relief;
}


void
gnome_cmd_data_set_filter_type (FilterType type)
{
	data->priv->filter_type = type;
}


FilterType
gnome_cmd_data_get_filter_type (void)
{
	return data->priv->filter_type;
}


void
gnome_cmd_data_set_device_only_icon    (gboolean value)
{
	data->priv->device_only_icon = value;
}


gboolean
gnome_cmd_data_get_device_only_icon    (void)
{
	return data->priv->device_only_icon;
}


gint
gnome_cmd_data_get_dir_cache_size      (void)
{
	return data->priv->dir_cache_size;
}


void
gnome_cmd_data_set_dir_cache_size      (gint size)
{
	data->priv->dir_cache_size = size;
}


gboolean
gnome_cmd_data_get_use_ls_colors (void)
{
	return data->priv->use_ls_colors;
}


void
gnome_cmd_data_set_use_ls_colors (gboolean value)
{
	data->priv->use_ls_colors = value;
}


void
gnome_cmd_data_set_use_fam (gboolean value)
{
	data->priv->use_fam = value;
}


gboolean
gnome_cmd_data_get_use_fam (void)
{
	return data->priv->use_fam;
}


SearchDefaults *
gnome_cmd_data_get_search_defaults (void)
{
	return data->priv->search_defaults;
}


GnomeCmdFtpServer *
gnome_cmd_data_get_quick_connect_server (void)
{
	return data->priv->quick_connect_server;
}


GnomeCmdBookmarkGroup *
gnome_cmd_data_get_local_bookmarks (void)
{
	return &data->priv->local_bookmark_group;
}


GList *
gnome_cmd_data_get_bookmark_groups (void)
{
	static GList *groups = NULL;
	GList *servers = gnome_cmd_data_get_ftp_servers ();

	if (groups) {
		g_list_free (groups);
		groups = NULL;
	}

	groups = g_list_append (groups, &data->priv->local_bookmark_group);

	while (servers) {
		groups = g_list_append (groups,
								gnome_cmd_ftp_server_get_bookmarks (
									(GnomeCmdFtpServer*)servers->data));
		servers = servers->next;
	}

	return groups;
}


gboolean
gnome_cmd_data_get_honor_expect_uris (void)
{
	return data->priv->honor_expect_uris;
}


void
gnome_cmd_data_set_honor_expect_uris (gboolean value)
{
	data->priv->honor_expect_uris = value;
}


gboolean
gnome_cmd_data_get_skip_mounting (void)
{
	return data->priv->skip_mounting;
}


void
gnome_cmd_data_set_skip_mounting (gboolean value)
{
	data->priv->skip_mounting = value;
}


gboolean
gnome_cmd_data_get_toolbar_visibility (void)
{
	return data->priv->toolbar_visibility;
}


gboolean
gnome_cmd_data_get_buttonbar_visibility (void)
{
	return data->priv->buttonbar_visibility;
}


void
gnome_cmd_data_set_toolbar_visibility (gboolean value)
{
	data->priv->toolbar_visibility = value;
}


void
gnome_cmd_data_set_buttonbar_visibility (gboolean value)
{
	data->priv->buttonbar_visibility = value;
}


AdvrenameDefaults *
gnome_cmd_data_get_advrename_defaults (void)
{
	return data->priv->advrename_defaults;
}

