/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-main-menu.h"
#include "gnome-cmd-types.h"
#include "gnome-cmd-bookmark-dialog.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "gnome-cmd-connection-funcs.h"
#include "gnome-cmd-data.h"
#include "utils.h"
#include "useractions.h"
#include "widget-factory.h"

#include "../pixmaps/menu_ftp_connect.xpm"
#include "../pixmaps/menu_ftp_disconnect.xpm"


/* Theese following types are slightly changed from the originals in the GnomeUI library
   We need special types because we neeed to place non-changeable shortcuts in the
   menus. Another differance is that we want only mouse-clicks in the menu to generate an
   action, keyboard shortcuts are caught by the different components by them self */
typedef enum {
	MENU_TYPE_END,        /* No more items, use it at the end of an array */
	MENU_TYPE_ITEM,		  /* Normal item, or radio item if it is inside a radioitems group */
	MENU_TYPE_BASIC,
	MENU_TYPE_TOGGLEITEM, /* Toggle (check box) item */
	MENU_TYPE_RADIOITEMS, /* Radio item group */
	MENU_TYPE_SUBTREE,    /* Item that defines a subtree/submenu */
	MENU_TYPE_SEPARATOR   /* Separator line (menus) or blank space (toolbars) */
} MenuType;


typedef struct {
	MenuType type;          /* Type of item */
	gchar *label;			/* The text to use for this menu-item */
	gchar *shortcut;		/* The shortcut for this menu-item */
	gchar *tooltip;         /* The tooltip of this menu-item */
	gpointer moreinfo;		/* For an item, toggleitem, this is a pointer to the 
							   function to call when the item is activated. */
	gpointer user_data;		/* Data pointer to pass to callbacks */
	GnomeUIPixmapType pixmap_type;	/* Type of pixmap for the item */
	gconstpointer pixmap_info;      /* Pointer to the pixmap information:
									 *
									 * For GNOME_APP_PIXMAP_STOCK, a 
									 * pointer to the stock icon name.
									 *
									 * For GNOME_APP_PIXMAP_DATA, a 
									 * pointer to the inline xpm data.
									 *
									 * For GNOME_APP_PIXMAP_FILENAME, a 
									 * pointer to the filename string.
									 */

	GtkWidget *widget;		/* Filled in by gnome_app_create*, you can use this
							   to tweak the widgets once they have been created */
} MenuData;

#define MENUTYPE_END { \
    MENU_TYPE_END, \
    NULL, NULL, NULL, \
    NULL, NULL, \
	(GnomeUIPixmapType) 0, NULL, \
    NULL }

#define MENUTYPE_SEPARATOR { \
    MENU_TYPE_SEPARATOR, \
    NULL, NULL, NULL, \
    NULL, NULL, \
	(GnomeUIPixmapType) 0, NULL, \
    NULL }




struct _GnomeCmdMainMenuPrivate
{
	GtkWidget *file_menu;
	GtkWidget *edit_menu;
	GtkWidget *mark_menu;
	GtkWidget *view_menu;
	GtkWidget *options_menu;
	GtkWidget *ftp_menu;
	GtkWidget *bookmarks_menu;
	GtkWidget *help_menu;

	GtkWidget *menu_ftp_disconnect;
	GtkWidget *menu_edit_paste;
	GtkWidget *menu_view_toolbar;
	GtkWidget *menu_view_buttonbar;
	GtkWidget *menu_view_back;
	GtkWidget *menu_view_forward;

	GList *bookmark_menuitems;
	GList *group_menuitems;

	GtkTooltips *tooltips;
};


static GtkMenuBarClass *parent_class = NULL;



/*******************************
 * Private functions
 *******************************/

static void
on_bookmark_selected (GtkMenuItem *menuitem, GnomeCmdBookmark *bookmark)
{
	g_return_if_fail (bookmark != NULL);
	
	gnome_cmd_bookmark_goto (bookmark);
}


static GtkWidget*
create_menu_item (GnomeCmdMainMenu *main_menu, GtkMenu *parent, MenuData *spec)
{
	GtkWidget *item=NULL;
	GtkWidget *desc, *shortcut;
	GtkWidget *content = NULL;
	GtkWidget *pixmap = NULL;
	GtkAccelGroup *accel_group;
	guint key;

	switch (spec->type) {
		case MENU_TYPE_BASIC:
			item = gtk_menu_item_new ();
			desc = create_label (GTK_WIDGET (main_win), spec->label);			
			
			accel_group = gtk_accel_group_new ();
			key = gtk_label_parse_uline (GTK_LABEL (desc), spec->label);
			gtk_widget_add_accelerator (
				item, "activate_item", accel_group,
				key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
			gtk_window_add_accel_group (GTK_WINDOW (main_win), accel_group);
			
			gtk_container_add (GTK_CONTAINER (item), desc);
			break;
			
		case MENU_TYPE_ITEM:
			item = gtk_pixmap_menu_item_new ();
			content = create_hbox (GTK_WIDGET (main_win), FALSE, 12);

			desc = create_label (GTK_WIDGET (main_win), spec->label);
			gtk_box_pack_start (GTK_BOX (content), desc, FALSE, FALSE, 0);

			accel_group = gtk_accel_group_new ();
			key = gtk_label_parse_uline (GTK_LABEL (desc), spec->label);
			gtk_widget_add_accelerator (
				item, "activate_item", accel_group,
				key, 0, (GtkAccelFlags) 0);
			gtk_accel_group_attach (accel_group, GTK_OBJECT (parent));
			
			shortcut = create_label (GTK_WIDGET (main_win), spec->shortcut);
			gtk_misc_set_alignment (GTK_MISC (shortcut), 1.0, 0.5);
			gtk_box_pack_start (GTK_BOX (content), shortcut, TRUE, TRUE, 0);

			if (spec->pixmap_type != 0 && spec->pixmap_info) {
				pixmap = create_ui_pixmap (GTK_WIDGET (main_win),
										   spec->pixmap_type,
										   spec->pixmap_info);
				if (pixmap) {
					gtk_widget_show (pixmap);
					gtk_pixmap_menu_item_set_pixmap (
						GTK_PIXMAP_MENU_ITEM (item), pixmap);
				}
			}
			if (spec->tooltip)
				gtk_tooltips_set_tip (main_menu->priv->tooltips, item, spec->tooltip, NULL);
			gtk_container_add (GTK_CONTAINER (item), content);
			break;

		case MENU_TYPE_TOGGLEITEM:
			item = gtk_check_menu_item_new_with_label (spec->label);
			gtk_signal_connect (GTK_OBJECT (item), "toggled",
								GTK_SIGNAL_FUNC (spec->moreinfo), spec->user_data);
			break;
			
		case MENU_TYPE_SEPARATOR:
			item = gtk_menu_item_new ();
			gtk_widget_set_sensitive (item, FALSE);
			break;

		default:
			g_warning ("This MENU_TYPE is not implemented\n");
			return NULL;
	}

	
	gtk_widget_show (item);


	if (spec->type == MENU_TYPE_ITEM) {
		/* Connect to the signal and set user data */	
		gtk_object_set_data (GTK_OBJECT (item),
							 GNOMEUIINFO_KEY_UIDATA, spec->user_data);
		
		gtk_signal_connect (GTK_OBJECT (item), "activate",
							GTK_SIGNAL_FUNC (spec->moreinfo), spec->user_data);
	}

	spec->widget = item;
	
	return item;
}


static GtkWidget*
create_menu (GnomeCmdMainMenu *main_menu, MenuData *spec, MenuData *childs)
{	
	gint i=0;
	GtkWidget *submenu, *item;

	submenu = gtk_menu_new ();
	item = create_menu_item (main_menu, NULL, spec);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);

	gtk_widget_ref (item);
	gtk_widget_show (item);

	while (childs[i].type != MENU_TYPE_END) {
		GtkWidget *child = create_menu_item (main_menu, GTK_MENU (submenu), &childs[i]);
		gtk_menu_shell_append (GTK_MENU_SHELL (submenu), child);
		i++;
	}

	return item;
}


static void
add_bookmark_menu_item (GnomeCmdMainMenu *main_menu, GtkMenuShell *menu, GnomeCmdBookmark *bookmark)
{
	GtkWidget *item, *label;
	GtkWidget *pixmap = NULL;

	g_return_if_fail (GTK_IS_MENU_SHELL (menu));
	g_return_if_fail (bookmark != NULL);
	
	item = gtk_pixmap_menu_item_new ();
	gtk_tooltips_set_tip (
		main_menu->priv->tooltips, item, bookmark->path, NULL);
	
	pixmap = gtk_pixmap_new (IMAGE_get_pixmap(PIXMAP_BOOKMARK),
							 IMAGE_get_mask(PIXMAP_BOOKMARK));
	if (pixmap) {
		gtk_widget_show (pixmap);
		gtk_pixmap_menu_item_set_pixmap (
			GTK_PIXMAP_MENU_ITEM (item), pixmap);
	}

	gtk_widget_show (item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

	/* Remeber this bookmarks item-widget so that we can remove it later */
	main_menu->priv->bookmark_menuitems = g_list_append (
		main_menu->priv->bookmark_menuitems,
		item);

	
	/* Create the contents of the menu item */
	label = gtk_label_new (bookmark->name);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_widget_show (label);
	gtk_container_add (GTK_CONTAINER (item), label);

	
	/* Connect to the signal and set user data */	
	gtk_object_set_data (GTK_OBJECT (item),
						 GNOMEUIINFO_KEY_UIDATA, bookmark);

	gtk_signal_connect (GTK_OBJECT (item), "activate",
						GTK_SIGNAL_FUNC (on_bookmark_selected), bookmark);
}


static void
add_bookmark_group (GnomeCmdMainMenu *main_menu, GtkMenuShell *menu, GnomeCmdBookmarkGroup *group)
{
	GtkWidget *submenu;
	GtkWidget *item, *label;
	GtkWidget *pixmap = NULL;
	GList *bookmarks = group->bookmarks;
	Pixmap pixmap_id;

	g_return_if_fail (GTK_IS_MENU_SHELL (menu));
	g_return_if_fail (bookmarks != NULL);
	
	item = gtk_pixmap_menu_item_new ();
	pixmap_id = group->server?PIXMAP_SERVER_SMALL:PIXMAP_HOME;
	pixmap = gtk_pixmap_new (IMAGE_get_pixmap(pixmap_id),
							 IMAGE_get_mask(pixmap_id));
	if (pixmap) {
		gtk_widget_show (pixmap);
		gtk_pixmap_menu_item_set_pixmap (
			GTK_PIXMAP_MENU_ITEM (item), pixmap);
	}

	gtk_widget_show (item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

	/* Remeber this bookmarks item-widget so that we can remove it later */
	main_menu->priv->group_menuitems = g_list_append (
		main_menu->priv->group_menuitems, item);
	
	/* Create the contents of the menu item */
	label = gtk_label_new (group->alias);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_widget_show (label);
	gtk_container_add (GTK_CONTAINER (item), label);

	
	submenu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);

	
	/* Add bookmarks for this group */
	while (bookmarks) {
		GnomeCmdBookmark *bookmark = (GnomeCmdBookmark*)bookmarks->data;
		add_bookmark_menu_item (main_menu, GTK_MENU_SHELL (submenu), bookmark);
		bookmarks = bookmarks->next;
	}
}



/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdMainMenu *menu = GNOME_CMD_MAIN_MENU (object);

	g_free (menu->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdMainMenuClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gtk_menu_bar_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdMainMenu *main_menu)
{
	MenuData files_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("Change _Owner/Group"), "", NULL,
			file_chown, NULL, 
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Change Per_missions"), "", NULL,
			file_chmod, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Advanced _Rename Tool"), "CTRL+T", NULL,
			file_advrename, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Create _Symbolic Link"), "CTRL+SHIFT+F5", NULL,
			file_create_symlink, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Properties..."), "ALT+ENTER", NULL,
			file_properties, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		MENUTYPE_SEPARATOR,
		{
			MENU_TYPE_ITEM, _("_Diff"), "", NULL,
			file_diff, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		MENUTYPE_SEPARATOR,
		{
			MENU_TYPE_ITEM, _("_Quit"), "CTRL+Q", NULL,
			file_exit, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT,
			NULL
		},
		MENUTYPE_END
	};

	MenuData mark_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("Select All"), "CTRL++", NULL,
			mark_select_all, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Unselect All"), "CTRL+-", NULL,
			mark_unselect_all, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Select with Pattern"), "+", NULL,
			mark_select_with_pattern, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Unselect with Pattern"), "-", NULL,
			mark_unselect_with_pattern, NULL, 
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Invert Selection"), "*", NULL,
			mark_invert_selection, NULL, 
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Restore Selection"), "", NULL,
			mark_restore_selection, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		MENUTYPE_END
	};
	
	MenuData edit_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("Cu_t"), "CTRL+X", NULL,
			file_cap_cut, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Copy"), "CTRL+C", NULL,
			file_cap_copy, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Paste"), "CTRL+V", NULL,
			file_cap_paste, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Delete"), "Delete", NULL,
			file_delete, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
			NULL
		},
		MENUTYPE_SEPARATOR,
		{
			MENU_TYPE_ITEM, _("_Search..."), "ALT+F7", NULL,
			edit_search, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Quick Search..."), "", NULL,
			edit_quick_search, NULL, 
			GNOME_APP_PIXMAP_NONE, 0,
			NULL
		},
		MENUTYPE_END
	};

	MenuData view_menu_uiinfo[] =
	{
		{
			MENU_TYPE_TOGGLEITEM, _("Toolbar"), "", NULL,
			view_toolbar, NULL, 
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		{
			MENU_TYPE_TOGGLEITEM, _("Buttonbar"), "", NULL,
			view_buttonbar, NULL, 
			GNOME_APP_PIXMAP_NONE, NULL,
			NULL
		},
		MENUTYPE_SEPARATOR,
		{
			MENU_TYPE_ITEM, _("_Back"), "ALT+Left", NULL,
			view_back, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Forward"), "ALT+Right", NULL,
			view_forward, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Refresh"), "CTRL+R", NULL,
			view_refresh, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
			NULL
		},
		MENUTYPE_END
	};

	MenuData bookmarks_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("_Bookmark this Directory..."), "", NULL,
			bookmarks_add_current, NULL, 
			GNOME_APP_PIXMAP_NONE, 0,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Edit Bookmarks..."), "CTRL+D", NULL,
			bookmarks_edit, NULL, 
			GNOME_APP_PIXMAP_NONE, 0,
			NULL
		},
		MENUTYPE_SEPARATOR,
		MENUTYPE_END
	};
	
	MenuData options_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("_Options..."), "CTRL+O", NULL,
			options_edit, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Edit _Mime Types..."), "", NULL,
			options_edit_mime_types, NULL, 
			GNOME_APP_PIXMAP_NONE, 0,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("Save _Position"), "", NULL,
			options_save_position, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
			NULL
		},
		MENUTYPE_END
	};
	
	MenuData ftp_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("_Connect..."), "CTRL+F", NULL,
			ftp_connect, NULL, 
			GNOME_APP_PIXMAP_DATA, menu_ftp_connect_xpm,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Quick Connect..."), "CTRL+G", NULL,
			ftp_quick_connect, NULL, 
			GNOME_APP_PIXMAP_DATA, menu_ftp_connect_xpm,
			NULL
		},
		{
			MENU_TYPE_ITEM, _("_Disconnect"), "CTRL+SHIFT+F", NULL,
			ftp_disconnect, NULL, 
			GNOME_APP_PIXMAP_DATA, menu_ftp_disconnect_xpm,
			NULL
		},
		MENUTYPE_END
	};

	MenuData help_menu_uiinfo[] =
	{
		{
			MENU_TYPE_ITEM, _("User _Manual..."), "F1", NULL,
			help_help, NULL, 
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_RED,
			NULL
		},
		MENUTYPE_SEPARATOR,
		{
			MENU_TYPE_ITEM, _("_About..."), "", NULL,
			help_about, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT,
			NULL
		},
		MENUTYPE_END
	};

	MenuData spec = { MENU_TYPE_BASIC, "", "", "", NULL, NULL, 0, NULL, NULL };

	main_menu->priv = g_new (GnomeCmdMainMenuPrivate, 1);
	main_menu->priv->tooltips = gtk_tooltips_new ();	
	main_menu->priv->bookmark_menuitems = NULL;
	main_menu->priv->group_menuitems = NULL;

	gtk_menu_bar_set_shadow_type (GTK_MENU_BAR (main_menu), GTK_SHADOW_NONE);

	spec.label = _("_File");
	main_menu->priv->file_menu = create_menu (main_menu, &spec, files_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->file_menu);
	
	spec.label = _("_Edit");
	main_menu->priv->edit_menu = create_menu (main_menu, &spec, edit_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->edit_menu);
	
	spec.label = _("_Mark");
	main_menu->priv->mark_menu = create_menu (main_menu, &spec, mark_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->mark_menu);
	
	spec.label = _("_View");
	main_menu->priv->view_menu = create_menu (main_menu, &spec, view_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->view_menu);
	
	spec.label = _("_Settings");
	main_menu->priv->options_menu = create_menu (main_menu, &spec, options_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->options_menu);
	
	spec.label = _("FT_P");
	main_menu->priv->ftp_menu = create_menu (main_menu, &spec, ftp_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->ftp_menu);
	
	spec.label = _("_Bookmarks");
	main_menu->priv->bookmarks_menu = create_menu (main_menu, &spec, bookmarks_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->bookmarks_menu);
	
	spec.label = _("_Help");
	main_menu->priv->help_menu = create_menu (main_menu, &spec, help_menu_uiinfo);
	gtk_menu_shell_append (GTK_MENU_SHELL (main_menu), main_menu->priv->help_menu);

	main_menu->priv->menu_ftp_disconnect = ftp_menu_uiinfo[2].widget;
	main_menu->priv->menu_edit_paste = edit_menu_uiinfo[2].widget;
	main_menu->priv->menu_view_toolbar = view_menu_uiinfo[0].widget;
	main_menu->priv->menu_view_buttonbar = view_menu_uiinfo[1].widget;
	main_menu->priv->menu_view_back = view_menu_uiinfo[3].widget;
	main_menu->priv->menu_view_forward = view_menu_uiinfo[4].widget;

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM (main_menu->priv->menu_view_toolbar),
		gnome_cmd_data_get_toolbar_visibility ());

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM (main_menu->priv->menu_view_buttonbar),
		gnome_cmd_data_get_buttonbar_visibility ());
	
	gnome_cmd_main_menu_update_bookmarks (main_menu);
}




/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_main_menu_new (void)
{
	return gtk_type_new (gnome_cmd_main_menu_get_type ());
}



GtkType
gnome_cmd_main_menu_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdMainMenu",
			sizeof (GnomeCmdMainMenu),
			sizeof (GnomeCmdMainMenuClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gtk_menu_bar_get_type (), &dlg_info);
	}
	return dlg_type;
}



void
gnome_cmd_main_menu_update_bookmarks (GnomeCmdMainMenu *main_menu)
{
	GList *groups;
	
	g_return_if_fail (GNOME_CMD_IS_MAIN_MENU (main_menu));

	/* Remove all old bookmark menu items */
	g_list_foreach (main_menu->priv->bookmark_menuitems, (GFunc)gtk_widget_destroy, NULL);
	g_list_free (main_menu->priv->bookmark_menuitems);
	main_menu->priv->bookmark_menuitems = NULL;

	/* Remove all old group menu items */
	g_list_foreach (main_menu->priv->group_menuitems, (GFunc)gtk_widget_destroy, NULL);
	g_list_free (main_menu->priv->group_menuitems);
	main_menu->priv->group_menuitems = NULL;
	
	/* Add bookmark groups */
	groups = gnome_cmd_data_get_bookmark_groups ();
	while (groups) {
		GnomeCmdBookmarkGroup *group = (GnomeCmdBookmarkGroup*)groups->data;
		GtkMenuShell *bookmarks_menu =
			GTK_MENU_SHELL (GTK_MENU_ITEM (main_menu->priv->bookmarks_menu)->submenu);
		if (group->bookmarks)
			add_bookmark_group (main_menu, bookmarks_menu, group);
		groups = groups->next;
	}
}


void
gnome_cmd_main_menu_update_sens (GnomeCmdMainMenu *main_menu)
{
	GnomeCmdFileSelector *fs;
	GnomeCmdConnection *con;
	
	g_return_if_fail (GNOME_CMD_IS_MAIN_MENU (main_menu));

	fs = gnome_cmd_main_win_get_active_fs (main_win);
	con = gnome_cmd_file_selector_get_connection (fs);

	gtk_widget_set_sensitive (main_menu->priv->menu_view_back,
							  gnome_cmd_file_selector_can_back (fs));
	gtk_widget_set_sensitive (main_menu->priv->menu_view_forward,
							  gnome_cmd_file_selector_can_forward (fs));
	gtk_widget_set_sensitive (main_menu->priv->menu_ftp_disconnect,
							  con->type == CON_TYPE_FTP);
}


