/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "widget-factory.h"
#include "gnome-cmd-string-dialog.h"
#include "utils.h"

struct _GnomeCmdStringDialogPrivate
{
	GnomeCmdStringDialogCallback ok_cb;
	GtkSignalFunc cancel_cb;
	gpointer data;
	GtkWidget *dialog;
	gchar *error_desc;
};


static GnomeDialogClass *parent_class = NULL;


static void
on_ok (GtkButton *button, GnomeCmdStringDialog *dialog)
{
	gboolean valid = TRUE;	
	
	if (dialog->priv->ok_cb) {
		gint i;
		gchar **values = (gchar**)g_new (gpointer, dialog->rows);

		for ( i=0 ; i<dialog->rows ; i++ )
			values[i] = gtk_entry_get_text (GTK_ENTRY (dialog->entries[i]));
								  
		
		valid = dialog->priv->ok_cb (dialog, (const gchar**)values, dialog->priv->data);
		if (!valid)
			create_error_dialog (dialog->priv->error_desc);
		g_free (values);
	}

	if (valid)
		gtk_widget_hide (GTK_WIDGET (dialog));
}


static void
on_cancel (GtkButton *button, GnomeCmdStringDialog *dialog)
{
	if (dialog->priv->cancel_cb)
		dialog->priv->cancel_cb (button, dialog->priv->data);
	
	gtk_widget_hide (GTK_WIDGET (dialog));
}





/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdStringDialog *dialog = GNOME_CMD_STRING_DIALOG (object);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);

	g_free (dialog->priv->error_desc);
	g_free (dialog->priv);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdStringDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}

static void
init (GnomeCmdStringDialog *string_dialog)
{
	string_dialog->priv = g_new (GnomeCmdStringDialogPrivate, 1);
	string_dialog->priv->ok_cb = NULL;
	string_dialog->priv->cancel_cb = NULL;
	string_dialog->priv->data = NULL;
	string_dialog->rows = -1;
	
	gtk_window_set_policy (GTK_WINDOW (string_dialog), FALSE, TRUE, FALSE);
	gtk_window_set_position (GTK_WINDOW (string_dialog), GTK_WIN_POS_CENTER);
}


static void
setup_widget (GnomeCmdStringDialog *string_dialog, gint rows)
{
	gint i;
	GtkWidget *dialog = GTK_WIDGET (string_dialog);
	GtkWidget *dialog_vbox1;
	GtkWidget *table;
	GtkWidget *button;

	string_dialog->rows = rows;
	string_dialog->labels = (GtkWidget**)g_new (gpointer, rows);
	string_dialog->entries = (GtkWidget**)g_new (gpointer, rows);
	string_dialog->priv->error_desc = g_strdup (_("No error description available"));

	dialog_vbox1 = GNOME_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog),
						 "dialog_vbox1", dialog_vbox1);
	gtk_box_set_spacing (GTK_BOX (dialog_vbox1), 12);
	gtk_widget_show (dialog_vbox1);

	
	table = create_table (dialog, rows, 2);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1), table, TRUE, TRUE, 0);

	for ( i=0 ; i<rows ; i++ ) {
		string_dialog->labels[i] = create_label (dialog, "");
		table_add (table, string_dialog->labels[i], 0, i, GTK_FILL|GTK_EXPAND);
		
		string_dialog->entries[i] = create_entry (dialog, "entry", "");
		table_add (table, string_dialog->entries[i], 1, i, GTK_FILL|GTK_EXPAND);
	}

	
	gnome_dialog_append_button (GNOME_DIALOG (dialog),
								GNOME_STOCK_BUTTON_OK);
	button = GTK_WIDGET (g_list_last (
		GNOME_DIALOG (dialog)->buttons)->data);
	gtk_widget_ref (button);
	gtk_object_set_data_full (GTK_OBJECT (dialog),
							  "button", button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
						GTK_SIGNAL_FUNC (on_ok), string_dialog);

	
	gnome_dialog_append_button (GNOME_DIALOG (dialog),
								GNOME_STOCK_BUTTON_CANCEL);
	button = GTK_WIDGET (g_list_last (
		GNOME_DIALOG (dialog)->buttons)->data);
	gtk_widget_ref (button);
	gtk_object_set_data_full (GTK_OBJECT (dialog),
							  "button", button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
						GTK_SIGNAL_FUNC (on_cancel), string_dialog);
	
	
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
	gtk_widget_grab_focus (string_dialog->entries[0]);
	gnome_dialog_editable_enters(GNOME_DIALOG(dialog), GTK_EDITABLE(string_dialog->entries[0]));
}




/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_string_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdStringDialog",
			sizeof (GnomeCmdStringDialog),
			sizeof (GnomeCmdStringDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}





GtkWidget* 
gnome_cmd_string_dialog_new_with_cancel (const gchar *title,
										 const gchar **labels,
										 gint rows,
										 GnomeCmdStringDialogCallback ok_cb,
										 GtkSignalFunc cancel_cb,
										 gpointer user_data)
{
	GnomeCmdStringDialog *dialog = gtk_type_new (gnome_cmd_string_dialog_get_type ());

	gnome_cmd_string_dialog_setup_with_cancel (
		dialog, title, labels, rows, 
		ok_cb, cancel_cb, user_data);
	
	return GTK_WIDGET (dialog);
}


GtkWidget* 
gnome_cmd_string_dialog_new (const gchar *title,
							 const gchar **labels,
							 gint rows,
							 GnomeCmdStringDialogCallback ok_cb,
							 gpointer user_data)
{
	return gnome_cmd_string_dialog_new_with_cancel (title, labels, rows,
													ok_cb, NULL, user_data);
}


void
gnome_cmd_string_dialog_setup_with_cancel (GnomeCmdStringDialog *dialog,
										   const gchar *title,
										   const gchar **labels,
										   gint rows,
										   GnomeCmdStringDialogCallback ok_cb,
										   GtkSignalFunc cancel_cb,
										   gpointer user_data)
{
	gint i;
	
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (title != NULL);
	g_return_if_fail (labels != NULL);
	g_return_if_fail (rows > 0);
	g_return_if_fail (ok_cb != NULL);

	setup_widget (dialog, rows);
	gnome_cmd_string_dialog_set_title (dialog, title);
	gnome_cmd_string_dialog_set_userdata (dialog, user_data);
	gnome_cmd_string_dialog_set_ok_cb (dialog, ok_cb);
	gnome_cmd_string_dialog_set_cancel_cb (dialog, cancel_cb);

	if (labels)
		for ( i=0 ; i<rows ; i++ )
			gnome_cmd_string_dialog_set_label (dialog, i, labels[i]);
}


void
gnome_cmd_string_dialog_setup (GnomeCmdStringDialog *dialog,
							   const gchar *title,
							   const gchar **labels,
							   gint rows,
							   GnomeCmdStringDialogCallback ok_cb,
							   gpointer user_data)
{
	gnome_cmd_string_dialog_setup_with_cancel (
		dialog, title, labels, rows, ok_cb, NULL, user_data);
}


void
gnome_cmd_string_dialog_set_hidden (GnomeCmdStringDialog *dialog, gint row, gboolean hidden)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (row >= 0 && row < dialog->rows);	

	gtk_entry_set_visibility (GTK_ENTRY (dialog->entries[row]), !hidden);	
}


void
gnome_cmd_string_dialog_set_title (GnomeCmdStringDialog *dialog, const gchar *title)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	
	gtk_window_set_title (GTK_WINDOW (dialog), title);
}


void
gnome_cmd_string_dialog_set_label (GnomeCmdStringDialog *dialog, gint row, const gchar *label)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (row >= 0 && row < dialog->rows);
	g_return_if_fail (label != NULL);
	
	gtk_label_set_text (GTK_LABEL (dialog->labels[row]), label);
}


void
gnome_cmd_string_dialog_set_userdata (GnomeCmdStringDialog *dialog, gpointer user_data)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	
	dialog->priv->data = user_data;
}


void
gnome_cmd_string_dialog_set_ok_cb (GnomeCmdStringDialog *dialog, GnomeCmdStringDialogCallback ok_cb)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (ok_cb != NULL);
	
	dialog->priv->ok_cb = ok_cb;
}


void
gnome_cmd_string_dialog_set_cancel_cb (GnomeCmdStringDialog *dialog, GtkSignalFunc cancel_cb)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	
	dialog->priv->cancel_cb = cancel_cb;
}


void
gnome_cmd_string_dialog_set_value (GnomeCmdStringDialog *dialog, gint row, const gchar *value)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (row >= 0 && row < dialog->rows);
	
	gtk_entry_set_text (GTK_ENTRY (dialog->entries[row]), value?value:"");
}


void
gnome_cmd_string_dialog_set_error_desc (GnomeCmdStringDialog *dialog, gchar *msg)
{
	g_return_if_fail (GNOME_CMD_IS_STRING_DIALOG (dialog));
	g_return_if_fail (msg != NULL);
	
	if (dialog->priv->error_desc)
		g_free (dialog->priv->error_desc);

	dialog->priv->error_desc = g_strdup (msg);
}



