/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/SqliteEPGStore.h>
#include <sqlite3.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gio/gio.h>
#include "src/Utils.h"




struct _DVBSqliteEPGStorePrivate {
	sqlite3_stmt* to_julian_statement;
	sqlite3_stmt* insert_event_statement;
	sqlite3_stmt* update_event_statement;
	sqlite3_stmt* delete_event_statement;
	sqlite3_stmt* has_event_statement;
	sqlite3_stmt* select_event_statement;
	sqlite3* db;
};

#define DVB_SQLITE_EPG_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SQLITE_EPG_STORE, DVBSqliteEPGStorePrivate))
enum  {
	DVB_SQLITE_EPG_STORE_DUMMY_PROPERTY
};
#define DVB_SQLITE_EPG_STORE_CREATE_EVENTS_TABLE_STATEMENT "CREATE TABLE events (group_id INTEGER,\n            sid INTEGER,\n            event_id INTEGER,\n            starttime JULIAN,\n            duration INTEGER,\n            running_status INTEGER(2),\n            free_ca_mode INTEGER(1),\n            name VARCHAR(255),\n            description VARCHAR(255),\n            extended_description TEXT,\n            PRIMARY KEY (sid, event_id))"
#define DVB_SQLITE_EPG_STORE_INSERT_EVENT_SQL "INSERT INTO events VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define DVB_SQLITE_EPG_STORE_DELETE_EVENT_STATEMENT "DELETE FROM events WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_SELECT_ALL_EVENTS_STATEMENT "SELECT event_id, datetime(starttime),\n            duration, running_status, free_ca_mode, name,\n            description, extended_description\n            FROM events WHERE group_id='%u' AND sid='%u'"
#define DVB_SQLITE_EPG_STORE_HAS_EVENT_STATEMENT "SELECT COUNT(*) FROM events WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_UPDATE_EVENT_SQL "UPDATE events SET starttime=?, duration=?, running_status=?,\n            free_ca_mode=?, name=?, description=?,\n            extended_description=? WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_TO_JULIAN_SQL "SELECT julianday(?)"
#define DVB_SQLITE_EPG_STORE_SELECT_EVENT_SQL "SELECT event_id, datetime(starttime),\n            duration, running_status, free_ca_mode, name,\n            description, extended_description\n            FROM events WHERE group_id=? AND sid=? AND event_id=?"
static gboolean dvb_sqlite_epg_store_real_add_or_update_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id);
static DVBEvent* dvb_sqlite_epg_store_real_get_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id);
static gboolean dvb_sqlite_epg_store_real_remove_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id);
static gboolean dvb_sqlite_epg_store_real_contains_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id);
static GeeList* dvb_sqlite_epg_store_real_get_events (DVBEPGStore* base, guint channel_sid, guint group_id);
static void _g_slist_free_dvb_event_audio_component_unref (GSList* self);
static void _g_slist_free_dvb_event_video_component_unref (GSList* self);
static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self);
static DVBEvent* dvb_sqlite_epg_store_create_event_from_statement (DVBSqliteEPGStore* self, sqlite3_stmt* statement);
static void dvb_sqlite_epg_store_print_last_error (DVBSqliteEPGStore* self);
static double dvb_sqlite_epg_store_to_julian (DVBSqliteEPGStore* self, guint year, guint month, guint day, guint hour, guint minute, guint second);
static char* dvb_sqlite_epg_store_escape (const char* text);
static sqlite3* dvb_sqlite_epg_store_get_db_handler (void);
static GObject * dvb_sqlite_epg_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_sqlite_epg_store_parent_class = NULL;
static DVBEPGStoreIface* dvb_sqlite_epg_store_dvb_epg_store_parent_iface = NULL;
static void dvb_sqlite_epg_store_finalize (GObject* obj);



static gboolean dvb_sqlite_epg_store_real_add_or_update_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gint _tmp1;
	gint free_ca_mode;
	char* name;
	char* desc;
	char* ext_desc;
	double julian_start;
	gboolean _tmp31;
	self = (DVBSqliteEPGStore*) base;
	g_return_val_if_fail (event != NULL, FALSE);
	if (self->priv->db == NULL) {
		g_critical ("SqliteEPGStore.vala:100: SQLite error: No database connection");
		return FALSE;
	}
	_tmp1 = 0;
	if (event->free_ca_mode) {
		_tmp1 = 1;
	} else {
		_tmp1 = 0;
	}
	free_ca_mode = _tmp1;
	name = dvb_sqlite_epg_store_escape (event->name);
	desc = dvb_sqlite_epg_store_escape (event->description);
	ext_desc = dvb_sqlite_epg_store_escape (event->extended_description);
	julian_start = dvb_sqlite_epg_store_to_julian (self, event->year, event->month, event->day, event->hour, event->minute, event->second);
	/* Check if start time got converted correctly*/
	if (julian_start <= 0) {
		gboolean _tmp2;
		return (_tmp2 = FALSE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp2);
	}
	if (dvb_epg_store_contains_event ((DVBEPGStore*) self, event, channel_sid, group_id)) {
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp5;
		gboolean _tmp6;
		gboolean _tmp7;
		gboolean _tmp8;
		gboolean _tmp9;
		gboolean _tmp10;
		gboolean _tmp11;
		sqlite3_reset (self->priv->update_event_statement);
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		_tmp5 = FALSE;
		_tmp6 = FALSE;
		_tmp7 = FALSE;
		_tmp8 = FALSE;
		_tmp9 = FALSE;
		_tmp10 = FALSE;
		_tmp11 = FALSE;
		if (sqlite3_bind_double (self->priv->update_event_statement, 1, julian_start) != SQLITE_OK) {
			_tmp11 = TRUE;
		} else {
			_tmp11 = sqlite3_bind_int (self->priv->update_event_statement, 2, (gint) event->duration) != SQLITE_OK;
		}
		if (_tmp11) {
			_tmp10 = TRUE;
		} else {
			_tmp10 = sqlite3_bind_int (self->priv->update_event_statement, 3, (gint) event->running_status) != SQLITE_OK;
		}
		if (_tmp10) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = sqlite3_bind_int (self->priv->update_event_statement, 4, free_ca_mode) != SQLITE_OK;
		}
		if (_tmp9) {
			_tmp8 = TRUE;
		} else {
			const char* _tmp12;
			_tmp12 = NULL;
			_tmp8 = sqlite3_bind_text (self->priv->update_event_statement, 5, (_tmp12 = name, (_tmp12 == NULL) ? NULL : g_strdup (_tmp12)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp8) {
			_tmp7 = TRUE;
		} else {
			const char* _tmp13;
			_tmp13 = NULL;
			_tmp7 = sqlite3_bind_text (self->priv->update_event_statement, 6, (_tmp13 = desc, (_tmp13 == NULL) ? NULL : g_strdup (_tmp13)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp7) {
			_tmp6 = TRUE;
		} else {
			const char* _tmp14;
			_tmp14 = NULL;
			_tmp6 = sqlite3_bind_text (self->priv->update_event_statement, 7, (_tmp14 = ext_desc, (_tmp14 == NULL) ? NULL : g_strdup (_tmp14)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp6) {
			_tmp5 = TRUE;
		} else {
			_tmp5 = sqlite3_bind_int (self->priv->update_event_statement, 8, (gint) group_id) != SQLITE_OK;
		}
		if (_tmp5) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = sqlite3_bind_int (self->priv->update_event_statement, 9, (gint) channel_sid) != SQLITE_OK;
		}
		if (_tmp4) {
			_tmp3 = TRUE;
		} else {
			_tmp3 = sqlite3_bind_int (self->priv->update_event_statement, 10, (gint) event->id) != SQLITE_OK;
		}
		if (_tmp3) {
			gboolean _tmp15;
			dvb_sqlite_epg_store_print_last_error (self);
			return (_tmp15 = FALSE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp15);
		}
		if (sqlite3_step (self->priv->update_event_statement) != SQLITE_DONE) {
			gboolean _tmp16;
			dvb_sqlite_epg_store_print_last_error (self);
			return (_tmp16 = FALSE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp16);
		}
	} else {
		gboolean _tmp17;
		gboolean _tmp18;
		gboolean _tmp19;
		gboolean _tmp20;
		gboolean _tmp21;
		gboolean _tmp22;
		gboolean _tmp23;
		gboolean _tmp24;
		gboolean _tmp25;
		sqlite3_reset (self->priv->insert_event_statement);
		_tmp17 = FALSE;
		_tmp18 = FALSE;
		_tmp19 = FALSE;
		_tmp20 = FALSE;
		_tmp21 = FALSE;
		_tmp22 = FALSE;
		_tmp23 = FALSE;
		_tmp24 = FALSE;
		_tmp25 = FALSE;
		if (sqlite3_bind_int (self->priv->insert_event_statement, 1, (gint) group_id) != SQLITE_OK) {
			_tmp25 = TRUE;
		} else {
			_tmp25 = sqlite3_bind_int (self->priv->insert_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
		}
		if (_tmp25) {
			_tmp24 = TRUE;
		} else {
			_tmp24 = sqlite3_bind_int (self->priv->insert_event_statement, 3, (gint) event->id) != SQLITE_OK;
		}
		if (_tmp24) {
			_tmp23 = TRUE;
		} else {
			_tmp23 = sqlite3_bind_double (self->priv->insert_event_statement, 4, julian_start) != SQLITE_OK;
		}
		if (_tmp23) {
			_tmp22 = TRUE;
		} else {
			_tmp22 = sqlite3_bind_int (self->priv->insert_event_statement, 5, (gint) event->duration) != SQLITE_OK;
		}
		if (_tmp22) {
			_tmp21 = TRUE;
		} else {
			_tmp21 = sqlite3_bind_int (self->priv->insert_event_statement, 6, (gint) event->running_status) != SQLITE_OK;
		}
		if (_tmp21) {
			_tmp20 = TRUE;
		} else {
			_tmp20 = sqlite3_bind_int (self->priv->insert_event_statement, 7, free_ca_mode) != SQLITE_OK;
		}
		if (_tmp20) {
			_tmp19 = TRUE;
		} else {
			const char* _tmp26;
			_tmp26 = NULL;
			_tmp19 = sqlite3_bind_text (self->priv->insert_event_statement, 8, (_tmp26 = name, (_tmp26 == NULL) ? NULL : g_strdup (_tmp26)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp19) {
			_tmp18 = TRUE;
		} else {
			const char* _tmp27;
			_tmp27 = NULL;
			_tmp18 = sqlite3_bind_text (self->priv->insert_event_statement, 9, (_tmp27 = desc, (_tmp27 == NULL) ? NULL : g_strdup (_tmp27)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp18) {
			_tmp17 = TRUE;
		} else {
			const char* _tmp28;
			_tmp28 = NULL;
			_tmp17 = sqlite3_bind_text (self->priv->insert_event_statement, 10, (_tmp28 = ext_desc, (_tmp28 == NULL) ? NULL : g_strdup (_tmp28)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp17) {
			gboolean _tmp29;
			dvb_sqlite_epg_store_print_last_error (self);
			return (_tmp29 = FALSE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp29);
		}
		if (sqlite3_step (self->priv->insert_event_statement) != SQLITE_DONE) {
			gboolean _tmp30;
			dvb_sqlite_epg_store_print_last_error (self);
			return (_tmp30 = FALSE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp30);
		}
	}
	return (_tmp31 = TRUE, name = (g_free (name), NULL), desc = (g_free (desc), NULL), ext_desc = (g_free (ext_desc), NULL), _tmp31);
}


static DVBEvent* dvb_sqlite_epg_store_real_get_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean _tmp0;
	gboolean _tmp1;
	gint rc;
	gboolean _tmp3;
	self = (DVBSqliteEPGStore*) base;
	sqlite3_reset (self->priv->select_event_statement);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (sqlite3_bind_int (self->priv->select_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = sqlite3_bind_int (self->priv->select_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = sqlite3_bind_int (self->priv->select_event_statement, 3, (gint) event_id) != SQLITE_OK;
	}
	if (_tmp0) {
		dvb_sqlite_epg_store_print_last_error (self);
		return NULL;
	}
	rc = sqlite3_step (self->priv->select_event_statement);
	_tmp3 = FALSE;
	if (rc != SQLITE_ROW) {
		_tmp3 = rc != SQLITE_DONE;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		dvb_sqlite_epg_store_print_last_error (self);
		return NULL;
	}
	/* ROW means there's data, DONE means there's none*/
	if (rc == SQLITE_DONE) {
		return NULL;
	} else {
		return dvb_sqlite_epg_store_create_event_from_statement (self, self->priv->select_event_statement);
	}
}


static gboolean dvb_sqlite_epg_store_real_remove_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean _tmp1;
	gboolean _tmp2;
	self = (DVBSqliteEPGStore*) base;
	if (self->priv->db == NULL) {
		g_critical ("SqliteEPGStore.vala:185: SQLite error: No database connection");
		return FALSE;
	}
	sqlite3_reset (self->priv->delete_event_statement);
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	if (sqlite3_bind_int (self->priv->delete_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = sqlite3_bind_int (self->priv->delete_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp2) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = sqlite3_bind_int (self->priv->delete_event_statement, 3, (gint) event_id) != SQLITE_OK;
	}
	if (_tmp1) {
		dvb_sqlite_epg_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->delete_event_statement) != SQLITE_DONE) {
		dvb_sqlite_epg_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


static gboolean dvb_sqlite_epg_store_real_contains_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean _tmp0;
	gboolean _tmp1;
	gint c;
	self = (DVBSqliteEPGStore*) base;
	g_return_val_if_fail (event != NULL, FALSE);
	sqlite3_reset (self->priv->has_event_statement);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (sqlite3_bind_int (self->priv->has_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = sqlite3_bind_int (self->priv->has_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = sqlite3_bind_int (self->priv->has_event_statement, 3, (gint) event->id) != SQLITE_OK;
	}
	if (_tmp0) {
		dvb_sqlite_epg_store_print_last_error (self);
		return FALSE;
	}
	c = 0;
	while (sqlite3_step (self->priv->has_event_statement) == SQLITE_ROW) {
		c = sqlite3_column_int (self->priv->has_event_statement, 0);
	}
	return c > 0;
}


static GeeList* dvb_sqlite_epg_store_real_get_events (DVBEPGStore* base, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	GeeList* events;
	char* statement_str;
	sqlite3_stmt* statement;
	sqlite3_stmt* _tmp3;
	gint _tmp2;
	sqlite3_stmt* _tmp1;
	GeeList* _tmp5;
	self = (DVBSqliteEPGStore*) base;
	events = (GeeList*) gee_array_list_new (DVB_TYPE_EVENT, (GBoxedCopyFunc) dvb_event_ref, dvb_event_unref, g_direct_equal);
	if (self->priv->db == NULL) {
		return events;
	}
	statement_str = g_strdup_printf (DVB_SQLITE_EPG_STORE_SELECT_ALL_EVENTS_STATEMENT, group_id, channel_sid);
	statement = NULL;
	_tmp3 = NULL;
	_tmp1 = NULL;
	if ((_tmp2 = sqlite3_prepare (self->priv->db, statement_str, -1, &_tmp1, NULL), statement = (_tmp3 = _tmp1, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp3), _tmp2) != SQLITE_OK) {
		GeeList* _tmp4;
		dvb_sqlite_epg_store_print_last_error (self);
		_tmp4 = NULL;
		return (_tmp4 = events, statement_str = (g_free (statement_str), NULL), (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp4);
	}
	while (sqlite3_step (statement) == SQLITE_ROW) {
		DVBEvent* event;
		event = dvb_sqlite_epg_store_create_event_from_statement (self, statement);
		gee_collection_add ((GeeCollection*) events, event);
		(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
	}
	_tmp5 = NULL;
	return (_tmp5 = events, statement_str = (g_free (statement_str), NULL), (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp5);
}


static void _g_slist_free_dvb_event_audio_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_audio_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_video_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_video_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_teletext_component_unref, NULL);
	g_slist_free (self);
}


static DVBEvent* dvb_sqlite_epg_store_create_event_from_statement (DVBSqliteEPGStore* self, sqlite3_stmt* statement) {
	DVBEvent* event;
	const char* starttime;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	GSList* _tmp3;
	GSList* _tmp4;
	GSList* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (statement != NULL, NULL);
	event = dvb_event_new ();
	event->id = (guint) sqlite3_column_int (statement, 0);
	starttime = sqlite3_column_text (statement, 1);
	sscanf (starttime, "%04u-%02u-%02u %02u:%02u:%02u", &event->year, &event->month, &event->day, &event->hour, &event->minute, &event->second, NULL);
	event->duration = (guint) sqlite3_column_int (statement, 2);
	event->running_status = (guint) sqlite3_column_int (statement, 3);
	event->free_ca_mode = sqlite3_column_int (statement, 4) == 1;
	/* Duplicate strings*/
	_tmp0 = NULL;
	event->name = (_tmp0 = g_strdup_printf ("%s", sqlite3_column_text (statement, 5)), event->name = (g_free (event->name), NULL), _tmp0);
	_tmp1 = NULL;
	event->description = (_tmp1 = g_strdup_printf ("%s", sqlite3_column_text (statement, 6)), event->description = (g_free (event->description), NULL), _tmp1);
	_tmp2 = NULL;
	event->extended_description = (_tmp2 = g_strdup_printf ("%s", sqlite3_column_text (statement, 7)), event->extended_description = (g_free (event->extended_description), NULL), _tmp2);
	/* We don't save those*/
	_tmp3 = NULL;
	event->audio_components = (_tmp3 = NULL, (event->audio_components == NULL) ? NULL : (event->audio_components = (_g_slist_free_dvb_event_audio_component_unref (event->audio_components), NULL)), _tmp3);
	_tmp4 = NULL;
	event->video_components = (_tmp4 = NULL, (event->video_components == NULL) ? NULL : (event->video_components = (_g_slist_free_dvb_event_video_component_unref (event->video_components), NULL)), _tmp4);
	_tmp5 = NULL;
	event->teletext_components = (_tmp5 = NULL, (event->teletext_components == NULL) ? NULL : (event->teletext_components = (_g_slist_free_dvb_event_teletext_component_unref (event->teletext_components), NULL)), _tmp5);
	return event;
}


static void dvb_sqlite_epg_store_print_last_error (DVBSqliteEPGStore* self) {
	g_return_if_fail (self != NULL);
	g_critical ("SqliteEPGStore.vala:275: SQLite error: %d, %s", sqlite3_errcode (self->priv->db), sqlite3_errmsg (self->priv->db));
}


static double dvb_sqlite_epg_store_to_julian (DVBSqliteEPGStore* self, guint year, guint month, guint day, guint hour, guint minute, guint second) {
	char* datetime_str;
	const char* _tmp0;
	double _tmp3;
	g_return_val_if_fail (self != NULL, 0.0);
	sqlite3_reset (self->priv->to_julian_statement);
	datetime_str = g_strdup_printf ("%04u-%02u-%02u %02u:%02u:%02u", year, month, day, hour, minute, second);
	_tmp0 = NULL;
	if (sqlite3_bind_text (self->priv->to_julian_statement, 1, (_tmp0 = datetime_str, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), -1, g_free) != SQLITE_OK) {
		double _tmp1;
		dvb_sqlite_epg_store_print_last_error (self);
		return (_tmp1 = (double) 0, datetime_str = (g_free (datetime_str), NULL), _tmp1);
	}
	if (sqlite3_step (self->priv->to_julian_statement) != SQLITE_ROW) {
		double _tmp2;
		dvb_sqlite_epg_store_print_last_error (self);
		return (_tmp2 = (double) 0, datetime_str = (g_free (datetime_str), NULL), _tmp2);
	}
	return (_tmp3 = sqlite3_column_double (self->priv->to_julian_statement, 0), datetime_str = (g_free (datetime_str), NULL), _tmp3);
}


/**
         * Replace "'" with "''"
         */
static char* dvb_sqlite_epg_store_escape (const char* text) {
	GError * inner_error;
	GRegex* regex;
	char* escaped_str;
	char* _tmp8;
	inner_error = NULL;
	if (text == NULL) {
		return g_strdup ("");
	}
	regex = NULL;
	{
		GRegex* _tmp1;
		_tmp1 = NULL;
		regex = (_tmp1 = g_regex_new ("'", G_REGEX_MULTILINE, 0, &inner_error), (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == G_REGEX_ERROR) {
				goto __catch35_g_regex_error;
			}
			goto __finally35;
		}
	}
	goto __finally35;
	__catch35_g_regex_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			const char* _tmp2;
			char* _tmp3;
			g_warning ("SqliteEPGStore.vala:313: RegexError: %s", e->message);
			_tmp2 = NULL;
			_tmp3 = NULL;
			return (_tmp3 = (_tmp2 = text, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp3);
		}
	}
	__finally35:
	if (inner_error != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	escaped_str = NULL;
	{
		char* _tmp4;
		char* _tmp5;
		_tmp4 = g_regex_replace_literal (regex, text, strlen (text), 0, "''", 0, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_REGEX_ERROR) {
				goto __catch36_g_regex_error;
			}
			goto __finally36;
		}
		_tmp5 = NULL;
		escaped_str = (_tmp5 = _tmp4, escaped_str = (g_free (escaped_str), NULL), _tmp5);
	}
	goto __finally36;
	__catch36_g_regex_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			const char* _tmp6;
			char* _tmp7;
			g_warning ("SqliteEPGStore.vala:322: RegexError: %s", e->message);
			_tmp6 = NULL;
			_tmp7 = NULL;
			return (_tmp7 = (_tmp6 = text, (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), escaped_str = (g_free (escaped_str), NULL), _tmp7);
		}
	}
	__finally36:
	if (inner_error != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		escaped_str = (g_free (escaped_str), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	_tmp8 = NULL;
	return (_tmp8 = escaped_str, (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp8);
}


static sqlite3* dvb_sqlite_epg_store_get_db_handler (void) {
	GError * inner_error;
	GFile* cache_dir;
	GFile* our_cache;
	GFile* eventsdb;
	gboolean create_tables;
	sqlite3* db;
	sqlite3* _tmp4;
	gint _tmp3;
	sqlite3* _tmp2;
	char* _tmp1;
	sqlite3* _tmp9;
	inner_error = NULL;
	cache_dir = g_file_new_for_path (g_get_user_cache_dir ());
	our_cache = g_file_get_child (cache_dir, "gnome-dvb-daemon");
	eventsdb = g_file_get_child (our_cache, "eventsdb.sqlite3");
	if (!g_file_query_exists (our_cache, NULL)) {
		{
			dvb_utils_mkdirs (our_cache, &inner_error);
			if (inner_error != NULL) {
				goto __catch37_g_error;
				goto __finally37;
			}
		}
		goto __finally37;
		__catch37_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				sqlite3* _tmp0;
				g_critical ("SqliteEPGStore.vala:339: Could not create directory: %s", e->message);
				_tmp0 = NULL;
				return (_tmp0 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL)), (our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL)), (eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL)), _tmp0);
			}
		}
		__finally37:
		if (inner_error != NULL) {
			(cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL));
			(our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL));
			(eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	create_tables = !g_file_query_exists (eventsdb, NULL);
	db = NULL;
	_tmp4 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = sqlite3_open (_tmp1 = g_file_get_path (eventsdb), &_tmp2);
	db = (_tmp4 = _tmp2, (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp4);
	_tmp3;
	_tmp1 = (g_free (_tmp1), NULL);
	if (create_tables) {
		char* errormsg;
		char* _tmp7;
		gint _tmp6;
		char* _tmp5;
		gint val;
		errormsg = NULL;
		_tmp7 = NULL;
		_tmp5 = NULL;
		val = (_tmp6 = sqlite3_exec (db, DVB_SQLITE_EPG_STORE_CREATE_EVENTS_TABLE_STATEMENT, NULL, NULL, &_tmp5), errormsg = (_tmp7 = _tmp5, errormsg = (g_free (errormsg), NULL), _tmp7), _tmp6);
		if (val != SQLITE_OK) {
			sqlite3* _tmp8;
			g_critical ("SqliteEPGStore.vala:353: SQLite error: %s", errormsg);
			_tmp8 = NULL;
			return (_tmp8 = NULL, errormsg = (g_free (errormsg), NULL), (cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL)), (our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL)), (eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL)), (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp8);
		}
		errormsg = (g_free (errormsg), NULL);
	}
	_tmp9 = NULL;
	return (_tmp9 = db, (cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL)), (our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL)), (eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL)), _tmp9);
}


DVBSqliteEPGStore* dvb_sqlite_epg_store_construct (GType object_type) {
	DVBSqliteEPGStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBSqliteEPGStore* dvb_sqlite_epg_store_new (void) {
	return dvb_sqlite_epg_store_construct (DVB_TYPE_SQLITE_EPG_STORE);
}


static GObject * dvb_sqlite_epg_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBSqliteEPGStoreClass * klass;
	GObjectClass * parent_class;
	DVBSqliteEPGStore * self;
	klass = DVB_SQLITE_EPG_STORE_CLASS (g_type_class_peek (DVB_TYPE_SQLITE_EPG_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SQLITE_EPG_STORE (obj);
	{
		sqlite3* _tmp0;
		sqlite3_stmt* _tmp3;
		gint _tmp2;
		sqlite3_stmt* _tmp1;
		sqlite3_stmt* _tmp6;
		gint _tmp5;
		sqlite3_stmt* _tmp4;
		sqlite3_stmt* _tmp9;
		gint _tmp8;
		sqlite3_stmt* _tmp7;
		sqlite3_stmt* _tmp12;
		gint _tmp11;
		sqlite3_stmt* _tmp10;
		sqlite3_stmt* _tmp15;
		gint _tmp14;
		sqlite3_stmt* _tmp13;
		sqlite3_stmt* _tmp18;
		gint _tmp17;
		sqlite3_stmt* _tmp16;
		_tmp0 = NULL;
		self->priv->db = (_tmp0 = dvb_sqlite_epg_store_get_db_handler (), (self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL)), _tmp0);
		_tmp3 = NULL;
		_tmp1 = NULL;
		_tmp2 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_TO_JULIAN_SQL, -1, &_tmp1, NULL);
		self->priv->to_julian_statement = (_tmp3 = _tmp1, (self->priv->to_julian_statement == NULL) ? NULL : (self->priv->to_julian_statement = (sqlite3_finalize (self->priv->to_julian_statement), NULL)), _tmp3);
		_tmp2;
		_tmp6 = NULL;
		_tmp4 = NULL;
		_tmp5 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_INSERT_EVENT_SQL, -1, &_tmp4, NULL);
		self->priv->insert_event_statement = (_tmp6 = _tmp4, (self->priv->insert_event_statement == NULL) ? NULL : (self->priv->insert_event_statement = (sqlite3_finalize (self->priv->insert_event_statement), NULL)), _tmp6);
		_tmp5;
		_tmp9 = NULL;
		_tmp7 = NULL;
		_tmp8 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_UPDATE_EVENT_SQL, -1, &_tmp7, NULL);
		self->priv->update_event_statement = (_tmp9 = _tmp7, (self->priv->update_event_statement == NULL) ? NULL : (self->priv->update_event_statement = (sqlite3_finalize (self->priv->update_event_statement), NULL)), _tmp9);
		_tmp8;
		_tmp12 = NULL;
		_tmp10 = NULL;
		_tmp11 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_DELETE_EVENT_STATEMENT, -1, &_tmp10, NULL);
		self->priv->delete_event_statement = (_tmp12 = _tmp10, (self->priv->delete_event_statement == NULL) ? NULL : (self->priv->delete_event_statement = (sqlite3_finalize (self->priv->delete_event_statement), NULL)), _tmp12);
		_tmp11;
		_tmp15 = NULL;
		_tmp13 = NULL;
		_tmp14 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_HAS_EVENT_STATEMENT, -1, &_tmp13, NULL);
		self->priv->has_event_statement = (_tmp15 = _tmp13, (self->priv->has_event_statement == NULL) ? NULL : (self->priv->has_event_statement = (sqlite3_finalize (self->priv->has_event_statement), NULL)), _tmp15);
		_tmp14;
		_tmp18 = NULL;
		_tmp16 = NULL;
		_tmp17 = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_SELECT_EVENT_SQL, -1, &_tmp16, NULL);
		self->priv->select_event_statement = (_tmp18 = _tmp16, (self->priv->select_event_statement == NULL) ? NULL : (self->priv->select_event_statement = (sqlite3_finalize (self->priv->select_event_statement), NULL)), _tmp18);
		_tmp17;
	}
	return obj;
}


static void dvb_sqlite_epg_store_class_init (DVBSqliteEPGStoreClass * klass) {
	dvb_sqlite_epg_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBSqliteEPGStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_sqlite_epg_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_sqlite_epg_store_finalize;
}


static void dvb_sqlite_epg_store_dvb_epg_store_interface_init (DVBEPGStoreIface * iface) {
	dvb_sqlite_epg_store_dvb_epg_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->add_or_update_event = dvb_sqlite_epg_store_real_add_or_update_event;
	iface->get_event = dvb_sqlite_epg_store_real_get_event;
	iface->remove_event = dvb_sqlite_epg_store_real_remove_event;
	iface->contains_event = dvb_sqlite_epg_store_real_contains_event;
	iface->get_events = dvb_sqlite_epg_store_real_get_events;
}


static void dvb_sqlite_epg_store_instance_init (DVBSqliteEPGStore * self) {
	self->priv = DVB_SQLITE_EPG_STORE_GET_PRIVATE (self);
}


static void dvb_sqlite_epg_store_finalize (GObject* obj) {
	DVBSqliteEPGStore * self;
	self = DVB_SQLITE_EPG_STORE (obj);
	(self->priv->to_julian_statement == NULL) ? NULL : (self->priv->to_julian_statement = (sqlite3_finalize (self->priv->to_julian_statement), NULL));
	(self->priv->insert_event_statement == NULL) ? NULL : (self->priv->insert_event_statement = (sqlite3_finalize (self->priv->insert_event_statement), NULL));
	(self->priv->update_event_statement == NULL) ? NULL : (self->priv->update_event_statement = (sqlite3_finalize (self->priv->update_event_statement), NULL));
	(self->priv->delete_event_statement == NULL) ? NULL : (self->priv->delete_event_statement = (sqlite3_finalize (self->priv->delete_event_statement), NULL));
	(self->priv->has_event_statement == NULL) ? NULL : (self->priv->has_event_statement = (sqlite3_finalize (self->priv->has_event_statement), NULL));
	(self->priv->select_event_statement == NULL) ? NULL : (self->priv->select_event_statement = (sqlite3_finalize (self->priv->select_event_statement), NULL));
	(self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL));
	G_OBJECT_CLASS (dvb_sqlite_epg_store_parent_class)->finalize (obj);
}


GType dvb_sqlite_epg_store_get_type (void) {
	static GType dvb_sqlite_epg_store_type_id = 0;
	if (dvb_sqlite_epg_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBSqliteEPGStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_sqlite_epg_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBSqliteEPGStore), 0, (GInstanceInitFunc) dvb_sqlite_epg_store_instance_init, NULL };
		static const GInterfaceInfo dvb_epg_store_info = { (GInterfaceInitFunc) dvb_sqlite_epg_store_dvb_epg_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_sqlite_epg_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBSqliteEPGStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_sqlite_epg_store_type_id, DVB_TYPE_EPG_STORE, &dvb_epg_store_info);
	}
	return dvb_sqlite_epg_store_type_id;
}




