/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <time.h>
#include <gio/gio.h>
#include <gst/gst.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


#define DVB_TYPE_ID_BUS_RECORDER (dvb_id_bus_recorder_get_type ())
#define DVB_ID_BUS_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_RECORDER, DVBIDBusRecorder))
#define DVB_IS_ID_BUS_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_RECORDER))
#define DVB_ID_BUS_RECORDER_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_RECORDER, DVBIDBusRecorderIface))

typedef struct _DVBIDBusRecorder DVBIDBusRecorder;
typedef struct _DVBIDBusRecorderIface DVBIDBusRecorderIface;

#define DVB_TYPE_OVERLAP_TYPE (dvb_overlap_type_get_type ())

#define DVB_TYPE_RECORDER (dvb_recorder_get_type ())
#define DVB_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDER, DVBRecorder))
#define DVB_RECORDER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDER, DVBRecorderClass))
#define DVB_IS_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDER))
#define DVB_IS_RECORDER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDER))
#define DVB_RECORDER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDER, DVBRecorderClass))

typedef struct _DVBRecorder DVBRecorder;
typedef struct _DVBRecorderClass DVBRecorderClass;
typedef struct _DVBRecorderPrivate DVBRecorderPrivate;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

#define DVB_TYPE_TIMER (dvb_timer_get_type ())
#define DVB_TIMER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMER, DVBTimer))
#define DVB_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TIMER, DVBTimerClass))
#define DVB_IS_TIMER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMER))
#define DVB_IS_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TIMER))
#define DVB_TIMER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TIMER, DVBTimerClass))

typedef struct _DVBTimer DVBTimer;
typedef struct _DVBTimerClass DVBTimerClass;

#define DVB_TYPE_RECORDING (dvb_recording_get_type ())
#define DVB_RECORDING(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDING, DVBRecording))
#define DVB_RECORDING_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDING, DVBRecordingClass))
#define DVB_IS_RECORDING(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDING))
#define DVB_IS_RECORDING_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDING))
#define DVB_RECORDING_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDING, DVBRecordingClass))

typedef struct _DVBRecording DVBRecording;
typedef struct _DVBRecordingClass DVBRecordingClass;

#define DVB_TYPE_SETTINGS (dvb_settings_get_type ())
#define DVB_SETTINGS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SETTINGS, DVBSettings))
#define DVB_SETTINGS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SETTINGS, DVBSettingsClass))
#define DVB_IS_SETTINGS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SETTINGS))
#define DVB_IS_SETTINGS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SETTINGS))
#define DVB_SETTINGS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SETTINGS, DVBSettingsClass))

typedef struct _DVBSettings DVBSettings;
typedef struct _DVBSettingsClass DVBSettingsClass;

#define DVB_TYPE_TIMERS_STORE (dvb_timers_store_get_type ())
#define DVB_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStore))
#define DVB_IS_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMERS_STORE))
#define DVB_TIMERS_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStoreIface))

typedef struct _DVBTimersStore DVBTimersStore;
typedef struct _DVBTimersStoreIface DVBTimersStoreIface;

#define DVB_TYPE_CHANGE_TYPE (dvb_change_type_get_type ())

#define DVB_TYPE_EPG_STORE (dvb_epg_store_get_type ())
#define DVB_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_STORE, DVBEPGStore))
#define DVB_IS_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_STORE))
#define DVB_EPG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_EPG_STORE, DVBEPGStoreIface))

typedef struct _DVBEPGStore DVBEPGStore;
typedef struct _DVBEPGStoreIface DVBEPGStoreIface;

#define DVB_TYPE_EVENT (dvb_event_get_type ())
#define DVB_EVENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT, DVBEvent))
#define DVB_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT, DVBEventClass))
#define DVB_IS_EVENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT))
#define DVB_IS_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT))
#define DVB_EVENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT, DVBEventClass))

typedef struct _DVBEvent DVBEvent;
typedef struct _DVBEventClass DVBEventClass;
typedef struct _DVBEventPrivate DVBEventPrivate;

#define DVB_EVENT_TYPE_AUDIO_COMPONENT (dvb_event_audio_component_get_type ())
#define DVB_EVENT_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponent))
#define DVB_EVENT_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))
#define DVB_EVENT_IS_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_IS_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_AUDIO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))

typedef struct _DVBEventAudioComponent DVBEventAudioComponent;
typedef struct _DVBEventAudioComponentClass DVBEventAudioComponentClass;

#define DVB_EVENT_TYPE_VIDEO_COMPONENT (dvb_event_video_component_get_type ())
#define DVB_EVENT_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponent))
#define DVB_EVENT_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))
#define DVB_EVENT_IS_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_IS_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_VIDEO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))

typedef struct _DVBEventVideoComponent DVBEventVideoComponent;
typedef struct _DVBEventVideoComponentClass DVBEventVideoComponentClass;

#define DVB_EVENT_TYPE_TELETEXT_COMPONENT (dvb_event_teletext_component_get_type ())
#define DVB_EVENT_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponent))
#define DVB_EVENT_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))
#define DVB_EVENT_IS_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_IS_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_TELETEXT_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))

typedef struct _DVBEventTeletextComponent DVBEventTeletextComponent;
typedef struct _DVBEventTeletextComponentClass DVBEventTeletextComponentClass;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_RECORDINGS_STORE (dvb_recordings_store_get_type ())
#define DVB_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStore))
#define DVB_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))
#define DVB_IS_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDINGS_STORE))
#define DVB_IS_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDINGS_STORE))
#define DVB_RECORDINGS_STORE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))

typedef struct _DVBRecordingsStore DVBRecordingsStore;
typedef struct _DVBRecordingsStoreClass DVBRecordingsStoreClass;

#define DVB_TYPE_CHANNEL_FACTORY (dvb_channel_factory_get_type ())
#define DVB_CHANNEL_FACTORY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactory))
#define DVB_CHANNEL_FACTORY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactoryClass))
#define DVB_IS_CHANNEL_FACTORY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_FACTORY))
#define DVB_IS_CHANNEL_FACTORY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_FACTORY))
#define DVB_CHANNEL_FACTORY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_FACTORY, DVBChannelFactoryClass))

typedef struct _DVBChannelFactory DVBChannelFactory;
typedef struct _DVBChannelFactoryClass DVBChannelFactoryClass;

#define DVB_TYPE_PLAYER_THREAD (dvb_player_thread_get_type ())
#define DVB_PLAYER_THREAD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_PLAYER_THREAD, DVBPlayerThread))
#define DVB_PLAYER_THREAD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_PLAYER_THREAD, DVBPlayerThreadClass))
#define DVB_IS_PLAYER_THREAD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_PLAYER_THREAD))
#define DVB_IS_PLAYER_THREAD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_PLAYER_THREAD))
#define DVB_PLAYER_THREAD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_PLAYER_THREAD, DVBPlayerThreadClass))

typedef struct _DVBPlayerThread DVBPlayerThread;
typedef struct _DVBPlayerThreadClass DVBPlayerThreadClass;

#define DVB_TYPE_SCHEDULE (dvb_schedule_get_type ())
#define DVB_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCHEDULE, DVBSchedule))
#define DVB_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCHEDULE, DVBScheduleClass))
#define DVB_IS_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCHEDULE))
#define DVB_IS_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCHEDULE))
#define DVB_SCHEDULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCHEDULE, DVBScheduleClass))

typedef struct _DVBSchedule DVBSchedule;
typedef struct _DVBScheduleClass DVBScheduleClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

typedef enum  {
	DVB_OVERLAP_TYPE_UNKNOWN,
	DVB_OVERLAP_TYPE_NONE,
	DVB_OVERLAP_TYPE_PARTIAL,
	DVB_OVERLAP_TYPE_COMPLETE
} DVBOverlapType;

struct _DVBIDBusRecorderIface {
	GTypeInterface parent_iface;
	guint32 (*AddTimer) (DVBIDBusRecorder* self, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
	guint32 (*AddTimerWithMargin) (DVBIDBusRecorder* self, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
	guint32 (*AddTimerForEPGEvent) (DVBIDBusRecorder* self, guint event_id, guint channel_sid);
	gboolean (*DeleteTimer) (DVBIDBusRecorder* self, guint32 timer_id);
	guint32* (*GetTimers) (DVBIDBusRecorder* self, int* result_length1);
	guint32* (*GetStartTime) (DVBIDBusRecorder* self, guint32 timer_id, int* result_length1);
	guint* (*GetEndTime) (DVBIDBusRecorder* self, guint32 timer_id, int* result_length1);
	guint (*GetDuration) (DVBIDBusRecorder* self, guint32 timer_id);
	char* (*GetChannelName) (DVBIDBusRecorder* self, guint32 timer_id);
	guint32* (*GetActiveTimers) (DVBIDBusRecorder* self, int* result_length1);
	gboolean (*IsTimerActive) (DVBIDBusRecorder* self, guint32 timer_id);
	gboolean (*HasTimer) (DVBIDBusRecorder* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
	DVBOverlapType (*HasTimerForEvent) (DVBIDBusRecorder* self, guint event_id, guint channel_sid);
};

/**
     * This class is responsible for managing upcoming recordings and
     * already recorded items for a single group of devices
     */
struct _DVBRecorder {
	GObject parent_instance;
	DVBRecorderPrivate * priv;
};

struct _DVBRecorderClass {
	GObjectClass parent_class;
};

struct _DVBRecorderPrivate {
	DVBDeviceGroup* _DeviceGroup;
	GeeSet* active_timers;
	gboolean have_check_timers_timeout;
	GeeHashMap* timers;
	GStaticRecMutex __lock_timers;
	GeeMap* recordings;
	GStaticRecMutex __lock_recordings;
};

struct _DVBTimersStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_timers_of_device_group) (DVBTimersStore* self, DVBDeviceGroup* dev);
	gboolean (*add_timer_to_device_group) (DVBTimersStore* self, DVBTimer* timer, DVBDeviceGroup* dev);
	gboolean (*remove_timer_from_device_group) (DVBTimersStore* self, guint timer_id, DVBDeviceGroup* dev);
};

typedef enum  {
	DVB_CHANGE_TYPE_ADDED = 0,
	DVB_CHANGE_TYPE_DELETED = 1,
	DVB_CHANGE_TYPE_UPDATED = 2
} DVBChangeType;

struct _DVBEPGStoreIface {
	GTypeInterface parent_iface;
	gboolean (*add_or_update_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	DVBEvent* (*get_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*remove_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*contains_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	GeeList* (*get_events) (DVBEPGStore* self, guint channel_sid, guint group_id);
	gboolean (*remove_events_of_group) (DVBEPGStore* self, guint group_id);
};

/**
     * Represents an EPG event (i.e. a show with all its information)
     */
struct _DVBEvent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventPrivate * priv;
	guint id;
	guint year;
	guint month;
	guint hour;
	guint day;
	guint minute;
	guint second;
	guint duration;
	guint running_status;
	gboolean free_ca_mode;
	char* name;
	char* description;
	char* extended_description;
	GSList* audio_components;
	GSList* video_components;
	GSList* teletext_components;
};

struct _DVBEventClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEvent *self);
};

typedef void (*DVBForcedStopNotify) (DVBChannel* channel);
struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


static gpointer dvb_recorder_parent_class = NULL;
static DVBIDBusRecorderIface* dvb_recorder_dvb_id_bus_recorder_parent_iface = NULL;
static GeeIterableIface* dvb_recorder_gee_iterable_parent_iface = NULL;

GType dvb_overlap_type_get_type (void);
GType dvb_id_bus_recorder_get_type (void);
GType dvb_recorder_get_type (void);
GType dvb_device_group_get_type (void);
GType dvb_timer_get_type (void);
GType dvb_recording_get_type (void);
#define DVB_RECORDER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_RECORDER, DVBRecorderPrivate))
enum  {
	DVB_RECORDER_DUMMY_PROPERTY,
	DVB_RECORDER_DEVICE_GROUP,
	DVB_RECORDER_COUNT
};
#define DVB_RECORDER_CHECK_TIMERS_INTERVAL 5
static void dvb_recorder_set_DeviceGroup (DVBRecorder* self, DVBDeviceGroup* value);
DVBRecorder* dvb_recorder_new (DVBDeviceGroup* dev);
DVBRecorder* dvb_recorder_construct (GType object_type, DVBDeviceGroup* dev);
static GType dvb_recorder_real_get_element_type (GeeIterable* base);
static GeeIterator* dvb_recorder_real_iterator (GeeIterable* base);
DVBTimer* dvb_recorder_create_timer (DVBRecorder* self, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
guint32 dvb_recorder_add_timer (DVBRecorder* self, DVBTimer* new_timer);
static guint32 dvb_recorder_real_AddTimer (DVBIDBusRecorder* base, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
GType dvb_settings_get_type (void);
DVBSettings* dvb_factory_get_settings (void);
gint dvb_settings_get_integer (DVBSettings* self, const char* group_name, const char* key, GError** error);
#define DVB_SETTINGS_TIMERS_SECTION "timers"
#define DVB_SETTINGS_MARGIN_START "margin_start"
#define DVB_SETTINGS_MARGIN_END "margin_end"
guint dvb_timer_get_Duration (DVBTimer* self);
void dvb_timer_set_Duration (DVBTimer* self, guint value);
void dvb_timer_add_to_start_time (DVBTimer* self, gint minutes);
static guint32 dvb_recorder_real_AddTimerWithMargin (DVBIDBusRecorder* base, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
gboolean dvb_timer_has_expired (DVBTimer* self);
gboolean dvb_timer_conflicts_with (DVBTimer* self, DVBTimer* t2);
DVBDeviceGroup* dvb_recorder_get_DeviceGroup (DVBRecorder* self);
gint dvb_device_group_get_size (DVBDeviceGroup* self);
char* dvb_timer_to_string (DVBTimer* self);
guint32 dvb_timer_get_Id (DVBTimer* self);
GType dvb_timers_store_get_type (void);
DVBTimersStore* dvb_factory_get_timers_store (void);
gboolean dvb_timers_store_add_timer_to_device_group (DVBTimersStore* self, DVBTimer* timer, DVBDeviceGroup* dev);
GType dvb_change_type_get_type (void);
static gboolean dvb_recorder_check_timers (DVBRecorder* self);
static gboolean _dvb_recorder_check_timers_gsource_func (gpointer self);
gpointer dvb_event_ref (gpointer instance);
void dvb_event_unref (gpointer instance);
GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_value_set_event (GValue* value, gpointer v_object);
gpointer dvb_value_get_event (const GValue* value);
GType dvb_event_get_type (void);
GType dvb_epg_store_get_type (void);
DVBEPGStore* dvb_factory_get_epg_store (void);
DVBEvent* dvb_epg_store_get_event (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
guint dvb_device_group_get_Id (DVBDeviceGroup* self);
struct tm dvb_event_get_local_start_time (DVBEvent* self);
guint32 dvb_id_bus_recorder_AddTimerWithMargin (DVBIDBusRecorder* self, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration);
gpointer dvb_event_audio_component_ref (gpointer instance);
void dvb_event_audio_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_audio_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_audio_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_audio_component (const GValue* value);
GType dvb_event_audio_component_get_type (void);
gpointer dvb_event_video_component_ref (gpointer instance);
void dvb_event_video_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_video_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_video_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_video_component (const GValue* value);
GType dvb_event_video_component_get_type (void);
gpointer dvb_event_teletext_component_ref (gpointer instance);
void dvb_event_teletext_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_teletext_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_teletext_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_teletext_component (const GValue* value);
GType dvb_event_teletext_component_get_type (void);
static guint32 dvb_recorder_real_AddTimerForEPGEvent (DVBIDBusRecorder* base, guint event_id, guint channel_sid);
gboolean dvb_id_bus_recorder_IsTimerActive (DVBIDBusRecorder* self, guint32 timer_id);
void dvb_recorder_stop_recording (DVBRecorder* self, DVBTimer* timer);
gboolean dvb_timers_store_remove_timer_from_device_group (DVBTimersStore* self, guint timer_id, DVBDeviceGroup* dev);
static gboolean dvb_recorder_real_DeleteTimer (DVBIDBusRecorder* base, guint32 timer_id);
static guint32* dvb_recorder_real_GetTimers (DVBIDBusRecorder* base, int* result_length1);
guint* dvb_timer_get_start_time (DVBTimer* self, int* result_length1);
static guint32* dvb_recorder_real_GetStartTime (DVBIDBusRecorder* base, guint32 timer_id, int* result_length1);
guint* dvb_timer_get_end_time (DVBTimer* self, int* result_length1);
static guint* dvb_recorder_real_GetEndTime (DVBIDBusRecorder* base, guint32 timer_id, int* result_length1);
static guint dvb_recorder_real_GetDuration (DVBIDBusRecorder* base, guint32 timer_id);
GType dvb_channel_get_type (void);
DVBChannel* dvb_timer_get_Channel (DVBTimer* self);
const char* dvb_channel_get_Name (DVBChannel* self);
static char* dvb_recorder_real_GetChannelName (DVBIDBusRecorder* base, guint32 timer_id);
static guint32* dvb_recorder_real_GetActiveTimers (DVBIDBusRecorder* base, int* result_length1);
static gboolean dvb_recorder_real_IsTimerActive (DVBIDBusRecorder* base, guint32 timer_id);
DVBOverlapType dvb_timer_get_overlap_local (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
static gboolean dvb_recorder_real_HasTimer (DVBIDBusRecorder* base, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
guint dvb_channel_get_Sid (DVBChannel* self);
DVBOverlapType dvb_timer_get_overlap_utc (DVBTimer* self, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration);
static DVBOverlapType dvb_recorder_real_HasTimerForEvent (DVBIDBusRecorder* base, guint event_id, guint channel_sid);
void dvb_recorder_stop (DVBRecorder* self);
GType dvb_channel_list_get_type (void);
DVBChannelList* dvb_device_group_get_Channels (DVBDeviceGroup* self);
gboolean dvb_channel_list_contains (DVBChannelList* self, guint sid);
GType dvb_recordings_store_get_type (void);
DVBRecordingsStore* dvb_recordings_store_get_instance (void);
guint32 dvb_recordings_store_get_next_id (DVBRecordingsStore* self);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
DVBTimer* dvb_timer_new (guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
DVBTimer* dvb_timer_construct (GType object_type, guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
static GFile* dvb_recorder_create_recording_dirs (DVBRecorder* self, DVBChannel* channel, guint* start, int start_length1);
void dvb_timer_set_sink (DVBTimer* self, GstElement* value);
GType dvb_channel_factory_get_type (void);
DVBChannelFactory* dvb_device_group_get_channel_factory (DVBDeviceGroup* self);
GType dvb_player_thread_get_type (void);
DVBPlayerThread* dvb_channel_factory_watch_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink_element, gboolean force, DVBForcedStopNotify notify_func);
GstElement* dvb_player_thread_get_pipeline (DVBPlayerThread* self);
static void dvb_recorder_on_eit_structure (DVBRecorder* self, DVBPlayerThread* player, const GstStructure* structure);
static void _dvb_recorder_on_eit_structure_dvb_player_thread_eit_structure (DVBPlayerThread* _sender, const GstStructure* structure, gpointer self);
DVBRecording* dvb_recording_new (void);
DVBRecording* dvb_recording_construct (GType object_type);
void dvb_recording_set_Name (DVBRecording* self, const char* value);
void dvb_recording_set_Description (DVBRecording* self, const char* value);
void dvb_recording_set_Id (DVBRecording* self, guint32 value);
void dvb_recording_set_ChannelSid (DVBRecording* self, guint value);
void dvb_recording_set_ChannelName (DVBRecording* self, const char* value);
struct tm dvb_timer_get_start_time_time (DVBTimer* self);
void dvb_recording_set_StartTime (DVBRecording* self, struct tm* value);
void dvb_recording_set_Location (DVBRecording* self, GFile* value);
guint32 dvb_recording_get_Id (DVBRecording* self);
gboolean dvb_recordings_store_add (DVBRecordingsStore* self, DVBRecording* rec, gboolean monitor);
void dvb_recorder_start_recording (DVBRecorder* self, DVBTimer* timer);
gint64 dvb_utils_difftime (const struct tm* t1, const struct tm* t2);
void dvb_recording_get_StartTime (DVBRecording* self, struct tm* value);
void dvb_recording_set_Length (DVBRecording* self, gint64 value);
const char* dvb_recording_get_ChannelName (DVBRecording* self);
gint64 dvb_recording_get_Length (DVBRecording* self);
void dvb_recording_save_to_disk (DVBRecording* self, GError** error);
gboolean dvb_channel_factory_stop_channel (DVBChannelFactory* self, DVBChannel* channel, GstElement* sink);
GstElement* dvb_timer_get_sink (DVBTimer* self);
void dvb_recordings_store_monitor_recording (DVBRecordingsStore* self, DVBRecording* rec);
char* dvb_utils_remove_nonalphanums (const char* text);
GFile* dvb_device_group_get_RecordingsDirectory (DVBDeviceGroup* self);
void dvb_utils_mkdirs (GFile* directory, GError** error);
gboolean dvb_timer_is_end_due (DVBTimer* self);
gboolean dvb_timer_is_start_due (DVBTimer* self);
gboolean dvb_id_bus_recorder_DeleteTimer (DVBIDBusRecorder* self, guint32 timer_id);
static void _g_slist_free_g_object_unref (GSList* self);
const char* dvb_recording_get_Name (DVBRecording* self);
guint dvb_recording_get_ChannelSid (DVBRecording* self);
GType dvb_schedule_get_type (void);
DVBSchedule* dvb_channel_get_Schedule (DVBChannel* self);
DVBEvent* dvb_schedule_get_running_event (DVBSchedule* self);
guint dvb_recorder_get_count (DVBRecorder* self);
void dvb_recordings_store_restore_from_dir (DVBRecordingsStore* self, GFile* recordingsbasedir);
static GObject * dvb_recorder_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void dvb_recorder_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_recorder_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_recorder_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_recorder_introspect (DVBRecorder* self, DBusConnection* connection, DBusMessage* message);
static void dvb_recorder_finalize (GObject* obj);
static void dvb_recorder_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_recorder_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_recorder_dbus_path_vtable = {_dvb_recorder_dbus_unregister, dvb_recorder_dbus_message};
static const _DBusObjectVTable _dvb_recorder_dbus_vtable = {dvb_recorder_dbus_register_object};


DVBRecorder* dvb_recorder_construct (GType object_type, DVBDeviceGroup* dev) {
	GParameter * __params;
	GParameter * __params_it;
	DVBRecorder * self;
	g_return_val_if_fail (dev != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "DeviceGroup";
	g_value_init (&__params_it->value, DVB_TYPE_DEVICE_GROUP);
	g_value_set_object (&__params_it->value, dev);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBRecorder* dvb_recorder_new (DVBDeviceGroup* dev) {
	return dvb_recorder_construct (DVB_TYPE_RECORDER, dev);
}


static GType dvb_recorder_real_get_element_type (GeeIterable* base) {
	DVBRecorder * self;
	GType result;
	self = (DVBRecorder*) base;
	result = DVB_TYPE_TIMER;
	return result;
}


static GeeIterator* dvb_recorder_real_iterator (GeeIterable* base) {
	DVBRecorder * self;
	GeeIterator* result;
	GeeCollection* _tmp0_;
	GeeIterator* _tmp1_;
	self = (DVBRecorder*) base;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_values ((GeeMap*) self->priv->timers))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	return result;
}


/**
         * @channel: Channel number
         * @start_year: The year when the recording should start
         * @start_month: The month when recording should start
         * @start_day: The day when recording should start
         * @start_hour: The hour when recording should start
         * @start_minute: The minute when recording should start
         * @duration: How long the channel should be recorded (in minutes)
         * @returns: The new timer's id on success, or 0 if timer couldn't
         * be created
         * 
         * Add a new timer
         */
static guint32 dvb_recorder_real_AddTimer (DVBIDBusRecorder* base, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration) {
	DVBRecorder * self;
	guint32 result;
	DVBTimer* new_timer;
	self = (DVBRecorder*) base;
	new_timer = dvb_recorder_create_timer (self, channel, start_year, start_month, start_day, start_hour, start_minute, duration);
	if (new_timer == NULL) {
		result = (guint32) 0;
		(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
		return result;
	} else {
		result = dvb_recorder_add_timer (self, new_timer);
		(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
		return result;
	}
	(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
}


/**
         * Works the same way as AddTimer() but adds a margin before and
         * after the timer.
         *
         * If the timer with added margins conflicts with a scheduled
         * recording the margins are removed and adding the timer will
         * be tried again.
         */
static guint32 dvb_recorder_real_AddTimerWithMargin (DVBIDBusRecorder* base, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration) {
	DVBRecorder * self;
	guint32 result;
	GError * _inner_error_;
	DVBTimer* new_timer;
	DVBSettings* _tmp0_;
	DVBSettings* settings;
	gint start_margin;
	guint end_margin;
	guint32 tid;
	self = (DVBRecorder*) base;
	_inner_error_ = NULL;
	new_timer = dvb_recorder_create_timer (self, channel, start_year, start_month, start_day, start_hour, start_minute, duration);
	if (new_timer == NULL) {
		result = (guint32) 0;
		(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
		return result;
	}
	_tmp0_ = NULL;
	settings = (_tmp0_ = dvb_factory_get_settings (), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	start_margin = 0;
	end_margin = (guint) 0;
	{
		gint _tmp1_;
		gint _tmp2_;
		_tmp1_ = dvb_settings_get_integer (settings, DVB_SETTINGS_TIMERS_SECTION, DVB_SETTINGS_MARGIN_START, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_KEY_FILE_ERROR) {
				goto __catch9_g_key_file_error;
			}
			goto __finally9;
		}
		start_margin = (-1) * _tmp1_;
		_tmp2_ = dvb_settings_get_integer (settings, DVB_SETTINGS_TIMERS_SECTION, DVB_SETTINGS_MARGIN_END, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_KEY_FILE_ERROR) {
				goto __catch9_g_key_file_error;
			}
			goto __finally9;
		}
		end_margin = 2 * ((guint) _tmp2_);
		dvb_timer_set_Duration (new_timer, dvb_timer_get_Duration (new_timer) + end_margin);
		dvb_timer_add_to_start_time (new_timer, start_margin);
	}
	goto __finally9;
	__catch9_g_key_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Recorder.vala:123: Could not retrieve start/end margins: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally9:
	if (_inner_error_ != NULL) {
		(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
		(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return 0U;
	}
	tid = dvb_recorder_add_timer (self, new_timer);
	if (tid == 0) {
		/* The timer conflicts, see what happens when we remove margins*/
		dvb_timer_set_Duration (new_timer, dvb_timer_get_Duration (new_timer) - end_margin);
		dvb_timer_add_to_start_time (new_timer, (-1) * start_margin);
		tid = dvb_recorder_add_timer (self, new_timer);
	}
	result = tid;
	(new_timer == NULL) ? NULL : (new_timer = (g_object_unref (new_timer), NULL));
	(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
	return result;
}


static gboolean _dvb_recorder_check_timers_gsource_func (gpointer self) {
	return dvb_recorder_check_timers (self);
}


guint32 dvb_recorder_add_timer (DVBRecorder* self, DVBTimer* new_timer) {
	guint32 result;
	guint32 timer_id;
	g_return_val_if_fail (self != NULL, 0U);
	g_return_val_if_fail (new_timer != NULL, 0U);
	if (dvb_timer_has_expired (new_timer)) {
		result = (guint32) 0;
		return result;
	}
	timer_id = (guint32) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		gboolean has_conflict;
		gint conflict_count;
		has_conflict = FALSE;
		conflict_count = 0;
		/* Check for conflicts*/
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _key_it;
			/* Check for conflicts*/
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_key_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->priv->timers))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			/* Check for conflicts*/
			while (TRUE) {
				guint32 key;
				DVBTimer* _tmp2_;
				gboolean _tmp3_;
				/* Check for conflicts*/
				if (!gee_iterator_next (_key_it)) {
					/* Check for conflicts*/
					break;
				}
				/* Check for conflicts*/
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				_tmp2_ = NULL;
				if ((_tmp3_ = dvb_timer_conflicts_with (_tmp2_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key)), new_timer), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_)) {
					conflict_count++;
					if (conflict_count >= dvb_device_group_get_size (self->priv->_DeviceGroup)) {
						char* _tmp5_;
						DVBTimer* _tmp4_;
						_tmp5_ = NULL;
						_tmp4_ = NULL;
						g_debug ("Recorder.vala:151: Timer is conflicting with another timer: %s", _tmp5_ = dvb_timer_to_string (_tmp4_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key))));
						_tmp5_ = (g_free (_tmp5_), NULL);
						(_tmp4_ == NULL) ? NULL : (_tmp4_ = (g_object_unref (_tmp4_), NULL));
						has_conflict = TRUE;
						break;
					}
				}
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
		if (!has_conflict) {
			gboolean _tmp6_;
			gee_map_set ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (dvb_timer_get_Id (new_timer)), new_timer);
			dvb_timers_store_add_timer_to_device_group (dvb_factory_get_timers_store (), new_timer, self->priv->_DeviceGroup);
			g_signal_emit_by_name ((DVBIDBusRecorder*) self, "changed", dvb_timer_get_Id (new_timer), (guint) DVB_CHANGE_TYPE_ADDED);
			_tmp6_ = FALSE;
			if (gee_map_get_size ((GeeMap*) self->priv->timers) == 1) {
				_tmp6_ = !self->priv->have_check_timers_timeout;
			} else {
				_tmp6_ = FALSE;
			}
			if (_tmp6_) {
				g_debug ("Recorder.vala:166: Creating new check timers");
				g_timeout_add_seconds ((guint) DVB_RECORDER_CHECK_TIMERS_INTERVAL, _dvb_recorder_check_timers_gsource_func, self);
				self->priv->have_check_timers_timeout = TRUE;
			}
			timer_id = dvb_timer_get_Id (new_timer);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = timer_id;
	return result;
}


/**
         * @event_id: id of the EPG event
         * @channel_sid: SID of channel
         * @returns: The new timer's id on success, or 0 if timer couldn't
         * be created
         */
static guint32 dvb_recorder_real_AddTimerForEPGEvent (DVBIDBusRecorder* base, guint event_id, guint channel_sid) {
	DVBRecorder * self;
	guint32 result;
	DVBEPGStore* epgstore;
	DVBEvent* event;
	struct tm start;
	self = (DVBRecorder*) base;
	epgstore = dvb_factory_get_epg_store ();
	event = dvb_epg_store_get_event (epgstore, event_id, channel_sid, dvb_device_group_get_Id (self->priv->_DeviceGroup));
	if (event == NULL) {
		g_debug ("Recorder.vala:190: Could not find event with id %u", event_id);
		result = (guint32) 0;
		(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		return result;
	}
	start = dvb_event_get_local_start_time (event);
	result = dvb_id_bus_recorder_AddTimerWithMargin ((DVBIDBusRecorder*) self, channel_sid, start.tm_year + 1900, start.tm_mon + 1, start.tm_mday, start.tm_hour, start.tm_min, event->duration / 60);
	(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
	return result;
}


/**
         * @timer_id: The id of the timer you want to delete
         * @returns: TRUE on success
         *
         * Delete timer. If the id belongs to the currently
         * active timer recording is aborted.
         */
static gboolean dvb_recorder_real_DeleteTimer (DVBIDBusRecorder* base, guint32 timer_id) {
	DVBRecorder * self;
	gboolean result;
	gboolean val;
	self = (DVBRecorder*) base;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		if (gee_map_contains ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id))) {
			if (dvb_id_bus_recorder_IsTimerActive ((DVBIDBusRecorder*) self, timer_id)) {
				DVBTimer* timer;
				/* Abort recording*/
				timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
				dvb_recorder_stop_recording (self, timer);
				(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
			}
			gee_map_remove ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
			dvb_timers_store_remove_timer_from_device_group (dvb_factory_get_timers_store (), (guint) timer_id, self->priv->_DeviceGroup);
			g_signal_emit_by_name ((DVBIDBusRecorder*) self, "changed", timer_id, (guint) DVB_CHANGE_TYPE_DELETED);
			val = TRUE;
		} else {
			val = FALSE;
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = val;
	return result;
}


/**
         * dvb_recorder_GetTimers
         * @returns: A list of all timer ids
         */
static guint32* dvb_recorder_real_GetTimers (DVBIDBusRecorder* base, int* result_length1) {
	DVBRecorder * self;
	guint32* result;
	gint timer_arr_size;
	gint timer_arr_length1;
	guint32* timer_arr;
	guint32* _tmp5_;
	self = (DVBRecorder*) base;
	timer_arr = (timer_arr_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		guint32* _tmp1_;
		gint _tmp0_;
		gint i;
		_tmp1_ = NULL;
		timer_arr = (_tmp1_ = g_new0 (guint32, _tmp0_ = gee_map_get_size ((GeeMap*) self->priv->timers)), timer_arr = (g_free (timer_arr), NULL), timer_arr_length1 = _tmp0_, timer_arr_size = timer_arr_length1, _tmp1_);
		i = 0;
		{
			GeeSet* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _key_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_key_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = gee_map_get_keys ((GeeMap*) self->priv->timers))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				guint32 key;
				DVBTimer* _tmp4_;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				_tmp4_ = NULL;
				timer_arr[i] = dvb_timer_get_Id (_tmp4_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key)));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (g_object_unref (_tmp4_), NULL));
				i++;
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	_tmp5_ = NULL;
	result = (_tmp5_ = timer_arr, *result_length1 = timer_arr_length1, _tmp5_);
	return result;
	timer_arr = (g_free (timer_arr), NULL);
}


/**
         * @timer_id: Timer's id
         * @returns: An array of length 5, where index 0 = year, 1 = month,
         * 2 = day, 3 = hour and 4 = minute.
         */
static guint32* dvb_recorder_real_GetStartTime (DVBIDBusRecorder* base, guint32 timer_id, int* result_length1) {
	DVBRecorder * self;
	guint32* result;
	gint val_size;
	gint val_length1;
	guint32* val;
	guint32* _tmp5_;
	self = (DVBRecorder*) base;
	val = (val_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		if (gee_map_contains ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id))) {
			guint32* _tmp2_;
			gint _tmp1_;
			DVBTimer* _tmp0_;
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp2_ = dvb_timer_get_start_time (_tmp0_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id)), &_tmp1_), val = (g_free (val), NULL), val_length1 = _tmp1_, val_size = val_length1, _tmp2_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		} else {
			guint32* _tmp4_;
			guint* _tmp3_;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			val = (_tmp4_ = (_tmp3_ = g_new0 (guint, 0), _tmp3_), val = (g_free (val), NULL), val_length1 = 0, val_size = val_length1, _tmp4_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	_tmp5_ = NULL;
	result = (_tmp5_ = val, *result_length1 = val_length1, _tmp5_);
	return result;
	val = (g_free (val), NULL);
}


/**
         * @timer_id: Timer's id
         * @returns: Same as dvb_recorder_GetStartTime()
         */
static guint* dvb_recorder_real_GetEndTime (DVBIDBusRecorder* base, guint32 timer_id, int* result_length1) {
	DVBRecorder * self;
	guint* result;
	gint val_size;
	gint val_length1;
	guint* val;
	guint* _tmp5_;
	self = (DVBRecorder*) base;
	val = (val_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		if (gee_map_contains ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id))) {
			guint* _tmp2_;
			gint _tmp1_;
			DVBTimer* _tmp0_;
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp2_ = dvb_timer_get_end_time (_tmp0_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id)), &_tmp1_), val = (g_free (val), NULL), val_length1 = _tmp1_, val_size = val_length1, _tmp2_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		} else {
			guint* _tmp4_;
			guint* _tmp3_;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			val = (_tmp4_ = (_tmp3_ = g_new0 (guint, 0), _tmp3_), val = (g_free (val), NULL), val_length1 = 0, val_size = val_length1, _tmp4_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	_tmp5_ = NULL;
	result = (_tmp5_ = val, *result_length1 = val_length1, _tmp5_);
	return result;
	val = (g_free (val), NULL);
}


/**
         * @timer_id: Timer's id
         * @returns: Duration in seconds or 0 if there's no timer with
         * the given id
         */
static guint dvb_recorder_real_GetDuration (DVBIDBusRecorder* base, guint32 timer_id) {
	DVBRecorder * self;
	guint result;
	guint val;
	self = (DVBRecorder*) base;
	val = (guint) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		if (gee_map_contains ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id))) {
			DVBTimer* _tmp0_;
			_tmp0_ = NULL;
			val = dvb_timer_get_Duration (_tmp0_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id)));
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = val;
	return result;
}


/**
         * @timer_id: Timer's id
         * @returns: The name of the channel the timer belongs to or an
         * empty string when a timer with the given id doesn't exist
         */
static char* dvb_recorder_real_GetChannelName (DVBIDBusRecorder* base, guint32 timer_id) {
	DVBRecorder * self;
	char* result;
	char* name;
	self = (DVBRecorder*) base;
	name = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		if (gee_map_contains ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id))) {
			DVBTimer* t;
			char* _tmp1_;
			const char* _tmp0_;
			t = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			name = (_tmp1_ = (_tmp0_ = dvb_channel_get_Name (dvb_timer_get_Channel (t)), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), name = (g_free (name), NULL), _tmp1_);
			(t == NULL) ? NULL : (t = (g_object_unref (t), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = name;
	return result;
}


/**
         * @returns: The currently active timers
         */
static guint32* dvb_recorder_real_GetActiveTimers (DVBIDBusRecorder* base, int* result_length1) {
	DVBRecorder * self;
	guint32* result;
	guint32* _tmp1_;
	gint val_size;
	gint val_length1;
	gint _tmp0_;
	guint32* val;
	gint i;
	guint32* _tmp2_;
	self = (DVBRecorder*) base;
	_tmp1_ = NULL;
	val = (_tmp1_ = g_new0 (guint32, _tmp0_ = gee_collection_get_size ((GeeCollection*) self->priv->active_timers)), val_length1 = _tmp0_, val_size = val_length1, _tmp1_);
	i = 0;
	{
		GeeIterator* _timer_id_it;
		_timer_id_it = gee_iterable_iterator ((GeeIterable*) self->priv->active_timers);
		while (TRUE) {
			guint32 timer_id;
			DVBTimer* timer;
			if (!gee_iterator_next (_timer_id_it)) {
				break;
			}
			timer_id = GPOINTER_TO_UINT (gee_iterator_get (_timer_id_it));
			timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
			val[i] = dvb_timer_get_Id (timer);
			i++;
			(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
		}
		(_timer_id_it == NULL) ? NULL : (_timer_id_it = (g_object_unref (_timer_id_it), NULL));
	}
	_tmp2_ = NULL;
	result = (_tmp2_ = val, *result_length1 = val_length1, _tmp2_);
	return result;
	val = (g_free (val), NULL);
}


/**
         * @timer_id: Timer's id
         * @returns: TRUE if timer is currently active
         */
static gboolean dvb_recorder_real_IsTimerActive (DVBIDBusRecorder* base, guint32 timer_id) {
	DVBRecorder * self;
	gboolean result;
	self = (DVBRecorder*) base;
	result = gee_collection_contains ((GeeCollection*) self->priv->active_timers, GUINT_TO_POINTER (timer_id));
	return result;
}


/**
         * @returns: TRUE if a timer is already scheduled in the given
         * period of time
         */
static gboolean dvb_recorder_real_HasTimer (DVBIDBusRecorder* base, guint start_year, guint start_month, guint start_day, guint start_hour, guint start_minute, guint duration) {
	DVBRecorder * self;
	gboolean result;
	gboolean val;
	self = (DVBRecorder*) base;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _key_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_key_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->priv->timers))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint32 key;
				DVBTimer* _tmp2_;
				DVBOverlapType _tmp3_;
				DVBOverlapType overlap;
				gboolean _tmp4_;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				_tmp2_ = NULL;
				overlap = (_tmp3_ = dvb_timer_get_overlap_local (_tmp2_ = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key)), start_year, start_month, start_day, start_hour, start_minute, duration), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
				_tmp4_ = FALSE;
				if (overlap == DVB_OVERLAP_TYPE_PARTIAL) {
					_tmp4_ = TRUE;
				} else {
					_tmp4_ = overlap == DVB_OVERLAP_TYPE_COMPLETE;
				}
				if (_tmp4_) {
					val = TRUE;
					break;
				}
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = val;
	return result;
}


static DVBOverlapType dvb_recorder_real_HasTimerForEvent (DVBIDBusRecorder* base, guint event_id, guint channel_sid) {
	DVBRecorder * self;
	DVBOverlapType result;
	DVBEPGStore* epgstore;
	DVBEvent* event;
	DVBOverlapType val;
	self = (DVBRecorder*) base;
	epgstore = dvb_factory_get_epg_store ();
	event = dvb_epg_store_get_event (epgstore, event_id, channel_sid, dvb_device_group_get_Id (self->priv->_DeviceGroup));
	if (event == NULL) {
		g_debug ("Recorder.vala:360: Could not find event with id %u", event_id);
		result = DVB_OVERLAP_TYPE_UNKNOWN;
		(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		return result;
	}
	val = DVB_OVERLAP_TYPE_NONE;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _key_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_key_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->priv->timers))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint32 key;
				DVBTimer* timer;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key));
				if (dvb_channel_get_Sid (dvb_timer_get_Channel (timer)) == channel_sid) {
					DVBOverlapType overlap;
					gboolean _tmp2_;
					overlap = dvb_timer_get_overlap_utc (timer, event->year, event->month, event->day, event->hour, event->minute, event->duration / 60);
					_tmp2_ = FALSE;
					if (overlap == DVB_OVERLAP_TYPE_PARTIAL) {
						_tmp2_ = TRUE;
					} else {
						_tmp2_ = overlap == DVB_OVERLAP_TYPE_COMPLETE;
					}
					if (_tmp2_) {
						val = overlap;
						(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
						break;
					}
				}
				(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = val;
	(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
	return result;
}


void dvb_recorder_stop (DVBRecorder* self) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		{
			GeeIterator* _timer_id_it;
			_timer_id_it = gee_iterable_iterator ((GeeIterable*) self->priv->active_timers);
			while (TRUE) {
				guint32 timer_id;
				DVBTimer* timer;
				if (!gee_iterator_next (_timer_id_it)) {
					break;
				}
				timer_id = GPOINTER_TO_UINT (gee_iterator_get (_timer_id_it));
				timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
				dvb_recorder_stop_recording (self, timer);
				(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
			}
			(_timer_id_it == NULL) ? NULL : (_timer_id_it = (g_object_unref (_timer_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
}


DVBTimer* dvb_recorder_create_timer (DVBRecorder* self, guint channel, gint start_year, gint start_month, gint start_day, gint start_hour, gint start_minute, guint duration) {
	DVBTimer* result;
	guint32 timer_id;
	DVBChannel* _tmp0_;
	DVBTimer* _tmp1_;
	DVBTimer* new_timer;
	g_return_val_if_fail (self != NULL, NULL);
	g_debug ("Recorder.vala:398: Creating new timer: channel: %u, start: %d-%d-%d %d:%d, duration: %u", channel, start_year, start_month, start_day, start_hour, start_minute, duration);
	if (!dvb_channel_list_contains (dvb_device_group_get_Channels (self->priv->_DeviceGroup), channel)) {
		g_warning ("Recorder.vala:403: No channel %u for device group %u", channel, dvb_device_group_get_Id (self->priv->_DeviceGroup));
		result = NULL;
		return result;
	}
	timer_id = dvb_recordings_store_get_next_id (dvb_recordings_store_get_instance ());
	/* TODO Get name for timer*/
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	new_timer = (_tmp1_ = dvb_timer_new (timer_id, _tmp0_ = dvb_channel_list_get_channel (dvb_device_group_get_Channels (self->priv->_DeviceGroup), channel), start_year, start_month, start_day, start_hour, start_minute, duration, NULL), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	result = new_timer;
	return result;
}


static void _dvb_recorder_on_eit_structure_dvb_player_thread_eit_structure (DVBPlayerThread* _sender, const GstStructure* structure, gpointer self) {
	dvb_recorder_on_eit_structure (self, _sender, structure);
}


/**
         * Start recording of specified timer
         */
void dvb_recorder_start_recording (DVBRecorder* self, DVBTimer* timer) {
	DVBChannel* _tmp0_;
	DVBChannel* channel;
	guint* _tmp3_;
	gint _tmp2__length1;
	guint* _tmp2_;
	gint _tmp1_;
	GFile* _tmp4_;
	GFile* location;
	GstElement* filesink;
	char* _tmp5_;
	DVBChannelFactory* _tmp6_;
	DVBChannelFactory* channel_factory;
	GstElement* _tmp7_;
	DVBPlayerThread* player;
	g_return_if_fail (self != NULL);
	g_return_if_fail (timer != NULL);
	_tmp0_ = NULL;
	channel = (_tmp0_ = dvb_timer_get_Channel (timer), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp4_ = NULL;
	location = (_tmp4_ = (_tmp3_ = (_tmp2_ = dvb_timer_get_start_time (timer, &_tmp1_), _tmp2__length1 = _tmp1_, _tmp2_), dvb_recorder_create_recording_dirs (self, channel, _tmp3_, _tmp1_)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp4_);
	if (location == NULL) {
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
		(location == NULL) ? NULL : (location = (g_object_unref (location), NULL));
		return;
	}
	filesink = gst_element_factory_make ("filesink", NULL);
	if (filesink == NULL) {
		g_critical ("Recorder.vala:432: Could not create filesink element");
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
		(location == NULL) ? NULL : (location = (g_object_unref (location), NULL));
		(filesink == NULL) ? NULL : (filesink = (gst_object_unref (filesink), NULL));
		return;
	}
	_tmp5_ = NULL;
	g_object_set ((GObject*) filesink, "location", _tmp5_ = g_file_get_path (location), NULL);
	_tmp5_ = (g_free (_tmp5_), NULL);
	dvb_timer_set_sink (timer, filesink);
	_tmp6_ = NULL;
	channel_factory = (_tmp6_ = dvb_device_group_get_channel_factory (self->priv->_DeviceGroup), (_tmp6_ == NULL) ? NULL : g_object_ref (_tmp6_));
	_tmp7_ = NULL;
	player = dvb_channel_factory_watch_channel (channel_factory, channel, (_tmp7_ = filesink, (_tmp7_ == NULL) ? NULL : gst_object_ref (_tmp7_)), TRUE, NULL);
	if (player != NULL) {
		GstElement* _tmp8_;
		DVBRecording* recording;
		struct tm _tmp9_ = {0};
		g_debug ("Recorder.vala:442: Setting pipeline to playing");
		_tmp8_ = NULL;
		gst_element_set_state (_tmp8_ = dvb_player_thread_get_pipeline (player), GST_STATE_PLAYING);
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (gst_object_unref (_tmp8_), NULL));
		g_signal_connect_object (player, "eit-structure", (GCallback) _dvb_recorder_on_eit_structure_dvb_player_thread_eit_structure, self, 0);
		recording = dvb_recording_new ();
		dvb_recording_set_Name (recording, NULL);
		dvb_recording_set_Description (recording, NULL);
		dvb_recording_set_Id (recording, dvb_timer_get_Id (timer));
		dvb_recording_set_ChannelSid (recording, dvb_channel_get_Sid (channel));
		dvb_recording_set_ChannelName (recording, dvb_channel_get_Name (channel));
		dvb_recording_set_StartTime (recording, (_tmp9_ = dvb_timer_get_start_time_time (timer), &_tmp9_));
		dvb_recording_set_Location (recording, location);
		g_static_rec_mutex_lock (&self->priv->__lock_recordings);
		{
			gee_map_set (self->priv->recordings, GUINT_TO_POINTER ((guint) dvb_recording_get_Id (recording)), recording);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
		dvb_recordings_store_add (dvb_recordings_store_get_instance (), recording, FALSE);
		(recording == NULL) ? NULL : (recording = (g_object_unref (recording), NULL));
	}
	gee_collection_add ((GeeCollection*) self->priv->active_timers, GUINT_TO_POINTER (dvb_timer_get_Id (timer)));
	g_signal_emit_by_name ((DVBIDBusRecorder*) self, "recording-started", dvb_timer_get_Id (timer));
	(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	(location == NULL) ? NULL : (location = (g_object_unref (location), NULL));
	(filesink == NULL) ? NULL : (filesink = (gst_object_unref (filesink), NULL));
	(channel_factory == NULL) ? NULL : (channel_factory = (g_object_unref (channel_factory), NULL));
	(player == NULL) ? NULL : (player = (g_object_unref (player), NULL));
}


/**
         * Stop recording of specified timer
         */
void dvb_recorder_stop_recording (DVBRecorder* self, DVBTimer* timer) {
	GError * _inner_error_;
	DVBRecording* rec;
	guint32 timer_id;
	g_return_if_fail (self != NULL);
	g_return_if_fail (timer != NULL);
	_inner_error_ = NULL;
	rec = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		DVBRecording* _tmp0_;
		struct tm _tmp6_ = {0};
		struct tm _tmp5_ = {0};
		time_t _tmp3_ = {0};
		time_t _tmp1_ = {0};
		struct tm _tmp2_ = {0};
		struct tm _tmp4_ = {0};
		DVBChannelFactory* _tmp7_;
		DVBChannelFactory* channel_factory;
		_tmp0_ = NULL;
		rec = (_tmp0_ = (DVBRecording*) gee_map_get (self->priv->recordings, GUINT_TO_POINTER ((guint) dvb_timer_get_Id (timer))), (rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL)), _tmp0_);
		dvb_recording_set_Length (rec, dvb_utils_difftime ((_tmp5_ = (localtime_r ((_tmp3_ = (time (&_tmp1_), _tmp1_), &_tmp3_), &_tmp2_), _tmp2_), &_tmp5_), (_tmp6_ = (dvb_recording_get_StartTime (rec, &_tmp4_), _tmp4_), &_tmp6_)));
		g_debug ("Recording of channel %s stopped after %" G_GINT64_FORMAT " seconds", dvb_recording_get_ChannelName (rec), dvb_recording_get_Length (rec));
		{
			dvb_recording_save_to_disk (rec, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch10_g_error;
				goto __finally10;
			}
		}
		goto __finally10;
		__catch10_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("Recorder.vala:485: Could not save recording: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally10:
		if (_inner_error_ != NULL) {
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return;
		}
		_tmp7_ = NULL;
		channel_factory = (_tmp7_ = dvb_device_group_get_channel_factory (self->priv->_DeviceGroup), (_tmp7_ == NULL) ? NULL : g_object_ref (_tmp7_));
		dvb_channel_factory_stop_channel (channel_factory, dvb_timer_get_Channel (timer), dvb_timer_get_sink (timer));
		gee_map_remove (self->priv->recordings, GUINT_TO_POINTER ((guint) dvb_timer_get_Id (timer)));
		(channel_factory == NULL) ? NULL : (channel_factory = (g_object_unref (channel_factory), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	timer_id = dvb_timer_get_Id (timer);
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		gee_collection_remove ((GeeCollection*) self->priv->active_timers, GUINT_TO_POINTER (timer_id));
		gee_map_remove ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	dvb_recordings_store_monitor_recording (dvb_recordings_store_get_instance (), rec);
	g_signal_emit_by_name ((DVBIDBusRecorder*) self, "changed", timer_id, (guint) DVB_CHANGE_TYPE_DELETED);
	g_signal_emit_by_name ((DVBIDBusRecorder*) self, "recording-finished", dvb_recording_get_Id (rec));
	(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
}


/**
         * @returns: File on success, NULL otherwise
         * 
         * Create directories and set location of recording
         */
static GFile* dvb_recorder_create_recording_dirs (DVBRecorder* self, DVBChannel* channel, guint* start, int start_length1) {
	GFile* result;
	GError * _inner_error_;
	char* channel_name;
	char* time;
	GFile* _tmp0_;
	GFile* _tmp1_;
	GFile* dir;
	char* attributes;
	GFileInfo* info;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (channel != NULL, NULL);
	_inner_error_ = NULL;
	channel_name = dvb_utils_remove_nonalphanums (dvb_channel_get_Name (channel));
	time = g_strdup_printf ("%u-%u-%u_%u-%u", start[0], start[1], start[2], start[3], start[4]);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	dir = (_tmp1_ = g_file_get_child (_tmp0_ = g_file_get_child (dvb_device_group_get_RecordingsDirectory (self->priv->_DeviceGroup), channel_name), time), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	if (!g_file_query_exists (dir, NULL)) {
		{
			dvb_utils_mkdirs (dir, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch11_g_error;
				goto __finally11;
			}
		}
		goto __finally11;
		__catch11_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp2_;
				_tmp2_ = NULL;
				g_error ("Recorder.vala:521: Could not create directory %s: %s", _tmp2_ = g_file_get_path (dir), e->message);
				_tmp2_ = (g_free (_tmp2_), NULL);
				result = NULL;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				channel_name = (g_free (channel_name), NULL);
				time = (g_free (time), NULL);
				(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
				return result;
			}
		}
		__finally11:
		if (_inner_error_ != NULL) {
			channel_name = (g_free (channel_name), NULL);
			time = (g_free (time), NULL);
			(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
	}
	attributes = g_strdup_printf ("%s,%s", G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_ATTRIBUTE_ACCESS_CAN_WRITE);
	info = NULL;
	{
		GFileInfo* _tmp3_;
		GFileInfo* _tmp4_;
		_tmp3_ = g_file_query_info (dir, attributes, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch12_g_error;
			goto __finally12;
		}
		_tmp4_ = NULL;
		info = (_tmp4_ = _tmp3_, (info == NULL) ? NULL : (info = (g_object_unref (info), NULL)), _tmp4_);
	}
	goto __finally12;
	__catch12_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Recorder.vala:533: Could not retrieve attributes: %s", e->message);
			result = NULL;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			channel_name = (g_free (channel_name), NULL);
			time = (g_free (time), NULL);
			(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
			attributes = (g_free (attributes), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			return result;
		}
	}
	__finally12:
	if (_inner_error_ != NULL) {
		channel_name = (g_free (channel_name), NULL);
		time = (g_free (time), NULL);
		(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
		attributes = (g_free (attributes), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	if (g_file_info_get_attribute_uint32 (info, G_FILE_ATTRIBUTE_STANDARD_TYPE) != G_FILE_TYPE_DIRECTORY) {
		char* _tmp5_;
		_tmp5_ = NULL;
		g_critical ("Recorder.vala:539: %s is not a directory", _tmp5_ = g_file_get_path (dir));
		_tmp5_ = (g_free (_tmp5_), NULL);
		result = NULL;
		channel_name = (g_free (channel_name), NULL);
		time = (g_free (time), NULL);
		(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
		attributes = (g_free (attributes), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return result;
	}
	if (!g_file_info_get_attribute_boolean (info, G_FILE_ATTRIBUTE_ACCESS_CAN_WRITE)) {
		char* _tmp6_;
		_tmp6_ = NULL;
		g_critical ("Recorder.vala:544: Cannot write to %s", _tmp6_ = g_file_get_path (dir));
		_tmp6_ = (g_free (_tmp6_), NULL);
		result = NULL;
		channel_name = (g_free (channel_name), NULL);
		time = (g_free (time), NULL);
		(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
		attributes = (g_free (attributes), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return result;
	}
	result = g_file_get_child (dir, "001.mpeg");
	channel_name = (g_free (channel_name), NULL);
	time = (g_free (time), NULL);
	(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
	attributes = (g_free (attributes), NULL);
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
	return result;
}


static void _g_slist_free_g_object_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) g_object_unref, NULL);
	g_slist_free (self);
}


static gboolean dvb_recorder_check_timers (DVBRecorder* self) {
	gboolean result;
	gboolean val;
	GSList* ended_recordings;
	g_return_val_if_fail (self != NULL, FALSE);
	g_debug ("Recorder.vala:552: Checking timers");
	val = FALSE;
	ended_recordings = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_timers);
	{
		GSList* deleteable_items;
		gboolean _tmp9_;
		{
			GeeIterator* _timer_id_it;
			_timer_id_it = gee_iterable_iterator ((GeeIterable*) self->priv->active_timers);
			while (TRUE) {
				guint32 timer_id;
				DVBTimer* timer;
				if (!gee_iterator_next (_timer_id_it)) {
					break;
				}
				timer_id = GPOINTER_TO_UINT (gee_iterator_get (_timer_id_it));
				timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (timer_id));
				if (dvb_timer_is_end_due (timer)) {
					DVBTimer* _tmp0_;
					_tmp0_ = NULL;
					ended_recordings = g_slist_prepend (ended_recordings, (_tmp0_ = timer, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)));
				}
				(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
			}
			(_timer_id_it == NULL) ? NULL : (_timer_id_it = (g_object_unref (_timer_id_it), NULL));
		}
		{
			gint i;
			/* Delete timers of recordings that have ended*/
			i = 0;
			{
				gboolean _tmp1_;
				_tmp1_ = TRUE;
				while (TRUE) {
					DVBTimer* _tmp2_;
					DVBTimer* timer;
					if (!_tmp1_) {
						i++;
					}
					_tmp1_ = FALSE;
					if (!(i < g_slist_length (ended_recordings))) {
						break;
					}
					_tmp2_ = NULL;
					timer = (_tmp2_ = (DVBTimer*) g_slist_nth_data (ended_recordings, (guint) i), (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_));
					dvb_recorder_stop_recording (self, timer);
					(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
				}
			}
		}
		/* Store items we want to delete in here*/
		deleteable_items = NULL;
		{
			GeeSet* _tmp3_;
			GeeIterator* _tmp4_;
			GeeIterator* _key_it;
			_tmp3_ = NULL;
			_tmp4_ = NULL;
			_key_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = gee_map_get_keys ((GeeMap*) self->priv->timers))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (g_object_unref (_tmp3_), NULL)), _tmp4_);
			while (TRUE) {
				guint32 key;
				DVBTimer* timer;
				char* _tmp5_;
				gboolean _tmp6_;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				timer = (DVBTimer*) gee_map_get ((GeeMap*) self->priv->timers, GUINT_TO_POINTER (key));
				_tmp5_ = NULL;
				g_debug ("Recorder.vala:578: Checking timer: %s", _tmp5_ = dvb_timer_to_string (timer));
				_tmp5_ = (g_free (_tmp5_), NULL);
				_tmp6_ = FALSE;
				if (dvb_timer_is_start_due (timer)) {
					_tmp6_ = !gee_collection_contains ((GeeCollection*) self->priv->active_timers, GUINT_TO_POINTER (dvb_timer_get_Id (timer)));
				} else {
					_tmp6_ = FALSE;
				}
				/* Check if we should start new recording and if we didn't
				 start it before*/
				if (_tmp6_) {
					dvb_recorder_start_recording (self, timer);
				} else {
					if (dvb_timer_has_expired (timer)) {
						char* _tmp7_;
						_tmp7_ = NULL;
						g_debug ("Recorder.vala:586: Removing expired timer: %s", _tmp7_ = dvb_timer_to_string (timer));
						_tmp7_ = (g_free (_tmp7_), NULL);
						deleteable_items = g_slist_prepend (deleteable_items, GUINT_TO_POINTER (key));
					}
				}
				(timer == NULL) ? NULL : (timer = (g_object_unref (timer), NULL));
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
		{
			gint i;
			/* Delete items from this.timers using this.DeleteTimer*/
			i = 0;
			{
				gboolean _tmp8_;
				_tmp8_ = TRUE;
				while (TRUE) {
					if (!_tmp8_) {
						i++;
					}
					_tmp8_ = FALSE;
					if (!(i < g_slist_length (deleteable_items))) {
						break;
					}
					dvb_id_bus_recorder_DeleteTimer ((DVBIDBusRecorder*) self, GPOINTER_TO_UINT (g_slist_nth_data (deleteable_items, (guint) i)));
				}
			}
		}
		_tmp9_ = FALSE;
		if (gee_map_get_size ((GeeMap*) self->priv->timers) == 0) {
			_tmp9_ = gee_collection_get_size ((GeeCollection*) self->priv->active_timers) == 0;
		} else {
			_tmp9_ = FALSE;
		}
		if (_tmp9_) {
			/* We don't have any timers and no recording is in progress*/
			g_debug ("Recorder.vala:598: No timers left and no recording in progress");
			self->priv->have_check_timers_timeout = FALSE;
			val = FALSE;
		} else {
			/* We still have timers*/
			g_debug ("Recorder.vala:603: %d timers and %d active recordings left", gee_map_get_size ((GeeMap*) self->priv->timers), gee_collection_get_size ((GeeCollection*) self->priv->active_timers));
			val = TRUE;
		}
		(deleteable_items == NULL) ? NULL : (deleteable_items = (g_slist_free (deleteable_items), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_timers);
	result = val;
	(ended_recordings == NULL) ? NULL : (ended_recordings = (_g_slist_free_g_object_unref (ended_recordings), NULL));
	return result;
}


static void dvb_recorder_on_eit_structure (DVBRecorder* self, DVBPlayerThread* player, const GstStructure* structure) {
	guint sid;
	g_return_if_fail (self != NULL);
	g_return_if_fail (player != NULL);
	g_return_if_fail (structure != NULL);
	sid = 0U;
	gst_structure_get_uint (structure, "service-id", &sid);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		/* Find name and description for recordings*/
		{
			GeeCollection* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _rec_it;
			/* Find name and description for recordings*/
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_rec_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_values (self->priv->recordings))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			/* Find name and description for recordings*/
			while (TRUE) {
				DVBRecording* rec;
				gboolean _tmp2_;
				/* Find name and description for recordings*/
				if (!gee_iterator_next (_rec_it)) {
					/* Find name and description for recordings*/
					break;
				}
				/* Find name and description for recordings*/
				rec = (DVBRecording*) gee_iterator_get (_rec_it);
				_tmp2_ = FALSE;
				if (dvb_recording_get_Name (rec) == NULL) {
					_tmp2_ = sid == dvb_recording_get_ChannelSid (rec);
				} else {
					_tmp2_ = FALSE;
				}
				if (_tmp2_) {
					DVBChannel* chan;
					DVBSchedule* _tmp3_;
					DVBSchedule* sched;
					DVBEvent* event;
					chan = dvb_channel_list_get_channel (dvb_device_group_get_Channels (self->priv->_DeviceGroup), sid);
					_tmp3_ = NULL;
					sched = (_tmp3_ = dvb_channel_get_Schedule (chan), (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
					event = dvb_schedule_get_running_event (sched);
					if (event != NULL) {
						char* _tmp4_;
						g_debug ("Recorder.vala:625: Found running event for active recording");
						dvb_recording_set_Name (rec, event->name);
						_tmp4_ = NULL;
						dvb_recording_set_Description (rec, _tmp4_ = g_strdup_printf ("%s\n%s", event->description, event->extended_description));
						_tmp4_ = (g_free (_tmp4_), NULL);
					}
					(chan == NULL) ? NULL : (chan = (g_object_unref (chan), NULL));
					(sched == NULL) ? NULL : (sched = (g_object_unref (sched), NULL));
					(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
				}
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
			}
			(_rec_it == NULL) ? NULL : (_rec_it = (g_object_unref (_rec_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
}


DVBDeviceGroup* dvb_recorder_get_DeviceGroup (DVBRecorder* self) {
	DVBDeviceGroup* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_DeviceGroup;
	return result;
}


static void dvb_recorder_set_DeviceGroup (DVBRecorder* self, DVBDeviceGroup* value) {
	g_return_if_fail (self != NULL);
	self->priv->_DeviceGroup = value;
	g_object_notify ((GObject *) self, "DeviceGroup");
}


guint dvb_recorder_get_count (DVBRecorder* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = (guint) gee_map_get_size (self->priv->recordings);
	return result;
}


static GObject * dvb_recorder_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBRecorderClass * klass;
	GObjectClass * parent_class;
	DVBRecorder * self;
	klass = DVB_RECORDER_CLASS (g_type_class_peek (DVB_TYPE_RECORDER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_RECORDER (obj);
	{
		GeeSet* _tmp0_;
		GeeHashMap* _tmp1_;
		GeeMap* _tmp2_;
		_tmp0_ = NULL;
		self->priv->active_timers = (_tmp0_ = (GeeSet*) gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal), (self->priv->active_timers == NULL) ? NULL : (self->priv->active_timers = (g_object_unref (self->priv->active_timers), NULL)), _tmp0_);
		_tmp1_ = NULL;
		self->priv->timers = (_tmp1_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_TIMER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->timers == NULL) ? NULL : (self->priv->timers = (g_object_unref (self->priv->timers), NULL)), _tmp1_);
		self->priv->have_check_timers_timeout = FALSE;
		dvb_recordings_store_restore_from_dir (dvb_recordings_store_get_instance (), dvb_device_group_get_RecordingsDirectory (self->priv->_DeviceGroup));
		_tmp2_ = NULL;
		self->priv->recordings = (_tmp2_ = (GeeMap*) gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_RECORDING, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL)), _tmp2_);
	}
	return obj;
}


void _dvb_recorder_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_recorder_introspect (DVBRecorder* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Recorder\">\n  <method name=\"AddTimer\">\n    <arg name=\"channel\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_year\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_month\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_day\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_hour\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_minute\" type=\"i\" direction=\"in\"/>\n    <arg name=\"duration\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"AddTimerWithMargin\">\n    <arg name=\"channel\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_year\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_month\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_day\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_hour\" type=\"i\" direction=\"in\"/>\n    <arg name=\"start_minute\" type=\"i\" direction=\"in\"/>\n    <arg name=\"duration\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"AddTimerForEPGEvent\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channel_sid\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"DeleteTimer\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetTimers\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetStartTime\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetEndTime\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDuration\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelName\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetActiveTimers\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"IsTimerActive\">\n    <arg name=\"timer_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"HasTimer\">\n    <arg name=\"start_year\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_month\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_day\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_hour\" type=\"u\" direction=\"in\"/>\n    <arg name=\"start_minute\" type=\"u\" direction=\"in\"/>\n    <arg name=\"duration\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"HasTimerForEvent\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channel_sid\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"i\" direction=\"out\"/>\n  </method>\n  <signal name=\"RecordingStarted\">\n    <arg name=\"timer_id\" type=\"u\"/>\n  </signal>\n  <signal name=\"RecordingFinished\">\n    <arg name=\"recording_id\" type=\"u\"/>\n  </signal>\n  <signal name=\"Changed\">\n    <arg name=\"timer_id\" type=\"u\"/>\n    <arg name=\"type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_recorder_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_recorder_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_recorder_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_recorder_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_recorder_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_recorder_dbus_register_object (connection, path, object);
}


static void dvb_recorder_class_init (DVBRecorderClass * klass) {
	dvb_recorder_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBRecorderPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_recorder_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_recorder_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_recorder_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_recorder_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_RECORDER_DEVICE_GROUP, g_param_spec_object ("DeviceGroup", "DeviceGroup", "DeviceGroup", DVB_TYPE_DEVICE_GROUP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_RECORDER_COUNT, g_param_spec_uint ("count", "count", "count", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_type_set_qdata (DVB_TYPE_RECORDER, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_recorder_dbus_vtable));
}


static void dvb_recorder_dvb_id_bus_recorder_interface_init (DVBIDBusRecorderIface * iface) {
	dvb_recorder_dvb_id_bus_recorder_parent_iface = g_type_interface_peek_parent (iface);
	iface->AddTimer = dvb_recorder_real_AddTimer;
	iface->AddTimerWithMargin = dvb_recorder_real_AddTimerWithMargin;
	iface->AddTimerForEPGEvent = dvb_recorder_real_AddTimerForEPGEvent;
	iface->DeleteTimer = dvb_recorder_real_DeleteTimer;
	iface->GetTimers = dvb_recorder_real_GetTimers;
	iface->GetStartTime = dvb_recorder_real_GetStartTime;
	iface->GetEndTime = dvb_recorder_real_GetEndTime;
	iface->GetDuration = dvb_recorder_real_GetDuration;
	iface->GetChannelName = dvb_recorder_real_GetChannelName;
	iface->GetActiveTimers = dvb_recorder_real_GetActiveTimers;
	iface->IsTimerActive = dvb_recorder_real_IsTimerActive;
	iface->HasTimer = dvb_recorder_real_HasTimer;
	iface->HasTimerForEvent = dvb_recorder_real_HasTimerForEvent;
}


static void dvb_recorder_gee_iterable_interface_init (GeeIterableIface * iface) {
	dvb_recorder_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = dvb_recorder_real_get_element_type;
	iface->iterator = dvb_recorder_real_iterator;
}


static void dvb_recorder_instance_init (DVBRecorder * self) {
	self->priv = DVB_RECORDER_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_timers);
	g_static_rec_mutex_init (&self->priv->__lock_recordings);
}


static void dvb_recorder_finalize (GObject* obj) {
	DVBRecorder * self;
	self = DVB_RECORDER (obj);
	(self->priv->active_timers == NULL) ? NULL : (self->priv->active_timers = (g_object_unref (self->priv->active_timers), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_timers);
	(self->priv->timers == NULL) ? NULL : (self->priv->timers = (g_object_unref (self->priv->timers), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_recordings);
	(self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL));
	G_OBJECT_CLASS (dvb_recorder_parent_class)->finalize (obj);
}


GType dvb_recorder_get_type (void) {
	static GType dvb_recorder_type_id = 0;
	if (dvb_recorder_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBRecorderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_recorder_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBRecorder), 0, (GInstanceInitFunc) dvb_recorder_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_recorder_info = { (GInterfaceInitFunc) dvb_recorder_dvb_id_bus_recorder_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) dvb_recorder_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_recorder_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBRecorder", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_recorder_type_id, DVB_TYPE_ID_BUS_RECORDER, &dvb_id_bus_recorder_info);
		g_type_add_interface_static (dvb_recorder_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
	}
	return dvb_recorder_type_id;
}


static void dvb_recorder_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBRecorder * self;
	gpointer boxed;
	self = DVB_RECORDER (object);
	switch (property_id) {
		case DVB_RECORDER_DEVICE_GROUP:
		g_value_set_object (value, dvb_recorder_get_DeviceGroup (self));
		break;
		case DVB_RECORDER_COUNT:
		g_value_set_uint (value, dvb_recorder_get_count (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_recorder_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBRecorder * self;
	self = DVB_RECORDER (object);
	switch (property_id) {
		case DVB_RECORDER_DEVICE_GROUP:
		dvb_recorder_set_DeviceGroup (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




