/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <gio/gio.h>
#include <time.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


#define DVB_TYPE_ID_BUS_RECORDINGS_STORE (dvb_id_bus_recordings_store_get_type ())
#define DVB_ID_BUS_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_RECORDINGS_STORE, DVBIDBusRecordingsStore))
#define DVB_IS_ID_BUS_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_RECORDINGS_STORE))
#define DVB_ID_BUS_RECORDINGS_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_RECORDINGS_STORE, DVBIDBusRecordingsStoreIface))

typedef struct _DVBIDBusRecordingsStore DVBIDBusRecordingsStore;
typedef struct _DVBIDBusRecordingsStoreIface DVBIDBusRecordingsStoreIface;

#define DVB_TYPE_RECORDING_INFO (dvb_recording_info_get_type ())
typedef struct _DVBRecordingInfo DVBRecordingInfo;

#define DVB_TYPE_RECORDINGS_STORE (dvb_recordings_store_get_type ())
#define DVB_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStore))
#define DVB_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))
#define DVB_IS_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDINGS_STORE))
#define DVB_IS_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDINGS_STORE))
#define DVB_RECORDINGS_STORE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))

typedef struct _DVBRecordingsStore DVBRecordingsStore;
typedef struct _DVBRecordingsStoreClass DVBRecordingsStoreClass;
typedef struct _DVBRecordingsStorePrivate DVBRecordingsStorePrivate;

#define DVB_TYPE_RECORDING (dvb_recording_get_type ())
#define DVB_RECORDING(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDING, DVBRecording))
#define DVB_RECORDING_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDING, DVBRecordingClass))
#define DVB_IS_RECORDING(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDING))
#define DVB_IS_RECORDING_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDING))
#define DVB_RECORDING_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDING, DVBRecordingClass))

typedef struct _DVBRecording DVBRecording;
typedef struct _DVBRecordingClass DVBRecordingClass;

#define DVB_TYPE_CHANGE_TYPE (dvb_change_type_get_type ())
typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DVBRecordingInfo {
	guint32 id;
	char* name;
	char* description;
	gint64 length;
	gint64 start_timestamp;
	char* channel;
	char* location;
};

struct _DVBIDBusRecordingsStoreIface {
	GTypeInterface parent_iface;
	guint32* (*GetRecordings) (DVBIDBusRecordingsStore* self, int* result_length1);
	char* (*GetLocation) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	char* (*GetName) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	char* (*GetDescription) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	guint* (*GetStartTime) (DVBIDBusRecordingsStore* self, guint32 rec_id, int* result_length1);
	gint64 (*GetStartTimestamp) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	gint64 (*GetLength) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	gboolean (*Delete) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	char* (*GetChannelName) (DVBIDBusRecordingsStore* self, guint32 rec_id);
	DVBRecordingInfo (*GetAllInformations) (DVBIDBusRecordingsStore* self, guint32 rec_id);
};

/**
     * This class manages the recordings off all devices
     */
struct _DVBRecordingsStore {
	GObject parent_instance;
	DVBRecordingsStorePrivate * priv;
};

struct _DVBRecordingsStoreClass {
	GObjectClass parent_class;
};

struct _DVBRecordingsStorePrivate {
	GeeHashMap* recordings;
	GStaticRecMutex __lock_recordings;
	guint32 last_id;
	GStaticRecMutex __lock_last_id;
};

typedef enum  {
	DVB_CHANGE_TYPE_ADDED = 0,
	DVB_CHANGE_TYPE_DELETED = 1,
	DVB_CHANGE_TYPE_UPDATED = 2
} DVBChangeType;

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


static DVBRecordingsStore* dvb_recordings_store_instance;
static DVBRecordingsStore* dvb_recordings_store_instance = NULL;
static GStaticRecMutex dvb_recordings_store_instance_mutex;
static GStaticRecMutex dvb_recordings_store_instance_mutex = {0};
static gpointer dvb_recordings_store_parent_class = NULL;
static DVBIDBusRecordingsStoreIface* dvb_recordings_store_dvb_id_bus_recordings_store_parent_iface = NULL;

GType dvb_recording_info_get_type (void);
DVBRecordingInfo* dvb_recording_info_dup (const DVBRecordingInfo* self);
void dvb_recording_info_free (DVBRecordingInfo* self);
void dvb_recording_info_copy (const DVBRecordingInfo* self, DVBRecordingInfo* dest);
void dvb_recording_info_destroy (DVBRecordingInfo* self);
GType dvb_id_bus_recordings_store_get_type (void);
GType dvb_recordings_store_get_type (void);
GType dvb_recording_get_type (void);
#define DVB_RECORDINGS_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStorePrivate))
enum  {
	DVB_RECORDINGS_STORE_DUMMY_PROPERTY
};
DVBRecordingsStore* dvb_recordings_store_new (void);
DVBRecordingsStore* dvb_recordings_store_construct (GType object_type);
DVBRecordingsStore* dvb_recordings_store_get_instance (void);
void dvb_recordings_store_shutdown (void);
void dvb_recordings_store_update_last_id (DVBRecordingsStore* self, guint32 new_last_id);
guint32 dvb_recording_get_Id (DVBRecording* self);
void dvb_recordings_store_monitor_recording (DVBRecordingsStore* self, DVBRecording* rec);
GType dvb_change_type_get_type (void);
gboolean dvb_recordings_store_add (DVBRecordingsStore* self, DVBRecording* rec, gboolean monitor);
GFile* dvb_recording_get_Location (DVBRecording* self);
static void dvb_recordings_store_on_recording_file_changed (DVBRecordingsStore* self, GFileMonitor* monitor, GFile* file, GFile* other_file, GFileMonitorEvent event);
static void _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed (GFileMonitor* _sender, GFile* file, GFile* other_file, GFileMonitorEvent event_type, gpointer self);
void dvb_recording_set_file_monitor (DVBRecording* self, GFileMonitor* value);
guint32 dvb_recordings_store_get_next_id (DVBRecordingsStore* self);
static guint32* dvb_recordings_store_real_GetRecordings (DVBIDBusRecordingsStore* base, int* result_length1);
static char* dvb_recordings_store_real_GetLocation (DVBIDBusRecordingsStore* base, guint32 rec_id);
const char* dvb_recording_get_Name (DVBRecording* self);
static char* dvb_recordings_store_real_GetName (DVBIDBusRecordingsStore* base, guint32 rec_id);
const char* dvb_recording_get_Description (DVBRecording* self);
static char* dvb_recordings_store_real_GetDescription (DVBIDBusRecordingsStore* base, guint32 rec_id);
guint* dvb_recording_get_start (DVBRecording* self, int* result_length1);
static guint* dvb_recordings_store_real_GetStartTime (DVBIDBusRecordingsStore* base, guint32 rec_id, int* result_length1);
void dvb_recording_get_StartTime (DVBRecording* self, struct tm* value);
static gint64 dvb_recordings_store_real_GetStartTimestamp (DVBIDBusRecordingsStore* base, guint32 rec_id);
gint64 dvb_recording_get_Length (DVBRecording* self);
static gint64 dvb_recordings_store_real_GetLength (DVBIDBusRecordingsStore* base, guint32 rec_id);
void dvb_utils_delete_dir_recursively (GFile* dir, GError** error);
static gboolean dvb_recordings_store_real_Delete (DVBIDBusRecordingsStore* base, guint32 rec_id);
const char* dvb_recording_get_ChannelName (DVBRecording* self);
static char* dvb_recordings_store_real_GetChannelName (DVBIDBusRecordingsStore* base, guint32 rec_id);
static DVBRecordingInfo dvb_recordings_store_real_GetAllInformations (DVBIDBusRecordingsStore* base, guint32 rec_id);
void dvb_recordings_store_restore_from_dir (DVBRecordingsStore* self, GFile* recordingsbasedir);
DVBRecording* dvb_recording_deserialize (GFile* file, GError** error);
static gboolean dvb_recordings_store_delete_recording_by_location (DVBRecordingsStore* self, const char* location);
static GObject * dvb_recordings_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void dvb_recordings_store_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_recordings_store_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_recordings_store_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_recordings_store_introspect (DVBRecordingsStore* self, DBusConnection* connection, DBusMessage* message);
static void dvb_recordings_store_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_recordings_store_dbus_path_vtable = {_dvb_recordings_store_dbus_unregister, dvb_recordings_store_dbus_message};
static const _DBusObjectVTable _dvb_recordings_store_dbus_vtable = {dvb_recordings_store_dbus_register_object};


DVBRecordingsStore* dvb_recordings_store_get_instance (void) {
	DVBRecordingsStore* result;
	g_static_rec_mutex_lock (&dvb_recordings_store_instance_mutex);
	if (dvb_recordings_store_instance == NULL) {
		DVBRecordingsStore* _tmp0_;
		_tmp0_ = NULL;
		dvb_recordings_store_instance = (_tmp0_ = dvb_recordings_store_new (), (dvb_recordings_store_instance == NULL) ? NULL : (dvb_recordings_store_instance = (g_object_unref (dvb_recordings_store_instance), NULL)), _tmp0_);
	}
	g_static_rec_mutex_unlock (&dvb_recordings_store_instance_mutex);
	result = dvb_recordings_store_instance;
	return result;
}


void dvb_recordings_store_shutdown (void) {
	DVBRecordingsStore* _tmp0_;
	DVBRecordingsStore* rs;
	g_static_rec_mutex_lock (&dvb_recordings_store_instance_mutex);
	_tmp0_ = NULL;
	rs = (_tmp0_ = dvb_recordings_store_instance, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	if (rs != NULL) {
		DVBRecordingsStore* _tmp1_;
		gee_abstract_map_clear ((GeeAbstractMap*) rs->priv->recordings);
		_tmp1_ = NULL;
		dvb_recordings_store_instance = (_tmp1_ = NULL, (dvb_recordings_store_instance == NULL) ? NULL : (dvb_recordings_store_instance = (g_object_unref (dvb_recordings_store_instance), NULL)), _tmp1_);
	}
	g_static_rec_mutex_unlock (&dvb_recordings_store_instance_mutex);
	(rs == NULL) ? NULL : (rs = (g_object_unref (rs), NULL));
}


void dvb_recordings_store_update_last_id (DVBRecordingsStore* self, guint32 new_last_id) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_last_id);
	{
		if (new_last_id > self->priv->last_id) {
			self->priv->last_id = new_last_id;
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
}


gboolean dvb_recordings_store_add (DVBRecordingsStore* self, DVBRecording* rec, gboolean monitor) {
	gboolean result;
	guint32 id;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (rec != NULL, FALSE);
	id = dvb_recording_get_Id (rec);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (id))) {
			g_critical ("RecordingsStore.vala:70: Recording with id %u already available", (guint) id);
			result = FALSE;
			return result;
		}
		if (monitor) {
			dvb_recordings_store_monitor_recording (self, rec);
		}
		gee_abstract_map_set ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (id), rec);
		g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", id, (guint) DVB_CHANGE_TYPE_ADDED);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = TRUE;
	return result;
}


static void _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed (GFileMonitor* _sender, GFile* file, GFile* other_file, GFileMonitorEvent event_type, gpointer self) {
	dvb_recordings_store_on_recording_file_changed (self, _sender, file, other_file, event_type);
}


void dvb_recordings_store_monitor_recording (DVBRecordingsStore* self, DVBRecording* rec) {
	GError * _inner_error_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (rec != NULL);
	_inner_error_ = NULL;
	{
		GFileMonitor* monitor;
		monitor = g_file_monitor_file (dvb_recording_get_Location (rec), 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch13_g_error;
			goto __finally13;
		}
		g_signal_connect_object (monitor, "changed", (GCallback) _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed, self, 0);
		dvb_recording_set_file_monitor (rec, monitor);
		(monitor == NULL) ? NULL : (monitor = (g_object_unref (monitor), NULL));
	}
	goto __finally13;
	__catch13_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_warning ("RecordingsStore.vala:91: Could not create FileMonitor: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally13:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
}


guint32 dvb_recordings_store_get_next_id (DVBRecordingsStore* self) {
	guint32 result;
	guint32 val;
	g_return_val_if_fail (self != NULL, 0U);
	val = 0U;
	g_static_rec_mutex_lock (&self->priv->__lock_last_id);
	{
		val = self->priv->last_id = self->priv->last_id + 1;
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
	result = val;
	return result;
}


/**
         * @returns: A list of ids for all recordings
         */
static guint32* dvb_recordings_store_real_GetRecordings (DVBIDBusRecordingsStore* base, int* result_length1) {
	DVBRecordingsStore * self;
	guint32* result;
	gint ids_size;
	gint ids_length1;
	guint32* ids;
	guint32* _tmp4_;
	self = (DVBRecordingsStore*) base;
	ids = (ids_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		guint32* _tmp1_;
		gint _tmp0_;
		gint i;
		_tmp1_ = NULL;
		ids = (_tmp1_ = g_new0 (guint32, _tmp0_ = gee_map_get_size ((GeeMap*) self->priv->recordings)), ids = (g_free (ids), NULL), ids_length1 = _tmp0_, ids_size = ids_length1, _tmp1_);
		i = 0;
		{
			GeeSet* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _key_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_key_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->recordings))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				guint32 key;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				ids[i] = key;
				i++;
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	_tmp4_ = NULL;
	result = (_tmp4_ = ids, *result_length1 = ids_length1, _tmp4_);
	return result;
	ids = (g_free (ids), NULL);
}


/**
         * @rec_id: The id of the recording
         * @returns: The location of the recording on the filesystem
         */
static char* dvb_recordings_store_real_GetLocation (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* result;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp1_;
			DVBRecording* _tmp0_;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp1_ = g_file_get_uri (dvb_recording_get_Location (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)))), val = (g_free (val), NULL), _tmp1_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: The name of the recording (e.g. the name of
         * a TV show)
         */
static char* dvb_recordings_store_real_GetName (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* result;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp2_;
			const char* _tmp1_;
			DVBRecording* _tmp0_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp2_ = (_tmp1_ = dvb_recording_get_Name (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), val = (g_free (val), NULL), _tmp2_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
			if (val == NULL) {
				char* _tmp3_;
				_tmp3_ = NULL;
				val = (_tmp3_ = g_strdup (""), val = (g_free (val), NULL), _tmp3_);
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: A short text describing the recorded item
         * (e.g. the description from EPG)
         */
static char* dvb_recordings_store_real_GetDescription (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* result;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp2_;
			const char* _tmp1_;
			DVBRecording* _tmp0_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp2_ = (_tmp1_ = dvb_recording_get_Description (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), val = (g_free (val), NULL), _tmp2_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
			if (val == NULL) {
				char* _tmp3_;
				_tmp3_ = NULL;
				val = (_tmp3_ = g_strdup (""), val = (g_free (val), NULL), _tmp3_);
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: The starting time of the recording
         */
static guint* dvb_recordings_store_real_GetStartTime (DVBIDBusRecordingsStore* base, guint32 rec_id, int* result_length1) {
	DVBRecordingsStore * self;
	guint* result;
	gint val_size;
	gint val_length1;
	guint* val;
	guint* _tmp5_;
	self = (DVBRecordingsStore*) base;
	val = (val_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			guint* _tmp2_;
			gint _tmp1_;
			DVBRecording* _tmp0_;
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp2_ = dvb_recording_get_start (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)), &_tmp1_), val = (g_free (val), NULL), val_length1 = _tmp1_, val_size = val_length1, _tmp2_);
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		} else {
			guint* _tmp4_;
			guint* _tmp3_;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			val = (_tmp4_ = (_tmp3_ = g_new0 (guint, 0), _tmp3_), val = (g_free (val), NULL), val_length1 = 0, val_size = val_length1, _tmp4_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	_tmp5_ = NULL;
	result = (_tmp5_ = val, *result_length1 = val_length1, _tmp5_);
	return result;
	val = (g_free (val), NULL);
}


/**
         * @rec_id: The id of the recording
         * @returns: Start time as UNIX timestamp
         */
static gint64 dvb_recordings_store_real_GetStartTimestamp (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	gint64 result;
	gint64 val;
	self = (DVBRecordingsStore*) base;
	val = (gint64) (-1);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			struct tm _tmp2_ = {0};
			DVBRecording* _tmp0_;
			struct tm _tmp1_ = {0};
			_tmp0_ = NULL;
			val = (gint64) mktime ((_tmp2_ = (dvb_recording_get_StartTime (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)), &_tmp1_), _tmp1_), &_tmp2_));
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: The length of the recording in seconds
         * or -1 if no recording with the given id exists
         */
static gint64 dvb_recordings_store_real_GetLength (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	gint64 result;
	gint64 val;
	self = (DVBRecordingsStore*) base;
	val = (gint64) (-1);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			DVBRecording* _tmp0_;
			_tmp0_ = NULL;
			val = dvb_recording_get_Length (_tmp0_ = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)));
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: TRUE on success, FALSE otherwises
         *
         * Delete the recording. This deletes all files in the directory
         * created by the Recorder
         */
static gboolean dvb_recordings_store_real_Delete (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	gboolean result;
	GError * _inner_error_;
	gboolean val;
	self = (DVBRecordingsStore*) base;
	_inner_error_ = NULL;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (!gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			val = FALSE;
		} else {
			DVBRecording* rec;
			g_debug ("RecordingsStore.vala:231: Deleting recording %u", (guint) rec_id);
			rec = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
			{
				GFile* _tmp0_;
				_tmp0_ = NULL;
				dvb_utils_delete_dir_recursively (_tmp0_ = g_file_get_parent (dvb_recording_get_Location (rec)), &_inner_error_);
				if (_inner_error_ != NULL) {
					goto __catch14_g_error;
					goto __finally14;
				}
				(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
				gee_abstract_map_remove ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id), NULL);
				val = TRUE;
			}
			goto __finally14;
			__catch14_g_error:
			{
				GError * e;
				e = _inner_error_;
				_inner_error_ = NULL;
				{
					g_critical ("RecordingsStore.vala:238: Could not delete recording: %s", e->message);
					val = FALSE;
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				}
			}
			__finally14:
			if (_inner_error_ != NULL) {
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
				g_clear_error (&_inner_error_);
				return FALSE;
			}
			g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", rec_id, (guint) DVB_CHANGE_TYPE_DELETED);
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = val;
	return result;
}


/**
         * @rec_id: The id of the recording
         * @returns: The channel's name or an empty string if
         * rec_id doesn't exist
         */
static char* dvb_recordings_store_real_GetChannelName (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* result;
	char* ret;
	self = (DVBRecordingsStore*) base;
	ret = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			DVBRecording* rec;
			char* _tmp1_;
			const char* _tmp0_;
			rec = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			ret = (_tmp1_ = (_tmp0_ = dvb_recording_get_ChannelName (rec), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), ret = (g_free (ret), NULL), _tmp1_);
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		} else {
			char* _tmp2_;
			_tmp2_ = NULL;
			ret = (_tmp2_ = g_strdup (""), ret = (g_free (ret), NULL), _tmp2_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = ret;
	return result;
}


static DVBRecordingInfo dvb_recordings_store_real_GetAllInformations (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	DVBRecordingInfo result;
	DVBRecordingInfo _tmp0_ = {0};
	DVBRecordingInfo info;
	self = (DVBRecordingsStore*) base;
	info = (memset (&_tmp0_, 0, sizeof (DVBRecordingInfo)), _tmp0_);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			DVBRecording* rec;
			const char* _tmp1_;
			char* _tmp3_;
			const char* _tmp2_;
			const char* _tmp4_;
			char* _tmp6_;
			const char* _tmp5_;
			char* _tmp7_;
			struct tm _tmp9_ = {0};
			struct tm _tmp8_ = {0};
			char* _tmp11_;
			const char* _tmp10_;
			rec = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
			_tmp1_ = NULL;
			if (dvb_recording_get_Name (rec) == NULL) {
				_tmp1_ = "";
			} else {
				_tmp1_ = dvb_recording_get_Name (rec);
			}
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			info.name = (_tmp3_ = (_tmp2_ = _tmp1_, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), info.name = (g_free (info.name), NULL), _tmp3_);
			info.id = rec_id;
			info.length = dvb_recording_get_Length (rec);
			_tmp4_ = NULL;
			if (dvb_recording_get_Description (rec) == NULL) {
				_tmp4_ = "";
			} else {
				_tmp4_ = dvb_recording_get_Description (rec);
			}
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			info.description = (_tmp6_ = (_tmp5_ = _tmp4_, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_)), info.description = (g_free (info.description), NULL), _tmp6_);
			_tmp7_ = NULL;
			info.location = (_tmp7_ = g_file_get_path (dvb_recording_get_Location (rec)), info.location = (g_free (info.location), NULL), _tmp7_);
			info.start_timestamp = (gint64) mktime ((_tmp9_ = (dvb_recording_get_StartTime (rec, &_tmp8_), _tmp8_), &_tmp9_));
			_tmp11_ = NULL;
			_tmp10_ = NULL;
			info.channel = (_tmp11_ = (_tmp10_ = dvb_recording_get_ChannelName (rec), (_tmp10_ == NULL) ? NULL : g_strdup (_tmp10_)), info.channel = (g_free (info.channel), NULL), _tmp11_);
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = info;
	return result;
}


/**
         * @recordingsbasedir: The directory to search
         *
         * Searches recursively in the given directory
         * for "info.rec" files, restores a new Recording
         * from that file and adds it to itsself.
         */
void dvb_recordings_store_restore_from_dir (DVBRecordingsStore* self, GFile* recordingsbasedir) {
	GError * _inner_error_;
	char* attrs;
	GFileInfo* info;
	GFileEnumerator* files;
	g_return_if_fail (self != NULL);
	g_return_if_fail (recordingsbasedir != NULL);
	_inner_error_ = NULL;
	if (!g_file_query_exists (recordingsbasedir, NULL)) {
		char* _tmp0_;
		_tmp0_ = NULL;
		g_debug ("RecordingsStore.vala:293: Directory %s does not exist", _tmp0_ = g_file_get_path (recordingsbasedir));
		_tmp0_ = (g_free (_tmp0_), NULL);
		return;
	}
	attrs = g_strdup_printf ("%s,%s,%s", G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_ATTRIBUTE_ACCESS_CAN_READ, G_FILE_ATTRIBUTE_STANDARD_NAME);
	info = NULL;
	{
		GFileInfo* _tmp1_;
		GFileInfo* _tmp2_;
		_tmp1_ = g_file_query_info (recordingsbasedir, attrs, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch15_g_error;
			goto __finally15;
		}
		_tmp2_ = NULL;
		info = (_tmp2_ = _tmp1_, (info == NULL) ? NULL : (info = (g_object_unref (info), NULL)), _tmp2_);
	}
	goto __finally15;
	__catch15_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("RecordingsStore.vala:303: Could not retrieve attributes: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			return;
		}
	}
	__finally15:
	if (_inner_error_ != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	if (g_file_info_get_file_type (info) != G_FILE_TYPE_DIRECTORY) {
		char* _tmp3_;
		_tmp3_ = NULL;
		g_critical ("RecordingsStore.vala:308: %s is not a directory", _tmp3_ = g_file_get_path (recordingsbasedir));
		_tmp3_ = (g_free (_tmp3_), NULL);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return;
	}
	if (!g_file_info_get_attribute_boolean (info, G_FILE_ATTRIBUTE_ACCESS_CAN_READ)) {
		char* _tmp4_;
		_tmp4_ = NULL;
		g_critical ("RecordingsStore.vala:313: Cannot read %s", _tmp4_ = g_file_get_path (recordingsbasedir));
		_tmp4_ = (g_free (_tmp4_), NULL);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return;
	}
	files = NULL;
	{
		GFileEnumerator* _tmp5_;
		GFileEnumerator* _tmp6_;
		_tmp5_ = g_file_enumerate_children (recordingsbasedir, attrs, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch16_g_error;
			goto __finally16;
		}
		_tmp6_ = NULL;
		files = (_tmp6_ = _tmp5_, (files == NULL) ? NULL : (files = (g_object_unref (files), NULL)), _tmp6_);
	}
	goto __finally16;
	__catch16_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("RecordingsStore.vala:322: Could not read directory: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
			return;
		}
	}
	__finally16:
	if (_inner_error_ != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	{
		GFileInfo* childinfo;
		childinfo = NULL;
		while (TRUE) {
			GFileInfo* _tmp7_;
			GFileInfo* _tmp8_;
			guint32 type;
			GFile* child;
			_tmp7_ = g_file_enumerator_next_file (files, NULL, &_inner_error_);
			if (_inner_error_ != NULL) {
				(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
				goto __catch17_g_error;
				goto __finally17;
			}
			_tmp8_ = NULL;
			if (!((childinfo = (_tmp8_ = _tmp7_, (childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL)), _tmp8_)) != NULL)) {
				break;
			}
			type = g_file_info_get_attribute_uint32 (childinfo, G_FILE_ATTRIBUTE_STANDARD_TYPE);
			child = g_file_get_child (recordingsbasedir, g_file_info_get_name (childinfo));
			switch (type) {
				case G_FILE_TYPE_DIRECTORY:
				{
					dvb_recordings_store_restore_from_dir (self, child);
					break;
				}
				case G_FILE_TYPE_REGULAR:
				{
					if (_vala_strcmp0 (g_file_info_get_name (childinfo), "info.rec") == 0) {
						DVBRecording* rec;
						rec = NULL;
						{
							DVBRecording* _tmp9_;
							DVBRecording* _tmp10_;
							_tmp9_ = dvb_recording_deserialize (child, &_inner_error_);
							if (_inner_error_ != NULL) {
								goto __catch19_g_error;
								goto __finally19;
							}
							_tmp10_ = NULL;
							rec = (_tmp10_ = _tmp9_, (rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL)), _tmp10_);
						}
						goto __finally19;
						__catch19_g_error:
						{
							GError * e;
							e = _inner_error_;
							_inner_error_ = NULL;
							{
								g_critical ("RecordingsStore.vala:346: Could not deserialize recording: %s", e->message);
								(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
							}
						}
						__finally19:
						if (_inner_error_ != NULL) {
							(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
							(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
							(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
							goto __catch17_g_error;
							goto __finally17;
						}
						if (rec != NULL) {
							char* _tmp11_;
							_tmp11_ = NULL;
							g_debug ("RecordingsStore.vala:351: Restored recording from %s", _tmp11_ = g_file_get_path (child));
							_tmp11_ = (g_free (_tmp11_), NULL);
							g_static_rec_mutex_lock (&self->priv->__lock_recordings);
							{
								dvb_recordings_store_add (self, rec, TRUE);
							}
							g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
							g_static_rec_mutex_lock (&self->priv->__lock_last_id);
							{
								if (dvb_recording_get_Id (rec) > self->priv->last_id) {
									self->priv->last_id = dvb_recording_get_Id (rec);
								}
							}
							g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
						}
						(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
					}
					break;
				}
			}
			(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
		}
		(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
	}
	goto __finally17;
	__catch17_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("RecordingsStore.vala:368: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally17:
	{
		{
			g_file_enumerator_close (files, NULL, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch18_g_error;
				goto __finally18;
			}
		}
		goto __finally18;
		__catch18_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("RecordingsStore.vala:373: Could not close file: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally18:
		if (_inner_error_ != NULL) {
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
			goto __finally17;
		}
	}
	if (_inner_error_ != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	attrs = (g_free (attrs), NULL);
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
	(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
}


/**
         * @location: Path to the .ts file of the recording
         * @returns: TRUE on success
         *
         * Delete a recording by the path of the recording
         */
static gboolean dvb_recordings_store_delete_recording_by_location (DVBRecordingsStore* self, const char* location) {
	gboolean result;
	gboolean _result_;
	guint32 rec_id;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (location != NULL, FALSE);
	_result_ = FALSE;
	rec_id = (guint32) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->recordings))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint32 id;
				DVBRecording* rec;
				char* _tmp2_;
				gboolean _tmp3_;
				if (!gee_iterator_next (_id_it)) {
					break;
				}
				id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
				rec = (DVBRecording*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (id));
				_tmp2_ = NULL;
				if ((_tmp3_ = _vala_strcmp0 (_tmp2_ = g_file_get_path (dvb_recording_get_Location (rec)), location) == 0, _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_)) {
					rec_id = id;
					(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
					break;
				}
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
			}
			(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
		}
		if (rec_id != 0) {
			g_debug ("RecordingsStore.vala:397: Deleting recording %u", (guint) rec_id);
			gee_abstract_map_remove ((GeeAbstractMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id), NULL);
			g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", rec_id, (guint) DVB_CHANGE_TYPE_DELETED);
			_result_ = TRUE;
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	result = _result_;
	return result;
}


static void dvb_recordings_store_on_recording_file_changed (DVBRecordingsStore* self, GFileMonitor* monitor, GFile* file, GFile* other_file, GFileMonitorEvent event) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (monitor != NULL);
	g_return_if_fail (file != NULL);
	if (event == G_FILE_MONITOR_EVENT_DELETED) {
		char* location;
		location = g_file_get_path (file);
		g_debug ("RecordingsStore.vala:411: %s has been deleted", location);
		dvb_recordings_store_delete_recording_by_location (self, location);
		g_file_monitor_cancel (monitor);
		location = (g_free (location), NULL);
	}
}


/**
     * This class manages the recordings off all devices
     */
DVBRecordingsStore* dvb_recordings_store_construct (GType object_type) {
	DVBRecordingsStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBRecordingsStore* dvb_recordings_store_new (void) {
	return dvb_recordings_store_construct (DVB_TYPE_RECORDINGS_STORE);
}


static GObject * dvb_recordings_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBRecordingsStoreClass * klass;
	GObjectClass * parent_class;
	DVBRecordingsStore * self;
	klass = DVB_RECORDINGS_STORE_CLASS (g_type_class_peek (DVB_TYPE_RECORDINGS_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_RECORDINGS_STORE (obj);
	{
		GeeHashMap* _tmp1_;
		_tmp1_ = NULL;
		self->priv->recordings = (_tmp1_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_RECORDING, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL)), _tmp1_);
		self->priv->last_id = (guint32) 0;
	}
	return obj;
}


void _dvb_recordings_store_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_recordings_store_introspect (DVBRecordingsStore* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.RecordingsStore\">\n  <method name=\"GetRecordings\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLocation\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetName\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDescription\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetStartTime\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetStartTimestamp\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"x\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLength\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"x\" direction=\"out\"/>\n  </method>\n  <method name=\"Delete\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelName\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetAllInformations\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"(ussxxss)\" direction=\"out\"/>\n  </method>\n  <signal name=\"Changed\">\n    <arg name=\"rec_id\" type=\"u\"/>\n    <arg name=\"type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_recordings_store_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_recordings_store_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_recordings_store_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_recordings_store_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_recordings_store_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_recordings_store_dbus_register_object (connection, path, object);
}


static void dvb_recordings_store_class_init (DVBRecordingsStoreClass * klass) {
	GStaticRecMutex _tmp0_ = {0};
	dvb_recordings_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBRecordingsStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_recordings_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_recordings_store_finalize;
	g_type_set_qdata (DVB_TYPE_RECORDINGS_STORE, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_recordings_store_dbus_vtable));
	dvb_recordings_store_instance_mutex = (g_static_rec_mutex_init (&_tmp0_), _tmp0_);
}


static void dvb_recordings_store_dvb_id_bus_recordings_store_interface_init (DVBIDBusRecordingsStoreIface * iface) {
	dvb_recordings_store_dvb_id_bus_recordings_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetRecordings = dvb_recordings_store_real_GetRecordings;
	iface->GetLocation = dvb_recordings_store_real_GetLocation;
	iface->GetName = dvb_recordings_store_real_GetName;
	iface->GetDescription = dvb_recordings_store_real_GetDescription;
	iface->GetStartTime = dvb_recordings_store_real_GetStartTime;
	iface->GetStartTimestamp = dvb_recordings_store_real_GetStartTimestamp;
	iface->GetLength = dvb_recordings_store_real_GetLength;
	iface->Delete = dvb_recordings_store_real_Delete;
	iface->GetChannelName = dvb_recordings_store_real_GetChannelName;
	iface->GetAllInformations = dvb_recordings_store_real_GetAllInformations;
}


static void dvb_recordings_store_instance_init (DVBRecordingsStore * self) {
	self->priv = DVB_RECORDINGS_STORE_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_recordings);
	g_static_rec_mutex_init (&self->priv->__lock_last_id);
}


static void dvb_recordings_store_finalize (GObject* obj) {
	DVBRecordingsStore * self;
	self = DVB_RECORDINGS_STORE (obj);
	g_static_rec_mutex_free (&self->priv->__lock_recordings);
	(self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_last_id);
	G_OBJECT_CLASS (dvb_recordings_store_parent_class)->finalize (obj);
}


GType dvb_recordings_store_get_type (void) {
	static GType dvb_recordings_store_type_id = 0;
	if (dvb_recordings_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBRecordingsStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_recordings_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBRecordingsStore), 0, (GInstanceInitFunc) dvb_recordings_store_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_recordings_store_info = { (GInterfaceInitFunc) dvb_recordings_store_dvb_id_bus_recordings_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_recordings_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBRecordingsStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_recordings_store_type_id, DVB_TYPE_ID_BUS_RECORDINGS_STORE, &dvb_id_bus_recordings_store_info);
	}
	return dvb_recordings_store_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




