/*
 * file-selector.c: implementation of GNOME::FileSelector
 *
 * Authors:
 *    Jacob Berkman  <jacob@ximian.com>
 *    Miguel de Icaza  <miguel@ximian.com>
 *
 * Copyright 2000, 2001 Ximian, Inc.
 */

#include <config.h>

#include "file-selector.h"

#include "widgets/flist-mime-icon.h"
#include "widgets/property-bag-item-handler.h"

#include "GNOME_FileList.h"

#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-item-handler.h>
#include <bonobo/bonobo-moniker-util.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <bonobo/bonobo-widget.h>

#include <gal/e-text/e-completion.h>
#include <gal/e-text/e-entry.h>

#include <gal/shortcut-bar/e-shortcut-bar.h>
#include <gal/util/e-util.h>

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include <glade/glade.h>

#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtksignal.h>

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>

#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-uidefs.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>

#define d(x)

static GtkObjectClass *parent_class;

enum {
	PROP_ACCEPT_DIRS,
	PROP_APPLICATION,
	PROP_ACTION_LABEL,
	PROP_CURRENT_URI,
	PROP_DEFAULT_FILE_NAME,
	PROP_DEFAULT_VIEW,
	PROP_LOCAL_URIS_ONLY,
	PROP_MIME_TYPES,
	PROP_MIME_TYPES_SEQUENCE,
	PROP_MULTIPLE_SELECTION,
	PROP_LAST
};

static const char *prop_name[] = {
	"AcceptDirectories",
	"Application",
	"ActionButtonLabel",
	"CurrentWorkingURI",
	"DefaultFileName",
	"DefaultView",
	"LocalURIsOnly",
	"MimeTypes",
	"MimeTypesSequence",
	"MultipleSelection"
};

static const char *prop_desc[] = {
	N_("Calling application can accept directories"),
	N_("Name of calling application"),
	N_("Label for the action button: typically Open or Save"),
	N_("URI currently displayed in the file list"),
	N_("File name to default to when changing directories"),
	N_("Default view mode"),
	N_("Restrict files to local URIs only"),
	N_("List of mime types calling application can accept"),
	N_("List of mime types calling application can accept"),
	N_("Calling application can accept multiple URIs"),
};

typedef Bonobo_EventSource_ListenerId ListenerId;

typedef struct {
	Bonobo_EventSource es;
	ListenerId         id;
} EventKit;

struct _FileSelctorPrivate {
	BonoboPropertyBag *properties;

	BonoboListener *flist_listener;
	BonoboListener *entry_listener;

	EventKit flist_kit[2];
	EventKit entry_kit;

	/* "us" */
	BonoboControl     *control;
	BonoboEventSource *event_source;

	GnomeVFSURI       *current_uri;
	GnomeVFSURI       *requested_uri;

	GnomeVFSURI       *home_uri;
	GnomeVFSURI       *desktop_uri;
	GnomeVFSURI       *docs_uri;
	
	GtkWidget         *dialog;

	GNOME_FileList     flist;

	GtkWidget *path_tree;
	GtkWidget *back;
	GtkWidget *up;
	GtkWidget *mkdir;
	GtkWidget *view_container;	

	GtkWidget *shortcut_bar;
	GtkWidget *file_list;

	GtkWidget *name_entry;
	GtkWidget *type_combo;

	GtkWidget *action_button;
	GtkWidget *cancel_button;

	GList *parent_uris;
	GList *favorite_uris;
	GList *recent_uris;

	char **mime_types;

	gboolean last_was_flist;

	gboolean accept_dirs;
	char *app_name;
	GtkWidget *action_label;
	char *default_name;
	char *default_view;

	gboolean entry_is_active;
	gboolean flist_is_active;
};

/*
 * Prototypes required because libglade will call us back, so they can not
 * be made static.
 */
GtkWidget *custom_shortcut_bar_new    (char *name, gchar *string1, gchar *string2, gint int1, gint int2);
GtkWidget *custom_flist_new           (char *name, gchar *string1, gchar *string2, gint int1, gint int2);

typedef enum {
	SHORTCUT_DESKTOP,
	SHORTCUT_DOCUMENTS,
	SHORTCUT_HOME,
	SHORTCUT_ROOT,
	SHORTCUT_HISTORY,
	SHORTCUT_NETWORK,
	SHORTCUT_LAST
} ShortcutType;

static struct {
	char *label;
	char *filename;
	ShortcutType type;
	char *url;
	GdkPixbuf *image;
} icons [] = {
	{ N_("Desktop"),        "my_desktop.png",               SHORTCUT_DESKTOP },
	{ N_("Documents"),      "my_documents.png",             SHORTCUT_DOCUMENTS },
	{ N_("Home"),           "my_home.png",                  SHORTCUT_HOME },
	{ N_("Computer"),       "my_computer_alt.png",          SHORTCUT_ROOT },
	{ NULL },
	{ N_("History"),        "my_documents.png",             SHORTCUT_HISTORY },
	{ N_("Network places"), "network.png",                  SHORTCUT_NETWORK },
};

static GdkPixbuf *
get_icon (EShortcutBar *shortcut_bar, const char *url, gpointer data)
{
	int i;

	/* Yea... this could be improved a bit */
	for (i = 0; icons[i].url; i++) {
		if (!strcmp (icons[i].url, url)) {
			return icons[i].image;
		}
	}
	
	return NULL;
}

GtkWidget *
custom_shortcut_bar_new (char *name, gchar *string1, gchar *string2, gint int1, gint int2)
{
	EShortcutModel *model;
	EShortcutBar   *bar;
	int i;

	model = e_shortcut_model_new ();
	e_shortcut_model_add_group (model, 0, _("My Shortcuts"));

	for (i = 0; icons[i].label; i++) {
		if (!icons[i].image) {
			GdkPixbuf *pix;
			char *fname = g_strconcat (IMAGEDIR, "/", icons[i].filename, NULL);

			pix = gdk_pixbuf_new_from_file (fname);
			g_free (fname);

			if (!pix)
				continue;
			
			icons[i].image = pix;
		}
		if (!icons[i].url) {
			switch (icons[i].type) {
			case SHORTCUT_DESKTOP:
				icons[i].url = gnome_util_prepend_user_home (".gnome-desktop");
				break;
			case SHORTCUT_DOCUMENTS:
				icons[i].url = gnome_util_prepend_user_home ("Documents");
				break;
			case SHORTCUT_HOME:
				icons[i].url = g_get_home_dir ();
				break;
			case SHORTCUT_ROOT:
				icons[i].url = "/";
				break;
			default:
				break;
			}
		}
		
		e_shortcut_model_add_item (model, 0, i, icons[i].url, _(icons[i].label));
	}

	bar = E_SHORTCUT_BAR (e_shortcut_bar_new ());
	e_shortcut_bar_set_icon_callback (bar, get_icon, NULL);
	e_shortcut_bar_set_model (bar, model);

	return GTK_WIDGET (bar);
}

GtkWidget *
custom_flist_new (char *name, gchar *string1, gchar *string2, gint int1, gint int2)
{
	return bonobo_widget_new_control ("OAFIID:GNOME_FileList", CORBA_OBJECT_NIL);
}

static void
filter_item_cb (GtkWidget *w, const char *mime_type)
{
	FileSelector *fs = gtk_object_get_user_data (GTK_OBJECT (w));

	g_print ("setting filter: %s\n", mime_type ? mime_type : _("All Files"));

	bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->file_list),
				    prop_name[PROP_MIME_TYPES],
				    mime_type,
				    NULL);
}

static void
add_mime_type (GtkMenu *menu, const char *description, const char *mime_type)
{
	GtkWidget *item;	
	const char *mime_desc = NULL;

	if (description && strlen (description))
		mime_desc = description;
	else if (mime_type && strlen (mime_type))
		mime_desc = gnome_vfs_mime_get_description (mime_type);
	else
		mime_desc = _("All Files");
	
	if (!mime_desc)
		mime_desc = _("Unknown Files");

	item = gtk_menu_item_new_with_label (mime_desc);
	gtk_object_set_user_data (GTK_OBJECT (item),
				  gtk_object_get_user_data (GTK_OBJECT (menu)));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (filter_item_cb),
			    (char *)mime_type);

	gtk_widget_show (item);
	gtk_menu_append (menu, item);
}

static void
fixup_type_combo (FileSelector *fs)
{
	GtkWidget *menu;
	int i;

	menu = gtk_menu_new ();
	gtk_object_set_user_data (GTK_OBJECT (menu), fs);
	
	if (fs->priv->mime_types) {
		for (i = 0; fs->priv->mime_types[i] && fs->priv->mime_types[i+1]; i+=2) {
			add_mime_type (GTK_MENU (menu),
				       fs->priv->mime_types[i],
				       fs->priv->mime_types[i+1]);
		}
	}
	add_mime_type (GTK_MENU (menu), NULL, NULL);

	gtk_widget_show (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (fs->priv->type_combo),
				  menu);
}

static void 
fs_set_prop (BonoboPropertyBag *bag,
	     const BonoboArg *arg,
	     guint arg_id,
	     CORBA_Environment *ev,
	     gpointer user_data)
{
	FileSelector *fs = FILE_SELECTOR (user_data);

	switch (arg_id) {
	case PROP_ACCEPT_DIRS:
		bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_ACCEPT_DIRS],
					    BONOBO_ARG_GET_BOOLEAN (arg),
					    NULL);
		break;
	case PROP_APPLICATION:
		g_free (fs->priv->app_name);
		fs->priv->app_name = g_strdup (BONOBO_ARG_GET_STRING (arg));
		/* FIXME: do something */
		break;
	case PROP_ACTION_LABEL:
		gtk_label_set_text (GTK_LABEL (fs->priv->action_label),
				    BONOBO_ARG_GET_STRING (arg));
		break;
	case PROP_DEFAULT_FILE_NAME:
		g_free (fs->priv->default_name);
		fs->priv->default_name = g_strdup (BONOBO_ARG_GET_STRING (arg));
		/* FIXME: do something */
		break;
	case PROP_DEFAULT_VIEW:
		g_free (fs->priv->default_view);
		fs->priv->default_view = g_strdup (BONOBO_ARG_GET_STRING (arg));
		/* FIXME: do something */
		break;
	case PROP_LOCAL_URIS_ONLY:
		bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_LOCAL_URIS_ONLY],
					    BONOBO_ARG_GET_BOOLEAN (arg),
					    NULL);
		break;
	case PROP_MIME_TYPES:
		if (fs->priv->mime_types)
			g_strfreev (fs->priv->mime_types);
		g_print ("setting mime types: %s\n", BONOBO_ARG_GET_STRING (arg));
		fs->priv->mime_types = g_strsplit (BONOBO_ARG_GET_STRING (arg),
						   ":", -1);
		fixup_type_combo (fs);
		break;
	case PROP_MIME_TYPES_SEQUENCE: {
		CORBA_sequence_CORBA_string *seq;

		if (fs->priv->mime_types)
			g_strfreev (fs->priv->mime_types);

		seq = arg->_value;
		g_print ("got sequence of length %d\n", seq->_length);
		/* FIXME: finish */
		break;
	}
	case PROP_MULTIPLE_SELECTION:
		bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_MULTIPLE_SELECTION],
					    BONOBO_ARG_GET_BOOLEAN (arg),
					    NULL);
		break;
	default:
		/* until bonobo sets this exception */
		if (arg_id >= 0 && arg_id < PROP_LAST)
			bonobo_exception_set (ev, ex_Bonobo_Property_ReadOnlyProperty);
		else
			g_assert_not_reached ();
		break;
	}
}

static void
fs_get_prop (BonoboPropertyBag *bag,
	     BonoboArg *arg,
	     guint arg_id,
	     CORBA_Environment *ev,
	     gpointer user_data)
{
	FileSelector *fs = FILE_SELECTOR (user_data);
	gboolean b;
	char *s;

	switch (arg_id) {
	case PROP_ACCEPT_DIRS:
		bonobo_widget_get_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_ACCEPT_DIRS],
					    &b, NULL);
		BONOBO_ARG_SET_BOOLEAN (arg, b);
		break;
	case PROP_APPLICATION:
		BONOBO_ARG_SET_STRING (arg, fs->priv->app_name);
		break;
	case PROP_ACTION_LABEL:
		BONOBO_ARG_SET_STRING (arg, GTK_LABEL (fs->priv->action_label)->label);
		break;
	case PROP_CURRENT_URI:
		bonobo_widget_get_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_CURRENT_URI],
					    &s, NULL);
		BONOBO_ARG_SET_STRING (arg, s);
		break;
	case PROP_DEFAULT_FILE_NAME:
		BONOBO_ARG_SET_STRING (arg, fs->priv->default_name);
		break;
	case PROP_DEFAULT_VIEW:
		BONOBO_ARG_SET_STRING (arg, fs->priv->default_view);
		break;
	case PROP_LOCAL_URIS_ONLY:
		bonobo_widget_get_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_LOCAL_URIS_ONLY],
					    &b, NULL);
		BONOBO_ARG_SET_BOOLEAN (arg, b);
		break;
	case PROP_MIME_TYPES:
		s = fs->priv->mime_types
			? g_strjoinv (":", fs->priv->mime_types)
			: NULL;
		BONOBO_ARG_SET_STRING (arg, s);
		g_free (s);
		break;
	case PROP_MIME_TYPES_SEQUENCE: {
		CORBA_sequence_CORBA_string *seq;
		int len;

		len = fs->priv->mime_types ? sizeof (fs->priv->mime_types) / sizeof (char *) : 0;
		g_print ("we have %d mime types\n", len);

		seq = CORBA_sequence_CORBA_string__alloc ();
		seq->_length = len;
		if (len) {
			int i;
			seq->_buffer = CORBA_sequence_CORBA_string_allocbuf (len);
			for (i = 0; i < len; i++)
				seq->_buffer[i] = CORBA_string_dup (fs->priv->mime_types[i]);
		}
		arg->_value = seq;
		break;
	}
	case PROP_MULTIPLE_SELECTION:
		bonobo_widget_get_property (BONOBO_WIDGET (fs->priv->file_list),
					    prop_name[PROP_MULTIPLE_SELECTION],
					    &b, NULL);
		BONOBO_ARG_SET_BOOLEAN (arg, b);
		break;
	default:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
		break;
	}
}

static void
file_selector_request_uri (FileSelector *fs, const char *uri, 
			   GNOME_FileList_RequestType rtype)
{
	CORBA_Environment ev;

	CORBA_exception_init (&ev);
	GNOME_FileList_requestURI (fs->priv->flist, uri, rtype, &ev);
	CORBA_exception_free (&ev);
}

static void
menu_activate_uri (GtkWidget *w, GnomeVFSURI *uri)
{
	FileSelector *fs;	
	char *uri_str;

	fs = gtk_object_get_user_data (GTK_OBJECT (w));
	uri_str = gnome_vfs_uri_to_string (uri, GNOME_VFS_URI_HIDE_NONE);
	g_print ("menu: got: %s\n", uri_str);

	file_selector_request_uri (fs, uri_str, GNOME_FileList_RequestAbsolut);
	
	g_free (uri_str);
}

static void
add_uri_to_menu_full (GtkMenu *menu, const char *label, GnomeVFSURI *uri)
{
	GtkWidget *hbox;
	GtkWidget *w;
	char *path, *file;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	path = gnome_vfs_uri_to_string (uri, GNOME_VFS_URI_HIDE_NONE);
	file = flist_mime_icon_get_file (path, GNOME_VFS_FILE_TYPE_DIRECTORY, NULL);
	g_free (path);

	if (file) {
		w = gnome_pixmap_new_from_file_at_size (file, 20, 20);
		gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
		g_free (file);
	}

	w = gtk_label_new (label);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);

	gtk_widget_show_all (hbox);

	w = gtk_menu_item_new ();
	gtk_container_add (GTK_CONTAINER (w), hbox);

	gtk_object_set_user_data (GTK_OBJECT (w),
				  gtk_object_get_user_data (GTK_OBJECT (menu)));
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC (menu_activate_uri),
			    uri);
	gtk_menu_append (menu, w);	
}

/* todo: get a pixmap based on the uri or something */
static void
add_uri_to_menu (GnomeVFSURI *uri, GtkMenu *menu)
{
	char *name;

	name = gnome_vfs_uri_extract_short_name (uri);
	add_uri_to_menu_full (menu, name, uri);
	g_free (name);
}

static void
add_dummy_to_menu (GtkWidget *w, const char *label)
{
	GtkWidget *menuitem;
	
	menuitem = label 
		? gtk_menu_item_new_with_label (label) 
		: gtk_menu_item_new ();
	gtk_widget_set_sensitive (menuitem, FALSE);
	gtk_widget_show (menuitem);
	gtk_menu_append (GTK_MENU (w), menuitem);
}

static void
free_uri_list (GList **list)
{
	gnome_vfs_uri_list_unref (*list);
	g_list_free (*list);
	*list = NULL;
}

static void
update_path_tree (FileSelector *fs, const char *uri)
{
	GnomeVFSURI *vfs_uri;
	GtkWidget *menu;

	free_uri_list (&fs->priv->parent_uris);

	gnome_vfs_uri_ref (fs->priv->current_uri);
	vfs_uri = fs->priv->current_uri;

	do {
		fs->priv->parent_uris = g_list_append (fs->priv->parent_uris,
						       vfs_uri);
		vfs_uri = gnome_vfs_uri_get_parent (vfs_uri);
	} while (vfs_uri);

	menu = gtk_menu_new ();
	gtk_object_set_user_data (GTK_OBJECT (menu), fs);
	g_list_foreach (fs->priv->parent_uris, (GFunc)add_uri_to_menu,  menu);

	add_dummy_to_menu (menu, NULL);
	/* these need custom names... */
	add_uri_to_menu_full (GTK_MENU (menu), _("Desktop"),   fs->priv->desktop_uri);
	add_uri_to_menu_full (GTK_MENU (menu), _("Documents"), fs->priv->docs_uri);
	add_uri_to_menu_full (GTK_MENU (menu), _("Home"),      fs->priv->home_uri);

	add_dummy_to_menu (menu, NULL);
	add_dummy_to_menu (menu, _("Favorites"));
	g_list_foreach (fs->priv->favorite_uris, (GFunc)add_uri_to_menu, menu);

	add_dummy_to_menu (menu, NULL);
	add_dummy_to_menu (menu, _("Recent"));
	g_list_foreach (fs->priv->recent_uris, (GFunc)add_uri_to_menu, menu);

	gtk_widget_show_all (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (fs->priv->path_tree), menu);
}

static void
update_action_enabled (FileSelector *fs)
{
	gtk_widget_set_sensitive (fs->priv->action_button,
				  fs->priv->last_was_flist
				  ? fs->priv->flist_is_active
				  : fs->priv->entry_is_active);
}

static void
flist_listener_cb (BonoboListener *listener,
	     char *event_name,
	     CORBA_any *any,
	     CORBA_Environment *ev,
	     FileSelector *fs)
{
	char *uri_str;
	char *type, *subtype;

	g_return_if_fail (fs != NULL);

	type    = bonobo_event_type (event_name);
	subtype = bonobo_event_subtype (event_name);

	if (!strcmp (type, "Bonobo/Property:change")) {
		gboolean proxy_event = FALSE;

		if (!strcmp (subtype, prop_name[PROP_CURRENT_URI])) {
			uri_str = BONOBO_ARG_GET_STRING (any);
			
			if (fs->priv->current_uri)
				gnome_vfs_uri_unref (fs->priv->current_uri);
			
			fs->priv->current_uri = gnome_vfs_uri_new (uri_str);
			gtk_widget_set_sensitive (fs->priv->up, 
						  gnome_vfs_uri_has_parent (fs->priv->current_uri));
			
			update_path_tree (fs, uri_str);
			bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->name_entry),
						    "Text", "", NULL);
			proxy_event = TRUE;
		} else if (!strcmp (subtype, "SelectedURICount")) {
			fs->priv->flist_is_active = BONOBO_ARG_GET_INT (any) > 0;
			update_action_enabled (fs);
		} else if (!strcmp (subtype, prop_name[PROP_ACCEPT_DIRS])) {
			fs->priv->accept_dirs = BONOBO_ARG_GET_BOOLEAN (any);
			proxy_event = TRUE;
		} else if (!strcmp (subtype, prop_name[PROP_MULTIPLE_SELECTION]) ||
			   !strcmp (subtype, prop_name[PROP_LOCAL_URIS_ONLY]))
			proxy_event = TRUE;

		if (proxy_event) {
			g_message ("proxying: %s", subtype);
			bonobo_property_bag_notify_listeners (
				fs->priv->properties,
				subtype, any, NULL);
		}
		
	} else if (!strcmp (type, "GNOME/FileList:URIsActivated")) {
		bonobo_event_source_notify_listeners_full (fs->priv->event_source,
							   "GNOME/FileSelector",
							   "ButtonClicked",
							   "Action",
							   any, NULL);
	} else if (!strcmp (type, "GNOME/FileList:Error")) {
		GtkWidget *d;

		d = gnome_error_dialog (BONOBO_ARG_GET_STRING (any));
		gnome_dialog_run_and_close (GNOME_DIALOG (d));
	} else {
		g_warning ("unknown event: %s\n", event_name);
	}

	g_free (type);
	g_free (subtype);
}

static void
entry_listener_cb (BonoboListener *listener,
	     char *event_name,
	     CORBA_any *any,
	     CORBA_Environment *ev,
	     FileSelector *fs)
{
	char *type, *subtype;

	g_return_if_fail (fs != NULL);

	type    = bonobo_event_type (event_name);
	subtype = bonobo_event_subtype (event_name);

	if (strcmp (type, "Bonobo/Property:change"))
		return;

	if (strcmp (subtype, "Text"))
		return;

	fs->priv->entry_is_active = strlen (BONOBO_ARG_GET_STRING (any)) > 0;
	update_action_enabled (fs);
}

/* this can't be a custom widget since we need to have a handle to the
 * flist */
static void
create_view_combo (GNOME_FileList flist, GladeXML *ui)
{
	GtkWidget *w, *w2;
	Bonobo_Control control;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);
	control = GNOME_FileList_getTypeSelectorControl (flist, &ev);
	CORBA_exception_free (&ev);

	w = glade_xml_get_widget (ui, "flist-view");
	w2 = bonobo_widget_new_control_from_objref (control, CORBA_OBJECT_NIL);
	gtk_widget_show (w2);
	gtk_container_add (GTK_CONTAINER (w), w2);
}

static GtkWidget *
create_name_entry (GNOME_FileList flist, GladeXML *ui)
{
	GtkWidget *w, *w2;
	Bonobo_Control control;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);
	control = GNOME_FileList_getFileCompletionControl (flist, &ev);
	CORBA_exception_free (&ev);

	w = glade_xml_get_widget (ui, "flist-completion");
	w2 = bonobo_widget_new_control_from_objref (control, CORBA_OBJECT_NIL);
	gtk_widget_show (w2);
	gtk_container_add (GTK_CONTAINER (w), w2);

	return w2;
}

static void
on_up_clicked (GtkWidget *w, FileSelector *fs)
{
#if 1
	file_selector_request_uri (fs, "..", GNOME_FileList_RequestRelative);
#else
	GnomeVFSURI *uri;
	
	char *uri_str;

	g_return_if_fail (gnome_vfs_uri_has_parent (fs->priv->current_uri));

	uri = gnome_vfs_uri_get_parent (fs->priv->current_uri);
	uri_str = gnome_vfs_uri_to_string (uri, GNOME_VFS_URI_HIDE_NONE);
	gnome_vfs_uri_unref (uri);
	
	bonobo_widget_set_property (BONOBO_WIDGET (fs->priv->file_list),
				    prop_name[PROP_REQUESTED_URI],
				    uri_str, NULL);

	g_free (uri_str);
#endif
}

static void
on_shorctut_bar_item_selected (EShortcutBar *bar, GdkEvent *event, 
			       gint group, gint item, FileSelector *fs)
{
	char *uri;

	if (item >= e_shortcut_model_get_num_items (bar->model, group))
		return;

	e_shortcut_model_get_item_info (bar->model, group, item, &uri, NULL);

	file_selector_request_uri (fs, uri, GNOME_FileList_RequestAbsolut);

	g_free (uri);
}

static void
on_action_clicked (GtkWidget *w, FileSelector *fs)
{
	CORBA_Environment ev;
	CORBA_sequence_CORBA_string *seq;
	CORBA_any any;
	
	/* FIXME:
	 *
	 * if AcceptDirectories is TRUE, then this is ok.
	 *
	 * Else, we need to:
	 *
	 *    1. if selected URI is a directory, request that URI from
	 *    the flist rather than do ButtonClicked:Action
	 *
	 *    2. remove all directory URIs.
	 *
	 * Better yet, we can just add a gboolean AcceptDirectories to
	 * getSelectedURIs() which will do all that for us, as the
	 * flist has the gnome-vfs info already!
	 *
	 */

	g_print ("last_was_flist = %d\n", fs->priv->last_was_flist);	

	if (fs->priv->last_was_flist) {
		CORBA_exception_init (&ev);
		seq = GNOME_FileList_getSelectedURIs (fs->priv->flist, &ev);
		CORBA_exception_free (&ev);
	} else {
		seq = CORBA_sequence_CORBA_string__alloc ();
		seq->_length = 1;
		seq->_buffer = CORBA_sequence_CORBA_string_allocbuf (1);
		seq->_release = 1;

		/* FIXME: should this really be CORBA_string_dup()'d ? */
		bonobo_widget_get_property (BONOBO_WIDGET (fs->priv->name_entry),
					    "Text", &seq->_buffer[0],
					    NULL);
	}

	any._type = TC_CORBA_sequence_CORBA_string;
	any._value = seq;
	any._release = FALSE;
	
	bonobo_event_source_notify_listeners_full (fs->priv->event_source,
						   "GNOME/FileSelector",
						   "ButtonClicked",
						   "Action",
						   &any, NULL);

	CORBA_free (seq);
}

static void
on_cancel_clicked (GtkWidget *w, FileSelector *fs)
{
	BonoboArg *arg;
	arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, "");
	bonobo_event_source_notify_listeners_full (fs->priv->event_source,
						   "GNOME/FileSelector",
						   "ButtonClicked",
						   "Cancel", arg, NULL);
	bonobo_arg_release (arg);
}

/* this is a pretty gross hack. */
static void
on_focus_child (GtkContainer *container, GtkWidget *widget, FileSelector *fs)
{
	GtkWidget *child, *grandchild;
	d(char *s);

	d(g_print ("on_focus_child(%p)   { %d, %p }\n", widget, fs->priv->last_was_flist, fs->priv->file_list));

	if (!widget)
		return;

	child      = GTK_IS_CONTAINER (widget) ? GTK_CONTAINER (widget)->focus_child : NULL;
	grandchild = GTK_IS_CONTAINER (child)  ? GTK_CONTAINER (child)->focus_child  : NULL;

	if (child == fs->priv->file_list)
		fs->priv->last_was_flist = TRUE;
	else if (grandchild != fs->priv->action_button)
		fs->priv->last_was_flist = FALSE;

	d(gtk_widget_path (widget, NULL, &s, NULL));

	d(g_print ("last_was_flist = %d (%s)\n", fs->priv->last_was_flist, s));
	d(g_free (s));

	update_action_enabled (fs);
}

static void
connect_ui_signals (FileSelector *fs)
{
	gtk_signal_connect (GTK_OBJECT (fs->priv->up), "clicked",
			    GTK_SIGNAL_FUNC (on_up_clicked), fs);
	gtk_signal_connect (GTK_OBJECT (fs->priv->shortcut_bar), "item_selected",
			    GTK_SIGNAL_FUNC (on_shorctut_bar_item_selected), fs);

	gtk_signal_connect (GTK_OBJECT (fs->priv->action_button), "clicked",
			    GTK_SIGNAL_FUNC (on_action_clicked), fs);
	gtk_signal_connect (GTK_OBJECT (fs->priv->cancel_button), "clicked",
			    GTK_SIGNAL_FUNC (on_cancel_clicked), fs);

	gtk_signal_connect (GTK_OBJECT (fs->priv->dialog), "set_focus_child",
			    GTK_SIGNAL_FUNC (on_focus_child), fs);
}

static void
file_selector_create_ui (FileSelector *fs)
{
	GladeXML *ui;
	BonoboObjectClient *object_client;

	ui = glade_xml_new (GLADEDIR "/selector-ui.glade", "toplevel");

#define W(v, n) (fs->priv->##v = glade_xml_get_widget (ui, (n)))
	W (dialog, "toplevel");
	W (path_tree, "tree-combo");
	W (back, "button-back");
	W (up, "button-up");
	W (mkdir, "button-mkdir");
	W (shortcut_bar, "iconbar");
	W (file_list, "flist");
	W (type_combo, "type-combo");
	W (action_button, "action-button");
	W (cancel_button, "cancel-button");
	W (action_label, "action-label");
#undef W

	/* we need this to be able to get the combo, and we can't do
	 * it before since we don't have the widget that we already
	 * created */
	object_client = bonobo_widget_get_server (BONOBO_WIDGET (fs->priv->file_list));
	fs->priv->flist = bonobo_object_client_query_interface (
		object_client, "IDL:GNOME/FileList:1.0", NULL);

	create_view_combo (fs->priv->flist, ui);
	fixup_type_combo  (fs);
	fs->priv->name_entry = create_name_entry (fs->priv->flist, ui);

	connect_ui_signals (fs);

	update_action_enabled (fs);

	gtk_widget_show (fs->priv->dialog);
	gtk_object_unref (GTK_OBJECT (ui));
}

#define ADD_PROP(prop, prop_type, flags) \
bonobo_property_bag_add (pb, prop_name[prop], prop, prop_type, NULL, _(prop_desc[prop]), flags)

#define RW BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE
#define RO BONOBO_PROPERTY_READABLE


static void
add_properties (FileSelector *fs)
{
	BonoboPropertyBag *pb;

	pb = bonobo_property_bag_new (fs_get_prop, fs_set_prop, fs);

	ADD_PROP (PROP_ACCEPT_DIRS,         BONOBO_ARG_BOOLEAN, RW);
	ADD_PROP (PROP_APPLICATION,         BONOBO_ARG_STRING,  RW);
	ADD_PROP (PROP_ACTION_LABEL,        BONOBO_ARG_STRING,  RW);
	ADD_PROP (PROP_CURRENT_URI,         BONOBO_ARG_STRING,  RO);
	ADD_PROP (PROP_DEFAULT_FILE_NAME,   BONOBO_ARG_STRING,  RW);
	ADD_PROP (PROP_DEFAULT_VIEW,        BONOBO_ARG_STRING,  RW);
	ADD_PROP (PROP_MIME_TYPES,          BONOBO_ARG_STRING,  RW);
	ADD_PROP (PROP_MIME_TYPES_SEQUENCE, TC_CORBA_sequence_CORBA_string, RW);
	ADD_PROP (PROP_MULTIPLE_SELECTION,  BONOBO_ARG_BOOLEAN, RW);
	ADD_PROP (PROP_LOCAL_URIS_ONLY,     BONOBO_ARG_BOOLEAN, RW);

	fs->priv->properties = pb;
	bonobo_control_set_properties (fs->priv->control, pb);
}

static void
add_cf_listener (GtkWidget *bw,
		 Bonobo_Unknown listener, 
		 EventKit *kit,
		 CORBA_Environment *ev)
{
	Bonobo_PropertyBag  pb;
	BonoboControlFrame *cf;

	cf = bonobo_widget_get_control_frame (BONOBO_WIDGET (bw));
	pb      = bonobo_control_frame_get_control_property_bag (cf, ev);
	kit->es = Bonobo_PropertyBag_getEventSource (pb, ev);
	kit->id = Bonobo_EventSource_addListener (kit->es, listener, ev);

	/* FIXME: why is this NULL? */
	bonobo_object_release_unref (pb, NULL);

	return;
}

static void
create_listeners (FileSelector *fs)
{
	Bonobo_Unknown listener;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	/* flist listener */
	fs->priv->flist_listener = bonobo_listener_new (
		(BonoboListenerCallbackFn)flist_listener_cb, fs);
	listener = BONOBO_OBJREF (fs->priv->flist_listener);

	/* flist property changes */
	add_cf_listener (fs->priv->file_list, listener, &fs->priv->flist_kit[0], &ev);

	/* flist events */
	fs->priv->flist_kit[1].es =
		Bonobo_Unknown_queryInterface (fs->priv->flist, 
					       "IDL:Bonobo/EventSource:1.0", 
					       &ev);
	fs->priv->flist_kit[1].id = 
		Bonobo_EventSource_addListener (fs->priv->flist_kit[1].es, 
						listener, &ev);

	/* entry listener */
	fs->priv->entry_listener = bonobo_listener_new (
		(BonoboListenerCallbackFn)entry_listener_cb, fs);
	listener = BONOBO_OBJREF (fs->priv->entry_listener);

	/* entry property changes */
	add_cf_listener (fs->priv->name_entry, listener, &fs->priv->entry_kit, &ev);	
	
	/* we actually don't need this as the entry talks to the
	 * flist */
#if 0
	es = Bonobo_Unknown_queryInterface (control, "IDL:Bonobo/EventSource:1.0", &ev);
	Bonobo_EventSource_addListener (es, listener, &ev);
	bonobo_object_release_unref (es, NULL);
#endif

	CORBA_exception_free (&ev);
}
static void
load_uri_lists (FileSelector *fs)
{
	char *s;
	/* load favorites / recent things here */

	fs->priv->home_uri = gnome_vfs_uri_new (g_get_home_dir ());

	s = gnome_util_prepend_user_home (".gnome-desktop");
	fs->priv->desktop_uri = gnome_vfs_uri_new (s);
	g_free (s);

	s = gnome_util_prepend_user_home ("Documents");
	fs->priv->docs_uri = gnome_vfs_uri_new (s);
	g_free (s);
}

static void
impl_dummy (PortableServer_Servant servant, CORBA_Environment *ev)
{
	/* dummy */
}

static void
free_kit (EventKit *kit, CORBA_Environment *ev)
{
	Bonobo_EventSource_removeListener (kit->es, kit->id, ev);

	bonobo_object_release_unref (kit->es, ev);
}

static void
file_selector_destroy (GtkObject *object)
{
	CORBA_Environment ev;
	FileSelector *fs;
	FileSelectorPrivate *priv;

	fs = FILE_SELECTOR (object);
	priv = fs->priv;

	g_print ("file_selector_destroy ()\n");

	CORBA_exception_init (&ev);

	bonobo_object_unref (BONOBO_OBJECT (priv->properties));

	free_kit (&priv->flist_kit[0], &ev);
	free_kit (&priv->flist_kit[1], &ev);
	free_kit (&priv->entry_kit, &ev);

	bonobo_object_unref (BONOBO_OBJECT (priv->flist_listener));
	bonobo_object_unref (BONOBO_OBJECT (priv->entry_listener));
	
	if (priv->current_uri)
		gnome_vfs_uri_unref (priv->current_uri);

	if (priv->requested_uri)
		gnome_vfs_uri_unref (priv->requested_uri);

	if (priv->home_uri)
		gnome_vfs_uri_unref (priv->home_uri);
	
	if (priv->desktop_uri)
		gnome_vfs_uri_unref (priv->desktop_uri);

	if (priv->docs_uri)
		gnome_vfs_uri_unref (priv->docs_uri);	
	
	bonobo_object_release_unref (priv->flist, &ev);

	if (priv->parent_uris)
		free_uri_list (&priv->parent_uris);

	if (priv->favorite_uris)
		free_uri_list (&priv->favorite_uris);

	if (priv->recent_uris)
		free_uri_list (&priv->recent_uris);

	if (priv->mime_types)
		g_strfreev (priv->mime_types);

	g_free (priv->app_name);
	g_free (priv->default_name);
	g_free (priv->default_view);

	CORBA_exception_free (&ev);

	g_free (FILE_SELECTOR (object)->priv);

	parent_class->destroy (object);

	g_print ("\n [ file selector destroyed ] \n\n");
}

static void
file_selector_class_init (GtkObjectClass *klass)
{
	parent_class = gtk_type_class (BONOBO_X_OBJECT_TYPE);

	FILE_SELECTOR_CLASS (klass)->epv.dummy = impl_dummy;

	klass->destroy = file_selector_destroy;

	glade_gnome_init ();
}


static void
file_selector_init (FileSelector *fs)
{
	fs->priv = g_new0 (FileSelectorPrivate, 1);

	file_selector_create_ui (fs);

	load_uri_lists (fs);

	fs->priv->control = bonobo_control_new (fs->priv->dialog);
	bonobo_object_add_interface (BONOBO_OBJECT (fs),
				     BONOBO_OBJECT (fs->priv->control));

	fs->priv->event_source = bonobo_event_source_new ();
	bonobo_object_add_interface (BONOBO_OBJECT (fs),
				     BONOBO_OBJECT (fs->priv->event_source));

	add_properties  (fs);

	property_bag_item_handler_new (BONOBO_OBJECT (fs), fs->priv->properties);


	create_listeners (fs);

#if 1
	/* have this here for now... */
	file_selector_request_uri (fs, g_get_home_dir (), GNOME_FileList_RequestAbsolut);
#endif
}

BONOBO_X_TYPE_FUNC_FULL (FileSelector, GNOME_FileSelector, BONOBO_X_OBJECT_TYPE, file_selector);

static BonoboObject *
create_file_selector (BonoboGenericFactory *factory, gpointer data)
{
	return BONOBO_OBJECT (gtk_type_new (FILE_SELECTOR_TYPE));
}

BONOBO_OAF_SHLIB_FACTORY ("OAFIID:GNOME_FileSelector_ControlFactory",
			  "File selector control",
			  create_file_selector,
			  NULL);
