/*
   Copyright (C) 2004 Fernando Herrera <fherrera@onirica.com>
   Copyright (C) 2004 Mariano Suárez-Alvarez <mariano@gnome.org>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <gconf/gconf-client.h>
#include <gnome-keyring.h>

#include <string.h>
#include <stdlib.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-password-dialog.h"
#include "gnome-keyring-manager-keyring-manager.h"
#include "gnome-keyring-manager-keyring-editor.h"

#define GKM_KEYRING_MANAGER_GCONF_DIR GNOME_KEYRING_MANAGER_GCONF_PREFIX "/keyring-manager"

#define LOCKED_ICON   "stock_lock"
/* FIXME: when gnome-icon-theme gets an unlocked icon, change this one; see bug #141458 */
#define UNLOCKED_ICON "stock_lock-broken"
/* FIXME: this icon is not very clear... */
#define DEFAULT_ICON  "stock_3d-light-on"

static const char *stock_icons[] =
{
  LOCKED_ICON,
  UNLOCKED_ICON,
  DEFAULT_ICON
};

enum
{
  COLUMN_KEYRING = 0,
  COLUMN_LOCK,
  COLUMN_LOCK_ON_IDLE,
  COLUMN_LOCK_TIMEOUT,
  COLUMN_MTIME,
  COLUMN_CTIME,
  COLUMN_DEFAULT,
  NUM_COLUMNS
};

struct _GKMKeyringManagerPrivate
{
  GtkWidget *tree_view;
  GtkListStore *keyrings;
  char *default_keyring;

  GtkWidget *create_dialog;
  GtkWidget *unlock_dialog;

  GtkUIManager *ui_manager;

  GSList *gconf_cnxn_ids;
};

static void gkm_keyring_manager_class_init (GKMKeyringManagerClass *class);
static void gkm_keyring_manager_init       (GKMKeyringManager *dialog);
static void gkm_keyring_manager_finalize   (GObject *object);
static void gkm_keyring_manager_destroy    (GtkObject *object);

static void gkm_keyring_manager_register_icons (void);
static void gkm_keyring_manager_initialize_tree_view (GKMKeyringManager *manager);

static void close_action_callback (GtkAction *action, GKMKeyringManager *manager);
static void new_keyring_action_callback (GtkWidget *button, GKMKeyringManager *manager);
static void open_keyring_action_callback (GtkWidget *button, GKMKeyringManager *manager);
static void lock_unlock_keyring_action_callback (GtkWidget *widget, GKMKeyringManager *manager);
static void set_default_keyring_action_callback (GtkWidget *widget, GKMKeyringManager *manager);
static void toggle_view_column_action_callback (GtkToggleAction *action, GtkWidget *widget);
static void about_action_callback (GtkAction *action G_GNUC_UNUSED, GKMKeyringManager *manager);

static gboolean keyring_tree_context_menu_callback (GtkWidget *tree_view, GKMKeyringManager *window);
static gboolean keyring_tree_button_press_callback (GtkWidget *tree_view, GdkEventButton *event, GKMKeyringManager *window);

static void gkm_keyring_manager_update_keyrings (GKMKeyringManager *transient_parent);
static void gkm_keyring_manager_update_keyring_info (GKMKeyringManager *manager, const char *keyring, GtkTreeIter *iter);
static void gkm_keyring_manager_update_default_keyring (GKMKeyringManager *manager);

/********************************************************************
 * GKMKeyringManager and GKMKeyringManagerClass
 */

static GtkWindowClass *parent_class = NULL;

static GConfClient *gconf_client = NULL;

GType
gkm_keyring_manager_get_type (void)
{
  static GType type = 0;
  
  if (!type)
    {
      static const GTypeInfo info =
      {
	sizeof (GKMKeyringManagerClass),
	NULL,           /* base_init */
	NULL,           /* base_finalize */
	(GClassInitFunc) gkm_keyring_manager_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data */
	sizeof (GKMKeyringManager),
	0,              /* n_preallocs */
	(GInstanceInitFunc) gkm_keyring_manager_init,
        0
      };

      type = g_type_register_static (GTK_TYPE_WINDOW, "GKMKeyringManager", &info, 0);
    }
  
  return type;
}

static void
gkm_keyring_manager_class_init (GKMKeyringManagerClass *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_keyring_manager_finalize;

  object_class->destroy = gkm_keyring_manager_destroy;

  gkm_keyring_manager_register_icons ();

  gconf_client = gconf_client_get_default ();
}

static GtkActionEntry entries[] =
{
    { "KeyringsMenu", NULL, N_("_Keyrings"), NULL, NULL, NULL },
    { "EditMenu", NULL, N_("_Edit"), NULL, NULL, NULL },
    { "ViewMenu", NULL, N_("_View"), NULL, NULL, NULL },
    { "HelpMenu", NULL, N_("_Help"), NULL, NULL, NULL },

    { "Close", GTK_STOCK_CLOSE, N_("_Close"), "<control>W", NULL, 
      G_CALLBACK (close_action_callback) },
    { "About", GNOME_STOCK_ABOUT, N_("_About"), NULL, NULL, 
      G_CALLBACK (about_action_callback) },
    { "NewKeyring", GTK_STOCK_NEW, N_("_New..."), "<control>N", N_("Create a new keyring"), 
      G_CALLBACK (new_keyring_action_callback) },
    { "DeleteKeyring", GTK_STOCK_DELETE, N_("_Delete"), NULL, N_("Delete keyring"), 
      NULL },
    { "KeyringContents", GTK_STOCK_OPEN, N_("_Contents"), NULL, N_("Display the contents of the selected keyring"), 
      G_CALLBACK (open_keyring_action_callback) },
    { "LockUnlockKeyring", NULL, N_("_Lock/Unlock"), "<control>L", N_("Lock or unlock the selected keyring"), 
      G_CALLBACK (lock_unlock_keyring_action_callback) },

    /* keyring context popup */
    { "LockKeyring", LOCKED_ICON, N_("_Lock"), "<control>L", N_("Lock the selected keyring"), 
      G_CALLBACK (lock_unlock_keyring_action_callback) },
    { "UnlockKeyring", UNLOCKED_ICON, N_("_Unlock"), "<control>U", N_("Unlock the selected keyring"), 
      G_CALLBACK (lock_unlock_keyring_action_callback) },
    { "MakeDefault", DEFAULT_ICON, N_("_Select as Default"), "<control>D", N_("Make the selected keyring the default one"), 
      G_CALLBACK (set_default_keyring_action_callback) }
};

static GtkToggleActionEntry toggle_entries[] =
{
    { "ToggleLockOnIdle", NULL, N_("Lock on Idle"), NULL, N_("Toggle display of keyring lock on idle"), 
      G_CALLBACK (toggle_view_column_action_callback), FALSE},
    { "ToggleLockTimeout", NULL, N_("Lock Timeout"), NULL, N_("Toggle display of keyring lock timeout"), 
      G_CALLBACK (toggle_view_column_action_callback), FALSE},
    { "ToggleModificationTime", NULL, N_("Modification Time"), NULL, N_("Toggle display of keyring modification time"), 
      G_CALLBACK (toggle_view_column_action_callback), FALSE },
    { "ToggleCreationTime", NULL, N_("Creation Time"), NULL, N_("Toggle display of keyring creation time"), 
      G_CALLBACK (toggle_view_column_action_callback), FALSE }
};

static const char *ui_description =
  "<ui>"
  "  <menubar name='KeyringManagerMenu'>"
  "    <menu action='KeyringsMenu'>"
  "      <menuitem action='NewKeyring'/>"
  "      <separator/>"
  "      <menuitem action='Close'/>"
  "    </menu>"
  "    <menu action='EditMenu'>"
  "      <menuitem action='KeyringContents'/>"
  "      <menuitem action='DeleteKeyring'/>"
  "      <menuitem action='LockUnlockKeyring'/>"
  "    </menu>"
  "    <menu action='ViewMenu'>"
  "      <menuitem action='ToggleLockOnIdle'/>"
  "      <menuitem action='ToggleLockTimeout'/>"
  "      <menuitem action='ToggleModificationTime'/>"
  "      <menuitem action='ToggleCreationTime'/>"
  "    </menu>"
  "    <menu action='HelpMenu'>"
  "      <menuitem action='About'/>"
  "    </menu>"
  "  </menubar>"
  "  <popup name='KeyringContextMenu'>"
  "    <menuitem action='KeyringContents'/>"
  "    <menuitem action='DeleteKeyring'/>"
  "    <menuitem action='LockKeyring'/>"
  "    <menuitem action='UnlockKeyring'/>"
  "    <menuitem action='MakeDefault'/>"
  "  </popup>"
  "</ui>";

static void
gkm_keyring_manager_init (GKMKeyringManager *manager)
{
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *scrolledwindow;
  GtkWidget *vbox;
  GtkWidget *vbuttonbox;
  GtkWidget *button;
  GtkWidget *statusbar;
  GtkAccelGroup *accel_group;
  GtkActionGroup *action_group;
  GtkWidget *menubar;
  GError *error;

  manager->priv = g_new0 (GKMKeyringManagerPrivate, 1);

  gtk_window_set_default_size (GTK_WINDOW (manager), 550, 400); 
  gtk_window_set_title (GTK_WINDOW (manager), _("Keyring Manager"));
  
  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (manager), main_vbox);
  
  action_group = gtk_action_group_new ("KeyringManagerMenuActions");
  gtk_action_group_add_actions (action_group, entries, G_N_ELEMENTS (entries), manager);
  gtk_action_group_add_toggle_actions (action_group, toggle_entries, G_N_ELEMENTS (toggle_entries), manager);
  
  manager->priv->ui_manager = gtk_ui_manager_new ();
  gtk_ui_manager_insert_action_group (manager->priv->ui_manager,
                                      action_group, 0);

  accel_group = gtk_ui_manager_get_accel_group (manager->priv->ui_manager);
  gtk_window_add_accel_group (GTK_WINDOW (manager), accel_group);

  error = NULL;
  if (!gtk_ui_manager_add_ui_from_string (manager->priv->ui_manager,
                                          ui_description, -1, &error))
    {
      g_message ("building menus failed: %s", error->message);
      g_error_free (error);
      exit (1);
    }

  menubar = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringManagerMenu");
  gtk_box_pack_start (GTK_BOX (main_vbox), menubar, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 18);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, TRUE, TRUE, 0);

  statusbar = gtk_statusbar_new ();
  gtk_box_pack_start (GTK_BOX (main_vbox), statusbar, FALSE, FALSE, 0);
  
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow, TRUE, TRUE, 0);
  
  manager->priv->tree_view = gtk_tree_view_new ();
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (manager->priv->tree_view), TRUE);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), manager->priv->tree_view);
  g_object_set_data (G_OBJECT (manager), "tree-view", manager->priv->tree_view);
  
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

  vbuttonbox = gtk_vbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (vbuttonbox), 12);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start (GTK_BOX (vbox), vbuttonbox, TRUE, TRUE, 0);
  
  button = gtk_button_new_from_stock (GTK_STOCK_OPEN);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), button, FALSE, FALSE, 0);
  g_signal_connect (button, "clicked", G_CALLBACK (open_keyring_action_callback), manager);
  
  button = gtk_button_new_with_mnemonic (_("_Set as Default"));
  gtk_box_pack_start (GTK_BOX (vbuttonbox), button, FALSE, FALSE, 0);
  g_signal_connect (button, "clicked", G_CALLBACK (set_default_keyring_action_callback), manager);
  
  button = gtk_button_new_with_mnemonic (_("_Lock/Unlock"));
  gtk_box_pack_start (GTK_BOX (vbuttonbox), button, FALSE, FALSE, 0);
  g_signal_connect (button, "clicked", G_CALLBACK (lock_unlock_keyring_action_callback), manager);
  
  button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), button, FALSE, FALSE, 0);
  
  gtk_box_pack_start (GTK_BOX (vbuttonbox), gtk_alignment_new (0,0,0,0), FALSE, FALSE, 0);

  button = gtk_button_new_from_stock (GTK_STOCK_NEW);
  g_signal_connect (button, "clicked", G_CALLBACK (new_keyring_action_callback), manager);
  gtk_container_add (GTK_CONTAINER (vbuttonbox), button);
  
  gkm_keyring_manager_initialize_tree_view (manager);
  g_signal_connect (manager->priv->tree_view, "button-press-event", G_CALLBACK (keyring_tree_button_press_callback), manager);
  g_signal_connect (manager->priv->tree_view, "popup-menu", G_CALLBACK (keyring_tree_context_menu_callback), manager);
  
  SetColumnPrefsData toggle[] = 
  {
    { COLUMN_LOCK_ON_IDLE, "ToggleLockOnIdle",       "lock-on-idle" },
    { COLUMN_LOCK_TIMEOUT, "ToggleLockTimeout",      "lock-timeout" },
    { COLUMN_MTIME,        "ToggleModificationTime", "mtime" },
    { COLUMN_CTIME,        "ToggleCreationTime",     "ctime" }
  };
  manager->priv->gconf_cnxn_ids = 
  set_column_visibility_preferences (toggle,
  				     G_N_ELEMENTS (toggle),
  				     GKM_KEYRING_MANAGER_GCONF_DIR,
  				     GTK_TREE_VIEW (manager->priv->tree_view), 
				     action_group,
				     gconf_client);
 
  gkm_keyring_manager_update_keyrings (manager); 
     /* FIXME: we can't use window as transient_parent here because it's not mapped yet
      *        Make things so that the store is initially updated -after- the window is
      *        first mapped, maybe?
      * FIXME^2: we -are- using manager as transient parent...
      */
  
  gtk_widget_show_all (main_vbox);
}

static void
gkm_keyring_manager_finalize (GObject *object)
{
  GKMKeyringManager *manager;

  g_return_if_fail (GKM_IS_KEYRING_MANAGER (object));

  manager = GKM_KEYRING_MANAGER (object);

  g_free (manager->priv->default_keyring);

  close_gconf_connections (manager->priv->gconf_cnxn_ids,
  			   GKM_KEYRING_MANAGER_GCONF_DIR,
			   gconf_client);
	
	g_object_unref (G_OBJECT (manager->priv->ui_manager));
  
  g_free (manager->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_keyring_manager_destroy (GtkObject *object)
{
  GKMKeyringManager *manager;

  g_return_if_fail (GKM_IS_KEYRING_MANAGER (object));

  manager = GKM_KEYRING_MANAGER (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

GtkWidget *
gkm_keyring_manager_new (void)
{
  GKMKeyringManager *manager;

  manager = g_object_new (GKM_TYPE_KEYRING_MANAGER, NULL);

  return GTK_WIDGET (manager);
}

static void
gkm_keyring_manager_register_icons (void)
{
  static gboolean been_here = FALSE;

  unsigned int i;
  GtkIconTheme *icon_theme;
  GtkIconFactory *factory;

  if (been_here)
    {
      return;
    }

  icon_theme = gtk_icon_theme_get_default ();
  factory = gtk_icon_factory_new ();

  /* FIXME: maybe handle errors here */
  /* FIXME: these 16s should actually be measured in ems */

  for (i = 0; i < G_N_ELEMENTS (stock_icons); i++)
    {
      GdkPixbuf *icon;
      GtkIconSet *icon_set;

      icon = gtk_icon_theme_load_icon (icon_theme, stock_icons[i], 16, 0, NULL);
      icon_set = gtk_icon_set_new_from_pixbuf (icon);
      gtk_icon_factory_add (factory, stock_icons[i], icon_set);
    }

  gtk_icon_factory_add_default (factory);

  been_here = TRUE;
}

static gint
compare_keyring_lock_status (GtkTreeModel *model,
			     GtkTreeIter  *a,
			     GtkTreeIter  *b,
			     gpointer      data)
{
  gint   sortcol;
  gint   ret;
  gchar *string1;
  gchar *string2;

  sortcol = GPOINTER_TO_INT (data);
  ret = 0;
 
  gtk_tree_model_get (model, a, sortcol, &string1, -1);
  gtk_tree_model_get (model, b, sortcol, &string2, -1);
  
  /* Put Unlocked keyrings first. */
  if (strcmp (string1, string2) != 0)
    {
      ret = (strcmp (string1, UNLOCKED_ICON) == 0) ? -1 : 1;
    }
  
  g_free (string1);
  g_free (string2);
  
  return ret;
}

static void
format_timeout_cell_data_func (GtkTreeViewColumn *column G_GNUC_UNUSED,
                               GtkCellRenderer   *cell,
                               GtkTreeModel      *model,
                               GtkTreeIter       *iter,
                               gpointer           data)
{
  gint column_id;
  guint32 timeout;
  gchar *readable_timeout;

  column_id = GPOINTER_TO_INT (data);

  gtk_tree_model_get(model, iter, column_id, &timeout, -1); 
  readable_timeout = g_strdup_printf (_("%d seconds"), timeout);
  
  g_object_set(cell, "text", readable_timeout, NULL);

  g_free (readable_timeout);
}

static void
gkm_keyring_manager_initialize_tree_view (GKMKeyringManager *manager)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  
  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));
  g_assert (manager->priv->keyrings == NULL);

  manager->priv->keyrings = gtk_list_store_new (NUM_COLUMNS, 
                                                G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UINT, 
                                                G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT,
                                                G_TYPE_STRING);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Keyring"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  gtk_tree_view_column_set_expand (column, TRUE);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", COLUMN_KEYRING);
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, FALSE);
  gtk_tree_view_column_add_attribute (column, renderer, "stock-id", COLUMN_DEFAULT);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_KEYRING,
				   (GtkTreeIterCompareFunc) tree_model_compare_strings,
				   GINT_TO_POINTER (COLUMN_KEYRING), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_KEYRING);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Locked"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "stock-id", COLUMN_LOCK);
  g_object_set (G_OBJECT (renderer), "xalign", 0.0, "xpad", 3, NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_LOCK,
				   (GtkTreeIterCompareFunc) compare_keyring_lock_status,
				   GINT_TO_POINTER (COLUMN_LOCK), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_LOCK);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Lock on Idle"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", COLUMN_LOCK_ON_IDLE);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_LOCK_ON_IDLE,
				   (GtkTreeIterCompareFunc) tree_model_compare_strings,
				   GINT_TO_POINTER (COLUMN_LOCK_ON_IDLE), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_LOCK_ON_IDLE);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Lock Timeout"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_timeout_cell_data_func,
				           GINT_TO_POINTER (COLUMN_LOCK_TIMEOUT), NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_LOCK_TIMEOUT,
				   (GtkTreeIterCompareFunc) tree_model_compare_uints,
				   GINT_TO_POINTER (COLUMN_LOCK_TIMEOUT), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_LOCK_TIMEOUT);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Modification Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (COLUMN_MTIME), NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_MTIME,
				   (GtkTreeIterCompareFunc) tree_model_compare_uints,
				   GINT_TO_POINTER (COLUMN_MTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_MTIME);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Creation Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (manager->priv->tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (COLUMN_CTIME), NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
  				   COLUMN_CTIME,
				   (GtkTreeIterCompareFunc) tree_model_compare_uints,
				   GINT_TO_POINTER (COLUMN_CTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_CTIME);

  gtk_tree_sortable_set_default_sort_func (GTK_TREE_SORTABLE (manager->priv->keyrings),
				           (GtkTreeIterCompareFunc) tree_model_compare_strings,
				           GINT_TO_POINTER (COLUMN_KEYRING), NULL);

  gtk_tree_view_set_model (GTK_TREE_VIEW (manager->priv->tree_view), GTK_TREE_MODEL (manager->priv->keyrings));
  g_object_unref (G_OBJECT (manager->priv->keyrings));
}

/********************************************************************
 * The About dialog
 */

static void
about_action_callback (GtkAction         *action G_GNUC_UNUSED, 
                       GKMKeyringManager *manager)
{
  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  gkm_application_open_about_dialog (GTK_WINDOW (manager));
}

/********************************************************************
 * The Close menu
 */

static void
close_action_callback (GtkAction         *action G_GNUC_UNUSED,
                       GKMKeyringManager *manager)
{
  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  gtk_widget_destroy (GTK_WIDGET (manager)); 
}

/********************************************************************
 * Toggle column display
 */

static void
toggle_view_column_action_callback (GtkToggleAction *toggle_action,
				    GtkWidget	    *widget G_GNUC_UNUSED)
{
  toggle_view_column_action (toggle_action, gconf_client);
}

/********************************************************************
 * Creating a keyring
 */

typedef struct _CreateKeyringCallbackData
{
  GKMKeyringManager *manager;
  gchar *keyring_name;
} CreateKeyringCallbackData;

static void
create_keyring_callback_data_free (CreateKeyringCallbackData *data)
{
  g_free (data->keyring_name);
  g_free (data);
}

static void
create_keyring_callback (GnomeKeyringResult         result, 
                         CreateKeyringCallbackData *data)
{
  GtkTreeIter iter;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (NULL, result);

      return;
    }

  gtk_list_store_append (data->manager->priv->keyrings, &iter);
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_KEYRING, data->keyring_name, -1);

  gkm_keyring_manager_update_keyring_info (data->manager, data->keyring_name, &iter);
}

static void
new_keyring_dialog_create_button_clicked_callback (GtkButton         *button G_GNUC_UNUSED,
                                                   GKMKeyringManager *manager)
{
  GtkWidget *entry;
  gchar *keyring_name;

  entry = g_object_get_data (G_OBJECT (manager->priv->create_dialog), "entry");

  keyring_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
    
  if (keyring_name != NULL) 
    {
      CreateKeyringCallbackData *data;

      data = g_new0 (CreateKeyringCallbackData, 1);
      data->manager = manager;
      data->keyring_name = keyring_name;

      /* FIXME: Use our dialog here to ask for the password */
      gnome_keyring_create (keyring_name, NULL, 
                            (GnomeKeyringOperationDoneCallback) create_keyring_callback, 
                            data, (GDestroyNotify) create_keyring_callback_data_free);
    }

  gtk_widget_destroy (manager->priv->create_dialog);
}

static void
new_keyring_dialog_text_changed_callback (GtkWidget *widget, 
                                          GtkDialog *dialog)
{
  const gchar *text;
  gboolean empty;

  g_return_if_fail (GTK_IS_DIALOG (dialog));

  text = gtk_entry_get_text (GTK_ENTRY (widget));

  empty = text == NULL || strcmp (text, "") == 0;

  gtk_dialog_set_response_sensitive (dialog, GTK_RESPONSE_ACCEPT, !empty);
}


static void
new_keyring_action_callback (GtkWidget         *widget G_GNUC_UNUSED,
                             GKMKeyringManager *manager)
{
  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  if (manager->priv->create_dialog == NULL)
    {
      GtkWidget *button;
      GtkWidget *entry;
      GtkWidget *label;
      GtkWidget *hbox;

      manager->priv->create_dialog = gtk_dialog_new_with_buttons (_("Create New Keyring"),
                                                           GTK_WINDOW (manager),
                                                           GTK_DIALOG_DESTROY_WITH_PARENT,
                                                           NULL);
      gtk_window_set_transient_for (GTK_WINDOW (manager->priv->create_dialog), GTK_WINDOW (manager));
      gtk_window_set_resizable(GTK_WINDOW (manager->priv->create_dialog), FALSE);
      gtk_container_set_border_width(GTK_CONTAINER (GTK_DIALOG (manager->priv->create_dialog)), 6);

      button = gtk_dialog_add_button (GTK_DIALOG (manager->priv->create_dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
      g_signal_connect_swapped (G_OBJECT (button), "clicked", G_CALLBACK (gtk_widget_destroy), manager->priv->create_dialog);

      button = gtk_dialog_add_button (GTK_DIALOG (manager->priv->create_dialog), _("_Create"), GTK_RESPONSE_ACCEPT);
      gtk_dialog_set_default_response (GTK_DIALOG (manager->priv->create_dialog), GTK_RESPONSE_ACCEPT);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (manager->priv->create_dialog), GTK_RESPONSE_ACCEPT, FALSE);
      g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (new_keyring_dialog_create_button_clicked_callback), manager);

      hbox = gtk_hbox_new (FALSE, 6);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (manager->priv->create_dialog)->vbox), hbox, TRUE, TRUE, 12);

      label = gtk_label_new_with_mnemonic (_("Keyring _Name:"));
      gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

      entry = gtk_entry_new ();
      gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
      gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
      g_signal_connect (G_OBJECT (entry), "changed", 
                        G_CALLBACK (new_keyring_dialog_text_changed_callback), manager->priv->create_dialog);
      g_object_set_data (G_OBJECT (manager->priv->create_dialog), "entry", entry);
      gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

      gtk_widget_show_all (hbox);

      g_object_add_weak_pointer (G_OBJECT (manager->priv->create_dialog), (void**) &manager->priv->create_dialog);
    }

  gtk_window_present (GTK_WINDOW (manager->priv->create_dialog));
}

/********************************************************************
 * Locking and unlocking keyrings
 */

typedef struct _LockUnlockKeyringCallbackData
{
  GtkTreeRowReference *row;
  GKMKeyringManager *manager;
  gchar *keyring_name;
} LockUnlockKeyringCallbackData;

static void
lock_unlock_keyring_callback_data_free (LockUnlockKeyringCallbackData *data)
{
  gtk_tree_row_reference_free (data->row);
  g_free (data->keyring_name);
  g_free (data);
}

static LockUnlockKeyringCallbackData *
lock_unlock_keyring_callback_data_clone (LockUnlockKeyringCallbackData *data)
{
  LockUnlockKeyringCallbackData *clone;

  clone = g_new (LockUnlockKeyringCallbackData, 1);
  clone->row = gtk_tree_row_reference_copy (data->row);
  clone->manager = data->manager;
  clone->keyring_name = g_strdup (data->keyring_name);

  return clone;
}

static void 
lock_keyring_callback (GnomeKeyringResult             result,
                       LockUnlockKeyringCallbackData *data)
{
  GtkTreePath *path;
  GtkTreeIter iter;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->manager), result);

      return;
    } 

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for it to be locked..."));

      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (data->manager->priv->keyrings), &iter, path);
  gtk_tree_path_free (path);

  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK, LOCKED_ICON, -1);
} 

static void 
unlock_keyring_callback (GnomeKeyringResult             result,
                         LockUnlockKeyringCallbackData *data)
{
  GtkTreePath *path;
  GtkTreeIter iter;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->manager), result);

      return;
    } 

  if (!gtk_tree_row_reference_valid (data->row))
    {
      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (data->manager->priv->keyrings), &iter, path))
    {
      g_assert_not_reached ();
    }
  gtk_tree_path_free (path);

  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK, UNLOCKED_ICON, -1);
}

static void
unlock_keyring_password_callback (GKMPasswordDialog             *dialog,
                                  gint                           response,
                                  LockUnlockKeyringCallbackData *data)
{
  g_return_if_fail (GKM_IS_PASSWORD_DIALOG (dialog));

  if (response == GTK_RESPONSE_ACCEPT)
    {
      char *password;

      password = gkm_password_dialog_get_password (dialog);
      gnome_keyring_unlock (data->keyring_name, password,
                            (GnomeKeyringOperationDoneCallback) unlock_keyring_callback, 
                            lock_unlock_keyring_callback_data_clone (data), 
                            (GDestroyNotify) lock_unlock_keyring_callback_data_free);
      g_free (password);
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
lock_unlock_get_keyring_info_callback (GnomeKeyringResult             result, 
                         	       GnomeKeyringInfo              *info,
				       LockUnlockKeyringCallbackData *data)
{
  if (result != GNOME_KEYRING_RESULT_OK) 
    {
      g_warning (_("Failed to get keyring info."));

      return;
    }

  if (gnome_keyring_info_get_is_locked (info))
    {
      char *secondary_text;

      if (data->manager->priv->unlock_dialog != NULL)
        {
          gtk_widget_destroy (data->manager->priv->unlock_dialog);
        }

      secondary_text = g_strdup_printf (_("A password is required in order to unlock the keyring '%s'."), data->keyring_name);
      data->manager->priv->unlock_dialog = gkm_password_dialog_new (NULL, GTK_WINDOW (data->manager), 
                                        GTK_DIALOG_DESTROY_WITH_PARENT,
                                        _("Password required"),
					secondary_text);
      g_free (secondary_text);
      gkm_password_dialog_accept_button_set_label (GKM_PASSWORD_DIALOG (data->manager->priv->unlock_dialog), _("_Unlock"));


      g_signal_connect_data (G_OBJECT (data->manager->priv->unlock_dialog), "response", 
                             G_CALLBACK (unlock_keyring_password_callback),
                             lock_unlock_keyring_callback_data_clone (data), 
                             (GClosureNotify) lock_unlock_keyring_callback_data_free,
                             0);
      g_object_add_weak_pointer (G_OBJECT (data->manager->priv->unlock_dialog), (void**) &data->manager->priv->unlock_dialog);

      gtk_window_present (GTK_WINDOW (data->manager->priv->unlock_dialog));
    }
  else
    {
      gnome_keyring_lock (data->keyring_name,
                          (GnomeKeyringOperationDoneCallback) lock_keyring_callback, 
                          lock_unlock_keyring_callback_data_clone (data), (GDestroyNotify) lock_unlock_keyring_callback_data_free); 
    }
}

static void
lock_unlock_keyring_action_callback (GtkWidget         *widget G_GNUC_UNUSED,
			             GKMKeyringManager *manager)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeView *tree_view;

  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  tree_view = g_object_get_data (G_OBJECT (manager), "tree-view");

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
        
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      LockUnlockKeyringCallbackData *data;
      GtkTreePath *path;
      gchar *keyring_name;

      gtk_tree_model_get (model, &iter, COLUMN_KEYRING, &keyring_name, -1);
      
      data = g_new (LockUnlockKeyringCallbackData, 1);

      path = gtk_tree_model_get_path (model, &iter);
      data->row = gtk_tree_row_reference_new (model, path);
      gtk_tree_path_free (path);

      data->keyring_name = keyring_name;
      data->manager = manager;

      gnome_keyring_get_info (keyring_name, 
                              (GnomeKeyringOperationGetKeyringInfoCallback) lock_unlock_get_keyring_info_callback, 
                              data, (GDestroyNotify) lock_unlock_keyring_callback_data_free);
    }
}

/********************************************************************
 * Opening keyrings
 */

static void
open_keyring_callback (GnomeKeyringResult             result,
                       LockUnlockKeyringCallbackData *data)
{
  GtkTreePath *path;
  GtkTreeIter iter;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->manager), result);

      return;
    } 

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for the data..."));

      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (data->manager->priv->keyrings), &iter, path))
    {
      g_assert_not_reached ();
    }
  gtk_tree_path_free (path);

  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK, UNLOCKED_ICON, -1);
  
  gkm_application_open_keyring_editor_for (data->keyring_name, NULL);
}

static void
open_keyring_password_callback (GKMPasswordDialog             *dialog,
                                gint                           response,
                                LockUnlockKeyringCallbackData *data)
{
  g_return_if_fail (GKM_IS_PASSWORD_DIALOG (dialog));

  if (response == GTK_RESPONSE_ACCEPT)
    {
      char *password;

      password = gkm_password_dialog_get_password (dialog);
      gnome_keyring_unlock (data->keyring_name, password,
                            (GnomeKeyringOperationDoneCallback) open_keyring_callback, 
                            lock_unlock_keyring_callback_data_clone (data), 
                            (GDestroyNotify) lock_unlock_keyring_callback_data_free);
      g_free (password);
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void 
open_get_keyring_info_callback (GnomeKeyringResult             result, 
                         	GnomeKeyringInfo              *info,
				LockUnlockKeyringCallbackData *data)
{
 if (result != GNOME_KEYRING_RESULT_OK) 
    {
      g_warning (_("Failed to get keyring info."));

      return;
    }

  if (gnome_keyring_info_get_is_locked (info))
    {
      gchar *primary_text;

      if (data->manager->priv->unlock_dialog != NULL)
        {
          gtk_widget_destroy (data->manager->priv->unlock_dialog);
        }

      primary_text = g_strdup_printf (_("Unlock the “%s” Keyring."), data->keyring_name);
      data->manager->priv->unlock_dialog = gkm_password_dialog_new (NULL, GTK_WINDOW (data->manager), 
                                        GTK_DIALOG_DESTROY_WITH_PARENT,
                                        primary_text,
                                        _("To view the contents of this keyring, you must first unlock it."));
      g_free (primary_text);

      g_signal_connect_data (G_OBJECT (data->manager->priv->unlock_dialog), "response", 
                             G_CALLBACK (open_keyring_password_callback),
                             lock_unlock_keyring_callback_data_clone (data), 
                             (GClosureNotify) lock_unlock_keyring_callback_data_free,
                             0);
      g_object_add_weak_pointer (G_OBJECT (data->manager->priv->unlock_dialog), (void**) &data->manager->priv->unlock_dialog);

      gtk_window_present (GTK_WINDOW (data->manager->priv->unlock_dialog));
    }
  else
    {
      gkm_application_open_keyring_editor_for (data->keyring_name, NULL);
    }
}

static void
open_keyring_action_callback (GtkWidget         *button G_GNUC_UNUSED,
			      GKMKeyringManager *manager)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (manager->priv->tree_view));
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      LockUnlockKeyringCallbackData *data;
      GtkTreePath *path;
      gchar *keyring_name;

      gtk_tree_model_get (model, &iter, COLUMN_KEYRING, &keyring_name, -1);
      
      data = g_new0 (LockUnlockKeyringCallbackData, 1);

      path = gtk_tree_model_get_path (model, &iter);
      data->row = gtk_tree_row_reference_new (model, path);
      gtk_tree_path_free (path);

      data->keyring_name = keyring_name;
      data->manager = manager;

      gnome_keyring_get_info (keyring_name, 
                              (GnomeKeyringOperationGetKeyringInfoCallback) open_get_keyring_info_callback, 
                              data, (GDestroyNotify) lock_unlock_keyring_callback_data_free);
    }

  return;
}

/********************************************************************
 * Set the default keyring
 */

static void
set_default_keyring_worker_callback (GnomeKeyringResult  result,
                                     GKMKeyringManager  *manager)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      /* FIXME: tell the user about this. */
      g_warning (_("Failed set the default keyring."));

      return;
    }

  gkm_keyring_manager_update_default_keyring (manager);
}

static void
set_default_keyring_action_callback (GtkWidget         *widget G_GNUC_UNUSED,
                                     GKMKeyringManager *manager)
{
  GtkTreeView *tree_view;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_return_if_fail (GTK_IS_WINDOW (manager));

  tree_view = g_object_get_data (G_OBJECT (manager), "tree-view");

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      char *keyring;

      gtk_tree_model_get (GTK_TREE_MODEL (manager->priv->keyrings), &iter, COLUMN_KEYRING, &keyring, -1);

      gnome_keyring_set_default_keyring (keyring, 
                                         (GnomeKeyringOperationDoneCallback) set_default_keyring_worker_callback,
                                         manager, NULL);

      g_free (keyring);
    }
}

/********************************************************************
 * Context menu for keyrings
 */

static void
keyring_tree_popup_menu (GtkWidget         *tree_view,
                         GtkTreeIter       *iter,
		         GdkEventButton    *event,
		         GKMKeyringManager *manager)
{
  GtkWidget *menu_item;
  GtkWidget *menu;
  char *state;
 
  g_return_if_fail (GTK_IS_TREE_VIEW (tree_view));
  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  /* This is getting the state from the keyrings ListStore, which is different from
   * what we do in lock_unlock_keyring_callback...
   */
  
  menu = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu");
  
  gtk_tree_model_get (GTK_TREE_MODEL (manager->priv->keyrings), iter, COLUMN_LOCK, &state, -1);
  if (strcmp (state, LOCKED_ICON) == 0)
    {
      menu_item = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu/LockKeyring");
      gtk_widget_hide (menu_item);
      menu_item = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu/UnlockKeyring");
      gtk_widget_show (menu_item);
    }
  else
    {
      menu_item = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu/UnlockKeyring");
      gtk_widget_hide (menu_item);
      menu_item = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu/LockKeyring");
      gtk_widget_show (menu_item);
    }
  g_free (state);


  gtk_tree_model_get (GTK_TREE_MODEL (manager->priv->keyrings), iter, COLUMN_DEFAULT, &state, -1);
  menu_item = gtk_ui_manager_get_widget (manager->priv->ui_manager, "/KeyringContextMenu/MakeDefault");
  gtk_widget_set_sensitive (menu_item, state == NULL);
  g_free (state);

  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
  		  (event != NULL) ? event->button : 0,
		  gdk_event_get_time ( (GdkEvent *) event));
}

static gboolean
keyring_tree_context_menu_callback (GtkWidget         *tree_view,
                                    GKMKeyringManager *manager)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      keyring_tree_popup_menu (tree_view, &iter, NULL, manager);
    }

  return TRUE;
}

static gboolean
keyring_tree_button_press_callback (GtkWidget         *tree_view,
				    GdkEventButton    *event,
				    GKMKeyringManager *manager)
{
  if (event->type == GDK_BUTTON_PRESS && event->button == 3)
    {
       GtkTreeSelection *selection;
       GtkTreePath *path;

       selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

       /* Only bring up the menu if there is a row under the mouse. */ 
       if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (tree_view),
                                          event->x, event->y,
                                          &path, NULL, NULL, NULL))
         {
           GtkTreeIter iter;

           gtk_tree_selection_unselect_all (selection);
           gtk_tree_selection_select_path (selection, path);
           gtk_tree_model_get_iter (GTK_TREE_MODEL (manager->priv->keyrings), &iter, path);
           gtk_tree_path_free (path);

	   keyring_tree_popup_menu (tree_view, &iter, event, manager);
	 }

      return TRUE;
    }
  else if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
    {
      open_keyring_action_callback (NULL, manager);
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

/********************************************************************
 * Searching for a keyring
 */

typedef struct _SearchKeyringCallbackData
{
  const char  *keyring;
  GtkTreeIter  iter;
  gboolean     found;
} SearchKeyringCallbackData;

static gboolean
search_keyring_worker (GtkTreeModel              *model,
                       GtkTreePath               *path G_GNUC_UNUSED,
                       GtkTreeIter               *iter,
                       SearchKeyringCallbackData *data)
{
  char *keyring;
  
  g_assert (data != NULL);

  gtk_tree_model_get (model, iter, COLUMN_KEYRING, &keyring, -1);

  if (strcmp (keyring, data->keyring) == 0) 
    {
      data->iter = *iter;
      data->found = TRUE;
    }

  g_free (keyring);

  return data->found;
}

/**
 * gkm_keyring_manager_search_keyring:
 * @manager: a #GKMKeyringManager.
 * @keyring: name of the keyring to look for.
 * @iter: a pointer to a #GtkTreeIter or %NULL.
 * @path: a pointer to a pointer to a #GtkTreePath or %NULL.
 *
 * Looks for @keyring in the keyrings #GtkTreeModel, and if it find it,
 * fills those of @iter, @path, and @row which are not %NULL so that 
 * they point to it.
 *
 * Returns: %TRUE if @keyring was found; %FALSE otherwise.
 **/

static gboolean
gkm_keyring_manager_search_keyring (GKMKeyringManager    *manager,
                                    const char           *keyring, 
                                    GtkTreeIter          *iter, 
                                    GtkTreePath         **path,
                                    GtkTreeRowReference **row)
{
  SearchKeyringCallbackData data;

  data.keyring = keyring;
  data.found = FALSE;

  gtk_tree_model_foreach (GTK_TREE_MODEL (manager->priv->keyrings), (GtkTreeModelForeachFunc) search_keyring_worker, &data);

  if (data.found)
    {
      if (iter != NULL)
        {
          *iter = data.iter;
        }
      if (path != NULL)
        {
          *path = gtk_tree_model_get_path (GTK_TREE_MODEL (manager->priv->keyrings), &data.iter);
        }
      if (row != NULL)
        {
          if (path != NULL)
            {
              *row = gtk_tree_row_reference_new (GTK_TREE_MODEL (manager->priv->keyrings), *path);
            }
          else
            {
              GtkTreePath *tmp_path;

              tmp_path = gtk_tree_model_get_path (GTK_TREE_MODEL (manager->priv->keyrings), &data.iter);

              *row = gtk_tree_row_reference_new (GTK_TREE_MODEL (manager->priv->keyrings), tmp_path);
              gtk_tree_path_free (tmp_path);
            }
        }
    }

  return data.found;
}

/********************************************************************
 * Getting the default keyring
 */

static void
get_default_keyring_callback (GnomeKeyringResult  result,
                              const char         *new_default_keyring,
                              GKMKeyringManager  *manager)
{
  GtkTreeIter iter;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      g_warning (_("Failed get the default keyring."));

      return;
    }

  if (manager->priv->default_keyring != NULL)
    {
      if (new_default_keyring != NULL && strcmp (manager->priv->default_keyring, new_default_keyring) == 0)
        {
          return;
        }
      else if (gkm_keyring_manager_search_keyring (manager, manager->priv->default_keyring, &iter, NULL, NULL))
        {
          gtk_list_store_set (manager->priv->keyrings, &iter, COLUMN_DEFAULT, NULL, -1);
        }
      else
        {
          g_warning (_("The default keyring is not in the list store!"));
        }

      g_free (manager->priv->default_keyring);
    }

  manager->priv->default_keyring = g_strdup (new_default_keyring);

  if (manager->priv->default_keyring != NULL 
      && gkm_keyring_manager_search_keyring (manager, manager->priv->default_keyring, &iter, NULL, NULL))
    {
      gtk_list_store_set (manager->priv->keyrings, &iter, COLUMN_DEFAULT, DEFAULT_ICON, -1);
    }
}

static void
gkm_keyring_manager_update_default_keyring (GKMKeyringManager *manager)
{
  gnome_keyring_get_default_keyring ((GnomeKeyringOperationGetStringCallback) get_default_keyring_callback, 
                                     manager, NULL);
}


/********************************************************************
 * Updating the info on a keyring
 */

typedef struct _UpdateKeyringCallbackData
{
  GKMKeyringManager *manager;
  GtkTreeRowReference *row;
} UpdateKeyringCallbackData;

static void
update_keyring_callback_data_free (UpdateKeyringCallbackData *data)
{
  gtk_tree_row_reference_free (data->row);
  g_free (data);
}

static void
update_keyring_info_worker_callback (GnomeKeyringResult         result,
                                     GnomeKeyringInfo          *info,
                                     UpdateKeyringCallbackData *data)
{
  static char *yes = N_("yes"), *no = N_("no");

  char *value;
  const char *stock_id;
  time_t time;
  guint32 seconds;
  GtkTreePath *path;
  GtkTreeIter iter;
  
  if (result != GNOME_KEYRING_RESULT_OK) 
    {
      g_warning (_("Failed to get keyring info."));

      return;
    }

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for the data..."));

      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (data->manager->priv->keyrings), &iter, path);
  gtk_tree_path_free (path);
  
  stock_id = (gnome_keyring_info_get_is_locked (info) == TRUE) ? LOCKED_ICON : UNLOCKED_ICON;
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK, stock_id, -1);
  
  value = (gnome_keyring_info_get_lock_on_idle (info) == TRUE) ? yes : no;
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK_ON_IDLE, value, -1);
  
  seconds =  gnome_keyring_info_get_lock_timeout (info);
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_LOCK_TIMEOUT, seconds, -1);
  
  time = gnome_keyring_info_get_ctime (info);
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_CTIME, time, -1);

  time = gnome_keyring_info_get_mtime (info);
  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_MTIME, time, -1);

  gtk_list_store_set (data->manager->priv->keyrings, &iter, COLUMN_DEFAULT, NULL, -1);
}

/**
 * gkm_keyring_manager_update_keyring_info:
 * @manager: a #GKMKeyringManager.
 * @keyring: the name of the keyring whose info is to be updated.
 * @iter: a #GtkTreeIter pointing to @keyring in the keyrings @GtkListStore, 
 * or %NULL.
 *
 * Update the information on @keyring in the keyrings liststore.
 **/

static void
gkm_keyring_manager_update_keyring_info (GKMKeyringManager *manager, 
                                         const char        *keyring, 
                                         GtkTreeIter       *iter)
{
  UpdateKeyringCallbackData *data;

  data = g_new0 (UpdateKeyringCallbackData, 1);
  data->manager = manager;

  if (iter == NULL)
    {
      if (!gkm_keyring_manager_search_keyring (manager, keyring, NULL, NULL, &data->row))
        {
          g_warning ("gkm_keyring_manager_update_keyring_info: Tried to update the information of a keyring we don't know about");
        }
    }
  else
    {
      GtkTreePath *path;

      path = gtk_tree_model_get_path (GTK_TREE_MODEL (manager->priv->keyrings), iter);
      data->row = gtk_tree_row_reference_new (GTK_TREE_MODEL (manager->priv->keyrings), path);
      gtk_tree_path_free (path);
    }

  gnome_keyring_get_info (keyring, 
                          (GnomeKeyringOperationGetKeyringInfoCallback) update_keyring_info_worker_callback, 
                          data, (GDestroyNotify) update_keyring_callback_data_free);
}

/********************************************************************
 * Getting the list of all keyrings
 */

static void
update_keyrings_worker_callback (GnomeKeyringResult  result, 
                                 GList              *list, 
                                 GKMKeyringManager  *manager)
{
  GList *tmp;

  g_return_if_fail (GKM_IS_KEYRING_MANAGER (manager));

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      /* FIXME: We pass here NULL because this will be called while the manager is initializing, so
       * the window will not be shown yet
       */ 
      complain_about_gnome_keyring_bad_result (NULL, result);

      return;
    }
       
  gtk_list_store_clear (manager->priv->keyrings);

  for (tmp = list; tmp != NULL; tmp = tmp->next)
    {
      GtkTreeIter iter;

      gtk_list_store_append (manager->priv->keyrings, &iter);
      gtk_list_store_set (manager->priv->keyrings, &iter, COLUMN_KEYRING, tmp->data, -1);

      gkm_keyring_manager_update_keyring_info (manager, tmp->data, &iter);
    }

  gkm_keyring_manager_update_default_keyring (manager);
}

/**
 * gkm_keyring_manager_update_keyrings:
 * @transient_parent: a #GtkWindow on which to make error dialogs transient_for, or NULL
 *
 * Update the contents of the global keyrings #GtkListStore.
 **/

static void
gkm_keyring_manager_update_keyrings (GKMKeyringManager *manager)
{
  gnome_keyring_list_keyring_names ((GnomeKeyringOperationGetListCallback) update_keyrings_worker_callback, 
                                    manager, NULL);
}
