/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanresolutionselector.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_RESOLUTION_SELECTOR_ERROR			(g_type_qname (GNOME_TYPE_SCAN_RESOLUTION_SELECTOR))
#define	GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)					(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_RESOLUTION_SELECTOR, GnomeScanResolutionSelectorPrivate))

typedef struct _GnomeScanResolutionSelectorPrivate		GnomeScanResolutionSelectorPrivate;

struct _GnomeScanResolutionSelectorPrivate {
  gboolean		dispose_has_run;
  GnomeScanContext	*context;
  GtkAdjustment		*adjustment;
  GtkWidget		*scale;
  GtkWidget		*spin;
  GtkWidget		*combo;
  GtkListStore		*liststore;
};

/********************************
 * 	      CALLBACKS		*
 ********************************/

void				gsrs_scanner_selected				(GnomeScanContext *context,
										 GnomeScanner *scanner,
										 GnomeScanResolutionSelector *gsrs);

gchar*				gsrs_format_value				(GtkScale *scale,
										 gdouble value);

void				gsrs_range_value_changed			(GtkRange *range,
										 GnomeScanResolutionSelector *gsrs);

void 				gsrs_combo_value_changed			(GtkComboBox *combo,
										 GnomeScanResolutionSelector *gsrs);

void				gsrs_chain_clicked				(GtkButton *button,
										 GnomeScanResolutionSelector *gsrs);

/********************************
 * 	       OTHERS		*
 ********************************/


/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_resolution_selector_set_property	(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_resolution_selector_get_property	(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_resolution_selector_finalize		(GObject *obj);

void				gnome_scan_resolution_selector_dispose		(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

enum
{
        ENUM_LABEL,
        ENUM_VALUE,
        ENUM_N_COLUMNS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (GnomeScanResolutionSelector, gnome_scan_resolution_selector, GTK_TYPE_TABLE);


void
gnome_scan_resolution_selector_class_init (GnomeScanResolutionSelectorClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scan_resolution_selector_set_property;
  gobject_class->get_property = gnome_scan_resolution_selector_get_property;
  gobject_class->dispose 	= gnome_scan_resolution_selector_dispose;
  /*   gobject_class->finalize 	= gnome_scan_resolution_selector_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanResolutionSelectorPrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */
}

void
gnome_scan_resolution_selector_init (GnomeScanResolutionSelector *widget)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  priv->dispose_has_run = FALSE;
  priv->context		= NULL;
  priv->adjustment	= NULL;
  priv->scale		= NULL;
  priv->spin		= NULL;
  priv->combo		= NULL;
}

void
gnome_scan_resolution_selector_dispose (GObject *obj)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);
  GnomeScanResolutionSelectorClass *b_klass = GNOME_SCAN_RESOLUTION_SELECTOR_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref objects */
  g_object_unref (priv->context);

  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_resolution_selector_finalize (GObject *obj)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorClass *b_klass = GNOME_SCAN_RESOLUTION_SELECTOR_GET_CLASS (obj);

  /*   GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS (b_klass)->finalize (obj); */
}

void
gnome_scan_resolution_selector_set_property (GObject *obj,
					     guint property_id,
					     const GValue *value,
					     GParamSpec *pspec)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    priv->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_resolution_selector_get_property (GObject *obj,
					     guint property_id,
					     GValue *value,
					     GParamSpec *pspec)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, priv->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_resolution_selector_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanResolutionSelector connected to @context.
 * 
 * Return value: a new #GnomeScanResolutionSelector
 **/
GtkWidget*
gnome_scan_resolution_selector_new (GnomeScanContext *context)
{
  GtkWidget *widget, *button, *icon;
  GnomeScanResolutionSelector *gsrs;
  GnomeScanResolutionSelectorPrivate *priv;
  GtkCellRenderer *renderer;	

  widget = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_RESOLUTION_SELECTOR,
				     "context", context,
				     NULL));

  gsrs = GNOME_SCAN_RESOLUTION_SELECTOR (widget);
  priv = GET_PRIVATE (gsrs);
  gtk_table_set_col_spacings (GTK_TABLE (widget), 6);
  gtk_table_set_row_spacings (GTK_TABLE (widget), 6);

  priv->scale = gtk_hscale_new (NULL);
  priv->adjustment = gtk_range_get_adjustment (GTK_RANGE (priv->scale));
  gtk_scale_set_draw_value (GTK_SCALE (priv->scale), FALSE);
  gtk_range_set_update_policy (GTK_RANGE (priv->scale), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_increments (GTK_RANGE (priv->scale),
			    5,	/* seems buggy, value is doubled */
			    100);

  priv->spin = gtk_spin_button_new (priv->adjustment, .5, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (priv->spin), GTK_UPDATE_CONTINUOUS);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (priv->spin), TRUE);

  priv->liststore = gtk_list_store_new (ENUM_N_COLUMNS,
  					G_TYPE_STRING,
					G_TYPE_VALUE);
  priv->combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL (priv->liststore));
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (priv->combo), renderer, TRUE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (priv->combo), renderer,
                                "text", ENUM_LABEL);

  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("Resolution")),
		    0, 1,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->scale,
		    1, 2,
		    0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->spin,
		    2, 3,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
                    priv->combo,
                    2, 3,
                    0, 1,
                    GTK_FILL,
                    GTK_FILL,
                    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("dpi")),
		    3, 4,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);

  g_signal_connect (GTK_RANGE (priv->scale),
		    "value-changed",
		    G_CALLBACK (gsrs_range_value_changed),
		    gsrs);

  g_signal_connect (GTK_COMBO_BOX (priv->combo),
                    "changed",
                    G_CALLBACK (gsrs_combo_value_changed),
                    gsrs);

  g_signal_connect (context,
		    "scanner-selected",
		    G_CALLBACK (gsrs_scanner_selected),
		    gsrs);

  return widget;
}

/********************************
 * 	      CALLBACKS		*
 ********************************/

void
gsrs_scanner_selected (GnomeScanContext *context,
		       GnomeScanner *scanner,
		       GnomeScanResolutionSelector *gsrs)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (gsrs);
  GnomeScanRange *range = gnome_scanner_get_resolution_range (scanner);
  GValueArray *enumeration =  gnome_scanner_get_resolution_enum (scanner);
  gdouble resolution;

  if (range || enumeration) {

    if (range) {
      gtk_widget_set_sensitive (GTK_WIDGET (gsrs), TRUE);
      gtk_widget_hide (priv->combo);
      gtk_widget_show (priv->scale);
      gtk_widget_show (priv->spin);
      gnome_scan_context_get_resolution (context, &resolution);
      gtk_range_set_range (GTK_RANGE (priv->scale), range->lower, range->upper);
      gtk_range_set_value (GTK_RANGE (priv->scale), resolution);
    }

    if (enumeration) {
      gtk_widget_set_sensitive (GTK_WIDGET (gsrs), TRUE);
      gtk_widget_hide (priv->scale);
      gtk_widget_hide (priv->spin);
      gtk_widget_show (priv->combo);
      
      GtkTreeIter iter;
      GValue *label, *value;
      gint i;

      label = g_new0(GValue, 1);
      g_value_init (label, G_TYPE_STRING);
      for (i = 0; i < enumeration->n_values; i++) {
        gtk_list_store_append (priv->liststore, &iter);
        gtk_list_store_set (priv->liststore, &iter,
                            ENUM_LABEL, g_strdup_printf("%.0f", g_value_get_double (enumeration->values+i)),
                            ENUM_VALUE, (enumeration->values+i),
                            -1);
      }

      gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo), 0);
    }

  }
  else {
    gtk_widget_set_sensitive (GTK_WIDGET (gsrs), FALSE);
  }
}

void
gsrs_range_value_changed (GtkRange *range,
		    GnomeScanResolutionSelector *gsrs)
{
  gnome_scan_context_set_resolution (GET_PRIVATE (gsrs)->context,
				     gtk_range_get_value (range));
}

void
gsrs_combo_value_changed (GtkComboBox *combo,
                    GnomeScanResolutionSelector *gsrs)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (gsrs);
  GtkTreeIter iter;
  GValue *value;

  gtk_combo_box_get_active_iter (GTK_COMBO_BOX (priv->combo), &iter);
  gtk_tree_model_get (GTK_TREE_MODEL (priv->liststore), &iter,
                                      ENUM_VALUE, &value,
                                      -1);
  gnome_scan_context_set_resolution (priv->context,
				     g_value_get_double (value));
}


/********************************
 * 	       OTHERS		*
 ********************************/

