/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION:gnome-scan-backend
 * @short_description: Probe scanners
 *
 * In Gnome Scan, the so called "backend" is only responsible to probe
 * devices. In fact, the scanner is only responsible to detect scanner
 * plug and unplug and trigger signal about that. Probing is just a
 * way to add already plugged scanner just like adding a hotplugged
 * scanner.
 *
 * The backend itseld does not maintain a public list of scanner, it's
 * up to the UI to maintain a list of scanner (e.g. through a
 * #GtkListStore).
 *
 * #GnomeScanBackend is the base class for real backend. If you want
 * to instanciate all backends, just initialize all #GnomeScanModules
 * and instanciate each g_type_children() of #GnomeScanBackend.
 **/
#include "gnome-scan-backend.h"
#include "gnome-scan-module.h"

enum
{
	PROP_0,
};

enum
{
	SCANNER_ADDED,
	SCANNER_REMOVED,
	PROBE_DONE,
	LAST_SIGNAL
};


static GObjectClass* parent_class = NULL;
static guint scan_backend_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GnomeScanBackend, gnome_scan_backend, G_TYPE_OBJECT);

static void
gnome_scan_backend_init (GnomeScanBackend *object)
{
}

static void
gnome_scan_backend_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

#if 0
static void
gnome_scan_backend_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_BACKEND (object));

	switch (prop_id)
	{
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gnome_scan_backend_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_BACKEND (object));

	switch (prop_id)
	{
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}
#endif

static void
gnome_scan_backend_scanner_added (GnomeScanBackend *backend, GnomeScanner *scanner, gpointer data)
{
	/* TODO: Add default signal handler implementation here */
}

static void
gnome_scan_backend_scanner_removed (GnomeScanBackend *backend, GnomeScanner *scanner, gpointer data)
{
	/* TODO: Add default signal handler implementation here */
}

static void
gnome_scan_backend_class_init (GnomeScanBackendClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

	object_class->finalize = gnome_scan_backend_finalize;

	klass->scanner_added = gnome_scan_backend_scanner_added;
	klass->scanner_removed = gnome_scan_backend_scanner_removed;

	/**
	 * GnomeScanBackend::scanner-added:
	 * @backend: the emitting #GnomeScanBackend
	 * @scanner: the newly added #GnomeScanner
	 *
	 * This backend is triggered each time it discover a new scanner
	 * during probe or on plug.
	 **/
	scan_backend_signals[SCANNER_ADDED] =
		g_signal_new ("scanner-added",
		              G_OBJECT_CLASS_TYPE (klass),
		              G_SIGNAL_RUN_LAST,
		              G_STRUCT_OFFSET (GnomeScanBackendClass, scanner_added),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__OBJECT,
		              G_TYPE_NONE, 1, GNOME_TYPE_SCANNER);

	/**
	 * GnomeScanBackend::scanner-removed:
	 * @backend: the emitting #GnomeScanBackend
	 * @scanner: the scanner removed
	 *
	 * This signal is emitted as a scanner is lost. Every reference to
	 * this scanner must be removed.
	 **/
	scan_backend_signals[SCANNER_REMOVED] =
		g_signal_new ("scanner-removed",
		              G_OBJECT_CLASS_TYPE (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (GnomeScanBackendClass, scanner_removed),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__OBJECT,
		              G_TYPE_NONE, 1, GNOME_TYPE_SCANNER);
	
	/**
	 * GnomeScanBackend::probe-done:
	 * @backend: the emitting #GnomeScanBackend
	 *
	 * This signal allow app to know if the backend has found a
	 * scanner during probe. If the #GnomeScanBackend::probe-done
	 * signal is emitted and no #GnomeScanBackend::scanner-added
	 * signal has been sent, this mean that no scanner has been found
	 * at all. Remember the backend does not handle a list of
	 * #GnomeScanner.
	 **/
	scan_backend_signals[PROBE_DONE] =
		g_signal_new ("probe-done",
					  G_OBJECT_CLASS_TYPE (klass),
					  G_SIGNAL_RUN_LAST,
					  G_STRUCT_OFFSET (GnomeScanBackendClass, probe_done),
					  NULL, NULL,
					  g_cclosure_marshal_VOID__VOID,
					  G_TYPE_NONE, 0);
}

/**
 * gnome_scan_backend_new:
 * @backend_type: the top level backend GType
 * 
 * Instanciate a new backend of type @type. It check wether @type is a
 * child type of #GnomeScanBackend.
 * 
 * Returns: a new #GnomeScanBackend.
 **/
GnomeScanBackend*
gnome_scan_backend_new (GType backend_type)
{
	g_return_val_if_fail (g_type_is_a (backend_type, GNOME_TYPE_SCAN_BACKEND), NULL);
	return GNOME_SCAN_BACKEND (g_object_new (backend_type, NULL));
}

/**
 * gnome_scan_backend_probe_scanners:
 * @backend: a #GnomeScanBackend
 * 
 * Trigger probe. Ideally called in a new thread through g_thread_create().
 *
 * See: g_thread_create()
 **/
void
gnome_scan_backend_probe_scanners (GnomeScanBackend *backend)
{
	GNOME_SCAN_BACKEND_GET_CLASS (backend)->probe_scanners (backend);
	gnome_scan_backend_probe_done (backend);
}

/**
 * gnome_scan_backend_add_scanner:
 * @backend: a #GnomeScanBackend
 * @scanner: a new #GnomeScanner
 * 
 * This function is used by child class to trigger
 * #GnomeScanBackend::scanner-added signal.
 **/
void
gnome_scan_backend_add_scanner (GnomeScanBackend *backend, GnomeScanner *scanner)
{
	g_signal_emit (backend, scan_backend_signals[SCANNER_ADDED], 0, scanner);
}

/**
 * gnome_scan_backend_remove_scanner:
 * @backend: a #GnomeScanBackend
 * @scanner: a deprecated #GnomeScanner
 * 
 * Trigger #GnomeScanBackend::scanner-removed upon @scanner. This
 * function is to be used only by child class.
 **/
void
gnome_scan_backend_remove_scanner (GnomeScanBackend *backend, GnomeScanner *scanner)
{
	g_signal_emit (backend, scan_backend_signals[SCANNER_REMOVED], 0, scanner);
}

/**
 * gnome_scan_backend_probe_done:
 * @backend: a #GnomeScanBackend
 * 
 * Trigger #GnomeScanBackend::probe-done signal. Use by child class.
 **/
void
gnome_scan_backend_probe_done (GnomeScanBackend *backend)
{
	g_signal_emit (backend, scan_backend_signals[PROBE_DONE], 0);
}
