/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>


#define GNOME_SCAN_TYPE_UNIT (gnome_scan_unit_get_type ())

#define GNOME_SCAN_TYPE_STATUS (gnome_scan_status_get_type ())

#define GNOME_SCAN_TYPE_POINT (gnome_scan_point_get_type ())
typedef struct _GnomeScanPoint GnomeScanPoint;

#define GNOME_SCAN_TYPE_EXTENT (gnome_scan_extent_get_type ())
typedef struct _GnomeScanExtent GnomeScanExtent;

#define GNOME_SCAN_TYPE_RANGE (gnome_scan_range_get_type ())
typedef struct _GnomeScanRange GnomeScanRange;

#define GNOME_SCAN_TYPE_FORMAT (gnome_scan_format_get_type ())
#define GNOME_SCAN_FORMAT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_FORMAT, GnomeScanFormat))
#define GNOME_SCAN_FORMAT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_FORMAT, GnomeScanFormatClass))
#define GNOME_SCAN_IS_FORMAT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_FORMAT))
#define GNOME_SCAN_IS_FORMAT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_FORMAT))
#define GNOME_SCAN_FORMAT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_FORMAT, GnomeScanFormatClass))

typedef struct _GnomeScanFormat GnomeScanFormat;
typedef struct _GnomeScanFormatClass GnomeScanFormatClass;
typedef struct _GnomeScanFormatPrivate GnomeScanFormatPrivate;

#define GNOME_SCAN_TYPE_ENUM_VALUE (gnome_scan_enum_value_get_type ())
typedef struct _GnomeScanEnumValue GnomeScanEnumValue;
#define _g_free0(var) (var = (g_free (var), NULL))
#define _g_type_class_unref0(var) ((var == NULL) ? NULL : (var = (g_type_class_unref (var), NULL)))
typedef struct _GnomeScanRectangle GnomeScanRectangle;

typedef enum  {
	GNOME_SCAN_UNIT_NONE = -1,
	GNOME_SCAN_UNIT_PIXEL,
	GNOME_SCAN_UNIT_POINTS,
	GNOME_SCAN_UNIT_INCH,
	GNOME_SCAN_UNIT_MM,
	GNOME_SCAN_UNIT_BIT,
	GNOME_SCAN_UNIT_DPI,
	GNOME_SCAN_UNIT_PERCENT,
	GNOME_SCAN_UNIT_MICROSECOND
} GnomeScanUnit;

typedef enum  {
	GNOME_SCAN_STATUS_UNKNOWN = 0,
	GNOME_SCAN_STATUS_FAILED = 100,
	GNOME_SCAN_STATUS_INITIALIZING = 200,
	GNOME_SCAN_STATUS_BUSY = 300,
	GNOME_SCAN_STATUS_UNCONFIGURED = 400,
	GNOME_SCAN_STATUS_READY = 500,
	GNOME_SCAN_STATUS_PROCESSING = 600,
	GNOME_SCAN_STATUS_DONE = 700
} GnomeScanStatus;

struct _GnomeScanPoint {
	double x;
	double y;
};

struct _GnomeScanExtent {
	double width;
	double height;
};

struct _GnomeScanRange {
	double min;
	double step;
	double max;
};

struct _GnomeScanFormat {
	GObject parent_instance;
	GnomeScanFormatPrivate * priv;
	const char* name;
	const char* desc;
	const char* domain;
	const char* icon_name;
	char** mime_types;
	gint mime_types_length1;
	char** suffixes;
	gint suffixes_length1;
};

struct _GnomeScanFormatClass {
	GObjectClass parent_class;
};

struct _GnomeScanEnumValue {
	GValue value;
	char* label;
	char* domain;
};

struct _GnomeScanRectangle {
	double x;
	double y;
	double width;
	double height;
};


static gpointer gnome_scan_format_parent_class = NULL;

GType gnome_scan_unit_get_type (void);
char* gnome_scan_unit_to_string (GnomeScanUnit unit);
GType gnome_scan_status_get_type (void);
GType gnome_scan_point_get_type (void);
GnomeScanPoint* gnome_scan_point_dup (const GnomeScanPoint* self);
void gnome_scan_point_free (GnomeScanPoint* self);
GType gnome_scan_extent_get_type (void);
GnomeScanExtent* gnome_scan_extent_dup (const GnomeScanExtent* self);
void gnome_scan_extent_free (GnomeScanExtent* self);
void gnome_scan_extent_init (GnomeScanExtent *self, double width, double height);
GType gnome_scan_range_get_type (void);
GnomeScanRange* gnome_scan_range_dup (const GnomeScanRange* self);
void gnome_scan_range_free (GnomeScanRange* self);
GType gnome_scan_format_get_type (void);
enum  {
	GNOME_SCAN_FORMAT_DUMMY_PROPERTY
};
static char** _vala_array_dup1 (char** self, int length);
static char** _vala_array_dup2 (char** self, int length);
GnomeScanFormat* gnome_scan_format_new (const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1);
GnomeScanFormat* gnome_scan_format_construct (GType object_type, const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1);
static void gnome_scan_format_finalize (GObject* obj);
GType gnome_scan_enum_value_get_type (void);
GnomeScanEnumValue* gnome_scan_enum_value_dup (const GnomeScanEnumValue* self);
void gnome_scan_enum_value_free (GnomeScanEnumValue* self);
void gnome_scan_enum_value_copy (const GnomeScanEnumValue* self, GnomeScanEnumValue* dest);
void gnome_scan_enum_value_destroy (GnomeScanEnumValue* self);
void gnome_scan_enum_value_init (GnomeScanEnumValue *self, const GValue* value, const char* label, const char* domain);
#define GNOME_SCAN_MM_PER_INCH 25.4
char* gnome_scan_enum_get_nick (GType enum_type, gint value);
char* gnome_scan_flags_get_nicks (GType flags_type, gint value);
void gnome_scan_warn_unsupported_unit (GnomeScanUnit unit);
double gnome_scan_convert_from_mm (double val, GnomeScanUnit to, double res);
double gnome_scan_convert_to_mm (double val, GnomeScanUnit from, double res);
double gnome_scan_convert (double val, GnomeScanUnit from, GnomeScanUnit to, double res);
void gnome_scan_rectangle_free (GnomeScanRectangle* self);
void gnome_scan_rectangle_rotate (GnomeScanRectangle* self, GnomeScanRectangle* area, gint angle);
void gnome_scan_rectangle_convert_to_mm (GnomeScanRectangle* self, GnomeScanUnit from, double res);
void gnome_scan_rectangle_convert_from_mm (GnomeScanRectangle* self, GnomeScanUnit to, double res);
void gnome_scan_rectangle_convert (GnomeScanRectangle* self, GnomeScanUnit from, GnomeScanUnit to, double res);
GnomeScanRectangle* gnome_scan_rectangle_new (void);
GnomeScanRectangle* gnome_scan_rectangle_new (void);
static void gnome_scan_rectangle_instance_init (GnomeScanRectangle * self);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);




GType gnome_scan_unit_get_type (void) {
	static GType gnome_scan_unit_type_id = 0;
	if (G_UNLIKELY (gnome_scan_unit_type_id == 0)) {
		static const GEnumValue values[] = {{GNOME_SCAN_UNIT_NONE, "GNOME_SCAN_UNIT_NONE", "none"}, {GNOME_SCAN_UNIT_PIXEL, "GNOME_SCAN_UNIT_PIXEL", "pixel"}, {GNOME_SCAN_UNIT_POINTS, "GNOME_SCAN_UNIT_POINTS", "points"}, {GNOME_SCAN_UNIT_INCH, "GNOME_SCAN_UNIT_INCH", "inch"}, {GNOME_SCAN_UNIT_MM, "GNOME_SCAN_UNIT_MM", "mm"}, {GNOME_SCAN_UNIT_BIT, "GNOME_SCAN_UNIT_BIT", "bit"}, {GNOME_SCAN_UNIT_DPI, "GNOME_SCAN_UNIT_DPI", "dpi"}, {GNOME_SCAN_UNIT_PERCENT, "GNOME_SCAN_UNIT_PERCENT", "percent"}, {GNOME_SCAN_UNIT_MICROSECOND, "GNOME_SCAN_UNIT_MICROSECOND", "microsecond"}, {0, NULL, NULL}};
		gnome_scan_unit_type_id = g_enum_register_static ("GnomeScanUnit", values);
	}
	return gnome_scan_unit_type_id;
}


char* gnome_scan_unit_to_string (GnomeScanUnit unit) {
	char* result;
	switch (unit) {
		case GNOME_SCAN_UNIT_PIXEL:
		{
			result = g_strdup (_ ("px"));
			return result;
		}
		case GNOME_SCAN_UNIT_POINTS:
		{
			result = g_strdup (_ ("pt"));
			return result;
		}
		case GNOME_SCAN_UNIT_MM:
		{
			result = g_strdup (_ ("mm"));
			return result;
		}
		case GNOME_SCAN_UNIT_BIT:
		{
			result = g_strdup (_ ("bit"));
			return result;
		}
		case GNOME_SCAN_UNIT_DPI:
		{
			result = g_strdup (_ ("dpi"));
			return result;
		}
		case GNOME_SCAN_UNIT_PERCENT:
		{
			result = g_strdup ("%");
			return result;
		}
		case GNOME_SCAN_UNIT_MICROSECOND:
		{
			result = g_strdup (_ ("µs"));
			return result;
		}
		default:
		{
			result = g_strdup ("");
			return result;
		}
	}
}



GType gnome_scan_status_get_type (void) {
	static GType gnome_scan_status_type_id = 0;
	if (G_UNLIKELY (gnome_scan_status_type_id == 0)) {
		static const GEnumValue values[] = {{GNOME_SCAN_STATUS_UNKNOWN, "GNOME_SCAN_STATUS_UNKNOWN", "unknown"}, {GNOME_SCAN_STATUS_FAILED, "GNOME_SCAN_STATUS_FAILED", "failed"}, {GNOME_SCAN_STATUS_INITIALIZING, "GNOME_SCAN_STATUS_INITIALIZING", "initializing"}, {GNOME_SCAN_STATUS_BUSY, "GNOME_SCAN_STATUS_BUSY", "busy"}, {GNOME_SCAN_STATUS_UNCONFIGURED, "GNOME_SCAN_STATUS_UNCONFIGURED", "unconfigured"}, {GNOME_SCAN_STATUS_READY, "GNOME_SCAN_STATUS_READY", "ready"}, {GNOME_SCAN_STATUS_PROCESSING, "GNOME_SCAN_STATUS_PROCESSING", "processing"}, {GNOME_SCAN_STATUS_DONE, "GNOME_SCAN_STATUS_DONE", "done"}, {0, NULL, NULL}};
		gnome_scan_status_type_id = g_enum_register_static ("GnomeScanStatus", values);
	}
	return gnome_scan_status_type_id;
}


GnomeScanPoint* gnome_scan_point_dup (const GnomeScanPoint* self) {
	GnomeScanPoint* dup;
	dup = g_new0 (GnomeScanPoint, 1);
	memcpy (dup, self, sizeof (GnomeScanPoint));
	return dup;
}


void gnome_scan_point_free (GnomeScanPoint* self) {
	g_free (self);
}


GType gnome_scan_point_get_type (void) {
	static GType gnome_scan_point_type_id = 0;
	if (gnome_scan_point_type_id == 0) {
		gnome_scan_point_type_id = g_boxed_type_register_static ("GnomeScanPoint", (GBoxedCopyFunc) gnome_scan_point_dup, (GBoxedFreeFunc) gnome_scan_point_free);
	}
	return gnome_scan_point_type_id;
}


void gnome_scan_extent_init (GnomeScanExtent *self, double width, double height) {
	memset (self, 0, sizeof (GnomeScanExtent));
	(*self).width = width;
	(*self).height = height;
}


GnomeScanExtent* gnome_scan_extent_dup (const GnomeScanExtent* self) {
	GnomeScanExtent* dup;
	dup = g_new0 (GnomeScanExtent, 1);
	memcpy (dup, self, sizeof (GnomeScanExtent));
	return dup;
}


void gnome_scan_extent_free (GnomeScanExtent* self) {
	g_free (self);
}


GType gnome_scan_extent_get_type (void) {
	static GType gnome_scan_extent_type_id = 0;
	if (gnome_scan_extent_type_id == 0) {
		gnome_scan_extent_type_id = g_boxed_type_register_static ("GnomeScanExtent", (GBoxedCopyFunc) gnome_scan_extent_dup, (GBoxedFreeFunc) gnome_scan_extent_free);
	}
	return gnome_scan_extent_type_id;
}


GnomeScanRange* gnome_scan_range_dup (const GnomeScanRange* self) {
	GnomeScanRange* dup;
	dup = g_new0 (GnomeScanRange, 1);
	memcpy (dup, self, sizeof (GnomeScanRange));
	return dup;
}


void gnome_scan_range_free (GnomeScanRange* self) {
	g_free (self);
}


GType gnome_scan_range_get_type (void) {
	static GType gnome_scan_range_type_id = 0;
	if (gnome_scan_range_type_id == 0) {
		gnome_scan_range_type_id = g_boxed_type_register_static ("GnomeScanRange", (GBoxedCopyFunc) gnome_scan_range_dup, (GBoxedFreeFunc) gnome_scan_range_free);
	}
	return gnome_scan_range_type_id;
}


static char** _vala_array_dup1 (char** self, int length) {
	char** result;
	int i;
	result = g_new0 (char*, length);
	for (i = 0; i < length; i++) {
		result[i] = g_strdup (self[i]);
	}
	return result;
}


static char** _vala_array_dup2 (char** self, int length) {
	char** result;
	int i;
	result = g_new0 (char*, length);
	for (i = 0; i < length; i++) {
		result[i] = g_strdup (self[i]);
	}
	return result;
}


GnomeScanFormat* gnome_scan_format_construct (GType object_type, const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1) {
	GnomeScanFormat * self;
	char** _tmp1_;
	char** _tmp0_;
	char** _tmp3_;
	char** _tmp2_;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (desc != NULL, NULL);
	g_return_val_if_fail (domain != NULL, NULL);
	g_return_val_if_fail (icon_name != NULL, NULL);
	self = (GnomeScanFormat*) g_object_new (object_type, NULL);
	self->name = name;
	self->desc = desc;
	self->domain = domain;
	self->icon_name = icon_name;
	self->mime_types = (_tmp1_ = (_tmp0_ = mime_types, (_tmp0_ == NULL) ? ((gpointer) _tmp0_) : _vala_array_dup1 (_tmp0_, mime_types_length1)), self->mime_types = (_vala_array_free (self->mime_types, self->mime_types_length1, (GDestroyNotify) g_free), NULL), self->mime_types_length1 = mime_types_length1, _tmp1_);
	self->suffixes = (_tmp3_ = (_tmp2_ = suffixes, (_tmp2_ == NULL) ? ((gpointer) _tmp2_) : _vala_array_dup2 (_tmp2_, suffixes_length1)), self->suffixes = (_vala_array_free (self->suffixes, self->suffixes_length1, (GDestroyNotify) g_free), NULL), self->suffixes_length1 = suffixes_length1, _tmp3_);
	return self;
}


GnomeScanFormat* gnome_scan_format_new (const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1) {
	return gnome_scan_format_construct (GNOME_SCAN_TYPE_FORMAT, name, desc, domain, icon_name, mime_types, mime_types_length1, suffixes, suffixes_length1);
}


static void gnome_scan_format_class_init (GnomeScanFormatClass * klass) {
	gnome_scan_format_parent_class = g_type_class_peek_parent (klass);
	G_OBJECT_CLASS (klass)->finalize = gnome_scan_format_finalize;
}


static void gnome_scan_format_instance_init (GnomeScanFormat * self) {
}


static void gnome_scan_format_finalize (GObject* obj) {
	GnomeScanFormat * self;
	self = GNOME_SCAN_FORMAT (obj);
	self->mime_types = (_vala_array_free (self->mime_types, self->mime_types_length1, (GDestroyNotify) g_free), NULL);
	self->suffixes = (_vala_array_free (self->suffixes, self->suffixes_length1, (GDestroyNotify) g_free), NULL);
	G_OBJECT_CLASS (gnome_scan_format_parent_class)->finalize (obj);
}


GType gnome_scan_format_get_type (void) {
	static GType gnome_scan_format_type_id = 0;
	if (gnome_scan_format_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GnomeScanFormatClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gnome_scan_format_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GnomeScanFormat), 0, (GInstanceInitFunc) gnome_scan_format_instance_init, NULL };
		gnome_scan_format_type_id = g_type_register_static (G_TYPE_OBJECT, "GnomeScanFormat", &g_define_type_info, 0);
	}
	return gnome_scan_format_type_id;
}


void gnome_scan_enum_value_init (GnomeScanEnumValue *self, const GValue* value, const char* label, const char* domain) {
	GValue _tmp1_;
	GValue _tmp0_ = {0};
	char* _tmp2_;
	char* _tmp3_;
	g_return_if_fail (label != NULL);
	memset (self, 0, sizeof (GnomeScanEnumValue));
	(*self).value = (_tmp1_ = G_IS_VALUE (&(*value)) ? (g_value_init (&_tmp0_, G_VALUE_TYPE (&(*value))), g_value_copy (&(*value), &_tmp0_), _tmp0_) : (*value), G_IS_VALUE (&(*self).value) ? (g_value_unset (&(*self).value), NULL) : NULL, _tmp1_);
	(*self).label = (_tmp2_ = g_strdup (label), _g_free0 ((*self).label), _tmp2_);
	(*self).domain = (_tmp3_ = g_strdup (domain), _g_free0 ((*self).domain), _tmp3_);
}


void gnome_scan_enum_value_copy (const GnomeScanEnumValue* self, GnomeScanEnumValue* dest) {
	GValue _tmp0_ = {0};
	dest->value = G_IS_VALUE (&self->value) ? (g_value_init (&_tmp0_, G_VALUE_TYPE (&self->value)), g_value_copy (&self->value, &_tmp0_), _tmp0_) : self->value;
	dest->label = g_strdup (self->label);
	dest->domain = g_strdup (self->domain);
}


void gnome_scan_enum_value_destroy (GnomeScanEnumValue* self) {
	G_IS_VALUE (&self->value) ? (g_value_unset (&self->value), NULL) : NULL;
	_g_free0 (self->label);
	_g_free0 (self->domain);
}


GnomeScanEnumValue* gnome_scan_enum_value_dup (const GnomeScanEnumValue* self) {
	GnomeScanEnumValue* dup;
	dup = g_new0 (GnomeScanEnumValue, 1);
	gnome_scan_enum_value_copy (self, dup);
	return dup;
}


void gnome_scan_enum_value_free (GnomeScanEnumValue* self) {
	gnome_scan_enum_value_destroy (self);
	g_free (self);
}


GType gnome_scan_enum_value_get_type (void) {
	static GType gnome_scan_enum_value_type_id = 0;
	if (gnome_scan_enum_value_type_id == 0) {
		gnome_scan_enum_value_type_id = g_boxed_type_register_static ("GnomeScanEnumValue", (GBoxedCopyFunc) gnome_scan_enum_value_dup, (GBoxedFreeFunc) gnome_scan_enum_value_free);
	}
	return gnome_scan_enum_value_type_id;
}


char* gnome_scan_enum_get_nick (GType enum_type, gint value) {
	char* result;
	GEnumClass* klass;
	GEnumValue* enum_value;
	klass = (GEnumClass*) g_type_class_ref (enum_type);
	enum_value = g_enum_get_value (klass, value);
	if (enum_value == NULL) {
		result = g_strdup_printf ("%i", value);
		_g_type_class_unref0 (klass);
		return result;
	} else {
		result = g_strdup (enum_value->value_nick);
		_g_type_class_unref0 (klass);
		return result;
	}
	_g_type_class_unref0 (klass);
}


char* gnome_scan_flags_get_nicks (GType flags_type, gint value) {
	char* result;
	GFlagsClass* klass;
	GFlagsValue* flags_value;
	char* nicks;
	klass = (GFlagsClass*) g_type_class_ref (flags_type);
	flags_value = NULL;
	nicks = g_strdup ("{");
	{
		gboolean _tmp0_;
		_tmp0_ = TRUE;
		while (TRUE) {
			gboolean _tmp1_;
			if (!_tmp0_) {
				value = value ^ flags_value->value;
			}
			_tmp0_ = FALSE;
			_tmp1_ = FALSE;
			if (value > 0) {
				_tmp1_ = (flags_value = g_flags_get_first_value (klass, (guint) value)) != NULL;
			} else {
				_tmp1_ = FALSE;
			}
			if (!_tmp1_) {
				break;
			}
			if (flags_value != NULL) {
				char* _tmp3_;
				if (g_utf8_strlen (nicks, -1) > 1) {
					char* _tmp2_;
					nicks = (_tmp2_ = g_strconcat (nicks, ",", NULL), _g_free0 (nicks), _tmp2_);
				}
				nicks = (_tmp3_ = g_strconcat (nicks, flags_value->value_nick, NULL), _g_free0 (nicks), _tmp3_);
			}
		}
	}
	result = g_strconcat (nicks, "}", NULL);
	_g_type_class_unref0 (klass);
	_g_free0 (nicks);
	return result;
}


void gnome_scan_warn_unsupported_unit (GnomeScanUnit unit) {
	char* _tmp0_;
	g_warning ("gnome-scan-common.vala:165: Unit %s conversion not supported.", _tmp0_ = gnome_scan_enum_get_nick (GNOME_SCAN_TYPE_UNIT, (gint) unit));
	_g_free0 (_tmp0_);
}


double gnome_scan_convert (double val, GnomeScanUnit from, GnomeScanUnit to, double res) {
	double result;
	if (from == to) {
		result = val;
		return result;
	}
	switch (from) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (from);
			result = val;
			return result;
		}
		default:
		{
			switch (to) {
				case GNOME_SCAN_UNIT_NONE:
				case GNOME_SCAN_UNIT_BIT:
				case GNOME_SCAN_UNIT_PERCENT:
				case GNOME_SCAN_UNIT_MICROSECOND:
				case GNOME_SCAN_UNIT_DPI:
				{
					gnome_scan_warn_unsupported_unit (to);
					result = val;
					return result;
				}
				default:
				{
					result = gnome_scan_convert_from_mm (gnome_scan_convert_to_mm (val, from, res), to, res);
					return result;
				}
			}
		}
	}
}


double gnome_scan_convert_from_mm (double val, GnomeScanUnit to, double res) {
	double result;
	switch (to) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (to);
			result = val;
			return result;
		}
		case GNOME_SCAN_UNIT_MM:
		{
			result = val;
			return result;
		}
		case GNOME_SCAN_UNIT_PIXEL:
		{
			result = val / (GNOME_SCAN_MM_PER_INCH / res);
			return result;
		}
	}
	result = (double) 0;
	return result;
}


double gnome_scan_convert_to_mm (double val, GnomeScanUnit from, double res) {
	double result;
	switch (from) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (from);
			result = val;
			return result;
		}
		case GNOME_SCAN_UNIT_MM:
		{
			result = val;
			return result;
		}
		case GNOME_SCAN_UNIT_PIXEL:
		{
			result = val * (GNOME_SCAN_MM_PER_INCH / res);
			return result;
		}
	}
	result = (double) 0;
	return result;
}


void gnome_scan_rectangle_rotate (GnomeScanRectangle* self, GnomeScanRectangle* area, gint angle) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (area != NULL);
	angle = angle % 360;
	switch (angle) {
		case 0:
		{
			break;
		}
		case 270:
		{
			self->width = self->height;
			self->height = self->width;
			self->x = self->y;
			self->y = MAX ((area->width - self->x) - self->width, area->x);
			break;
		}
		case 180:
		{
			self->x = MAX ((area->width - self->x) - self->width, area->x);
			self->y = MAX ((area->height - self->y) - self->height, area->y);
			break;
		}
		case 90:
		{
			self->width = self->height;
			self->height = self->width;
			self->y = self->x;
			self->x = MAX ((area->height - self->y) - self->height, area->y);
			break;
		}
		default:
		{
			g_warning ("gnome-scan-common.vala:284: %i degree rotation is not supported", angle);
			break;
		}
	}
}


void gnome_scan_rectangle_convert (GnomeScanRectangle* self, GnomeScanUnit from, GnomeScanUnit to, double res) {
	g_return_if_fail (self != NULL);
	gnome_scan_rectangle_convert_to_mm (self, from, res);
	gnome_scan_rectangle_convert_from_mm (self, to, res);
}


void gnome_scan_rectangle_convert_to_mm (GnomeScanRectangle* self, GnomeScanUnit from, double res) {
	g_return_if_fail (self != NULL);
	self->x = gnome_scan_convert_to_mm (self->x, from, res);
	self->y = gnome_scan_convert_to_mm (self->y, from, res);
	self->width = gnome_scan_convert_to_mm (self->width, from, res);
	self->height = gnome_scan_convert_to_mm (self->height, from, res);
}


void gnome_scan_rectangle_convert_from_mm (GnomeScanRectangle* self, GnomeScanUnit to, double res) {
	g_return_if_fail (self != NULL);
	self->x = gnome_scan_convert_from_mm (self->x, to, res);
	self->y = gnome_scan_convert_from_mm (self->y, to, res);
	self->width = gnome_scan_convert_from_mm (self->width, to, res);
	self->height = gnome_scan_convert_from_mm (self->height, to, res);
}


GnomeScanRectangle* gnome_scan_rectangle_new (void) {
	GnomeScanRectangle* self;
	self = g_slice_new0 (GnomeScanRectangle);
	gnome_scan_rectangle_instance_init (self);
	return self;
}


static void gnome_scan_rectangle_instance_init (GnomeScanRectangle * self) {
}


void gnome_scan_rectangle_free (GnomeScanRectangle* self) {
	g_slice_free (GnomeScanRectangle, self);
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}




