/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* smb-method.h - VFS modules for SMB
 *
 *  Copyright (C) 2001 Red Hat Inc,
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Alexander Larsson <alexl@redhat.com>
 *         Tambet Ingo <tambet@ximian.com>
 *
 * Some inspiration comes from the smbfs implementation in midnight commander
 * by Wayne Roberts <wroberts1@home.com>. 
 */


#include "config.h"
#include "gnome-vfs-extras-i18n.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <time.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <glib/gutils.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-module-callback-module-api.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include "smb-method.h"

#undef PACKAGE
#undef VERSION
#include "samba/include/config.h"
/* don't load crap in "samba/include/includes.h" we don't use and which 
   conflicts with definitions in other includes */
#undef HAVE_LIBREADLINE
#define NO_CONFIG_H
#define BOOL_DEFINED
#undef MAX
#undef MIN

#include "samba/include/includes.h"
#undef strcpy

typedef struct _SmbConnection SmbConnection;
typedef struct _SmbHandle SmbHandle;
typedef struct _SmbDirHandle SmbDirHandle;
typedef struct _SmbDirContent SmbDirContent;
typedef struct _SmbURIInfo SmbURIInfo;
typedef struct _SmbLookupData SmbLookupData;

typedef enum {
	SMB_VIRTUAL_TYPE_ROOT,
	SMB_VIRTUAL_TYPE_WORKGROUP,
	SMB_VIRTUAL_TYPE_WORKGROUP_FILE,
	SMB_VIRTUAL_TYPE_HOST,
	SMB_VIRTUAL_TYPE_HOST_FILE,
	SMB_VIRTUAL_TYPE_SHARE,
	SMB_VIRTUAL_TYPE_SHARE_DIR,
	SMB_VIRTUAL_TYPE_SHARE_FILE,

	SMB_VIRTUAL_TYPE_UNKNOWN
} SmbVirtualFileType;

struct _SmbConnection {
	struct cli_state *cli;
	char *server;
	char *share;
	gboolean needs_auth;
	char *user;
	char *password;
};

struct _SmbHandle {
	SmbConnection *connection;
	gboolean is_data;
	char *file_data;
	int fnum;
	GnomeVFSFileOffset offset;
	GnomeVFSFileOffset file_size;
};

struct _SmbDirContent {
	char *name;
	time_t mtime;
	time_t atime;
	time_t ctime;
	SmbVirtualFileType virtual_type;
	GnomeVFSFilePermissions permissions;
	uid_t uid;
	gid_t gid;
	SMB_OFF_T size;
};

struct _SmbDirHandle {
	char *server;
	GList *contents;
};

struct _SmbURIInfo {
	gchar *host_name;
	gchar *share_name;
	gchar *path;
	gchar *user_name;
	gchar *password;

	SmbVirtualFileType type;
};

struct _SmbLookupData {
	gchar *path;
	SmbVirtualFileType type;
	SmbConnection *connection;
};

/* Global vars: */
static GPrivate *dir_key = NULL;
static GMutex *samba_lock = NULL;

extern pstring global_myname;

static GNode *connection_root = NULL;

#undef DEBUG_SMB_ENABLE
#undef DEBUG_SMB_LOCKS
#undef DEBUG_SMB_2

#ifdef DEBUG_SMB_ENABLE
#define DEBUG_SMB(x) g_print x
#else
#define DEBUG_SMB(x) 
#endif

#ifdef DEBUG_SMB_LOCKS
#define LOCK_SAMBA() 	{g_mutex_lock (samba_lock); g_print ("LOCK %s\n", G_GNUC_PRETTY_FUNCTION);}
#define UNLOCK_SAMBA() 	{g_print ("UNLOCK %s\n", G_GNUC_PRETTY_FUNCTION); g_mutex_unlock (samba_lock);}
#else
#define LOCK_SAMBA() 	g_mutex_lock (samba_lock)
#define UNLOCK_SAMBA() 	g_mutex_unlock (samba_lock)
#endif

#ifdef DEBUG_SMB_2
#define DEBUG_SMB2(x) g_print x
#else
#define DEBUG_SMB2(x) 
#endif

static void smb_connection_free_all (void);
static GnomeVFSResult smb_host_connection (SmbURIInfo *info,
					   gboolean find_only,
					   gboolean ask_password,
					   SmbConnection **host);

static char *
unix_filename_to_dos (const char *filename)
{
	int len, i;
	char *dos_filename;
	
	dos_filename = g_strdup (filename);
	len = strlen (dos_filename);
	for (i = 0; i < len; i++) {
		if (dos_filename[i] == '/') {
			dos_filename[i] = '\\';
		}
	}
	
	return dos_filename;
}

static gchar *
get_default_user_name (void)
{
	return g_strdup (g_get_user_name());
}

static GnomeVFSResult
gnome_vfs_result_from_cli (struct cli_state *cli)
{
	int err;
	GnomeVFSResult res;
	guint8 errclass;
	guint32 errnum;
	guint32 nt_rpc_error;
	
	err = cli_error (cli, &errclass, &errnum, &nt_rpc_error);

	/* Stupid hack: If remote end closes connection and read/write error
	 * occurs, cli_error doesn't doesn't show anything. */
	if (err == 0) {
		res = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
	} else if (errclass == ERRSRV && errnum == ERRbadpw) {
		res = GNOME_VFS_ERROR_LOGIN_FAILED;
	} else {
		res = gnome_vfs_result_from_errno_code (err);
	}

	return res;
}

static SmbConnection *
smb_connection_new (struct cli_state *cli, const gchar *server, const gchar *share)
{
	SmbConnection *smbcon;

	smbcon = g_new0 (SmbConnection, 1);
	smbcon->cli = cli;
	smbcon->server = g_strdup (server);
	smbcon->needs_auth = FALSE;
	if (share) {
		smbcon->share = g_strdup (share);
	}

	return smbcon;
}

static void
smb_connection_free_one (SmbConnection *connection)
{
	DEBUG_SMB (("smb_connection_free: freeing connection %s -> %s\n",
		    connection->server, connection->share));

	cli_shutdown (connection->cli);
	g_free (connection->cli);
	g_free (connection->server);
	g_free (connection->share);
	g_free (connection->user);
	g_free (connection->password);

	g_free (connection);
}

static void
connection_free_cb (GNode *node, gpointer data)
{
	smb_connection_free_one (node->data);
	g_node_destroy (node);
}

static void
smb_connection_free (SmbConnection *connection)
{
	SmbConnection *c;
	GNode *node;

	/* Fisrt, check whether it's root connection */
	if (connection_root->data == connection) {
		smb_connection_free_all ();
		return;
	}

	/* Find the host connection and free it recursively */
	node = g_node_first_child (connection_root);
	while (node) {
		c = node->data;

		if ((g_strcasecmp (c->server, connection->server)) == 0) {
			break;
		}

		node = node->next;
	}

	if (node == NULL) {
		g_warning ("smb_connection_free: couldn't find host connection %s",
			   connection->server);
		return;
	}

	g_node_children_foreach (node, G_TRAVERSE_ALL, connection_free_cb, NULL);

	smb_connection_free_one (node->data);
	g_node_destroy (node);
}

static void
smb_connection_free_if_necessary (SmbConnection *connection, GnomeVFSResult result)
{
	g_warning ("smb_connection_free_if_necessary: %d\n", (guint) result);
	if (result == GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE)
		smb_connection_free (connection);
}

static void
smb_connection_free_all (void)
{
	GNode *node;

	node = g_node_first_child (connection_root);
	while (node) {
		SmbConnection *connection = node->data;

		smb_connection_free (connection);
		node = node->next;
	}

	smb_connection_free_one (node->data);
	g_node_destroy (node);
}

static GnomeVFSResult
smb_server_connection_new_from_ip (struct in_addr *ip,
				   const char *fallback_server_name,
				   const char *share,
				   const char *user,
				   const char *password,
				   SmbConnection **connection)
{
	struct cli_state *cli;
	struct nmb_name called, calling;
	char servername[256];
	pstring workgroup;
	SmbConnection *smbcon;
	const char *server;
	gchar *user_copy;
	GnomeVFSResult res;

	DEBUG_SMB (("smb_server_connection_new_from_ip (ip: %s, share: %s, user: %s, password: %s)\n",
		    inet_ntoa (*ip), (share)?share:"NULL", user, password));

	if (name_status_find (0x20, *ip, servername)) {
		server = servername;
	} else {
		server = fallback_server_name;
	}

	make_nmb_name (&calling, global_myname, 0x0);
	make_nmb_name (&called , server, 0x20);

	DEBUG_SMB (("smb_server_connection_new_from_ip: caller.name: %s, called.name: %s\n",
		    calling.name, called.name));
	
	/* Read the workgroup from the config file */
	pstrcpy (workgroup, lp_workgroup ());

	cli = cli_initialise (NULL);
	
	if (!cli) {
		g_warning ("Couldn't initialize cli");
		return GNOME_VFS_ERROR_INTERNAL;
	}	
	if (!cli_set_port (cli, SMB_PORT)) {
		g_warning ("Couldn't set port");
		res = gnome_vfs_result_from_cli (cli);
		cli_shutdown (cli);
		free (cli);
		return res;
	}

	if (!cli_connect (cli, server, ip)) {
		g_warning ("Couldn't connect to server");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	cli->protocol = PROTOCOL_NT1;

	if (!cli_session_request (cli, &calling, &called)) {
		g_warning ("Couldn't request session");
		res = gnome_vfs_result_from_cli (cli);
		/* TODO: smbclient tries some alternative names here */ 
		cli_shutdown (cli);
		free (cli);
		return res;
	}

	if (!cli_negprot (cli)) {
		g_warning ("Couldn't negotiate protocol");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_LOGIN_FAILED;
	}

	if (user == NULL) {
		user_copy = get_default_user_name ();
		password = "";
	} else {
		user_copy = g_strdup (user);
	}

	if (!cli_session_setup (cli,
				/* Username: */
				user_copy,
				/* Password: */
				(char *)password, strlen (password),
				/* NT Password: */
				(char *)password, strlen (password),
				/* Workgroup: */
				workgroup)) {
		/* Try anon login */
		if (!cli_session_setup (cli,
					"",
					"", 0,
					"", 0,
					workgroup)) { 
			cli_shutdown (cli);
			g_free (user_copy);
			free (cli);
			return GNOME_VFS_ERROR_LOGIN_FAILED;
		}
		/* Anon login ok */
		g_free (user_copy);
		user_copy = g_strdup ("");
		password = "";
	}

	/* Connect to the share */
	if (share && !cli_send_tconX (cli,
				      (char *)share,
				      "?????",
				      (char *)password, strlen (password)+1)) {
		g_warning ("couldn't send tconX: res: %s",
			   cli_errstr (cli));
		res = gnome_vfs_result_from_cli (cli);
		cli_shutdown (cli);
		g_free (user_copy);
		free(cli);
		return res;
	}

	smbcon = smb_connection_new (cli, server, share);

	smbcon->user = user_copy;
	smbcon->password = g_strdup (password);

	*connection = smbcon;

	return GNOME_VFS_OK;
}


static GnomeVFSResult
smb_server_connection_new (const char *server,
			   const char *share,
			   const char *user,
			   const char *password,
			   SmbConnection **connection)
{
	struct in_addr ip;

	DEBUG_SMB (("smb_server_connection_new (server: %s, share: %s, user: %s, password: %s)\n",
		    server, (share)?share:"NULL", user, password));

	if (!resolve_name (server, &ip, 0x20)) {
		DEBUG_SMB (("smb_server_connection_new: Couldn't resolve name %s\n", server));
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	return smb_server_connection_new_from_ip (&ip, server, share,
						  user, password, connection);
}

static GnomeVFSResult
smb_server_connection_new_ask_password (const char *server, const char *share, SmbConnection **connection)
{
	gchar *user, *password;
	GnomeVFSModuleCallbackAuthenticationIn in_args;
	GnomeVFSModuleCallbackAuthenticationOut out_args;
	GnomeVFSResult result = GNOME_VFS_ERROR_ACCESS_DENIED;

	memset (&in_args, 0, sizeof (in_args));
	in_args.uri = g_strdup_printf ("smb://%s", server);

	while (TRUE) {
		memset (&out_args, 0, sizeof (out_args));
		/* We can't keep the lock here, as we may deadlock calling back into
		   the app */
		UNLOCK_SAMBA();
		gnome_vfs_module_callback_invoke (GNOME_VFS_MODULE_CALLBACK_AUTHENTICATION,
						  &in_args, sizeof (in_args), 
						  &out_args, sizeof (out_args));

		LOCK_SAMBA();
		if (out_args.username == NULL)
			break;

		user = out_args.username;
		password = out_args.password;

		result = smb_server_connection_new (server, share,
						    user, password, connection);
		if (result == GNOME_VFS_OK) {
			(*connection)->needs_auth = FALSE;
			break;
		}

		g_free (user);
		g_free (password);
	}

	g_free (in_args.uri);

	return result;
}

static void
smb_dir_content_destroy (SmbDirContent *content)
{
	g_free (content->name);
	g_free (content);
}

static void
smb_dir_handle_destroy (SmbDirHandle *handle)
{
	GList *list;
	SmbDirContent *dir_content;

	if (!handle)
		return;
	
	list = handle->contents;
	while (list) {
		dir_content = (SmbDirContent *)list->data;
		smb_dir_content_destroy (dir_content);

		list = g_list_next (list);
	}
	g_free (handle->server);
	g_list_free (handle->contents);

	g_free (handle);
}

static gboolean
get_a_master_browser (struct in_addr *ip)
{
	struct in_addr *bcast, *ip_list;
	int iface;
	int count;
	int fd;

	fd = open_socket_in (SOCK_DGRAM,
			     0, 3,
                             interpret_addr(lp_socket_address()),
			     TRUE);

	if (fd < 0)
		return FALSE;

	set_socket_options (fd, "SO_BROADCAST");

	for (iface = iface_count () -1; iface >= 0; iface--) {
		bcast = iface_n_bcast (iface);
		ip_list = name_query (fd,
				      "\01\02__MSBROWSE__\02", 1,
				      TRUE,
				      TRUE,
				      *bcast,
				      &count);
		if (ip_list) {
			close (fd);
			*ip = ip_list[0];
			free (ip_list);
			return TRUE;
		}
	}
	close (fd);
	return FALSE;
}

static char *
get_workgroup_data (const char *name)
{
	return g_strdup_printf ("[Desktop Entry]\n"
				"Encoding=UTF-8\n"
				"Name=%s\n"
				"Type=Link\n"
				"URL=smb://%s/\n"
				"Icon=gnome-fs-network\n",
				name, name);
}

static char *
get_computer_data (const char *name)
{
	return g_strdup_printf ("[Desktop Entry]\n"
				"Encoding=UTF-8\n"
				"Name=%s\n"
				"Type=Link\n"
				"URL=smb://%s/\n"
				"Icon=gnome-fs-server\n",
				name, name);
}

/* Pffft. Stupid SAMBA, no user-data */
static gboolean workgroup_find_helper = FALSE;
static gchar *workgroup_name_helper = NULL;
static gchar *workgroup_master_helper = NULL;

static void
workgroup_connection_find_helper (const char *workgroup, uint32 m, const char *master)
{
	if (workgroup == NULL) {
		return;
	}

	if (workgroup_find_helper == FALSE &&
	    g_strcasecmp (workgroup, workgroup_name_helper) == 0) {
		workgroup_find_helper = TRUE;
		workgroup_master_helper = g_strdup (master);
	}
}

static GnomeVFSResult
smb_root_connection (SmbConnection **connection)
{
	struct in_addr ip;
	GnomeVFSResult result;
	SmbConnection *root;

	root = connection_root->data;
	if (root) {
		*connection = root;
		return GNOME_VFS_OK;
	}

	if (!get_a_master_browser (&ip)) {
		g_warning ("Couldn't find a master browser");
		return GNOME_VFS_ERROR_NO_MASTER_BROWSER;
	}

	result = smb_server_connection_new_from_ip (&ip, "*SMBSERVER", "IPC$",
						    NULL, NULL, connection);

	if (result != GNOME_VFS_OK) {
		return result;
	}

	connection_root->data = *connection;

	return result;
}

static GnomeVFSResult
smb_workgroup_connection_find (SmbURIInfo *info, gboolean can_fail, SmbConnection **connection)
{
	SmbConnection *root;
	GnomeVFSResult result;

	*connection = NULL;

	result = smb_root_connection (&root);
	if (result != GNOME_VFS_OK) {
		return result;
	}

	workgroup_find_helper = FALSE;
	workgroup_name_helper = info->host_name;

	if (!cli_NetServerEnum (root->cli, root->cli->server_domain,
				SV_TYPE_DOMAIN_ENUM,
				workgroup_connection_find_helper)) {

		result = gnome_vfs_result_from_cli (root->cli);
		smb_connection_free_if_necessary (root, result);

		return result;
	}

	if (workgroup_find_helper == TRUE) {
		if (g_strcasecmp (root->server, workgroup_master_helper) != 0) {
			if (can_fail) {
				result = GNOME_VFS_OK;
				*connection = root;
			} else {
				SmbURIInfo *fake_info;

				fake_info = g_new0 (SmbURIInfo, 1);
				fake_info->host_name = workgroup_master_helper;
				fake_info->user_name = info->user_name;
				fake_info->password = info->password;
				result = smb_host_connection (fake_info,
							      FALSE, FALSE,
							      connection);
				g_free (fake_info);
			}
		} else {
			result = GNOME_VFS_OK;
			*connection = root;
		}

		g_free (workgroup_master_helper);
		workgroup_master_helper = NULL;
	}

	return result;
}

/* Samba host connection */

static gboolean
smb_host_connection_auth_needed (SmbConnection *connection)
{
	if (connection->cli->sec_mode & 1) {
		return TRUE;
	}

	return FALSE;
}

static GnomeVFSResult
smb_host_connection_new (SmbURIInfo *info, gboolean ask_password, SmbConnection **host)
{
	GnomeVFSResult result;

	result = smb_server_connection_new (info->host_name, "IPC$",
					    info->user_name, info->password, host);

	if (*host && smb_host_connection_auth_needed (*host)) {
		if (ask_password) {
			smb_connection_free_one (*host);
			result = smb_server_connection_new_ask_password (info->host_name,
									 "IPC$", host);
		} else {
			(*host)->needs_auth = TRUE;
		}
	}

	if (result == GNOME_VFS_OK) {
		g_node_append_data (connection_root, *host);
	}

	return result;
}

static SmbConnection *
smb_host_connection_find (SmbURIInfo *info)
{
	GNode *node;
	SmbConnection *connection;

	if (info->host_name == NULL) {
		return NULL;
	}

	node = g_node_first_child (connection_root);
	while (node) {
		connection = node->data;

		/* TODO: Check that user name & password also match */
		if ((g_strcasecmp (connection->server, info->host_name)) == 0)
			return connection;

		node = node->next;
	}

	return NULL;
}

static GnomeVFSResult
smb_host_connection (SmbURIInfo *info, gboolean find_only, gboolean ask_password, SmbConnection **host)
{
	GnomeVFSResult result = GNOME_VFS_ERROR_HOST_NOT_FOUND;

	*host = smb_host_connection_find (info);

	if (ask_password == TRUE && *host != NULL && (*host)->needs_auth == TRUE) {
		smb_connection_free (*host);
		*host = NULL;
	}

	if (*host != NULL) {
		result = GNOME_VFS_OK;
	} else if (find_only == FALSE) {
		result = smb_host_connection_new (info, ask_password, host);
	}

	return result;
}

/* Samba share connection */

static GnomeVFSResult
smb_share_connection_new (SmbURIInfo *info,
			  SmbConnection *host,
			  SmbConnection **share)
{
	gchar *user_name, *password;
	GnomeVFSResult result;

	/* TODO: Check if user_name & password are given from URI and prefer those. */

	if (smb_host_connection_auth_needed (host)) {
		user_name = host->user;
		password = host->password;
	} else {
		user_name = info->user_name;
		password = info->password;
	}

	result = smb_server_connection_new (info->host_name, info->share_name,
					    user_name, password, share);

	if (result == GNOME_VFS_ERROR_LOGIN_FAILED) {
		result = smb_server_connection_new_ask_password (info->host_name,
								 info->share_name,
								 share);
	}

	if (result == GNOME_VFS_OK) {
		GNode *node;

		node = g_node_find_child (connection_root, G_TRAVERSE_ALL, host);
		g_assert (node != NULL);

		g_node_append_data (node, *share);
	}

	return result;
}

static SmbConnection *
smb_share_connection_find (SmbURIInfo *info, SmbConnection *host)
{
	GNode *node;
	SmbConnection *connection = NULL;

	if (info->share_name == NULL) {
		return NULL;
	}

	node = g_node_find_child (connection_root, G_TRAVERSE_NON_LEAFS, host);
	if (node) {
		node = node->children;
	}

	while (node) {
		connection = node->data;

		/* TODO: Check that user name & password also match */
		if ((g_strcasecmp (connection->share, info->share_name)) == 0) {
			break;
		}

		node = node->next;
		connection = NULL;
	}

	return connection;
}

static GnomeVFSResult
smb_share_connection (SmbURIInfo *info,
		      SmbConnection *host,
		      gboolean can_fail,
		      SmbConnection **share)
{
	GnomeVFSResult result = GNOME_VFS_OK;

	*share = smb_share_connection_find (info, host);
	if (*share == NULL) {
		if (can_fail) {
			*share = host;
			return GNOME_VFS_OK;
		}

		result = smb_share_connection_new (info, host, share);
	}

	return result;
}

/* End of share connection */

static gchar **
uri_get_share_and_path (GnomeVFSURI *uri)
{
	const gchar *path;

	path = gnome_vfs_uri_get_path (uri);
	if (path == NULL) {
		return NULL;
	}

	if (*path != GNOME_VFS_URI_PATH_CHR) {
		return NULL;
	}

	while (*path == GNOME_VFS_URI_PATH_CHR) {
		path++;
	}

	return g_strsplit (path, GNOME_VFS_URI_PATH_STR, 2);
}

static void
smb_lookup_data_free (SmbLookupData *data)
{
	g_free (data->path);
	g_free (data);
}

GnomeVFSResult
get_file_type (SmbURIInfo *info, gboolean can_fail, SmbLookupData **data)
{
	GnomeVFSResult result;
	SmbConnection *host, *share;
	SmbVirtualFileType type;

	*data = NULL;

	if (info->host_name == NULL &&
	    info->share_name == NULL &&
	    info->path == NULL) {
		/* Root ('smb:///') */
		
		result = smb_root_connection (&host);
		if (result == GNOME_VFS_OK) {
			*data = g_new (SmbLookupData, 1);
			(*data)->path = g_strdup (GNOME_VFS_URI_PATH_STR);
			(*data)->type = SMB_VIRTUAL_TYPE_ROOT;
			(*data)->connection = host;
		}

		return result;
	}

	if (info->host_name == NULL) {
		return GNOME_VFS_ERROR_BAD_PARAMETERS;
	}

	/* Host name or Workgroup */

	type = SMB_VIRTUAL_TYPE_HOST;
	
	result = smb_host_connection (info, TRUE, FALSE, &host);
	if (result != GNOME_VFS_OK) {
		result = smb_workgroup_connection_find (info, can_fail, &host);
		if (result == GNOME_VFS_OK) {
			type = (info->type == SMB_VIRTUAL_TYPE_UNKNOWN) ?
				SMB_VIRTUAL_TYPE_WORKGROUP : info->type;
		}
	}
	
	if (host == NULL ||
	    (type != SMB_VIRTUAL_TYPE_WORKGROUP &&
	     type != SMB_VIRTUAL_TYPE_WORKGROUP_FILE &&
	     can_fail == FALSE &&
	     host->needs_auth == TRUE)) {
		result = smb_host_connection (info, FALSE, TRUE, &host);
	}
	
	if (result != GNOME_VFS_OK) {
		return result;
	}
	
	*data = g_new (SmbLookupData, 1);
	(*data)->path = NULL;
	(*data)->type = type;
	(*data)->connection = host;

	if (info->share_name == NULL) {
		(*data)->path = g_strdup (info->host_name);
		return result;
	}

	if (info->path == NULL) {
		if ((*data)->type == SMB_VIRTUAL_TYPE_HOST) {
			result = smb_share_connection (info, host, can_fail, &share);
			if (result == GNOME_VFS_OK) {
				(*data)->path = g_strdup (GNOME_VFS_URI_PATH_STR);
				(*data)->type = SMB_VIRTUAL_TYPE_SHARE;
				(*data)->connection = share;
			} else {
				g_free (*data);
			}
		} else if ((*data)->type == SMB_VIRTUAL_TYPE_WORKGROUP) {
			SmbURIInfo *fake_info;

			fake_info = g_new0 (SmbURIInfo, 1);
			fake_info->host_name = info->share_name;
			fake_info->user_name = info->user_name;
			fake_info->password = info->password;

			result = smb_host_connection (fake_info, FALSE, FALSE, &host);
			g_free (fake_info);

			if (result == GNOME_VFS_OK) {
				(*data)->path = g_strdup (info->share_name);
				(*data)->type = SMB_VIRTUAL_TYPE_HOST_FILE;
				(*data)->connection = host;
			} else
				g_free (*data);
		} else {
			result = GNOME_VFS_ERROR_BAD_PARAMETERS;
		}

		return result;
	}

	if ((*data)->type == SMB_VIRTUAL_TYPE_WORKGROUP) {
		g_free (*data);
		return GNOME_VFS_ERROR_BAD_PARAMETERS;
	}

	result = smb_share_connection (info, host, FALSE, &share);
	if (result == GNOME_VFS_OK) {
		(*data)->path = g_strdup (info->path);
		(*data)->type = SMB_VIRTUAL_TYPE_SHARE_FILE;
		(*data)->connection = share;

		return GNOME_VFS_OK;
	}

	g_free (*data);

	return result;
}

static SmbURIInfo *
smb_uri_info_new (GnomeVFSURI *uri)
{
	SmbURIInfo *info;
	gchar **tmp;

	info = g_new0 (SmbURIInfo, 1);
	info->type = SMB_VIRTUAL_TYPE_UNKNOWN;

	info->host_name = gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri), NULL);

	tmp = uri_get_share_and_path (uri);
	if (tmp) {
		if (info->host_name == NULL) {
			if (tmp[0]) {
				info->host_name = gnome_vfs_unescape_string (tmp[0], NULL);
				info->type = SMB_VIRTUAL_TYPE_WORKGROUP_FILE;
				if (tmp[1]) {
					info->share_name = gnome_vfs_unescape_string (tmp[1], NULL);
					info->type = SMB_VIRTUAL_TYPE_HOST_FILE;
				}
			}
		} else {
			if (tmp[0]) {
				info->share_name = gnome_vfs_unescape_string (tmp[0], NULL);
				if (tmp[1]) {
					info->path = gnome_vfs_unescape_string (tmp[1], NULL);
				}
			}
		}

		g_strfreev (tmp);
	}

	info->user_name = gnome_vfs_unescape_string (gnome_vfs_uri_get_user_name (uri), NULL);
	info->password = gnome_vfs_unescape_string (gnome_vfs_uri_get_password (uri), NULL);

	return info;
}

static void
smb_uri_info_free (SmbURIInfo *info)
{
	g_free (info->host_name);
	g_free (info->share_name);
	g_free (info->path);
	g_free (info->user_name);
	g_free (info->password);
	g_free (info);
}

static GnomeVFSResult
lookup_uri (GnomeVFSURI *uri, SmbLookupData **data, gboolean can_fail)
{
	SmbURIInfo *info;
	GnomeVFSResult result;

	DEBUG_SMB (("lookup_uri (uri: %s)\n", gnome_vfs_uri_to_string (uri, 0)));

	info = smb_uri_info_new (uri);
	result = get_file_type (info, can_fail, data);
	smb_uri_info_free (info);

	return result;
}

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbHandle *handle;
	SmbLookupData *data;
	char *path_remainder;
	int fnum;
	char *dos_filename;
	int flags;
	GnomeVFSFileOffset file_size;

	LOCK_SAMBA();

	DEBUG_SMB (("do_open() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));

	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE)) {
		flags = O_RDWR;
	} else if (mode & GNOME_VFS_OPEN_READ) {
		flags = O_RDONLY;
	} else if (mode & GNOME_VFS_OPEN_WRITE) {
		flags = O_WRONLY;
	} else {
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;
	}

	res = lookup_uri (uri, &data, FALSE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	path_remainder = data->path;
	connection = data->connection;
	switch (data->type) {
	case SMB_VIRTUAL_TYPE_WORKGROUP:
	case SMB_VIRTUAL_TYPE_WORKGROUP_FILE:
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = get_workgroup_data (data->path);
		handle->file_size = strlen (handle->file_data);

		break;
	case SMB_VIRTUAL_TYPE_HOST:
	case SMB_VIRTUAL_TYPE_HOST_FILE:
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = get_computer_data (data->path);
		handle->file_size = strlen (handle->file_data);

		break;
	case SMB_VIRTUAL_TYPE_SHARE_FILE:
		dos_filename = unix_filename_to_dos (path_remainder);
		fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
		g_free (dos_filename);
		if (fnum == -1) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			smb_lookup_data_free (data);
			UNLOCK_SAMBA();
			return res;
		}

		if (mode & GNOME_VFS_OPEN_RANDOM) {
			size_t size;
			uint16 dummy1;
			time_t dummy2;
			if (cli_getattrE (connection->cli, fnum,
					  &dummy1, &size, &dummy2, 
					  &dummy2, &dummy2)) {
				file_size = size;
			} else {
				res = gnome_vfs_result_from_cli (connection->cli);
				smb_connection_free_if_necessary (connection, res);
				smb_lookup_data_free (data);
				UNLOCK_SAMBA();
				return GNOME_VFS_ERROR_NOT_SUPPORTED;
			}
		} else {
			file_size = -1;
		}
		
		
		handle = g_new (SmbHandle, 1);
		handle->is_data = FALSE;
		handle->connection = connection;
		handle->fnum = fnum;
		handle->offset = 0;
		handle->file_size = file_size;
		
		break;
	default:
		g_warning ("do_open: Unhandled case: %d", data->type);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	smb_lookup_data_free (data);

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;

	LOCK_SAMBA();
	DEBUG_SMB (("do_close() %p\n", method_handle));

	if (handle->is_data) {
		g_free (handle->file_data);
	} else {
		cli_close  (handle->connection->cli, handle->fnum);

	}
	
	g_free (handle);
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	SmbHandle *handle = (SmbHandle *)method_handle;
	GnomeVFSResult result;
	size_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_read() %p, num_bytes: %d\n", method_handle, (guint) num_bytes));

	if (handle->is_data) {
		link_len = strlen (handle->file_data);
		if (handle->offset >= link_len) {
			n = 0;
		} else {
			n = MIN (num_bytes, link_len - handle->offset);
			memcpy (buffer, handle->file_data + handle->offset, n);
		}
	} else {
		n = cli_read (handle->connection->cli,
			      handle->fnum,
			      buffer,
			      handle->offset,
			      num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_read = 0;
		result = gnome_vfs_result_from_cli (handle->connection->cli);
		smb_connection_free_if_necessary (handle->connection, result);
	}

	*bytes_read = n;

	if (n == 0) {
		return GNOME_VFS_ERROR_EOF;
	}

	handle->offset += n;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
	SmbHandle *handle = (SmbHandle *)method_handle;
	GnomeVFSResult result;
	ssize_t n;

	LOCK_SAMBA();
	DEBUG_SMB (("do_write() %p\n", method_handle));

	if (handle->is_data) {
	  return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	n = cli_write (handle->connection->cli,
		       handle->fnum,
		       0,
		       (char *)buffer,
		       handle->offset,
		       num_bytes);

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_written = 0;
		result = gnome_vfs_result_from_cli (handle->connection->cli);
		smb_connection_free_if_necessary (handle->connection, result);
	}
	
	*bytes_written = n;
	
	if (n == 0) {
		return GNOME_VFS_ERROR_EOF;
	}
	
	handle->offset += n;
	if (handle->offset > handle->file_size) {
		handle->file_size = handle->offset;
	}
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	SmbLookupData *data;
	SmbConnection *connection;
	SmbHandle *handle;
	int fnum;
	char *path_remainder;
	char *dos_filename;
	int flags;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_create() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE)) {
		flags = O_RDWR;
	} else if (mode & GNOME_VFS_OPEN_READ) {
		flags = O_RDONLY;
	} else if (mode & GNOME_VFS_OPEN_WRITE) {
		flags = O_WRONLY;
	} else {
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;
	}


	flags |= O_CREAT;
	if (exclusive) {
		flags |= O_EXCL;
	}
	
	res = lookup_uri (uri, &data, FALSE);

	if (res) {
		UNLOCK_SAMBA ();
		if (res == GNOME_VFS_ERROR_NOT_FOUND) {
			res = GNOME_VFS_ERROR_ACCESS_DENIED;
		}
		return res;
	}

	path_remainder = data->path;
	connection = data->connection;

	if (data->type != SMB_VIRTUAL_TYPE_SHARE_FILE) {
		smb_lookup_data_free (data);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	/* SMB_VIRTUAL_TYPE_SHARE */
	dos_filename = unix_filename_to_dos (path_remainder);
	DEBUG_SMB (("tryping to cli_open '%s', flags: %d\n", dos_filename, flags));
	fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
	g_free (dos_filename);
	if (fnum == -1) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_free_if_necessary (connection, res);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
		return res;
	}

	handle = g_new (SmbHandle, 1);
	handle->is_data = FALSE;
	handle->connection = connection;
	handle->fnum = fnum;
	handle->offset = 0;

	smb_lookup_data_free (data);

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *vfs_file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbLookupData *data;
	gchar *path;
	SmbConnection *connection;

	LOCK_SAMBA();
	DEBUG_SMB2 (("do_get_file_info() %s\n", gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri (uri, &data, TRUE);

	if (res) {
		UNLOCK_SAMBA();
		return res;
	}

	path = data->path;
	connection = data->connection;

	switch (data->type) {
	case SMB_VIRTUAL_TYPE_SHARE_FILE: {
		char *dos_filename;
		uint16 attr;
		size_t size;
		time_t time;

		dos_filename = unix_filename_to_dos (path);

		if ((cli_getatr (connection->cli, dos_filename, &attr, &size, &time)) != TRUE) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			g_free (dos_filename);
			smb_lookup_data_free (data);
			UNLOCK_SAMBA();
			return res;
		}

		g_free (dos_filename);

		vfs_file_info->name = g_strdup (g_basename (path));

		if (IS_DOS_DIR (attr)) {
			vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			vfs_file_info->mime_type = g_strdup("x-directory/normal");
		} else {
			vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			vfs_file_info->mime_type =
				g_strdup (gnome_vfs_mime_type_from_name_or_default (vfs_file_info->name,
										    GNOME_VFS_MIME_TYPE_UNKNOWN));
		}

		if (IS_DOS_READONLY (attr)) {
			vfs_file_info->permissions =
				GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
				GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
				GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
		} else {
			vfs_file_info->permissions =
				GNOME_VFS_PERM_USER_ALL |
				GNOME_VFS_PERM_GROUP_ALL |
				GNOME_VFS_PERM_OTHER_ALL;
		}

		vfs_file_info->size = size;
		vfs_file_info->uid = getuid ();
		vfs_file_info->gid = getgid ();

		vfs_file_info->mtime = vfs_file_info->atime = vfs_file_info->ctime = time;

		vfs_file_info->flags = 0;

		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_SIZE |
			GNOME_VFS_FILE_INFO_FIELDS_ATIME |
			GNOME_VFS_FILE_INFO_FIELDS_MTIME |
			GNOME_VFS_FILE_INFO_FIELDS_CTIME |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		break;
	}

	case SMB_VIRTUAL_TYPE_WORKGROUP_FILE:
	case SMB_VIRTUAL_TYPE_HOST_FILE:
		vfs_file_info->name = g_strdup (path);
		vfs_file_info->mime_type = g_strdup ("application/x-gnome-app-info");
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		vfs_file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		break;

	default:
		vfs_file_info->name = g_strdup (path);
		vfs_file_info->mime_type = g_strdup ("x-directory/normal");
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		vfs_file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		vfs_file_info->permissions =
			GNOME_VFS_PERM_USER_ALL |
			GNOME_VFS_PERM_GROUP_ALL |
			GNOME_VFS_PERM_OTHER_ALL;
		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	}

	smb_lookup_data_free (data);

	UNLOCK_SAMBA ();
	return res;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)
{
	return FALSE;
}

static void
build_root_helper (const char *workgroup, uint32 m, const char *master)
{
	SmbDirHandle *handle;
	SmbDirContent *content;

	handle = g_private_get (dir_key);

	content = g_new0 (SmbDirContent, 1);
	content->name = g_strdup (workgroup);
	content->virtual_type = SMB_VIRTUAL_TYPE_WORKGROUP;
	content->permissions =  GNOME_VFS_PERM_USER_ALL |
		GNOME_VFS_PERM_GROUP_ALL |
		GNOME_VFS_PERM_OTHER_ALL;

	handle->contents = g_list_prepend (handle->contents, content);
}

static void
build_workgroup_helper (const char *host, uint32 m, const char *comment)
{
	SmbDirContent *content;
	SmbDirHandle *handle;

	handle = g_private_get (dir_key);

	content = g_new0 (SmbDirContent, 1);
	content->name = g_strdup (host);
	content->virtual_type = SMB_VIRTUAL_TYPE_HOST;
	content->permissions = GNOME_VFS_PERM_USER_ALL |
		GNOME_VFS_PERM_GROUP_ALL |
		GNOME_VFS_PERM_OTHER_ALL;
	handle->contents = g_list_prepend (handle->contents, content);
}

static void
open_dir_helper (file_info *info, const char *dir)
{
	SmbDirHandle *handle;
	SmbDirContent *content;

	handle = g_private_get (dir_key);

	content = g_new (SmbDirContent, 1);
	content->name = g_strdup (info->name);
	content->mtime = info->mtime;
	content->atime = info->atime;
	content->ctime = info->ctime;

	if (IS_DOS_DIR (info->mode)) {
		content->virtual_type = SMB_VIRTUAL_TYPE_SHARE_DIR;
	} else {
		content->virtual_type = SMB_VIRTUAL_TYPE_SHARE_FILE;
	}

	if (IS_DOS_READONLY (info->mode)) {
		content->permissions =
			GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
			GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
			GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
	} else {
		content->permissions =
			GNOME_VFS_PERM_USER_ALL |
			GNOME_VFS_PERM_GROUP_ALL |
			GNOME_VFS_PERM_OTHER_ALL;
	}

	content->size = info->size;
	content->uid = info->uid;
	content->gid = info->gid;
	handle->contents = g_list_prepend (handle->contents, content);
}

static void
open_host_helper (const char *share, uint32 type, const char *comment)
{
	SmbDirContent *content;
	SmbDirHandle *handle;
	int len;

	if (type != STYPE_DISKTREE) {
		return; /* Not a share */
	}

	/* Avoid empty and hidden shares */
	len = strlen (share);
	if (len == 0 || share[len-1]=='$') {
		return;
	}
	
	handle = g_private_get (dir_key);

	content = g_new0 (SmbDirContent, 1);
	content->name = g_strdup (share);
	content->virtual_type = SMB_VIRTUAL_TYPE_SHARE;
	content->permissions = GNOME_VFS_PERM_USER_ALL |
		GNOME_VFS_PERM_GROUP_ALL |
		GNOME_VFS_PERM_OTHER_ALL;
	handle->contents = g_list_prepend (handle->contents, content);
}

static char *
dirname_to_mask (const char *dirname)
{
	char *mask;
	int dirname_len;
	int i;

	dirname_len = strlen (dirname);
	mask = g_malloc (dirname_len + 3);
	strcpy (mask, dirname);
	for (i = 0; i < dirname_len; i++) {
		if (mask[i]=='/') {
			mask[i] = '\\';
		}
	}

	if (mask[dirname_len-1] != '\\') {
		dirname_len++;
		mask[dirname_len-1] = '\\';
	}
	mask[dirname_len] = '*';
	mask[dirname_len+1] = 0;
	
	return mask;
}

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbDirHandle *handle;
	SmbConnection *connection;
	SmbLookupData *data;
	char *path_remainder;
	char *mask;

	LOCK_SAMBA();
	DEBUG_SMB2 (("do_open_directory() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	res = lookup_uri (uri, &data, FALSE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	path_remainder = data->path;
	connection = data->connection;

	switch (data->type) {
	case SMB_VIRTUAL_TYPE_HOST:
		handle = g_new (SmbDirHandle, 1);
		handle->contents = NULL;
		handle->server = NULL;

		g_private_set (dir_key, handle);

		if (cli_RNetShareEnum (connection->cli, open_host_helper) < 0) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			g_free (handle);
			smb_lookup_data_free (data);
			g_private_set (dir_key, NULL);
			UNLOCK_SAMBA();
			return res;
		}
		break;
	case SMB_VIRTUAL_TYPE_SHARE:
	case SMB_VIRTUAL_TYPE_SHARE_FILE:
	case SMB_VIRTUAL_TYPE_SHARE_DIR:
		handle = g_new (SmbDirHandle, 1);
		handle->contents = NULL;
		handle->server = NULL;
	
		g_private_set (dir_key, handle);

		mask = dirname_to_mask (path_remainder);
		if (cli_list (connection->cli, mask,
			      aDIR | aSYSTEM | aHIDDEN,
			      open_dir_helper) == -1) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			g_free (mask);
			g_free (handle);
			smb_lookup_data_free (data);
			g_private_set (dir_key, NULL);
			g_warning ("open_share_dir: cli_list returned -1, returning %d %s",
				   res, gnome_vfs_result_to_string (res));
			UNLOCK_SAMBA();
			return res;
		}

		g_free (mask);
		g_private_set (dir_key, NULL);

		break;
	case SMB_VIRTUAL_TYPE_WORKGROUP:
		handle = g_new (SmbDirHandle, 1);
		handle->contents = NULL;
		handle->server = NULL;
	
		g_private_set (dir_key, handle);

		if (cli_NetServerEnum (connection->cli,
				       connection->cli->server_domain,
				       SV_TYPE_ALL, build_workgroup_helper) < 0) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			smb_lookup_data_free (data);
			g_free (handle);
			g_private_set (dir_key, NULL);
			UNLOCK_SAMBA();
			return res;
		}

		break;
	case SMB_VIRTUAL_TYPE_ROOT:
		handle = g_new (SmbDirHandle, 1);
		handle->contents = NULL;
		handle->server = NULL;
	
		g_private_set (dir_key, handle);

		if (cli_NetServerEnum (connection->cli,
				       connection->cli->server_domain,
				       SV_TYPE_DOMAIN_ENUM, build_root_helper) < 0) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			smb_lookup_data_free (data);
			g_free (handle);
			g_private_set (dir_key, NULL);
			UNLOCK_SAMBA();
			return res;
		}

		break;
	default:
		g_warning ("do_open_directory: not handled, type %d\n", data->type);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	smb_lookup_data_free (data);

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;

	DEBUG_SMB2 (("do_close_directory() %p\n", handle));

	smb_dir_handle_destroy (handle);

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	SmbDirContent *content;
	GList *node;
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;

	DEBUG_SMB2 (("do_read_directory()\n"));

	if (handle->contents) {
		node = handle->contents;
		content = (SmbDirContent *)node->data;
		handle->contents =
			g_list_remove_link (handle->contents, node);

		file_info->name = content->name;
		g_assert (file_info->name != NULL);

		switch (content->virtual_type) {
		case SMB_VIRTUAL_TYPE_WORKGROUP:
		case SMB_VIRTUAL_TYPE_HOST:
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type = g_strdup ("application/x-gnome-app-info");

			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_TYPE |
				GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
				GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

			break;
		case SMB_VIRTUAL_TYPE_SHARE:
		case SMB_VIRTUAL_TYPE_SHARE_DIR:
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			if (content->virtual_type == SMB_VIRTUAL_TYPE_SHARE) {
				file_info->mime_type = g_strdup("x-directory/smb-share");
			} else {
				file_info->mime_type = g_strdup("x-directory/normal");
			}

			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_TYPE |
				GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
				GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

			break;
		default:
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type =
				g_strdup (gnome_vfs_mime_type_from_name_or_default (file_info->name,
										    GNOME_VFS_MIME_TYPE_UNKNOWN));

			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_TYPE |
				GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
				GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
				GNOME_VFS_FILE_INFO_FIELDS_SIZE |
				GNOME_VFS_FILE_INFO_FIELDS_ATIME |
				GNOME_VFS_FILE_INFO_FIELDS_MTIME |
				GNOME_VFS_FILE_INFO_FIELDS_CTIME |
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		}

		file_info->permissions = content->permissions;
		file_info->uid = content->uid;
		file_info->gid = content->gid;
		file_info->mtime = content->mtime;
		file_info->ctime = content->ctime;
		file_info->atime = content->atime;
		file_info->size = content->size;

		file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;

		g_free (content);
		return GNOME_VFS_OK;
	} else {
		return GNOME_VFS_ERROR_EOF;
	}
}

static GnomeVFSResult
do_seek (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSSeekPosition whence,
	 GnomeVFSFileOffset offset,
	 GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;

	DEBUG_SMB (("do_seek() %d from %d\n", (guint) offset, whence));
	
	switch (whence) {
	case GNOME_VFS_SEEK_START:
		handle->offset = offset;
		break;
	case GNOME_VFS_SEEK_CURRENT:
		handle->offset += offset;
		break;
	case GNOME_VFS_SEEK_END:
		if (handle->file_size >= 0) {
			handle->offset = handle->file_size - offset;
		} else {
			return GNOME_VFS_ERROR_NOT_SUPPORTED;
		}
 		break;
	default:
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_tell (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSFileOffset *offset_return)


{
	SmbHandle *handle = (SmbHandle *)method_handle;

	*offset_return = handle->offset;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbLookupData *data;
	SmbConnection *connection;
	char *path_remainder;
	char *dos_filename;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_unlink() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri (uri, &data, FALSE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (data->type != SMB_VIRTUAL_TYPE_SHARE_FILE) {
		smb_lookup_data_free (data);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	path_remainder = data->path;
	connection = data->connection;

	dos_filename = unix_filename_to_dos (path_remainder);
	ret = cli_unlink (connection->cli, dos_filename);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_free_if_necessary (connection, res);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
		return res;
	}

	smb_lookup_data_free (data);

	UNLOCK_SAMBA();
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	char *p1, *p2;

	DEBUG_SMB (("do_check_same_fs()\n"));

	server1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (a),
				     NULL);
	server2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (b),
				     NULL);
	path1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (a), NULL);
	path2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (b), NULL);

	if (!server1 || !server2 || !path1 || !path2 ||
	    (strcmp (server1, server2) != 0)) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}

	p1 = path1;
	p2 = path2;
	if (*p1 == '/') {
		p1++;
	}
	if (*p2 == '/') {
		p2++;
	}

	/* Make sure both URIs are on the same share: */
	while (*p1 && *p2 && *p1 == *p2 && *p1 != '/') {
		p1++;
		p2++;
	}
	if (*p1 == 0 || *p2 == 0 || *p1 != *p2) {
		*same_fs_return = FALSE;
	} else {
		*same_fs_return = TRUE;
	}
	
	g_free (server1);
	g_free (server2);
	g_free (path1);
	g_free (path2);
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	SmbLookupData *data1;
	SmbLookupData *data2;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_move() %s %s\n",
		    gnome_vfs_uri_to_string (old_uri, 0),
		    gnome_vfs_uri_to_string (new_uri, 0)));

	res = lookup_uri (old_uri, &data1, TRUE);
	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (data1->type != SMB_VIRTUAL_TYPE_SHARE_FILE || data1->path == NULL) {
		smb_lookup_data_free (data1);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	res = lookup_uri (new_uri, &data2, TRUE);
	if (res) {
		smb_lookup_data_free (data1);
		UNLOCK_SAMBA ();
		return res;
	}

	if (data2->path == NULL || data1->type != data2->type) {
		smb_lookup_data_free (data1);
		smb_lookup_data_free (data2);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	filename1 = data1->path;
	filename2 = data2->path;
	connection = data1->connection;

	dos_filename1 = unix_filename_to_dos (filename1);
	dos_filename2 = unix_filename_to_dos (filename2);
	ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
	g_free (dos_filename1);
	g_free (dos_filename2);

	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_free_if_necessary (connection, res);

		smb_lookup_data_free (data1);
		smb_lookup_data_free (data2);

		UNLOCK_SAMBA();
		return res;
	}

	smb_lookup_data_free (data1);
	smb_lookup_data_free (data2);

	UNLOCK_SAMBA();
	
	return GNOME_VFS_OK;
}

#if 0
static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	/* FIXME: need to update handle->file_size cache here */
	g_warning ("do_truncate_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_get_file_info_from_handle (GnomeVFSMethod *method,
			      GnomeVFSMethodHandle *method_handle,
			      GnomeVFSFileInfo *file_info,
			      GnomeVFSFileInfoOptions options,
			      GnomeVFSContext *context)


{
	/* FIXME: need to update handle->file_size cache here */
	g_warning ("do_get_file_info_from_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}
#endif

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbLookupData *data;
	BOOL ret;
	char *dos_filename;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_make_dir() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri (uri, &data, TRUE);
	if (res) {
		UNLOCK_SAMBA ();
		if (res == GNOME_VFS_ERROR_NOT_FOUND) {
			res = GNOME_VFS_ERROR_ACCESS_DENIED;
		}
		return res;
	}

	if (data->type != SMB_VIRTUAL_TYPE_SHARE_FILE || data->path == NULL) {
		smb_lookup_data_free (data);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	connection = data->connection;

	dos_filename = unix_filename_to_dos (data->path);
	ret = cli_mkdir (data->connection->cli, dos_filename);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_free_if_necessary (connection, res);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
		return res;
	}
	
	smb_lookup_data_free (data);

	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbLookupData *data;
	BOOL ret;
	char *dos_filename;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_remove_directory() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	res = lookup_uri (uri, &data, TRUE);
	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}
	
	if (data->type != SMB_VIRTUAL_TYPE_SHARE_FILE || data->path == NULL) {
		smb_lookup_data_free (data);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	connection = data->connection;

	dos_filename = unix_filename_to_dos (data->path);
	ret = cli_rmdir (connection->cli, dos_filename);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_free_if_necessary (connection, res);
		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
		return res;
	}

	smb_lookup_data_free (data);
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	SmbLookupData *data;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	char *end;
	BOOL ret;

	DEBUG_SMB (("do_set_file_info mask: %d\n", mask));

	if (mask &
	    (GNOME_VFS_SET_FILE_INFO_PERMISSIONS |
	     GNOME_VFS_SET_FILE_INFO_OWNER |
	     GNOME_VFS_SET_FILE_INFO_TIME)) {
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		LOCK_SAMBA();
		DEBUG_SMB (("set_info: set new name: %s\n", info->name));
		
		res = lookup_uri (uri, &data, TRUE);
		if (res) {
			UNLOCK_SAMBA ();
			return res;
		}

		if (data->type != SMB_VIRTUAL_TYPE_SHARE_FILE || data->path == NULL) {
			smb_lookup_data_free (data);
			UNLOCK_SAMBA ();
			return GNOME_VFS_ERROR_ACCESS_DENIED;
		}

		filename1 = data->path;
		connection = data->connection;

		dos_filename1 = unix_filename_to_dos (filename1);
		filename2 = g_malloc (strlen (filename1) + strlen (info->name) + 1);

		strcpy (filename2, filename1);
		end = strrchr (filename2, '/');
		if (!end) {
			strcpy (filename2, info->name);
		} else {
			strcpy (end+1, info->name);
		}
		
		dos_filename2 = unix_filename_to_dos (filename2);
		DEBUG_SMB (("cli_rename:ing %s to %s\n", dos_filename1, dos_filename2));
		ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
		g_free (filename2);
		g_free (dos_filename1);
		g_free (dos_filename2);
		if (!ret) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_free_if_necessary (connection, res);
			smb_lookup_data_free (data);
			UNLOCK_SAMBA();
			return res;
		}

		smb_lookup_data_free (data);
		UNLOCK_SAMBA();
	}

	return GNOME_VFS_OK;
}

#if 0
static GnomeVFSResult
do_truncate (GnomeVFSMethod *method,
	     GnomeVFSURI *uri,
	     GnomeVFSFileSize where,
	     GnomeVFSContext *context)

{
	g_warning ("do_truncate\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_find_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *near_uri,
		   GnomeVFSFindDirectoryKind kind,
		   GnomeVFSURI **result_uri,
		   gboolean create_if_needed,
		   gboolean find_if_needed,
		   guint permissions,
		   GnomeVFSContext *context)
{
	g_warning ("do_find_directory\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_create_symbolic_link (GnomeVFSMethod *method,
			 GnomeVFSURI *uri,
			 const char *target_reference,
			 GnomeVFSContext *context)
{
	g_warning ("do_create_symbolic_link\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}
#endif

static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,
	do_close,
	do_read,
	do_write,
	do_seek,
	do_tell,
	NULL, /* do_truncate_handle, */
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	NULL, /* do_get_file_info_from_handle, */
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	NULL, /* do_truncate, */
	NULL, /* do_find_directory, */
	NULL /* do_create_symbolic_link */
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	TimeInit ();
	charset_initialise ();

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
	if (!lp_load ("/etc/samba/smb.conf", TRUE, FALSE, FALSE)) {
		DEBUG_SMB (("Couldn't load smb config file"));
	}

	codepage_initialise (lp_client_code_page());

	load_interfaces ();

	get_myname((*global_myname)?NULL:global_myname);

	dir_key = g_private_new ((GDestroyNotify)smb_dir_handle_destroy);
	connection_root = g_node_new (NULL);
	samba_lock = g_mutex_new();

	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	DEBUG_SMB (("<-- smb module shutdown called -->\n"));

	smb_connection_free_all ();
}

