/*
 * manager.c - GNOME Volume Manager
 *
 * Robert Love <rml@ximian.com>
 *
 * Licensed under the GNU GPL v2.  See COPYING.
 *
 * gnome-volume-manager is a simple policy agent that sits on top of the stack
 * of system-level components including the kernel, hotplug, udev, and HAL.
 * Responding to HAL events, gnome-volume-manager implements automount,
 * autorun, autoplay, automatic photo management, and so on.
 */

#include "config.h"

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <gnome.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkx.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <libhal.h>

#include "gvm.h"

#ifdef ENABLE_NLS
#  include <libintl.h>
#  define _(String) gettext (String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop (String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define _(String)
#  define N_(String) (String)
#endif

#define GVM_DEBUG

#ifdef GVM_DEBUG
# define dbg(fmt,arg...) fprintf(stderr, "%s/%d: " fmt,__FILE__,__LINE__,##arg)
#else
# define dbg(fmt,arg...) do { } while(0)
#endif

#define warn(fmt,arg...) g_warning("%s/%d: " fmt,__FILE__,__LINE__,##arg)

#define BIN_MOUNT	"/bin/mount"	/* what we mount with */

struct gvm_configuration config;

/*
 * clipboard_get_func - dummy get_func for gtk_clipboard_set_with_data ()
 */
static void
clipboard_get_func (GtkClipboard *clipboard, GtkSelectionData *selection_data,
		    guint info, gpointer user_data_or_owner)
{
	(void) clipboard;
	(void) selection_data;
	(void) info;
	(void) user_data_or_owner;
}

/*
 * clipboard_clear_func - dummy clear_func for gtk_clipboard_set_with_data ()
 */
static void
clipboard_clear_func (GtkClipboard *clipboard, gpointer user_data_or_owner)
{
	(void) clipboard;
	(void) user_data_or_owner;
}

/*
 * gvm_get_clipboard - try and get the CLIPBOARD_NAME clipboard
 *
 * Returns TRUE if successfully retrieved and FALSE otherwise.
 */
static gboolean
gvm_get_clipboard (void)
{
	static const GtkTargetEntry targets[] = { {CLIPBOARD_NAME, 0, 0} };
	gboolean retval = FALSE;
	GtkClipboard *clipboard;
	Atom atom;

	atom = gdk_x11_get_xatom_by_name (CLIPBOARD_NAME);

	XGrabServer (GDK_DISPLAY ());

	if (XGetSelectionOwner (GDK_DISPLAY (), atom) != None)
		goto out;

	clipboard = gtk_clipboard_get (gdk_atom_intern (CLIPBOARD_NAME, FALSE));

	if (gtk_clipboard_set_with_data (clipboard, targets,
					 G_N_ELEMENTS (targets),
					 clipboard_get_func,
					 clipboard_clear_func, NULL))
		retval = TRUE;

out:
	XUngrabServer (GDK_DISPLAY ());
	gdk_flush ();

	return retval;
}

/*
 * gvm_load_config - synchronize gconf => config structure
 */
static void
gvm_load_config (void)
{
	config.automount_drives = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_drives", NULL);
	config.automount_media = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_media", NULL);
	config.autoplay_cda = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_cda", NULL);
	config.autorun = gconf_client_get_bool(config.client,
			GCONF_ROOT "autorun", NULL);
	config.autophoto = gconf_client_get_bool(config.client,
			GCONF_ROOT "autophoto", NULL);
	config.autoplay_dvd = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_dvd", NULL);
	config.autoplay_cda_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_cda_command", NULL);
	config.autorun_path = gconf_client_get_string (config.client,
			GCONF_ROOT "autorun_path", NULL);
	config.autoplay_dvd_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_dvd_command", NULL);
	config.autoburn_cdr = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoburn_cdr", NULL);
	config.autoburn_cdr_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoburn_cdr_command", NULL);
	config.autophoto_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autophoto_command", NULL);

	/*
	 * If all of the options that control our policy are disabled, then we
	 * have no point in living.  Save the user some memory and exit.
	 */
	if (!(config.automount_drives || config.autorun || config.autoplay_cda
			|| config.autoplay_dvd || config.autophoto)) {
		dbg ("daemon exit: no point living\n");
		exit (EXIT_SUCCESS);
	}
}

/*
 * gvm_config_changed - gconf_client_notify_add () call back to reload config
 */
static void
gvm_config_changed (GConfClient *client, guint id,
		    GConfEntry *entry, gpointer data)
{
	(void) client;
	(void) id;
	(void) entry;
	(void) data;

	dbg ("Configuration: changed!\n");

	g_free (config.autoplay_cda_command);
	g_free (config.autorun_path);
	g_free (config.autoplay_dvd_command);
	g_free (config.autoburn_cdr_command);
	g_free (config.autophoto_command);

	gvm_load_config ();
}

/*
 * gvm_init_config - initialize gconf client and load config data
 */
static void
gvm_init_config (void)
{
	config.client = gconf_client_get_default ();

	gconf_client_add_dir (config.client, GCONF_ROOT_SANS_SLASH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

	gvm_load_config ();

	gconf_client_notify_add (config.client, GCONF_ROOT_SANS_SLASH,
				 gvm_config_changed, NULL, NULL, NULL);
}

/*
 * gvm_run_command - run the given command, replacing %d with the device node
 * and %m with the given path
 */
static void
gvm_run_command (const char *device, const char *command, const char *path)
{
	char *argv[4];
	gchar *new_command;
	GError *error = NULL;
	GString *exec = g_string_new (NULL);
	char *p, *q;

	/* perform s/%d/device/ and s/%m/path/ */
	new_command = g_strdup (command);
	q = new_command;
	p = new_command;
	while ((p = strchr (p, '%')) != NULL) {
		if (*(p + 1) == 'd') {
			*p = '\0';
			g_string_append (exec, q);
			g_string_append (exec, device);
			q = p + 2;
			p = p + 2;
		}
		if (*(p + 1) == 'm') {
			*p = '\0';
			g_string_append (exec, q);
			g_string_append (exec, path);
			q = p + 2;
			p = p + 2;
		}
	}
	g_string_append (exec, q);

	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[2] = exec->str;
	argv[3] = NULL;

	g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL, NULL,
		       NULL, &error);
	if (error)
		warn ("failed to exec %s: %s\n", exec->str, error->message);

	g_string_free (exec, TRUE);
	g_free (new_command);
}

/*
 * gvm_ask_autorun - ask the user if they want to autorun a specific file
 *
 * Returns TRUE if the user selected 'Yes' and FALSE otherwise
 */
static gboolean
gvm_ask_autorun (const char *path)
{
	GtkWidget *askme;
	gboolean retval;

	askme = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
					_("Do you want to run %s?"), path);
	gtk_dialog_set_default_response (GTK_DIALOG (askme), GTK_RESPONSE_YES);

	switch (gtk_dialog_run (GTK_DIALOG (askme))) {
	case GTK_RESPONSE_YES:
		retval = TRUE;
		break;
	case GTK_RESPONSE_NO:
	default:
		retval = FALSE;
		break;
	}

	gtk_widget_destroy (askme);

	return retval;
}

/*
 * gvm_check_dvd - is this a Video DVD?  If so, do something about it.
 *
 * Returns TRUE if this was a Video DVD and FALSE otherwise.
 */
static gboolean
gvm_check_dvd (const char *device, const char *mount_point)
{
	char *path;
	gboolean retval;

	path = g_build_path (G_DIR_SEPARATOR_S, mount_point, "video_ts", NULL);
	retval = g_file_test (path, G_FILE_TEST_IS_DIR);
	g_free (path);

	/* try the other name, if needed */
	if (retval == FALSE) {
		path = g_build_path (G_DIR_SEPARATOR_S, mount_point,
				     "VIDEO_TS", NULL);
		retval = g_file_test (path, G_FILE_TEST_IS_DIR);
		g_free (path);
	}

	if (retval && config.autoplay_dvd)
		gvm_run_command (device, config.autoplay_dvd_command,
				 mount_point);

	return retval;
}

#define ASK_PHOTOS_MSG	"There are photographs on this device.\n\n" \
			"Would you like to import these photographs " \
			"into your photo album?"

/*
 * gvm_check_photos - check if this device is a digital camera or a storage
 * unit from a digital camera (e.g., a compact flash card).  If it is, then
 * ask the user if he wants to import the photos.
 *
 * Returns TRUE if there were photos on this device, FALSE otherwise
 */
static gboolean
gvm_check_photos (const char *udi, const char *device, const char *mount_point)
{
	char *dcim_path;
	enum { IMPORT } action = -1;
	GtkWidget *askme;
	int retval = FALSE;

	dcim_path = g_build_path (G_DIR_SEPARATOR_S, mount_point, "dcim", NULL);

	if (!g_file_test (dcim_path, G_FILE_TEST_IS_DIR))
		goto out;

	retval = TRUE;
	dbg ("Photos detected: %s\n", dcim_path);

	/* add the "camera" capability to this device */
	if (!hal_device_add_capability (udi, "camera"))
		warn ("failed to add camera capability to %s\n", device);

	if (config.autophoto) {
		askme = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_NONE,
						_(ASK_PHOTOS_MSG));
		gtk_dialog_add_buttons (GTK_DIALOG (askme),
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL, _("_Import"),
					IMPORT, NULL);
		action = gtk_dialog_run (GTK_DIALOG (askme));
		gtk_widget_destroy (askme);

		if (action == IMPORT)
			gvm_run_command (device, config.autophoto_command,
					 dcim_path);
	}

out:
	g_free (dcim_path);

	return retval;
}

/*
 * gvm_device_autorun - automatically execute stuff on the given UDI
 *
 * we currently autorun: autorun files, video DVD's, and digital photos
 */
static void
gvm_device_autorun (const char *udi)
{
	char *device = NULL, *mount_point = NULL;

	device = hal_device_get_property_string (udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	mount_point = hal_device_get_property_string (udi, "block.mount_point");
	if (!mount_point) {
		warn ("cannot get block.mount_point\n");
		goto out;
	}

	if (gvm_check_dvd (device, mount_point))
		goto out;

	if (gvm_check_photos (udi, device, mount_point))
		goto out;

	if (config.autorun == TRUE) {
		char **autorun_fns;
		int i;

		autorun_fns = g_strsplit (config.autorun_path, ":", -1);

		for (i = 0; autorun_fns[i]; i++) {
			char *path, *argv[2];

			path = g_strdup_printf ("%s/%s", mount_point,
						autorun_fns[i]);
			argv[0] = path;
			argv[1] = NULL;

			if (access (path, X_OK))
				continue;

			if (gvm_ask_autorun (path)) {
				GError *error = NULL;

				g_spawn_async (g_get_home_dir (), argv, NULL,
					       0, NULL, NULL, NULL, &error);
				if (error)
					warn ("failed to exec %s: %s\n", path,
					      error->message);
				g_free (path);
				break;
			}

			g_free (path);
		}

		g_strfreev (autorun_fns);
	}

out:
	hal_free_string (device);
	hal_free_string (mount_point);
}

/*
 * gvm_device_mount - use BIN_MOUNT to mount the given device node.
 *
 * Note that this requires that the given device node is in /etc/fstab.  This
 * is intentional.
 */
static void
gvm_device_mount (char *device)
{
	char *argv[3];
	GError *error = NULL;

	argv[0] = BIN_MOUNT;
	argv[1] = device;
	argv[2] = NULL;

	g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL,
		       NULL, NULL, &error);
	if (error)
		warn ("failed to exec " BIN_MOUNT ": %s\n", error->message);
}

/*
 * gvm_run_cdplay - if so configured, execute the user-specified CD player on
 * the given device node
 */
static void
gvm_run_cdplayer (const char *device, const char *mount_point)
{
	if (config.autoplay_cda)
		gvm_run_command (device, config.autoplay_cda_command,
				 mount_point);
}

#define ASK_MIXED_MSG	"This CD has both audio tracks and data files.\n" \
			"Would you like to play the audio tracks or " \
			"browse the data files?"

/*
 * gvm_ask_mixed - if a mixed mode CD (CD Plus) is inserted, we can either
 * mount the data tracks or play the audio tracks.  How we handle that depends
 * on the user's configuration.
 */
static void
gvm_ask_mixed (const char *udi)
{
	enum { MOUNT, PLAY } action = -1;
	char *device = NULL, *mount_point = NULL;

	device = hal_device_get_property_string (udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	if (config.automount_media && config.autoplay_cda) {
		GtkWidget *askme;

		askme = gtk_message_dialog_new (NULL, 0,
						GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_NONE,
						_(ASK_MIXED_MSG));
		gtk_dialog_add_buttons (GTK_DIALOG (askme),
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL, _("_Play"),
					PLAY, _("_Browse"), MOUNT, NULL);
		action = gtk_dialog_run (GTK_DIALOG (askme));
		gtk_widget_destroy (askme);
	} else if (config.automount_media)
		action = MOUNT;
	else if (config.autoplay_cda)
		action = PLAY;

	switch (action) {
	case MOUNT:
		gvm_device_mount (device);
		break;
	case PLAY:
		gvm_run_cdplayer (device, device);
		break;
	default:
		break;
	}

out:
	hal_free_string (device);
	hal_free_string (mount_point);
}

/*
 * gvm_run_cdburner - execute the user-specified CD burner command on the
 * given device node, if so configured
 */
static void
gvm_run_cdburner (const char *device, const char *mount_point)
{
	if (config.autoburn_cdr)
		gvm_run_command (device, config.autoburn_cdr_command,
				 mount_point);
}

#define BURNER (CDC_CD_R | CDC_CD_RW | CDC_DVD_R | CDC_DVD_RAM)

/*
 * gvm_device_is_writer - is this device capable of writing CDs?
 */
static gboolean
gvm_device_is_writer (const char *device)
{
	int type;
	int fd;

	fd = open (device, O_RDONLY | O_NONBLOCK | O_EXCL);
	if (fd < 0) {
		warn ("unable to open %s\n", device);
		return FALSE;
	}

	type = ioctl (fd, CDROM_GET_CAPABILITY, CDSL_CURRENT);
	close (fd);

	if (type < 0)
		return FALSE;
	else
		return ((type & BURNER) != 0);
}

/*
 * gvm_cdrom_policy - There has been a media change event on the CD-ROM
 * associated with the UDI.  Enforce policy.
 */
static void
gvm_cdrom_policy (const char *udi)
{
	char *device = NULL, *mount_point = NULL;
	int type, fd;

	device = hal_device_get_property_string (udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	fd = open (device, O_RDONLY | O_NONBLOCK | O_EXCL);
	if (fd < 0) {
		warn ("unable to open %s\n", device);
		goto out;
	}

	type = ioctl (fd, CDROM_DISC_STATUS, CDSL_CURRENT);
	close (fd);

	switch (type) {
	case CDS_AUDIO:		/* audio CD */
		gvm_run_cdplayer (device, device);
		break;
	case CDS_MIXED:		/* mixed mode CD (CD Plus) */
		gvm_ask_mixed (udi);
		break;
	case CDS_DATA_1:	/* data CD */
	case CDS_DATA_2:
	case CDS_XA_2_1:
	case CDS_XA_2_2:
		if (config.automount_media)
			gvm_device_mount (device);
		break;
	case CDS_NO_INFO:	/* blank or invalid CD */
		if (gvm_device_is_writer (device))
			gvm_run_cdburner (device, device);
		break;
	default:		/* should never see this */
		break;
	}

out:
	hal_free_string (device);
	hal_free_string (mount_point);
}

/*
 * gvm_cdrom_has_media - ensure that the CD-ROM associated with the given UDI
 * indeed has media.
 *
 * Returns TRUE if there is media in the given device and FALSE otherwise.
 */
static gboolean
gvm_cdrom_has_media (const char *udi)
{
	gboolean has_media = FALSE;
	char *device;
	int fd;

	device = hal_device_get_property_string (udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		return has_media;
	}

	fd = open (device, O_RDONLY | O_NONBLOCK | O_EXCL);
	if (fd < 0) {
		warn ("cannot open %s\n", device);
		goto out;
	}

	if (ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT) == CDS_DISC_OK) {
		int disctype;

		disctype = ioctl (fd, CDROM_DISC_STATUS, CDSL_CURRENT);
		switch (disctype) {
		case CDS_NO_INFO:
		case CDS_AUDIO:
		case CDS_DATA_1:
		case CDS_DATA_2:
		case CDS_XA_2_1:
		case CDS_XA_2_2:
		case CDS_MIXED:
			has_media = TRUE;
			break;
		default:
			/* mo media detected */
			break;
		}
	}

	close (fd);
out:
	hal_free_string (device);

	return has_media;
}

/*
 * gvm_media_changed - generic media change handler.
 *
 * This is called on a UDI in response to a media change event.  We have to
 * decipher the storage media type to run the appropriate media-present check.
 * Then, if there is indeed media in the drive, we enforce the appropriate
 * policy.
 *
 * At the moment, we only handle CD-ROM and DVD drives.
 */
static void
gvm_media_changed (const char *udi)
{
	char *media_type;

	/* get HAL's interpretation of our media type */
	media_type = hal_device_get_property_string (udi, "storage.media");
	if (!media_type) {
		warn ("cannot get storage.media\n");
		return;
	}

	if (!g_strcasecmp (media_type, "cdrom"))
		if (gvm_cdrom_has_media (udi) == TRUE)
			gvm_cdrom_policy (udi);

	/* other media_types go here */

	hal_free_string (media_type);
}

/*
 * handle_device_added - handle HAL's DeviceAdded signal
 *
 * This is where we learn about new device's, so this is where we perform
 * user-configurable policy such as auto-mounting.
 */
static DBusHandlerResult
handle_device_added (DBusMessage *message)
{
	char *udi;
	DBusError error;

	dbus_error_init (&error);

	if (dbus_message_get_args (message, &error,
				   DBUS_TYPE_STRING, &udi,
				   DBUS_TYPE_INVALID)) {
		char *bus, *device = NULL, *storage_device = NULL;

		bus = hal_device_get_property_string (udi, "info.bus");
		if (!bus) {
			warn ("cannot get info.bus\n");
			goto out;
		}

		dbg ("New Device: %s\n", udi);

		/* is this a block device ? */
		if (g_strcasecmp (bus, "block"))
			goto out;

		/* is this a mountable volume ? */
		if (!hal_device_get_property_bool (udi, "block.is_volume"))
			goto out;

		/* if it is a volume, it must have a device node */
		device = hal_device_get_property_string (udi, "block.device");
		if (!device) {
			dbg ("cannot get block.device\n");
			goto out;
		}

		/* get the backing storage device */
		storage_device = hal_device_get_property_string (udi,
				"block.storage_device");
		if (!storage_device) {
			dbg ("cannot get block.storage_device\n");
			goto out;
		}

		/*
		 * Does this device support removable media?  Note that we
		 * check storage_device and not our own UDI
		 */
		if (hal_device_get_property_bool (storage_device,
						  "storage.removable")) {
			/* we handle media change events separately */
			dbg ("Changed: %s\n", device);
			gvm_media_changed (udi);
			goto out;
		}

		/* folks, we have a new device! */
		dbg ("Added: %s\n", device);

		if (config.automount_drives)
			gvm_device_mount (device);

out:
		hal_free_string (bus);
		hal_free_string (device);
		hal_free_string (storage_device);
	} else
		dbg ("dbus_message_get_args failed to parse message\n");

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * handle_device_removed - respond to HAL's DeviceRemoved signal
 *
 * We currently do not do anything special here, except print out debug
 * messages.  In the future, this is where we will learn about new devices.
 */
static DBusHandlerResult
handle_device_removed (DBusMessage *message)
{
	char *udi;
	DBusError error;

	dbus_error_init (&error);

	if (dbus_message_get_args (message, &error,
				   DBUS_TYPE_STRING, &udi,
				   DBUS_TYPE_INVALID))
		dbg ("Device removed: %s\n", udi);
	else
		dbg ("dbus_message_get_args failed to parse message\n");

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * handle_property_modify - respond to HAL's PropertyModified signal
 *
 * We listen to this bad-boy to learn about change in mount status.  We handle
 * auto-run as a response to auto-mount, which is pretty cunning.
 */
static DBusHandlerResult
handle_property_modify (DBusMessage *message)
{
	dbus_int32_t nr;
	char *key;
	dbus_bool_t added, removed;
	DBusMessageIter iter;
	const char *udi;
	int i;

	udi = dbus_message_get_path (message);

	dbus_message_iter_init (message, &iter);
	nr = dbus_message_iter_get_int32 (&iter);
	dbus_message_iter_next (&iter);

	for (i = 0; i < nr; i++) {
		key = dbus_message_iter_get_string (&iter);
		dbus_message_iter_next (&iter);
		removed = dbus_message_iter_get_boolean (&iter);
                dbus_message_iter_next (&iter);
                added = dbus_message_iter_get_boolean (&iter);
                dbus_message_iter_next (&iter);

		if (!g_strcasecmp (key, "block.is_mounted")) {
			dbus_bool_t val;

			val = hal_device_get_property_bool (udi, key);
			if (val == TRUE) {
				dbg ("Mounted: %s\n", udi);
				gvm_device_autorun (udi);
			} else
				dbg ("Unmounted: %s\n", udi);
		}

		dbus_free (key);
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

#define MAN_DBUS_OBJ		"org.freedesktop.Hal.Manager"
#define DEV_DBUS_OBJ		"org.freedesktop.Hal.Device"
#define DBUS_DBUS_OBJ		"org.freedesktop.DBus"
#define DEVICE_ADDED_SIG	"DeviceAdded"
#define DEVICE_REMOVED_SIG	"DeviceRemoved"
#define PROPERTY_MOD_SIG	"PropertyModified"

/*
 * dbus_filter - generic filter handler for the signals we listen on
 */
static DBusHandlerResult
dbus_filter (DBusConnection *connection,
	     DBusMessage *message, gpointer user_data)
{
	(void) connection;
	(void) user_data;

	if (dbus_message_is_signal (message, MAN_DBUS_OBJ, DEVICE_ADDED_SIG))
		return handle_device_added (message);

	if (dbus_message_is_signal (message, MAN_DBUS_OBJ, DEVICE_REMOVED_SIG))
		return handle_device_removed (message);

	if (dbus_message_is_signal (message, DEV_DBUS_OBJ, PROPERTY_MOD_SIG))
		return handle_property_modify (message);

	dbg ("unhandled dbus message: %s from %s\n",
	     dbus_message_get_member (message),
	     dbus_message_get_sender (message));

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * gvm_init_dbus - initialize D-BUS and listen to the appropriate signals
 *
 * Returns TRUE is successful and FALSE if any step fails.  Since we cannot
 * continue without a successful D-BUS setup, a FALSE return value is fatal.
 */
static gboolean
gvm_init_dbus (void)
{
	DBusConnection *bus;
	DBusError error;

	dbus_error_init (&error);

	/* get on the system bus */
	bus = dbus_bus_get_with_g_main (DBUS_BUS_SYSTEM, 0);
	if (!bus) {
		dbg ("cannot connect to system bus: %s\n", error.message);
		dbus_error_free (&error);
		return FALSE;
	}

#define HAL_MANAGER_RULE "type='signal',interface='org.freedesktop.Hal.Manager',path='/org/freedesktop/Hal/Manager'"
	dbus_bus_add_match (bus, HAL_MANAGER_RULE, &error);
	if (dbus_error_is_set (&error)) {
		dbg ("failed to add match: %s\n", error.message);
		dbus_error_free (&error);
		return FALSE;
	}

#define HAL_DEVICE_RULE "type='signal',interface='org.freedesktop.Hal.Device'"
	dbus_bus_add_match (bus, HAL_DEVICE_RULE, &error);
	if (dbus_error_is_set (&error)) {
		dbg ("failed to add match: %s\n", error.message);
		dbus_error_free (&error);
		return FALSE;
	}

#define KERNEL_RULE "type='signal',interface='org.kernel.drivers.cdrom.change',path='/org/kernel/drivers/cdrom/change'"
	dbus_bus_add_match (bus, KERNEL_RULE, &error);
	if (dbus_error_is_set (&error)) {
		dbg ("failed to add match: %s\n", error.message);
		dbus_error_free (&error);
		return FALSE;
	}

	if (!dbus_connection_add_filter (bus, dbus_filter, NULL, NULL)) {
		dbg ("failed to add filter\n");
		return FALSE;
	}

	dbus_connection_flush (bus);

	return TRUE;
}

int
main (int argc, char *argv[])
{
	GnomeClient *client;

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
			    argc, argv, GNOME_PARAM_NONE);


	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

	client = gnome_master_client ();
	if (gvm_get_clipboard ())
		gnome_client_set_restart_style (client, GNOME_RESTART_ANYWAY);
	else {
		gnome_client_set_restart_style (client, GNOME_RESTART_NEVER);
		warn ("already running?\n");
		return 1;
	}

	gtk_signal_connect (GTK_OBJECT (client), "die",
			    GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

	if (gvm_init_dbus () == FALSE) {
		warn ("failed to initialize D-BUS!\n");
		return 1;
	}

	if (hal_initialize (NULL, FALSE)) {
		warn ("failed to initialize HAL!\n");
		return 1;
	}

	gvm_init_config ();

	gtk_main ();

	return 0;
}
