/* gok-output.h
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
 
#include <X11/keysym.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include "gok-output.h"
#include "gok-log.h"
#include "gok-word-complete.h"
#include "main.h"

/**
* gok_output_delete_all
* @pOutput: Pointer to the GokOutput that you want deleted.
*
* Deletes the given GokOutput and other GokOutputs that are in its list.
**/
void gok_output_delete_all (GokOutput* pOutput)
{
	GokOutput* pOutputTemp;
	
	while (pOutput != NULL)
	{
		pOutputTemp = pOutput;
		pOutput = pOutput->pOutputNext;
		g_free (pOutputTemp->Name);
		g_free (pOutputTemp);
	}
}

/**
* gok_output_new
* @Type: Type of output (e.g. keysym or keycode).
* @Name: Keycode or Keysym string.
* @Flag:
*
* Creates a new GokOutput and initializes it to the given values.
*
* returns: A pointer to the new GokOutput, NULL if not created. 
**/
GokOutput* gok_output_new (gint Type, gchar* Name, AccessibleKeySynthType Flag)
{
	GokOutput* pNewOutput;
	
	if (Name == NULL)
	{
		return NULL;
	}

	pNewOutput = (GokOutput*)g_malloc (sizeof (GokOutput));
	pNewOutput->pOutputNext = NULL;
	pNewOutput->Flag = Flag;
	pNewOutput->Type = Type;
	pNewOutput->Name = (gchar*)g_malloc (strlen (Name) + 1);
	strcpy (pNewOutput->Name, Name);

	return pNewOutput;
}

/**
* gok_output_new_from_xml
* @pNode: Pointer to the XML node.
*
* Creates a new GokOutput and initializes it from the given XML node.
*
* returns: A pointer to the new GokOutput, NULL if not created. 
**/
GokOutput* gok_output_new_from_xml (xmlNode* pNode)
{
	xmlChar* pStringAttributeValue;
	xmlChar* pStringFlag;
	gint typeOutput;
	AccessibleKeySynthType flagOutput;

	g_assert (pNode != NULL);
		
	pStringAttributeValue = xmlGetProp (pNode, (const xmlChar *) "type");
	if (pStringAttributeValue == NULL)
	{
		gok_log_x ("Output '%s' has no type!\n", xmlNodeGetContent (pNode));
		return NULL;
	}
	
	if (xmlStrcmp (pStringAttributeValue, (const xmlChar *)"keycode") == 0)
	{
		typeOutput = OUTPUT_KEYCODE;
		
		pStringFlag = xmlGetProp (pNode, (const xmlChar *) "flag");
		if (pStringFlag == NULL)
		{
			gok_log_x ("Output '%s'is keycode but has no flag!\n", xmlNodeGetContent (pNode));
		}
		else
		{
			if (xmlStrcmp (pStringFlag, (const xmlChar *)"press") == 0)
			{
				flagOutput = SPI_KEY_PRESS;
			}
			else if (xmlStrcmp (pStringFlag, (const xmlChar *)"release") == 0)
			{
				flagOutput = SPI_KEY_RELEASE;
			}
			else if (xmlStrcmp (pStringFlag, (const xmlChar *)"pressrelease") == 0)
			{
				flagOutput = SPI_KEY_PRESSRELEASE;
			}
			else
			{
				gok_log_x ("Output flag '%s' is invalid!\n", pStringFlag);
			}
		}
	}
	else if (xmlStrcmp (pStringAttributeValue, (const xmlChar *)"keysym") == 0)
	{
		typeOutput = OUTPUT_KEYSYM;
		flagOutput = SPI_KEY_PRESSRELEASE;
	}
	else if (xmlStrcmp (pStringAttributeValue, (const xmlChar *)"exec")
		== 0)
	{
		typeOutput = OUTPUT_EXEC;
		flagOutput = SPI_KEY_PRESSRELEASE;
	}
	else
	{
		gok_log_x ("Output type '%s' not correct!\n", pStringAttributeValue);
		return NULL;
	}
	
	return gok_output_new (typeOutput, (gchar *) xmlNodeGetContent (pNode), flagOutput);
}

/**
* gok_output_send_to_system
* @pOutput: Pointer to a GokOutput that will be sent to the system.
*
* Sends the given GokOutput to the system. All other GokOutputs that are
* linked to this output are also sent.
**/
void gok_output_send_to_system (GokOutput* pOutput, gboolean bWordCompletion)
{
	long keysym;
	long keycode;

	gok_log_enter();
	while (pOutput != NULL)
	{
		if (pOutput->Type == OUTPUT_KEYSYM)
		{
			keysym = XStringToKeysym (pOutput->Name);

			gok_log ("Sending a keysym");
			gok_log ("pOutput->Name = %s", pOutput->Name);
			gok_log ("SPI_KEY_SYM = %d", (int) SPI_KEY_SYM);

			SPI_generateKeyboardEvent ((long) keysym, NULL, SPI_KEY_SYM);

			/* update the word prediction keys with the new output */
			if (bWordCompletion == TRUE)
			{
				gok_wordcomplete_predict (keysym);
			}
		}
		else if (pOutput->Type == OUTPUT_KEYCODE)
		{
			keycode = strtol (pOutput->Name, NULL, 0);

			gok_log ("Sending a keycode");
			gok_log ("pOutput->Name = %s", pOutput->Name);
			gok_log ("pOutput->Flag = %d", (int) pOutput->Flag);

			SPI_generateKeyboardEvent ((long) keycode, NULL, pOutput->Flag);
			/* update the word prediction keys with the new output */
			if (bWordCompletion == TRUE && 
			    !gok_key_modifier_for_keycode (
				    gok_main_display (),
				    gok_keyboard_get_xkb_desc (),
				    keycode)) 
			{
				gulong mask = gok_spy_get_modmask ();
			        int index = (mask & ShiftMask) ?
					((mask & LockMask) ? 0 : 1) : 
					((mask & LockMask) ? 1 : 0);

				/* TODO: get the correct index from XKB */
			        keysym = XKeycodeToKeysym (gok_main_display (), 
							   keycode, index);
				gok_wordcomplete_predict (keysym);
			}
		}
		else if (pOutput->Type == OUTPUT_EXEC)
		{
			gok_log ("g_spawn_command_line_async(\"%s\")",
				pOutput->Name);

			g_spawn_command_line_async (pOutput->Name, NULL);
		}
		else
		{
			gok_log_x ("Output has invalid type '%d'!\n", pOutput->Type);
		}
		pOutput = pOutput->pOutputNext;
	}
	gok_log_leave();
}
