/* GStreamer Wavpack decoder plugin
 * Copyright (C) 2004 Ronald Bultje <rbultje@ronald.bitfreak.net>
 * Copyright (C) 2005 Tim-Philipp Müller <tim centricular net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gstwavpackreader.h"

#include <gst/gst.h>
#include <string.h>
#include <stdio.h>

/***************************************************************************
 *
 *   gst_wavpack_reader_peek
 *
 ***************************************************************************/

static int
gst_wavpack_reader_peek (void *this, void *ptr, gint size)
{
  GstByteStream *bs = (GstByteStream *) this;
  guint8 *data;
  gint read;

  do {
    read = gst_bytestream_peek_bytes (bs, &data, size);

    if (read != size) {
      GstEvent *event;
      guint32 remaining;

      gst_bytestream_get_status (bs, &remaining, &event);
      if (!event) {
        GST_ELEMENT_ERROR (gst_pad_get_parent (bs->pad),
            RESOURCE, READ, (NULL), (NULL));
        goto done;
      }

      switch (GST_EVENT_TYPE (event)) {
        case GST_EVENT_INTERRUPT:
          gst_event_unref (event);
          goto done;
        case GST_EVENT_EOS:
          gst_event_unref (event);
          goto done;
        case GST_EVENT_FLUSH:
          gst_event_unref (event);
          break;
        case GST_EVENT_DISCONTINUOUS:
          gst_event_unref (event);
          break;
        default:
          gst_pad_event_default (bs->pad, event);
          break;
      }
    }
  } while (read != size);

done:
  if (read != 0) {
    memcpy (ptr, data, read);
  }

  return read;
}

/***************************************************************************
 *
 *   gst_wavpack_reader_read_bytes
 *
 ***************************************************************************/

static long
gst_wavpack_reader_read_bytes (void *id, void *data, long bcount)
{
  GstByteStream *bs = (GstByteStream *) id;
  gulong read;

  /* read = peek + flush */
  read = gst_wavpack_reader_peek (bs, data, bcount);
  if (read > 0) {
    gst_bytestream_flush_fast (bs, read);
  }

  return (long) read;
}

/***************************************************************************
 *
 *   gst_wavpack_reader_get_pos
 *
 ***************************************************************************/

static ulong
gst_wavpack_reader_get_pos (void *id)
{
  return gst_bytestream_tell ((GstByteStream *) id);
}


/***************************************************************************
 *
 *   gst_wavpack_reader_set_pos_rel
 *
 ***************************************************************************/
static int
gst_wavpack_reader_set_pos_rel (void *id, long pos, int mode)
{
  GstByteStream *bs = (GstByteStream *) id;
  guint8 dummy;
  int whence;

  /* hacky hack - if we're after typefind, we'll fail because
   * typefind is still typefinding (heh :) ). So read first. */
  gst_wavpack_reader_peek (bs, &dummy, 1);

  switch (mode) {
    case SEEK_SET:
      whence = GST_SEEK_METHOD_SET;
      break;
    case SEEK_END:
      whence = GST_SEEK_METHOD_END;
      break;
    case SEEK_CUR:
      whence = GST_SEEK_METHOD_CUR;
      break;
    default:
      return -1;
  }

  /* seek */
  if (!gst_bytestream_seek (bs, pos, whence))
    return -1;

  /* get discont */
  if (gst_wavpack_reader_peek (bs, &dummy, 1) != 1)
    return -1;

  return 0;                     /* just like fseek */
}


/***************************************************************************
 *
 *   gst_wavpack_reader_set_pos_abs
 *
 ***************************************************************************/
static int
gst_wavpack_reader_set_pos_abs (void *id, ulong pos)
{
  return gst_wavpack_reader_set_pos_rel (id, (long) pos, SEEK_SET);
}

/***************************************************************************
 *
 *   gst_wavpack_reader_push_back_byte
 *
 ***************************************************************************/

static int
gst_wavpack_reader_push_back_byte (void *id, int c)
{
  /* only required if we want to be able to decode old v3 
   * wavpack files as well, which we do not at the moment */
#ifdef VER3
# error need to implement gst_wavpack_reader_push_back_byte
#endif
  g_return_val_if_reached (-1);
}


/***************************************************************************
 *
 *   gst_wavpack_reader_get_length
 *
 ***************************************************************************/
static ulong
gst_wavpack_reader_get_length (void *id)
{
  return gst_bytestream_length ((GstByteStream *) id);
}


/***************************************************************************
 *
 *   gst_wavpack_reader_can_seek
 *
 ***************************************************************************/
static int
gst_wavpack_reader_can_seek (void *id)
{
  return TRUE;
}


/***************************************************************************
 *
 *   gst_wavpack_reader_init
 *
 ***************************************************************************/
void
gst_wavpack_reader_init (stream_reader * r)
{
  r->read_bytes = gst_wavpack_reader_read_bytes;
  r->get_pos = gst_wavpack_reader_get_pos;
  r->set_pos_abs = gst_wavpack_reader_set_pos_abs;
  r->set_pos_rel = gst_wavpack_reader_set_pos_rel;
  r->push_back_byte = gst_wavpack_reader_push_back_byte;
  r->get_length = gst_wavpack_reader_get_length;
  r->can_seek = gst_wavpack_reader_can_seek;
}
