/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <stdio.h>
#include <string.h>
#include "ccd-node.h"
#include "ccd-selector.h"
#include "ccd-style.h"
#include "ccd-stylesheet.h"

typedef enum {
	CCD_SELECTOR_MODALITY_UNIVERSAL,	/* Universal selector. */
	CCD_SELECTOR_MODALITY_TYPE,		/* By element type. */
	CCD_SELECTOR_MODALITY_BASE_TYPE,	/* By element type. */
	CCD_SELECTOR_MODALITY_CLASS,		/* By element class. */
	CCD_SELECTOR_MODALITY_ID,		/* By element ID. */
	CCD_SELECTOR_MODALITY_ATTRIBUTE,	/* By element attribute. */
	CCD_SELECTOR_MODALITY_PSEUDO_CLASS	/* By pseudo class. */
} ccd_selector_modality_t;

/*
 * Abstract base selector.
 * See http://www.w3.org/TR/CSS21/cascade.html#specificity for an explantion 
 * of the meaning of the fields `a', `b', `c' and `d'. Field `e' (lowest priority)
 * is used for merging base styles. Field `a' is not used currently.
 */
struct ccd_selector_ {
	ccd_selector_modality_t	 modality;
	uint8_t			 a;
	uint8_t			 b;
	uint8_t			 c;
	uint8_t			 d;
	uint8_t			 e;
	struct ccd_selector_	*refinement;
	struct ccd_selector_	*container;
	struct ccd_selector_	*antecessor;
	ccd_block_t const	*block;
};

static void
selector_sync (ccd_selector_t const	*self,
	       ccd_selector_t		*to)
{
	g_assert (self && to);

	to->modality = self->modality;
	to->a = self->a;
	to->b = self->b;
	to->c = self->c;
	to->d = self->d;
	to->e = self->e;
	to->refinement = NULL;
	to->container = NULL;
	to->antecessor = NULL;
	to->block = self->block;
}

/*
 * Universal selector.
 */
typedef ccd_selector_t ccd_universal_selector_t;

ccd_selector_t * 
ccd_universal_selector_new (void)
{
	ccd_universal_selector_t *self;

	self = g_new0 (ccd_universal_selector_t, 1);
	self->modality = CCD_SELECTOR_MODALITY_UNIVERSAL;

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
universal_selector_dup (ccd_universal_selector_t const *original)
{
	ccd_universal_selector_t *self;

	self = g_new0 (ccd_universal_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, self);

	return (ccd_selector_t *) self;
}

static void
universal_selector_free (ccd_universal_selector_t *self)
{
	g_assert (self);

	g_free (self);
}

static void
universal_selector_dump (ccd_universal_selector_t const *self)
{
	printf (" *");
}

/*
 * Select by element type.
 */
typedef struct {
	ccd_selector_t	 parent;
	char		*type_name;
} ccd_type_selector_t;

ccd_selector_t *
ccd_type_selector_new (char const *type_name)
{
	ccd_type_selector_t *self;

	g_assert (type_name);

	self = g_new0 (ccd_type_selector_t, 1);
	self->parent.modality = CCD_SELECTOR_MODALITY_TYPE;
	self->parent.d = 1;
	self->type_name = g_strdup (type_name);

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
type_selector_dup (ccd_type_selector_t const *original)
{
	ccd_type_selector_t *self;

	self = g_new0 (ccd_type_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, &self->parent);
	self->type_name = g_strdup (original->type_name);

	return (ccd_selector_t *) self;
}

static void
type_selector_free (ccd_type_selector_t *self)
{
	g_assert (self);

	g_free (self->type_name);
	g_free (self);
}

static void
type_selector_dump (ccd_type_selector_t const *self)
{
	printf (" %s", self->type_name);
}

/*
 * Select by element's base type.
 * Derived from the type selector
 */
ccd_selector_t * 
ccd_base_type_selector_new (char const		*type_name,
			    unsigned int	 specificity_e)
{
	ccd_selector_t *self;

	self = ccd_type_selector_new (type_name);
	self->modality = CCD_SELECTOR_MODALITY_BASE_TYPE;
	self->a = 0;
	self->b = 0;
	self->c = 0;
	self->d = 0;
	self->e = specificity_e;

	return self;
}

/*
 * Select by element class.
 */
typedef struct {
	ccd_selector_t	 parent;
	char		*class_name;
} ccd_class_selector_t;

ccd_selector_t *
ccd_class_selector_new (char const *class_name)
{
	ccd_class_selector_t *self;

	g_assert (class_name);

	self = g_new0 (ccd_class_selector_t, 1);
	self->parent.modality = CCD_SELECTOR_MODALITY_CLASS;
	self->parent.c = 1;
	self->class_name = g_strdup (class_name);

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
class_selector_dup (ccd_class_selector_t const *original)
{
	ccd_class_selector_t *self;

	self = g_new0 (ccd_class_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, &self->parent);
	self->class_name = g_strdup (original->class_name);

	return (ccd_selector_t *) self;
}

static void
class_selector_free (ccd_class_selector_t *self)
{
	g_assert (self);

	g_free (self->class_name);
	g_free (self);
}

static void
class_selector_dump (ccd_class_selector_t const *self)
{
	printf (".%s", self->class_name);
}

/*
 * Select by element ID.
 */
typedef struct {
	ccd_selector_t	 parent;
	char		*id;
} ccd_id_selector_t;

ccd_selector_t *
ccd_id_selector_new (char const *id)
{
	ccd_id_selector_t *self;

	g_assert (id);

	self = g_new0 (ccd_id_selector_t, 1);
	self->parent.modality = CCD_SELECTOR_MODALITY_ID;
	self->parent.b = 1;
	self->id = g_strdup (id);

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
id_selector_dup (ccd_id_selector_t const *original)
{
	ccd_id_selector_t *self;

	self = g_new0 (ccd_id_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, &self->parent);
	self->id = g_strdup (original->id);

	return (ccd_selector_t *) self;
}

static void
id_selector_free (ccd_id_selector_t *self)
{
	g_assert (self);

	g_free (self->id);
	g_free (self);
}

static void
id_selector_dump (ccd_id_selector_t const *self)
{
	printf ("#%s", self->id);
}

/*
 * Select by attribute.
 */
typedef struct {
	ccd_selector_t			 parent;
	char				*name;
	char				*value;
	ccd_attribute_selector_match_t	 match;
} ccd_attribute_selector_t;

ccd_selector_t *
ccd_attribute_selector_new (char const				*name,
			    char const				*value,
			    ccd_attribute_selector_match_t	 match)
{
	ccd_attribute_selector_t *self;

	g_assert (name && value);

	self = g_new0 (ccd_attribute_selector_t, 1);
	self->parent.modality = CCD_SELECTOR_MODALITY_ATTRIBUTE;
	self->name = g_strdup (name);
	self->value = g_strdup (value);
	self->match = match;

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
attribute_selector_dup (ccd_attribute_selector_t const *original)
{
	ccd_attribute_selector_t *self;

	self = g_new0 (ccd_attribute_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, &self->parent);
	self->name = g_strdup (original->name);
	self->value = g_strdup (original->value);
	self->match = original->match;

	return (ccd_selector_t *) self;
}

static void
attribute_selector_free (ccd_attribute_selector_t *self)
{
	g_assert (self);

	g_free (self->name);
	g_free (self->value);
	g_free (self);
}

static void
attribute_selector_dump (ccd_attribute_selector_t const *self)
{
	switch (self->match) {
	case CCD_ATTRIBUTE_SELECTOR_MATCH_EXISTS:
		printf ("[%s]", self->name);
		break;
	case CCD_ATTRIBUTE_SELECTOR_MATCH_EQUALS:
		printf ("[%s=%s]", self->name, self->value);
		break;
	default:
		g_assert_not_reached ();
	}
}

/*
 * Select by pseudo class.
 */
typedef struct {
	ccd_selector_t	 parent;
	char		*pseudo_class;
} ccd_pseudo_class_selector_t;

ccd_selector_t *
ccd_pseudo_class_selector_new (char const *pseudo_class)
{
	ccd_pseudo_class_selector_t *self;

	g_assert (pseudo_class);

	self = g_new0 (ccd_pseudo_class_selector_t, 1);
	self->parent.modality = CCD_SELECTOR_MODALITY_PSEUDO_CLASS;
	self->pseudo_class = g_strdup (pseudo_class);

	return (ccd_selector_t *) self;
}

static ccd_selector_t *
pseudo_class_selector_dup (ccd_pseudo_class_selector_t const *original)
{
	ccd_pseudo_class_selector_t *self;

	self = g_new0 (ccd_pseudo_class_selector_t, 1);
	selector_sync ((ccd_selector_t const *) original, &self->parent);
	self->pseudo_class = g_strdup (original->pseudo_class);

	return (ccd_selector_t *) self;
}

static void
pseudo_class_selector_free (ccd_pseudo_class_selector_t *self)
{
	g_assert (self);

	g_free (self->pseudo_class);
	g_free (self);
}

static void
pseudo_class_selector_dump (ccd_pseudo_class_selector_t const *self)
{
	printf (":%s", self->pseudo_class);
}

ccd_selector_t *
ccd_selector_copy (ccd_selector_t const *original)
{
	ccd_selector_t *self;

	g_assert (original);

	switch (original->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
		self = universal_selector_dup ((ccd_universal_selector_t const *) original);
		break;
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
		self = type_selector_dup ((ccd_type_selector_t const *) original);
		break;
	case CCD_SELECTOR_MODALITY_CLASS:
		self = class_selector_dup ((ccd_class_selector_t const *) original);
		break;
	case CCD_SELECTOR_MODALITY_ID:
		self = id_selector_dup ((ccd_id_selector_t const *) original);
		break;
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
		self = attribute_selector_dup ((ccd_attribute_selector_t const *) original);
		break;
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
		self = pseudo_class_selector_dup ((ccd_pseudo_class_selector_t const *) original);
		break;
	default:
		g_warning ("Unknown selector modality %d", original->modality);
		self = NULL;
		return self;
	}

	if (original->refinement) {
		self->refinement = ccd_selector_copy (original->refinement);
	}

	if (original->container) {
		self->container = ccd_selector_copy (original->container);
	}

	if (original->antecessor) {
		self->antecessor = ccd_selector_copy (original->antecessor);
	}

	return self;
}

/*
 * Create a selector based on an existing one.
 */
ccd_selector_t *
ccd_selector_copy_as_base (ccd_selector_t const	*original,
			   int			 specificity_e)
{
	ccd_selector_t *self;

	g_assert (original && original->modality == CCD_SELECTOR_MODALITY_TYPE);

	self = ccd_selector_copy (original);
	self->modality = CCD_SELECTOR_MODALITY_BASE_TYPE;

	if (specificity_e > -1) {
		if (original->d > 0) {
			self->d--;
		} else {
			g_warning ("Specificity d == 0");
		}

		if (original->e == 0) {
#if DEBUG
			if (specificity_e > CCD_SELECTOR_MAX_SPECIFICITY) {
				//g_assert_not_reached ();
				printf ("foo\n");
			}
#endif
			g_assert (specificity_e <= CCD_SELECTOR_MAX_SPECIFICITY);
			self->e = (unsigned) specificity_e;
		} else {
			g_warning ("Specificity e != 0");
		}
	}

	return self;
}

/*
 * Free the whole selector chain.
 */
void
ccd_selector_free (ccd_selector_t *self)
{
	g_assert (self);

	if (self->refinement) {
		ccd_selector_free (self->refinement), self->refinement = NULL;
	}

	if (self->container) {
		ccd_selector_free (self->container), self->container = NULL;
	}

	if (self->antecessor) {
		ccd_selector_free (self->antecessor), self->antecessor = NULL;
	}

	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
		universal_selector_free ((ccd_universal_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
		type_selector_free ((ccd_type_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_CLASS:
		class_selector_free ((ccd_class_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_ID:
		id_selector_free ((ccd_id_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
		attribute_selector_free ((ccd_attribute_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
		pseudo_class_selector_free ((ccd_pseudo_class_selector_t *) self);
		break;
	default:
		g_warning ("Unknown selector modality %d", self->modality);
	}
}

/*
 * Does it matter that the refinements order is reversed?
 */
void
ccd_selector_refine (ccd_selector_t *self,
		     ccd_selector_t *selector)
{
	g_assert (self && self->refinement == NULL && selector);

	self->refinement = selector;

	/* propagate specificity */
	self->a += selector->a;
	self->b += selector->b;
	self->c += selector->c;
	self->d += selector->d;
	/* `e' is not relevant for refinements.
	 * self->e += selector->e; */
}

/*
 * Add a selector to the chain.
 * This can change the selector instance, make sure to regard the return value.
 */
ccd_selector_t *
ccd_selector_append_child (ccd_selector_t *self,
			   ccd_selector_t *selector)
{
	/* propagate specificity */
	selector->a += self->a;
	selector->b += self->b;
	selector->c += self->c;
	selector->d += self->d;
	/* TODO: invert the meaing of `e' and only apply
	 * "MAX - e" when calculating specificity.
	 * Will this let us propagate correctly?
	 * selector->e += self->e; */

	selector->container = self;

	return selector;
}

/*
 * Add a selector to the chain.
 * This can change the selector instance, make sure to regard the return value.
 */
ccd_selector_t *
ccd_selector_append_descendant (ccd_selector_t *self,
				ccd_selector_t *selector)
{
	/* propagate specificity */
	selector->a += self->a;
	selector->b += self->b;
	selector->c += self->c;
	selector->d += self->d;
	/* TODO: invert the meaing of `e' and only apply
	 * "MAX - e" when calculating specificity.
	 * Will this let us propagate correctly?
	 * selector->e += self->e; */

	selector->antecessor = self;

	return selector;
}

bool
ccd_selector_is_type (ccd_selector_t const *self)
{
	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
	case CCD_SELECTOR_MODALITY_TYPE:
		return true;
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
	case CCD_SELECTOR_MODALITY_CLASS:
	case CCD_SELECTOR_MODALITY_ID:
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
	default:
		return false;
	}
}

bool
ccd_selector_is_class (ccd_selector_t const *self)
{
	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_CLASS:
		return true;		
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
	case CCD_SELECTOR_MODALITY_ID:
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
	default:
		return false;
	}
}

bool
ccd_selector_is_id (ccd_selector_t const *self)
{
	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_ID:
		return true;		
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
	case CCD_SELECTOR_MODALITY_CLASS:
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
	default:
		return false;
	}
}

ccd_block_t const *
ccd_selector_get_block (ccd_selector_t const *self)
{
	g_assert (self);

	return self->block;
}

void
ccd_selector_set_block (ccd_selector_t		*self,
			ccd_block_t const	*block)
{
	g_assert (self);

	self->block = block;
}

/*
 * Depending on the modality of the selector this may return NULL.
 */
char const *
ccd_selector_get_key (ccd_selector_t const *self)
{
	g_return_val_if_fail (self, NULL);

	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
		return "*";
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
		return ((ccd_type_selector_t *) self)->type_name;
	case CCD_SELECTOR_MODALITY_CLASS:
		return ((ccd_class_selector_t *) self)->class_name;
	case CCD_SELECTOR_MODALITY_ID:
		return ((ccd_id_selector_t *) self)->id;
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
	default:
		return NULL;
	}
}

uint32_t
ccd_selector_get_specificity (ccd_selector_t const *self)
{
	g_assert (self->a <= CCD_SELECTOR_MAX_SPECIFICITY &&
		  self->b <= CCD_SELECTOR_MAX_SPECIFICITY &&
		  self->c <= CCD_SELECTOR_MAX_SPECIFICITY &&
		  self->d <= CCD_SELECTOR_MAX_SPECIFICITY &&
		  self->e <= CCD_SELECTOR_MAX_SPECIFICITY);

	return self->e | (self->d << 6) | (self->c << 12) | (self->b << 18) | (self->a << 24);
}

void
ccd_selector_get_specificity_values (ccd_selector_t const	*self, 
				     unsigned int		*a,
				     unsigned int		*b,
				     unsigned int		*c,
				     unsigned int		*d,
				     unsigned int		*e)
{
	g_assert (self);

	if (a)
		*a = self->a;

	if (b)
		*b = self->b;

	if (c)
		*c = self->c;

	if (d)
		*d = self->d;

	if (e)
		*e = self->e;
}

static bool
match_antecessor_r (ccd_selector_t const	*self, 
		    ccd_node_t const		*node)
{
	ccd_node_class_t const	*node_class;
	ccd_node_t		*container;
	bool			 is_matching;

	node_class = ccd_node_get_class ();

	container = node_class->get_container (node);
	if (container) {
		is_matching = ccd_selector_query_apply (self, container, NULL);
		if (!is_matching) {
			is_matching = match_antecessor_r (self, container);
		}
		node_class->release (container);
	} else {
		is_matching = false;
	}

	return is_matching;
}

/*
 * `style' may be NULL, in which case this just tests for a match.
 */
bool
ccd_selector_query_apply (ccd_selector_t const	*self, 
			  ccd_node_t const	*node,
			  ccd_style_t		*style)
{
	ccd_node_class_t const	*node_class;
	char const	*name;
	char		*value;
	bool		 is_matching;

	g_return_val_if_fail (self && node, false);

	node_class = ccd_node_get_class ();

	is_matching = false;
	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
		is_matching = true;
		break;
	case CCD_SELECTOR_MODALITY_TYPE:
		is_matching = node_class->is_a (node, ((ccd_type_selector_t *) self)->type_name);
		break;
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
		/* HACK warning: let's just say it matches, because the base type selectors have
		 * been set up internally -- that is in the fixup run after loading the stylesheet. */
		is_matching = true;
		break;
	case CCD_SELECTOR_MODALITY_CLASS:
		name = node_class->get_class (node);
		is_matching = name ? 0 == strcmp (name, ((ccd_class_selector_t *) self)->class_name) : false;
		break;
	case CCD_SELECTOR_MODALITY_ID:
		name = node_class->get_id (node);
		is_matching = name ? 0 == strcmp (name, ((ccd_id_selector_t *) self)->id) : false;
		break;
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
		name = ((ccd_attribute_selector_t *) self)->name;
		value = node_class->get_attribute (node, name);
		switch (((ccd_attribute_selector_t *) self)->match) {
		case CCD_ATTRIBUTE_SELECTOR_MATCH_EXISTS:
			is_matching = value ? true : false;
			break;
		case CCD_ATTRIBUTE_SELECTOR_MATCH_EQUALS:
			is_matching = value ? 0 == strcmp (value, ((ccd_attribute_selector_t *) self)->value) : false;			
			break;
		default:
			g_assert_not_reached ();
			is_matching = false;
		}
		g_free (value), value = NULL;
		break;
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
		name = node_class->get_pseudo_class (node);
		is_matching = name ? 0 == strcmp (name, ((ccd_pseudo_class_selector_t *) self)->pseudo_class) : false;
		break;
	default:
		g_assert_not_reached ();
	}

	if (!is_matching) {
		return false;
	}

	/* recursively match refinements */
	if (self->refinement) {
		is_matching = ccd_selector_query_apply (self->refinement, node, style);
		if (!is_matching) {
			return false;
		}
	}

	/* match container */
	if (self->container) {
		ccd_node_t *container;
		container = node_class->get_container (node);
		is_matching = false;
		if (container) {
			is_matching = ccd_selector_query_apply (self->container, container, style);
			node_class->release (container);
		}
		if (!is_matching) {
			return false;
		}
	}

	/* recursively match antecessor */
	if (self->antecessor) {
		is_matching = match_antecessor_r (self->antecessor, node);
		if (!is_matching) {
			return false;
		}
	}

	/* merge */
	if (self->block && style) {
		ccd_selector_apply (self, style);
	}

	return true;
}

bool
ccd_selector_apply (ccd_selector_t const	*self,
		    ccd_style_t			*style)
{
	g_return_val_if_fail (self && self->block && style, false);

	/* apply */

	switch (self->block->background.bg_color.spec) {
	case CCD_PROPERTY_SPEC_UNSET:
		/* do nothing */
		break;
	case CCD_PROPERTY_SPEC_NONE:
		/* reset */
		style->bg_color = NULL;
		break;
	case CCD_PROPERTY_SPEC_INHERIT:
		/* not implemented */
		g_assert_not_reached ();
		break;
	case CCD_PROPERTY_SPEC_SET:
		/* use */
		style->bg_color = &self->block->background.bg_color;
		break;
	}

	switch (self->block->background.bg_image.spec) {
	case CCD_PROPERTY_SPEC_UNSET:
		/* do nothing */
		break;
	case CCD_PROPERTY_SPEC_NONE:
		/* reset */
		style->bg_image = NULL;
		break;
	case CCD_PROPERTY_SPEC_INHERIT:
		/* not implemented */
		g_assert_not_reached ();
		break;
	case CCD_PROPERTY_SPEC_SET:
		/* use */
		style->bg_image = &self->block->background.bg_image;
		break;
	}

	if (self->block->border.left.flags)
		CCD_BORDER_STROKE_ASSIGN (style->left, self->block->border.left);
	if (self->block->border.top.flags)
		CCD_BORDER_STROKE_ASSIGN (style->top, self->block->border.top);
	if (self->block->border.right.flags)
		CCD_BORDER_STROKE_ASSIGN (style->right, self->block->border.right);
	if (self->block->border.bottom.flags)
		CCD_BORDER_STROKE_ASSIGN (style->bottom, self->block->border.bottom);

	switch (self->block->color_spec) {
	case CCD_PROPERTY_SPEC_UNSET:
		/* do nothing */
		break;
	case CCD_PROPERTY_SPEC_NONE:
		/* reset */
		style->color = NULL;
		style->color_spec = self->block->color_spec;
		break;
	case CCD_PROPERTY_SPEC_INHERIT:
		/* not implemented */
		g_assert_not_reached ();
		break;
	case CCD_PROPERTY_SPEC_SET:
		/* use */
		style->color = &self->block->color;
		style->color_spec = self->block->color_spec;
		break;
	}

	/* fixup */

	if (style->color) {
		if (CCD_PROPERTY_SPEC_UNSET == style->left.color_spec) {
			CCD_COLOR_ASSIGN (style->left.color, *style->color);
			style->left.color_spec = style->color_spec;
		}
		if (CCD_PROPERTY_SPEC_UNSET == style->top.color_spec) {
			CCD_COLOR_ASSIGN (style->top.color, *style->color);
			style->top.color_spec = style->color_spec;
		}
		if (CCD_PROPERTY_SPEC_UNSET == style->right.color_spec) {
			CCD_COLOR_ASSIGN (style->right.color, *style->color);
			style->right.color_spec = style->color_spec;
		}
		if (CCD_PROPERTY_SPEC_UNSET == style->bottom.color_spec) {
			CCD_COLOR_ASSIGN (style->bottom.color, *style->color);
			style->bottom.color_spec = style->color_spec;
		}
	}

	return true;
}

void 
ccd_selector_dump (ccd_selector_t const *self)
{
	g_assert (self);

	printf ("%p: ", (void *) self);

	if (self->container) {
		ccd_selector_t const *container;
		printf ("( ");
		container = self->container;
		while (container) {
			printf("%s < ", ccd_selector_get_key (container));
			container = container->container;
		}
		printf (")");
	}

	if (self->antecessor) {
		ccd_selector_t const *container;
		printf ("( ");
		container = self->antecessor;
		while (container) {
			printf("%s", ccd_selector_get_key (container));
			container = container->antecessor;
		}
		printf (")");
	}

	switch (self->modality) {
	case CCD_SELECTOR_MODALITY_UNIVERSAL:
		universal_selector_dump ((ccd_universal_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_TYPE:
	case CCD_SELECTOR_MODALITY_BASE_TYPE:
		type_selector_dump ((ccd_type_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_CLASS:
		class_selector_dump ((ccd_class_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_ID:
		id_selector_dump ((ccd_id_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_ATTRIBUTE:
		attribute_selector_dump ((ccd_attribute_selector_t *) self);
		break;
	case CCD_SELECTOR_MODALITY_PSEUDO_CLASS:
		pseudo_class_selector_dump ((ccd_pseudo_class_selector_t *) self);
		break;
	default:
		g_warning ("Unknown selector modality %d", self->modality);
	}

	if (self->refinement) {
		ccd_selector_dump (self->refinement);
	}

	if (self->block) {
		printf (" {\n");
		ccd_block_dump (self->block);
		printf ("}");
	}

	printf (" # modality: %d, specificity: %d,%d,%d,%d,%d\n", 
		self->modality, self->a, self->b, self->c, self->d, self->e);
}

